/*
 * $Id: DirEntry.java,v 1.5 2004/04/07 10:22:28 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: DirEntry.java,v $
 * Revision 1.5  2004/04/07 10:22:28  wrossi
 * Added documentation
 *
 * Revision 1.4  2003/11/20 11:55:45  wrossi
 * Removed printouts
 *
 * Revision 1.3  2003/11/10 12:59:27  wrossi
 * Can now read directory items
 *
 * Revision 1.2  2003/11/08 02:05:03  wrossi
 * Working on directory listing
 *
 * Revision 1.1  2003/11/06 23:35:01  wrossi
 * Initial revision
 *
 * Revision 1.1  2003/11/01 21:46:59  wrossi
 * Initial checkin
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a Directory Entry.<p>
 *
 * An item is an abstract data holder within a block.<p>
 *
 * A directory has an array of directory entry items at the beginning, the names corresponding to
 * those entries are packed in at the end of the item. <p>
 *
 * These directory entries are at the beginning of the directory item.  For convienece when
 * these are loaded from a directory item, the name is associated with the entry is also parsed.<p>
 * 
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Offset (hash code)          |
 *        --------------------------------------
 *  0004  |        Parent Dir Id               |
 *        --------------------------------------
 *  0008  |        Object Id                   |
 *        --------------------------------------
 *  000C  |  Location        |     State       |
 *        --------------------------------------
 * </pre>
 */

public class DirEntry 
{
  protected long hash;
  protected long parentDirId;
  protected long objectId;
  protected int location;
  protected int state;
  protected String name;

  private int dirhsize = 0x10;  // Size of a directory entry header 

  public DirEntry()
  { 
  }

  /** 
   * Get the hash code for the directory name.  ReiserFS uses this for indexed
   * directory lookups. 
   * @return the hash
   */
  public long getHash() { return hash; }
  public void setHash(long aHash) { hash=aHash; }

  /**
   * Get the identifier of this object's parent directory.  Together with
   * the object ID, this forms the key into the tree to find the object.
   * @return parent's ID
   */
  public long getParentDirId() { return parentDirId; }
  public void setParentDirId(long aParentDirId) { parentDirId=aParentDirId; }

  /**
   * Get the identifier of this object.  Together with the parent directory
   * ID, this forms the key into the tree to find the object.
   * @return long
   */
  public long getObjectId() { return objectId; }
  public void setObjectId(long aObjectId) { objectId=aObjectId; }

  /** 
   * Location of this directory entry within the directory item.
   * @return location 
   */
  public int getLocation() { return location; }
  public void setLocation(int aLocation) { location=aLocation; }

  public int getState() { return state; }
  public void setState(int aState) { state=aState; }

  /**
   * Name of the item in the directory.  The filename.
   * @return the filename
   */
  public String getName() { return name; }
  public void setName(String aName) { name=aName; }

  /** Loads a directory entry. 
   *  @param buffer Buffer holding our item.  
   *  @param itemoffset Item offset is an offset within buffer to the start of the item data.  
   *  @param entrynum is the ordinal of the direntry to load.
   *  @param end offset where the directory name must end
   */
  public void loadFromBuffer(byte[] buffer, int itemoffset, int entrynum, int end) throws FsException
  {
    setHash(FsUtils.getU32(buffer, itemoffset+entrynum*dirhsize+0x0));
    setParentDirId(FsUtils.getU32(buffer, itemoffset+entrynum*dirhsize+0x4));
    setObjectId(FsUtils.getU32(buffer, itemoffset+entrynum*dirhsize+0x8));
    setLocation(FsUtils.getU16(buffer, itemoffset+entrynum*dirhsize+0xC));
    setState(FsUtils.getU16(buffer, itemoffset+entrynum*dirhsize+0xE));

    //System.out.println("Hash = "+hash);
    //System.out.println("ParentDirId = "+parentDirId);
    //System.out.println("Object Id = "+objectId);
    //System.out.println("Location = "+location);

    /* Get the name */
    try
    {
      name = new String(buffer, itemoffset+getLocation(), end-getLocation(), "ISO-8859-1");
      int zeroidx = name.indexOf('\u0000');
      if (zeroidx > 0)
        name = name.substring(0, zeroidx);
      //System.out.println("Name = "+name+"  length = "+name.length());
    }
    catch (Exception ex)
    {
      FsException fsex = new FsException("Error parsing filename");
      fsex.initCause(ex);
      throw fsex;
    }
  }
}
