/*
 * $Id: InternalNode.java,v 1.6 2004/04/07 19:47:40 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: InternalNode.java,v $
 * Revision 1.6  2004/04/07 19:47:40  wrossi
 * Adding javadoc
 *
 * Revision 1.5  2003/11/13 21:17:03  wrossi
 * Fixed error in tree traversal
 *
 * Revision 1.4  2003/11/06 23:34:39  wrossi
 * Fixed sort order
 *
 * Revision 1.3  2003/11/01 21:45:43  wrossi
 * Added reference to super block
 *
 * Revision 1.2  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.1  2003/10/30 12:46:10  wrossi
 * B+Tree internal node
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of an internal node of the B+Tree.
 *
 * Following the block header (at offset 0x18), is an array of N Keys, followed by
 * an array of (N+1) DiskChild pointers.  The remainder of the block is free space.
 */

public class InternalNode extends FBlock
{
  protected Key keys[];
  protected DiskChild ptrs[];
  protected byte[] freeSpaceData;   // free space in the block

  /**
   * Create an internal node. 
   *
   * @param sb   ReiserSuperBlock - superblock of this filesystem
   * @param blocknum   block number of this block
   */
  public InternalNode(ReiserSuperBlock sb, int blocknum)
  {
    super(sb, blocknum);
  }

  /**
   * Get the keys contained in this internal node.
   * @return array of keys
   */
  public Key[] getKeys() { return keys; }
  public void setKeys(Key aKeys[]) { keys=aKeys; }

  /**
   * Get the disk child pointers in this node.
   * @return array of DiskChild pointers
   */
  public DiskChild[] getPtrs() { return ptrs; }
  public void setPtrs(DiskChild aPtrs[]) { ptrs=aPtrs; }

  /**
   * Get the free space in this node.  
   * @return buffer of free space.
   */
  public byte[] getFreeSpaceData() { return freeSpaceData; }
  public void setFreeSpaceData(byte[] aFreeSpace) { freeSpaceData=aFreeSpace; }

  /** 
   * Convieince method for searching the tree.  
   * @return the DiskChild pointer for the given key.
   */
  public DiskChild getPointerForKey(Key k)
  {
    for (int i=0; i<keys.length; i++)
    {
      if (k.compareTo(keys[i]) > 0)
        return ptrs[i];
    }

    return ptrs[keys.length];
  }

  /**
   * @see DiskStructure#loadFromBuffer(byte[], int)
   */
  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    super.loadFromBuffer(buffer, offset);

    keys = new Key[getNumItems()];
    ptrs = new DiskChild[getNumItems() + 1];

    int keysize = new KeyV1().getDataSize();
    int ptrsize = new DiskChild().getDataSize();
    int ptroffset = offset+0x18+(keysize*getNumItems());
    int keyoffset = offset+0x18;

    for (int i=0; i<getNumItems(); i++)
      keys[i] = Key.createKeyFromBuffer(buffer, keyoffset+(i*keysize));

    for (int i=0; i<=getNumItems(); i++)
    {
      DiskChild child = new DiskChild();
      child.loadFromBuffer(buffer, ptroffset+i*ptrsize);
      ptrs[i] = child;
    }

    /* Get the freespace */
    int endoffset = ptroffset+(getNumItems()+1)*ptrsize;
    int freesize = getDataSize() - endoffset;
    freeSpaceData = new byte[freesize];
    System.arraycopy(buffer, endoffset, freeSpaceData, 0, freesize);
  }
}
