/*
 * $Id: Key.java,v 1.9 2004/04/08 12:33:42 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Key.java,v $
 * Revision 1.9  2004/04/08 12:33:42  wrossi
 * Adding javadocs
 *
 * Revision 1.8  2003/11/25 01:31:28  wrossi
 * Made cloneable
 *
 * Revision 1.7  2003/11/18 11:01:39  wrossi
 * Added generic fs interface
 *
 * Revision 1.6  2003/11/08 02:05:03  wrossi
 * Working on directory listing
 *
 * Revision 1.5  2003/11/06 23:34:04  wrossi
 * Fixed sort order
 *
 * Revision 1.4  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.3  2003/10/30 00:47:56  wrossi
 * Added static method to create a key object given a key in a buffer.  Don't need to
 * know what type it is up front.
 *
 * Revision 1.2  2003/10/29 20:42:35  wrossi
 * Added key type definition and compareTo method
 *
 * Revision 1.1  2003/10/29 00:31:38  wrossi
 * Work in progress
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.InodePtr;

/**
 * Representation of a key.  There are two versions, KeyV1 and KeyV2.<p>
 *
 * The two versions differ in the way they represent offset and type  
 * on disk.<p>
 */

public abstract class Key implements DiskStructure, InodePtr, Comparable, Cloneable
{
  /** Indicates Key format 3.5 */
  public final static int KEY_FORMAT_3_5 = 0;
  /** Indicates Key format 3.6 */
  public final static int KEY_FORMAT_3_6 = 1;

  /** Type indicating Stat Item. */
  public final static int TYPE_STAT_DATA = 0;
  /** Type indicating Indirect Item. */
  public final static int TYPE_INDIRECT = 1;
  /** Type indicating Direct Item. */
  public final static int TYPE_DIRECT = 2;
  /** Type indicating Directory Item. */
  public final static int TYPE_DIRENTRY = 3;
  public final static int TYPE_MAXTYPE = 3;

  protected long parentDirId;     // Object ID of parent directory
  protected long objectId;        // our object ID
  protected long offset;          // offset -- When an object is split across multiple
                                  // items, this indicates the byte offset within the object
                                  // that this key represents
  protected int type;             // Type of item this key points to.

  public int getDataSize() { return 0x10; }

  /**
   * Get the identifier for the parent directory.
   * @return parent id.
   */
  public long getParentDirId() { return parentDirId; }
  public void setParentDirId(long aParentDirId) { parentDirId=aParentDirId; }

  /**
   * Get the identifier for this object.
   * @return identifier.
   */
  public long getObjectId() { return objectId; }
  public void setObjectId(long aObjectId) { objectId=aObjectId; }

  /**
   * Get the offset.  When an object is split across multiple
   * items, this indicates the byte offset within the object
   * that this key represents
   * @return the offset.
   */
  public long getOffset() { return offset; }
  public void setOffset(long aOffset) { offset=aOffset; }

  /**
   * Get the item type.  This will control which Item subclass is used to
   * represent the item.
   * @return item type.
   */
  public int getType() { return type; }
  public void setType(int aType) { type=aType; }

  /** Creates a key object given a buffer.  Trys to first create a 3.6 format key (KeyV2),
      and if that fails, then trys to make a 3.5 key (KeyV1) */
  public static Key createKeyFromBuffer(byte[] buffer, int blkoffset) throws FsException
  {
    try  // Try for a 3.6 key
    {
      KeyV2 k2 = new KeyV2();
      k2.loadFromBuffer(buffer, blkoffset);
      return k2;
    }
    catch (FsException ex)  // Maybe its a 3.5 key?
    {
      KeyV1 k1 = new KeyV1();
      k1.loadFromBuffer(buffer, blkoffset);
      return k1;
    }
  }

  /**
   * Defines the sort order of Keys.  Keys are stored in the B-tree in sorted order, 
   * so finding an item relys on this order.  Keys are first ordered by parent ID, then
   * by object Id, then by offset, and lastly type.   
   * @param key to compare to.
   * @return -1 if other < this, 1 if other > this, 0 if other == this
   */
  public int compareTo(Object other)
  {
    Key otherKey = (Key) other;

    if (parentDirId > otherKey.parentDirId)
      return -1;

    if (parentDirId < otherKey.parentDirId)
      return 1;

    if (objectId > otherKey.objectId)
      return -1;

    if (objectId < otherKey.objectId)
      return 1;

    if (offset > otherKey.offset)
      return -1;

    if (offset < otherKey.offset)
      return 1;

    if (type > otherKey.type)
      return -1;

    if (type < otherKey.type)
      return 1;

    return 0;
  }

  public Object clone() throws CloneNotSupportedException
  {
    return super.clone();
  }
}
