/*
 * $Id: KeyV1.java,v 1.5 2004/04/08 12:33:42 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: KeyV1.java,v $
 * Revision 1.5  2004/04/08 12:33:42  wrossi
 * Adding javadocs
 *
 * Revision 1.4  2003/11/20 11:57:35  wrossi
 * Bug fix
 *
 * Revision 1.3  2003/11/05 17:31:26  wrossi
 * Improved exception reporting
 *
 * Revision 1.2  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.1  2003/10/30 00:47:11  wrossi
 * Key implementation
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a format 3.5 key.<p>
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Parent Dir ID               |
 *        --------------------------------------
 *  0004  |        Object Id                   |
 *        --------------------------------------
 *  0008  |        Offset                      |
 *        --------------------------------------
 *  000C  |        Uniqueness                  |
 *        --------------------------------------
 * </pre>
 */

public class KeyV1 extends Key
{
  /** Uniqueness value for Stat Items */
  public static long V1_SD_UNIQUENESS = 0;
  /** Uniqueness value for Indirect Items */
  public static long V1_INDIRECT_UNIQUENESS = 0xfffffffeL;
  /** Uniqueness value for Direct Items */
  public static long V1_DIRECT_UNIQUENESS = 0xffffffffL;
  /** Uniqueness value for Directory Items */
  public static long V1_DIRENTRY_UNIQUENESS = 500;

  public void loadFromBuffer(byte buffer[], int blkoffset) throws FsException
  {
    setParentDirId(FsUtils.getU32(buffer, blkoffset+0x0));
    setObjectId(FsUtils.getU32(buffer, blkoffset+0x4));
    setOffset(FsUtils.getU32(buffer, blkoffset+0x8));

    long uniq = FsUtils.getU32(buffer, blkoffset+0xC);
    
    if (uniq == V1_SD_UNIQUENESS)
    {
      setType(TYPE_STAT_DATA);
      return;
    }

    if (uniq == V1_INDIRECT_UNIQUENESS) 
    {
      setType(TYPE_INDIRECT); 
      return;
    }

    if (uniq == V1_DIRECT_UNIQUENESS)
    {
      setType(TYPE_DIRECT);
      return;
    }

    if (uniq == V1_DIRENTRY_UNIQUENESS) 
    {
      setType(TYPE_DIRENTRY);
      return;
    }
    
    throw new FsException("Unknown uniqueness on format 3.5 key: 0x"+Long.toHexString(uniq));
  }
}
