/*
 * $Id: ReiserSuperBlock.java,v 1.5 2004/03/12 12:33:12 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: ReiserSuperBlock.java,v $
 * Revision 1.5  2004/03/12 12:33:12  wrossi
 * Test for version less than one
 *
 * Revision 1.4  2003/11/30 16:48:47  wrossi
 * Changed version test for superblock to support 3.5 version fs.
 *
 * Revision 1.3  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.2  2003/10/27 11:57:33  wrossi
 * Added FsException and FsUtils.  Added code to ReiserSuperBlock to parse
 * a buffer and do some sanity checks.
 *
 * Revision 1.1  2003/10/22 23:20:11  wrossi
 * Initial checkin
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;

/** Representation of ReiserFS superblock.
 *
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Block Count                 |
 *        --------------------------------------
 *  0004  |        Free Blocks                 |
 *        --------------------------------------
 *  0008  |        Root Block                  |
 *        --------------------------------------
 *  000C  |        Journal Block               |
 *        --------------------------------------
 *  0010  |        Journal Device              |
 *        --------------------------------------
 *  0014  |        Orig Journal Size           |
 *        --------------------------------------
 *  0018  |        Journal Trans Max           |
 *        --------------------------------------
 *  001C  |        Journal Block Count         |
 *        --------------------------------------
 *  0020  |        Journal Max Batch           |
 *        --------------------------------------
 *  0024  |        Journal Commit Age          |
 *        --------------------------------------
 *  0028  |        Journal Trans Age           |
 *        --------------------------------------
 *  002C  |     Block Size    | Old Max Size   |
 *        --------------------------------------
 *  0030  |     Curr Size     |   State        |
 *        --------------------------------------
 *  0034  |       Magic                        |
 *        --------------------------------------
 *  0038  |       Magic  (cont)                |
 *        --------------------------------------
 *  003C  |       Magic  (cont)                |
 *        --------------------------------------
 *  0040  |             Hash Func              | For V1, this is omitted
 *        --------------------------------------
 *  0044  |     Tree Height   | Bitmap Blocks  |
 *        --------------------------------------
 *  0048  |     Version       |  Reserved      | Remaining fields omitted in V1
 *        --------------------------------------
 *  004C  |              Inode Gen             | 
 *        --------------------------------------
 *  0050  |               Flags                |  
 *        --------------------------------------
 *  0054  |     Filesystem Unique Id           |  
 *        --------------------------------------
 *  0058  |     Filesystem Unique Id           |  
 *        --------------------------------------
 *  005C  |     Filesystem Unique Id           |  
 *        --------------------------------------
 *  0060` |     Filesystem Unique Id           |  
 *        --------------------------------------
 *  0064  |        Volume Label                |  
 *        --------------------------------------
 *  0068  |        Volume Label                |  
 *        --------------------------------------
 *  006C  |        Volume Label                |  
 *        --------------------------------------
 *  0070  |        Volume Label                |  
 *        --------------------------------------
 *
 * </pre>
 */

public class ReiserSuperBlock implements SuperBlock
{
  protected long blockCount;      
  protected long freeBlocks;     
  protected long rootBlock;
  protected long journalBlock;
  protected long journalDev;
  protected long origJournalSize;
  protected long journalTransMax;
  protected long journalBlockCount;
  protected long journalMaxBatch;
  protected long journalCommitAge;
  protected long journalTransAge;
  protected int blockSize;
  protected int oldMaxSize;
  protected int currSize;
  protected int state;
  protected byte[] magic;
  protected long hashFunc;
  protected int treeHeight;
  protected int bitmapBlocks;
  protected int version;
  protected int reserved;
  protected long inodeGen;
  protected long flags;
  protected byte[] uniqueID;
  protected byte[] label;

  public ReiserSuperBlock()
  {
  }

  public int getDataSize() { return 0x74+88; }

  public long getBlockCount() { return blockCount; }
  public void setBlockCount(long aBlockCount) { blockCount=aBlockCount; }

  public long getFreeBlocks() { return freeBlocks; }
  public void setFreeBlocks(long aFreeBlocks) { freeBlocks=aFreeBlocks; }

  public long getRootBlock() { return rootBlock; }
  public void setRootBlock(long aRootBlock) { rootBlock=aRootBlock; }

  public long getJournalBlock() { return journalBlock; }
  public void setJournalBlock(long aJournalBlock) { journalBlock=aJournalBlock; }

  public long getJournalDev() { return journalDev; }
  public void setJournalDev(long aJournalDev) { journalDev=aJournalDev; }

  public long getOrigJournalSize() { return origJournalSize; }
  public void setOrigJournalSize(long aOrigJournalSize) { origJournalSize=aOrigJournalSize; }

  public long getJournalTransMax() { return journalTransMax; }
  public void setJournalTransMax(long aJournalTransMax) { journalTransMax=aJournalTransMax; }

  public long getJournalBlockCount() { return journalBlockCount; }
  public void setJournalBlockCount(long aJournalBlockCount) { journalBlockCount=aJournalBlockCount; }

  public long getJournalMaxBatch() { return journalMaxBatch; }
  public void setJournalMaxBatch(long aJournalMaxBatch) { journalMaxBatch=aJournalMaxBatch; }

  public long getJournalCommitAge() { return journalCommitAge; }
  public void setJournalCommitAge(long aJournalCommitAge) { journalCommitAge=aJournalCommitAge; }

  public long getJournalTransAge() { return journalTransAge; }
  public void setJournalTransAge(long aJournalTransAge) { journalTransAge=aJournalTransAge; }

  public int getBlockSize() { return blockSize; }
  public void setBlockSize(int aBlockSize) { blockSize=aBlockSize; }

  public int getOldMaxSize() { return oldMaxSize; }
  public void setOldMaxSize(int aOldMaxSize) { oldMaxSize=aOldMaxSize; }

  public int getCurrSize() { return currSize; }
  public void setCurrSize(int aCurrSize) { currSize=aCurrSize; }

  public int getState() { return state; }
  public void setState(int aState) { state=aState; }

  public byte[] getMagic() { return magic; }
  public void setMagic(byte[] aMagic) { magic=aMagic; }

  public long getHashFunc() { return hashFunc; }
  public void setHashFunc(long aHashFunc) { hashFunc=aHashFunc; }

  public int getTreeHeight() { return treeHeight; }
  public void setTreeHeight(int aTreeHeight) { treeHeight=aTreeHeight; }

  public int getBitmapBlocks() { return bitmapBlocks; }
  public void setBitmapBlocks(int aBitmapBlocks) { bitmapBlocks=aBitmapBlocks; }

  public int getVersion() { return version; }
  public void setVersion(int aVersion) { version=aVersion; }

  public int getReserved() { return reserved; }
  public void setReserved(int aReserved) { reserved=aReserved; }

  public long getInodeGen() { return inodeGen; }
  public void setInodeGen(long aInodeGen) { inodeGen=aInodeGen; }

  public long getFlags() { return flags; }
  public void setFlags(long aFlags) { flags=aFlags; }

  public byte[] getUniqueID() { return uniqueID; }
  public void setUniqueID(byte[] aUniqueID) { uniqueID=aUniqueID; }

  public byte[] getLabel() { return label; }
  public void setLabel(byte[] aLabel) { label=aLabel; }

  /** Load from a buffer containint the superblock.  We assume that the superblock
      begins at the specified offset within the buffer */
  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    /* Make sure we have enough buffer to do this.  We need 116 bytes to read the
       superblock */

    if ( buffer.length < (offset+116))
      throw new FsException("Insufficinet buffer to load super-block");

    setBlockCount(FsUtils.getU32(buffer, offset+0x0));
    setFreeBlocks(FsUtils.getU32(buffer, offset+0x4));
    setRootBlock(FsUtils.getU32(buffer, offset+0x8));
    setJournalBlock(FsUtils.getU32(buffer, offset+0xC));
    setJournalDev(FsUtils.getU32(buffer, offset+0x10));
    setOrigJournalSize(FsUtils.getU32(buffer, offset+0x14));
    setJournalTransMax(FsUtils.getU32(buffer, offset+0x18));
    setJournalBlockCount(FsUtils.getU32(buffer, offset+0x1C));
    setJournalMaxBatch(FsUtils.getU32(buffer, offset+0x20));
    setJournalCommitAge(FsUtils.getU32(buffer, offset+0x24));
    setJournalTransAge(FsUtils.getU32(buffer, offset+0x28));
    setBlockSize(FsUtils.getU16(buffer, offset+0x2C));
    setOldMaxSize(FsUtils.getU16(buffer, offset+0x2E));
    setCurrSize(FsUtils.getU16(buffer, offset+0x30));
    setState(FsUtils.getU16(buffer, offset+0x32));
    magic = new byte[12];
    System.arraycopy(buffer, offset+0x34, magic, 0, 12);
    setHashFunc(FsUtils.getU32(buffer, offset+0x40));
    setTreeHeight(FsUtils.getU16(buffer, offset+0x44));
    setBitmapBlocks(FsUtils.getU16(buffer, offset+0x46));
    setVersion(FsUtils.getU16(buffer, offset+0x48));
    setReserved(FsUtils.getU16(buffer, offset+0x4A));
    setInodeGen(FsUtils.getU32(buffer, offset+0x4C));
    setFlags(FsUtils.getU32(buffer, offset+0x50));
    uniqueID = new byte[16];
    System.arraycopy(buffer, offset+0x54, uniqueID, 0, 16);
    label = new byte[16];
    System.arraycopy(buffer, offset+0x64, label, 0, 16);

    /* Check the magic field to make sure we have a valid superblock */
    try
    {
      String magics = new String(magic, "ISO-8859-1");
      if (version == 1 && !magics.startsWith("ReIsErFs"))
        throw new FsException("Invalid magic for version 1 "+magic);

      if (version == 2 && !magics.startsWith("ReIsEr2Fs"))
        throw new FsException("Invalid magic for version 1 "+magic);

      if (version > 2 || version < 1)
        throw new FsException("Unsupported version "+version);
    }
    catch (Exception ex)
    {
      FsException fex = new FsException("Error processing superblock.");
      fex.initCause(ex);
      throw fex;
    }
  }
}
