/*
 * $Id: BlockIO.java,v 1.6 2004/02/28 20:49:16 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: BlockIO.java,v $
 * Revision 1.6  2004/02/28 20:49:16  wrossi
 * Added ability to close a filesystem.
 *
 * Revision 1.5  2004/02/17 21:16:45  wrossi
 * Added optional offset into constructor on BlockIO
 *
 * Revision 1.4  2003/11/30 16:38:15  wrossi
 * Fix integer overflow error
 *
 * Revision 1.3  2003/11/24 21:14:27  wrossi
 * Generalized the cache
 *
 * Revision 1.2  2003/11/05 17:34:04  wrossi
 * Use int instead of long for block numbers
 *
 * Revision 1.1  2003/11/04 12:10:36  wrossi
 * Block manaipulation routines
 *
 *
 */

package rossi.fstools.io;

import java.io.RandomAccessFile;
import java.io.IOException;

public class BlockIO implements BlockReader
{
  protected RandomAccessFile file;
  protected int blockSize;
  protected BlockCache cache;
  protected long offset;  // offset to start of filesystem in bytes

  private final static int CACHE_SIZE = 10;

  public BlockIO(RandomAccessFile f, int blockSize)
  {
    this(f, blockSize, 0);
  }

  public BlockIO(RandomAccessFile f, int blockSize, long offset)
  {
    this.file = f;
    this.blockSize = blockSize;
    this.offset = offset;
    cache = new BlockCache(CACHE_SIZE);
  }

  public int getBlockSize() { return blockSize; }

  public synchronized void setBlockSize(int aBlockSize) 
  { 
    cache.clearCache();
    blockSize=aBlockSize; 
  }

  private byte[] getBlockFromCache(int blkno)
  {
    return (byte[]) cache.getBlockFromCache(blkno);
  }

  private byte[] getBlockFromDisk(int blkno) throws IOException
  {
    byte[] buffer;
 
    buffer = new byte[blockSize];
    file.seek((long)blkno*(long)blockSize + offset);
    file.readFully(buffer);

    /* Cache the block */
    cache.addBlockToCache(blkno, buffer);

    return buffer;
  }

  public synchronized byte[] getBlock(int blkno) throws IOException
  {
    byte[] buffer;

    buffer = getBlockFromCache(blkno);
    if (buffer == null)
      buffer = getBlockFromDisk(blkno);

    return buffer;
  }

  /** Close this file */
  public void close() throws IOException
  {
    file.close();
  }
}
