/*
 * $Id: BlockPointerInputStream.java,v 1.3 2004/03/11 00:39:56 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: BlockPointerInputStream.java,v $
 * Revision 1.3  2004/03/11 00:39:56  wrossi
 * Performance boost.  Allow a single BlockPointer to point to multiple
 * contigous blocks.  This reduces our memory footprint considerably
 *
 * Revision 1.2  2003/11/30 15:48:38  wrossi
 * bug fixes
 *
 * Revision 1.1  2003/11/25 21:37:36  wrossi
 * IO for reading files
 *
 *
 */

package rossi.fstools.io;

import java.io.InputStream;
import java.io.IOException;
import java.io.EOFException;

import java.util.List;
import java.util.Iterator;

/** An input stream composed of a list of block pointers and a block reader */
public class BlockPointerInputStream extends InputStream
{
  protected List blocks;
  protected BlockReader reader;
  protected int currentBlock;          /* Current index to a block pointer in blocks */
  protected int currentBlockNum;       /* Block number of the current block */
  protected int currentOffset;         /* current offset within a block pointer */
  protected int bufferHead;            /* Where the data starts in the buffer */
  protected byte[] currentBuffer;      /* A single block of a data */
  protected boolean isEOF = false;

  public BlockPointerInputStream(BlockReader reader, List blockList)
  {
    this.reader = reader;
    this.blocks = blockList;
    currentBlock = 0;
    currentOffset = 0;
  }

  public Iterator getBlocks()
  {
    return blocks.iterator();
  }

  public synchronized int read(byte[] buffer, int offset, int len) throws IOException
  {
    BlockPointer bp;
    int blockSize;
    int blocknum;

    if (isEOF)
      throw new EOFException();

    if (currentBlock == blocks.size())
    {
      isEOF = true;
      return -1;
    }

    bp = (BlockPointer) blocks.get(currentBlock);
    blockSize = bp.getBlockSize();

    blocknum = (currentOffset / blockSize);
    if (currentBuffer == null || blocknum != currentBlockNum)
    {
      if (reader.getBlockSize() != blockSize)
        throw new IOException("Block size mismatch");

      currentBuffer = reader.getBlock(bp.getBlockNum() + (currentOffset / blockSize));     
      currentBlockNum = blocknum;

      if (currentOffset < blockSize)
        bufferHead = bp.getOffset();            // if this is the first block of the block pointer
      else
        bufferHead = 0;                         // following consecutive blocks
    }

    /* Dont read past the end of the current buffer */
    if (len > (blockSize - (currentOffset % blockSize) - bufferHead))
      len = (blockSize - (currentOffset % blockSize) - bufferHead);

    /* for simplicity sake, we wont read past the end of the current block */
    if (len >= (bp.getLength()-currentOffset))
    {
      int readlen = bp.getLength() - currentOffset;
      
      System.arraycopy(currentBuffer, bufferHead+(currentOffset%blockSize), buffer, offset, readlen);
      /* Setup for the next block */
      currentBuffer = null;
      currentBlock++;
      currentOffset = 0;
      return readlen;
    }

    /* Read the requested length */
    System.arraycopy(currentBuffer, bufferHead+(currentOffset%blockSize), buffer, offset, len);
    currentOffset += len;
    return len;
  }

  public synchronized int read(byte[] buffer) throws IOException
  {
    return read(buffer, 0, buffer.length);
  }

  public synchronized int read() throws IOException
  {
    byte[] buffer = new byte[1];
   
    int bytesRead = read(buffer);

    if (bytesRead == -1)
      return -1;

    return buffer[0];
  }
}
