/*
 * $Id: PartitionTable.java,v 1.2 2004/02/28 21:46:09 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: PartitionTable.java,v $
 * Revision 1.2  2004/02/28 21:46:09  wrossi
 * Got partition parsing working
 *
 * Revision 1.1  2004/02/28 20:44:43  wrossi
 * Partition support
 *
 *
 */

package rossi.fstools.partition;

import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.FsUtils;

/*
 * Representation of a partition table 
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |         Boot Loader                |
 *        --------------------------------------
 *  0004  |         Boot Loader                |
 *        --------------------------------------
 *                      .
 *                      .
 *        --------------------------------------
 *  01bc  |                  |  Partition 0    |
 *        --------------------------------------
 *  01c0  |            Partition 0             |
 *        --------------------------------------
 *  01c4  |            Partition 0             |
 *        --------------------------------------
 *  01c8  |            Partition 0             |
 *        --------------------------------------
 *  01cc  |  Partition 0     |  Partition 1    |
 *        --------------------------------------
 *  01d0  |            Partition 1             |
 *        --------------------------------------
 *  01d4  |            Partition 1             |
 *        --------------------------------------
 *  01d8  |            Partition 1             |
 *        --------------------------------------
 *  01dc  |  Partition 1     |  Partition 2    |
 *        --------------------------------------
 *  01e0  |            Partition 2             |
 *        --------------------------------------
 *  01e4  |            Partition 2             |
 *        --------------------------------------
 *  01e8  |            Partition 2             |
 *        --------------------------------------
 *  01ec  |  Partition 2     |  Partition 3    |
 *        --------------------------------------
 *  01f0  |            Partition 3             |
 *        --------------------------------------
 *  01f4  |            Partition 3             |
 *        --------------------------------------
 *  01f8  |            Partition 3             |
 *        --------------------------------------
 *  01fc  |  Partition 3     |    Magic        |
 *        --------------------------------------
 * </pre>
 */

public class PartitionTable implements DiskStructure
{
  protected Partition partitions[] = new Partition[4];

  public Partition[] getPartitions() { return partitions; }
  public void setPartitions(Partition aPartitions[]) { partitions=aPartitions; }
  
  /** Returns the size in bytes of the structure */
  public int getDataSize()
  {
    return 256;
  }

  /** Loads the object from a memory buffer */
  public void loadFromBuffer(byte[] buffer, int offset) throws FsException
  {
    /* Check the magic number.  Should be 0x55 , 0xaa*/
    if ((buffer[0x1fe]&0xff) != 0x55 || (buffer[0x1ff]&0xff) != 0xaa)
    {
      throw new FsException("Invalid parition table");
    }

    for (int i=0; i<4; i++)
    {
      partitions[i] = new Partition();
      partitions[i].loadFromBuffer(buffer, offset+0x1be + i*partitions[i].getDataSize());
    }
  }
}
