/*
 * $Id: InodeTablePanel.java,v 1.14 2004/04/09 21:04:53 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: InodeTablePanel.java,v $
 * Revision 1.14  2004/04/09 21:04:53  wrossi
 * Fix bug with long files ( > 2^32 bytes)
 *
 * Revision 1.13  2004/03/05 12:43:33  wrossi
 * Added RCS header and copyright
 *
 *
 */

package rossi.fstoolsui.fstree;

import rossi.fstools.fs.Inode;
import rossi.fstools.fs.FileSystem;
import rossi.fstools.fs.Directory;
import rossi.fstools.fs.Device;
import rossi.fstools.fs.FsObject;
import rossi.fstools.fs.FsException;
import javax.swing.table.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.*;
import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.Component;
import java.awt.Color;
import java.util.*;
import java.awt.event.*;
import java.io.IOException;

public class InodeTablePanel extends JPanel implements MouseListener
{
  /** The table itself */
  protected JTable tbl;
  /** The table model */
  protected TableModel model;
  /** The map of inodes */
  protected Map inodeMap;
  /** List of file names */
  protected List fileNameList;
  /** FileSystem object we are operating on */
  protected FileSystem fs = null;
  /** current tree node */
  protected FsObjectTreeNode currentNode = null;
  /** actionlisteners */
  protected List actionListeners = new Vector();

  public InodeTablePanel()
  {
    initialize();
  }

  protected void initialize()
  {
    fileNameList = new Vector();
    inodeMap = new HashMap();

    model = getModel(fileNameList, inodeMap);
    TableColumnModel cm = getColumnModel(model);

    tbl = new JTable(model, cm);
    tbl.setSurrendersFocusOnKeystroke(true);

    tbl.setAutoResizeMode(tbl.AUTO_RESIZE_OFF);

    tbl.addMouseListener(this);
    tbl.getTableHeader().addMouseListener(this);

    setLayout(new BorderLayout());

    JScrollPane sp = new JScrollPane(tbl);

    add(sp, "Center");
  }
  
  public FileSystem getFileSystem() { return fs; }

  public void setFileSystem(FileSystem aFs) 
  {  
    fs=aFs; 
    ((InodeTableModel)model).setFileSystem(fs);
  }

  protected TableModel getModel(List nameList, Map inodeMap)
  {
    return new InodeTableModel(nameList, inodeMap);
  }

  protected TableColumnModel getColumnModel(TableModel model)
  {
    DefaultTableColumnModel cm = new DefaultTableColumnModel();
    DefaultTableCellRenderer def = new DefaultTableCellRenderer();
    TableCellRenderer rend[] = {new NameColumnRenderer(), new ModeColumnRenderer(), def, def, 
                                def, new SizeColumnRenderer(), def, def, def};
    int width[] = {200, 55, 45, 45, 45, 65, 150, 150, 150};

    for (int i=0; i<model.getColumnCount(); i++)
    {
      TableColumn c = new TableColumn(i, width[i]);
      c.setHeaderValue(model.getColumnName(i));
      c.setCellRenderer(rend[i]);
      cm.addColumn(c);
    }

    return cm;
  }

  /** MouseListener interface */
  public void mousePressed(MouseEvent evt)
  {
  }

  /** MouseListener interface */
  public void mouseReleased(MouseEvent evt)
  {
  }

  /** MouseListener interface */
  public void mouseEntered(MouseEvent evt)
  {
  }

  /** MouseListener interface */
  public void mouseExited(MouseEvent evt)
  {
  }

  /** MouseListener interface */
  public void mouseClicked(MouseEvent evt)
  {
    if (SwingUtilities.isLeftMouseButton(evt))
    {
      if (evt.getSource() == tbl.getTableHeader())
      {
        int col = tbl.columnAtPoint(new Point(evt.getX(), evt.getY()));
 
        sortModel(col);
      }
      else
      {
        if (evt.getClickCount() == 2)
        {
          handleDoubleClick();
        }
      }
    }
  }

  protected void handleDoubleClick()
  {
    int row;

    row = tbl.getSelectedRow();
    if (row == -1)
      return;

    Inode inode = (Inode) inodeMap.get(fileNameList.get(row));
    if ( (inode.getMode() & Inode.FILETYPE_MASK) == Inode.DIRECTORY_MODE)
    {
      try
      {
        Directory dir = (Directory) fs.getObject(inode);
      
        FsObjectTreeNode fsn = new FsObjectTreeNode(fs, currentNode, dir, inode, (String) fileNameList.get(row));
        //setData(fsn.getChildList(), fsn.getInodeMap());
        setFsObjectNode(fsn);
      }
      catch (Exception ex)
      {
      }
    }
  }

  public FsObject getSelectedFsObject() throws IOException, FsException
  {
    int row;

    row = tbl.getSelectedRow();
    if (row == -1)
      return null;

    Inode inode = (Inode) inodeMap.get(fileNameList.get(row));
    return fs.getObject(inode);
  }

  public String getSelectedFilename()
  {
    int row;

    row = tbl.getSelectedRow();
    if (row == -1)
      return null;

    return (String)fileNameList.get(row);
  }

  /** Get the table so our panent component can add listeners to it */
  public JTable getTable()
  {
    return tbl;
  }

  public FsObjectTreeNode getFsObjectNode()
  {
    return currentNode;
  }

  public void setFsObjectNode(FsObjectTreeNode node)
  {
    currentNode = node;
    setData(node.getChildList(), node.getInodeMap());
    fireActionEvent();
  }

  protected void fireActionEvent()
  {
    ActionEvent evt = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, "cmd" );
    synchronized(actionListeners)
    {
      for (Iterator itr=actionListeners.iterator(); itr.hasNext(); )
      {
        ActionListener al = (ActionListener) itr.next();
        al.actionPerformed(evt);
      }
    }
  }

  public void addActionListener(ActionListener al)
  {
    synchronized (actionListeners)
    {
      actionListeners.add(al);
    }
  }

  protected void setData(List nameList, Map iMap)
  {
    while (fileNameList.size() != 0)
      fileNameList.remove(0);

    fileNameList.addAll(nameList);
    //System.out.println(". inode = "+iMap.get("."));

    inodeMap.clear();
    inodeMap.putAll(iMap);
    //System.out.println(". inode = "+inodeMap.get("."));

    ((AbstractTableModel)tbl.getModel()).fireTableDataChanged();
  }

  protected void sortModel(final int col)
  {
    Collections.sort(fileNameList,  

                     new Comparator()
                     {
                       public int compare(Object o1, Object o2)
                       {
                         String name1, name2;
                         Comparable cell1, cell2;

                         name1 = (String) o1;
                         name2 = (String) o2;

                         cell1 = (Comparable) ((InodeTableModel)model).cellFromName(name1, col);
                         cell2 = (Comparable) ((InodeTableModel)model).cellFromName(name2, col);

                         return cell1.compareTo(cell2);
                       }
                     }
                    );

    ((AbstractTableModel)model).fireTableDataChanged();
    ((AbstractTableModel)model).fireTableChanged(new TableModelEvent(model));
  }

  class NameColumnRenderer extends DefaultTableCellRenderer
  {
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, 
                                                   boolean hasFocus, int row, int column) 
    {
      TableModel model = table.getModel();
      JLabel lbl;
      int mode = ((Integer)model.getValueAt(row, 1)).intValue();

      lbl = (JLabel) super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

      /* The default rederer creates a JLabel with the toString() of the node
         as the text */

      lbl.setForeground(Color.black);

      if ((mode & Inode.FILETYPE_MASK) == Inode.CHARDEV_MODE)
        lbl.setForeground(Color.red);

      if ((mode & Inode.FILETYPE_MASK) == Inode.BLOCKDEV_MODE)
        lbl.setForeground(Color.red.darker());

      if ((mode & Inode.FILETYPE_MASK) == Inode.FIFO_MODE)
        lbl.setForeground(Color.green);

      if ((mode & Inode.FILETYPE_MASK) == Inode.DIRECTORY_MODE)
        lbl.setForeground(Color.blue);

      if ((mode & Inode.FILETYPE_MASK) == Inode.SOCKET_MODE)
        lbl.setForeground(Color.orange);

      if ((mode & Inode.FILETYPE_MASK) == Inode.SYMLINK_MODE)
      {
        //lbl.setText(value+" -> "+((SymLink)node.getFsObject()).getLink());
        lbl.setForeground(Color.cyan);
      }

      return lbl;
    }
  }

  class ModeColumnRenderer extends DefaultTableCellRenderer
  {
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, 
                                                   boolean hasFocus, int row, int column) 
    {
      JLabel lbl;
      TableModel model = table.getModel();
      int mode = ((Integer)model.getValueAt(row, 1)).intValue();
      String modestr = "";

      lbl = (JLabel) super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);

      modestr = "-";
      if ((mode & Inode.FILETYPE_MASK) == Inode.CHARDEV_MODE)
        modestr = "c";

      if ((mode & Inode.FILETYPE_MASK) == Inode.BLOCKDEV_MODE)
        modestr = "b";

      if ((mode & Inode.FILETYPE_MASK) == Inode.FIFO_MODE)
        modestr = "p";

      if ((mode & Inode.FILETYPE_MASK) == Inode.DIRECTORY_MODE)
        modestr = "d";

      if ((mode & Inode.FILETYPE_MASK) == Inode.SOCKET_MODE)
        modestr = "s";

      if ((mode & Inode.FILETYPE_MASK) == Inode.SYMLINK_MODE)
        modestr = "l";

      int wr = 0400;
      int rd = 0200;
      int ex = 0100;

      for (int i=0; i<3; i++)
      {
        if ((mode & wr) != 0) modestr+="w"; else modestr+="-";
        if ((mode & rd) != 0) modestr+="r"; else modestr+="-";
        if ((mode & ex) != 0) modestr+="x"; else modestr+="-";
      }

      lbl.setText(modestr);
      lbl.setToolTipText("0"+Integer.toOctalString(mode));

      return lbl;
    }
  }

  class SizeColumnRenderer extends DefaultTableCellRenderer
  {
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, 
                                                   boolean hasFocus, int row, int column) 
    {
      JLabel lbl;
      TableModel model = table.getModel();
      int mode = ((Integer)model.getValueAt(row, 1)).intValue();
      long size = ((Long)model.getValueAt(row, 5)).longValue();

      lbl = (JLabel) super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
      lbl.setHorizontalAlignment(SwingConstants.RIGHT);

      if ((mode & Inode.FILETYPE_MASK) == Inode.CHARDEV_MODE ||
          (mode & Inode.FILETYPE_MASK) == Inode.BLOCKDEV_MODE)
      {
        try
        {
          Inode inode = (Inode) inodeMap.get(fileNameList.get(row));
          Device dev = (Device) fs.getObject(inode);
          String devstr = Integer.toString(dev.getMajor())+", "+Integer.toString(dev.getMinor());
          lbl.setText(devstr);
        }
        catch (Exception ex)
        {
          lbl.setText(ex.getClass().getName()+": "+ex.getMessage());
        }
      }
      else
      {
        lbl.setText(Long.toString(size));
      }

      return lbl;
    }
  }
}
