/*
 * $Id: FileSystem.java,v 1.3 2004/04/05 20:38:33 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FileSystem.java,v $
 * Revision 1.3  2004/04/05 20:38:33  wrossi
 * Add javadocs
 *
 * Revision 1.2  2004/02/28 20:49:16  wrossi
 * Added ability to close a filesystem.
 *
 * Revision 1.1  2003/11/18 11:01:39  wrossi
 * Added generic fs interface
 *
 *
 */

package rossi.fstools.fs;

import rossi.fstools.io.BlockReader;
import java.io.IOException;

/** 
 * Representation of a file system.  This is the main interface applications will 
 * use to interface to a filesytem.   Methods are provided here for navigating 
 * the filesystem. 
 */
public interface FileSystem
{
  /**
   * Open a file system for reading.  The BlockReader should be initialized to point
   * to the phyical storage of the desired filesystem.
   *
   * @param br   a BlockReader for accessing the physical filesystem.
   * @return the filesystem's SuperBlock
   * @exception FsException if the superblock cannot be parsed.
   * @exception IOException
   */
  public SuperBlock open(BlockReader br) throws FsException, IOException;

  /**
   * Close the underlying BlockReader used by this file system.
   * @exception IOException
   */
  public void close() throws IOException;

  /**
   * Get a pointer to the root directory's inode.
   *
   * @return InodePtr pointing to the root directory.
   */
  public InodePtr getRootDir();

  /**
   * Get an Inode given a point to it.  May return null if the Inode does not exist.
   *
   * @param ptr   an InodePtr 
   * @return Inode pointed to by ptr
   * @exception FsException if the Inode cannot be parsed.
   * @exception IOException if an IO error occurs
   */
  public Inode getInode(InodePtr ptr) throws FsException, IOException;

  /**
   * Get the filesystem object represented by an inode.
   *
   * @param inode   an Inode
   * @return FsObject that is represented by the inode
   * @exception FsException if the object cannot be parsed.
   * @exception IOException if an IO error occurs.
   */
  public FsObject getObject(Inode inode) throws FsException, IOException;
}
