/*
 * $Id: DirEntry.java,v 1.5 2004/04/09 21:11:25 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: DirEntry.java,v $
 * Revision 1.5  2004/04/09 21:11:25  wrossi
 * Adding javadocs
 *
 * Revision 1.4  2004/04/04 17:11:24  wrossi
 * Fix for directory type information and directories with null records.
 *
 * Revision 1.3  2004/04/03 20:33:49  wrossi
 * Fix bug in reading directories
 *
 * Revision 1.2  2004/04/03 18:59:03  wrossi
 * Can read directories almost
 *
 * Revision 1.1  2004/03/26 21:35:52  wrossi
 * Directory entry and group descriptor
 *
 * Revision 1.6  2004/03/25 21:14:44  wrossi
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;

/** 
 * Representation of a directory entry.<p>
 * <pre>
 *  On disk the directory entry looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |           Inode                    |
 *        --------------------------------------
 *  0004  |  Record Len       |NameLen | Type  |
 *        --------------------------------------
 *  0008  |        Name...                     |
 *        --------------------------------------
 * </pre>
 */

public class DirEntry implements DiskStructure
{
  protected long inode;
  protected int recordLen;
  protected int nameLen;
  protected int type;
  protected String name;

  /** File type is unknown */
  public final static int EXT2_FT_UNKNOWN   =  0;
  /** Regular file type */
  public final static int EXT2_FT_REG_FILE  =  1;
  /** Directory type */
  public final static int EXT2_FT_DIR       =  2;
  /** Character device type */
  public final static int EXT2_FT_CHRDEV    =  3;
  /** Block device type */
  public final static int EXT2_FT_BLKDEV    =  4;
  /** Fifo type */
  public final static int EXT2_FT_FIFO      =  5;
  /** Socket type */
  public final static int EXT2_FT_SOCK      =  6;
  /** Symlink type */
  public final static int EXT2_FT_SYMLINK   =  7;


  /**
   * Get the size of this DirEntry.  Note that dir entries are variable sized.
   * @return size
   */
  public int getDataSize() { return getRecordLen(); } 

  /**
   * Get the Inode number associated with this direntry.
   * @return inode number
   */
  public long getInode() { return inode; }
  public void setInode(long aInode) { inode=aInode; }

  /**
   * Get the length of this record.
   * @return record length
   */
  public int getRecordLen() { return recordLen; }
  public void setRecordLen(int aRecordLen) { recordLen=aRecordLen; }

  /**
   * Get the length of the name in this record.  
   * @return name length
   */
  public int getNameLen() { return nameLen; }
  public void setNameLen(int aNameLen) { nameLen=aNameLen; }

  public String getName() { return name; }
  public void setName(String aName) { name=aName; }

  /**
   * Get the type of this file for EXT3 filesystem which support it.  Must have
   * feature EXT2_FEATURE_INCOMPAT_FILETYPE set in the superblock.
   * @return type
   */
  public int getType() { return type; }
  public void setType(int aType) { type=aType; }

  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    setInode(FsUtils.getU32(buffer, offset+0x0));
    setRecordLen(FsUtils.getU16(buffer, offset+0x4));
    setNameLen( ((int)(buffer[offset+0x6])) & 0xff);
    setType( ((int)(buffer[offset+0x7])) & 0xff);

    /* Get the name */
    try
    {
      name = new String(buffer, offset+8, getNameLen(), "ISO-8859-1");
      int zeroidx = name.indexOf('\u0000');
      if (zeroidx > 0)
        name = name.substring(0, zeroidx);
      //System.out.println("Name = "+name+"  length = "+name.length()+" namelen = "+getNameLen()+" reclen= "+getRecordLen());
    }
    catch (Exception ex)
    {
      FsException fsex = new FsException("Error parsing filename");
      fsex.initCause(ex);
      throw fsex;
    }
  }
}
