/*
 * $Id: Ext2Directory.java,v 1.5 2004/04/09 21:11:25 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Ext2Directory.java,v $
 * Revision 1.5  2004/04/09 21:11:25  wrossi
 * Adding javadocs
 *
 * Revision 1.4  2004/04/04 17:11:24  wrossi
 * Fix for directory type information and directories with null records.
 *
 * Revision 1.3  2004/04/03 21:06:51  wrossi
 * Redesign of directory reading
 *
 * Revision 1.2  2004/04/03 20:33:49  wrossi
 * Fix bug in reading directories
 *
 * Revision 1.1  2004/04/03 17:55:36  wrossi
 * Adding directory support
 *
 * Revision 1.1  2004/03/26 21:35:52  wrossi
 * Directory entry and group descriptor
 *
 * Revision 1.6  2004/03/25 21:14:44  wrossi
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.Directory;
import rossi.fstools.fs.File;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.InodePtr;

import java.util.Map;
import java.util.HashMap;

import java.io.InputStream;
import java.io.IOException;

/** 
 * Represents an EXT2 directory. <p>
 * Direcoties on EXT2 are stored as files containing a linked list of DirEntries. <p>
 */
public class Ext2Directory implements Directory
{
  private Map dirMap;
  private boolean atEOF = false;

  /**
   * Create an Ext2Directory given a File.
   *
   * @param file   a File containting directory data.
   * @exception IOException
   * @exception FsException
   */
  public Ext2Directory(File file) throws IOException, FsException
  {
    dirMap = new HashMap();
    load(file);
  }

  private void load(File file) throws IOException, FsException
  {
    InputStream in = file.getData();
    DirEntry dirEntry = new DirEntry();
    int bytesRead, offset = 0;
    int availData = 0;
    // buffer big enough to hold the max block size since an entry won't span a block. 
    // so it would seem anyways.  Sometimes to eat up space in the the block there will be
    // a direntry that is as big as a block.  I think this is how they keep dir indexing read
    // compatible
    byte buffer[] = new byte[Ext2SuperBlock.EXT2_MAX_BLOCK_SIZE];
    
    while (true)
    {
      bytesRead = fillBuffer(in, buffer, offset);
      availData += bytesRead;

      if (availData < 8)  // End of data (a direntry header is 8 bytes)
        break;

      dirEntry.loadFromBuffer(buffer, 0);

      InodePtr inodePtr = new Ext2InodePtr(dirEntry.getInode());
  
      // Its possible to have an empty record
      if (dirEntry.getNameLen() > 0)
      {
        String name = dirEntry.getName();
        dirMap.put(name, inodePtr);
      }

      offset = buffer.length - dirEntry.getRecordLen();
      // Shift the contents of the buffer dirEntry.getRecordLen bytes
      System.arraycopy(buffer, dirEntry.getRecordLen(), buffer, 0, offset);
      availData -= dirEntry.getRecordLen();
      //System.out.println("AvailData = "+availData);
    }
  }

  /** Read in up to end of buffer bytes or to end of file whatever comes first. */
  private int fillBuffer(InputStream in, byte[] buffer, int startoffset) throws IOException
  {
    int totalBytesRead = startoffset;
    int bytesToRead = buffer.length;

    if (atEOF)
      return 0;

    while (totalBytesRead < bytesToRead)
    {
      int bytesRead = in.read(buffer, totalBytesRead, bytesToRead-totalBytesRead);
      if (bytesRead == -1)
      {
        atEOF = true;
        return (totalBytesRead - startoffset);
      }
      totalBytesRead += bytesRead;
      //System.out.println("bytesRead = "+bytesRead);
    }

    return (totalBytesRead - startoffset);
  }

  public Map getInodePtrMap()
  {
    return dirMap;
  }
}
