/*
 * $Id: Ext2SuperBlock.java,v 1.10 2004/04/09 21:41:05 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Ext2SuperBlock.java,v $
 * Revision 1.10  2004/04/09 21:41:05  wrossi
 * Added javadocs.
 *
 * Revision 1.9  2004/04/09 21:11:25  wrossi
 * Adding javadocs
 *
 * Revision 1.8  2004/03/31 21:45:33  wrossi
 * Starting file system
 *
 * Revision 1.7  2004/03/31 12:33:08  wrossi
 * Added comments and inode pointer class
 *
 * Revision 1.6  2004/03/25 21:14:44  wrossi
 * Only read in dynamic rev data if on dynamic rev
 *
 * Revision 1.5  2004/03/24 20:35:31  wrossi
 * Compiles now
 *
 * Revision 1.4  2004/03/24 20:24:20  wrossi
 * wip
 *
 * Revision 1.3  2004/03/23 21:13:03  wrossi
 * work in progress
 *
 * Revision 1.2  2004/03/22 21:43:19  wrossi
 * Adding fields
 *
 * Revision 1.1  2004/03/22 21:33:32  wrossi
 * The beginnings of the ext2 filesystem
 *
 *
 */

package rossi.fstools.fs.ext2fs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;

/** 
 * Representation of ext2/3 superblock.
 *
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Inode Count                 |
 *        --------------------------------------
 *  0004  |        Blocks Count                |
 *        --------------------------------------
 *  0008  |        Reserved Blocks Count       |
 *        --------------------------------------
 *  000C  |        Free Blocks Count           |
 *        --------------------------------------
 *  0010  |        Free Inode Count            |
 *        --------------------------------------
 *  0014  |        First Data Block            |
 *        --------------------------------------
 *  0018  |        Block Size                  |
 *        --------------------------------------
 *  001C  |        Fragment Size               |
 *        --------------------------------------
 *  0020  |        Blocks Per Group            |
 *        --------------------------------------
 *  0024  |        Fragments Per Group         |
 *        --------------------------------------
 *  0028  |        Inodes Per Group            |
 *        --------------------------------------
 *  002C  |        Mount time                  |
 *        --------------------------------------
 *  0030  |        Write Time                  |
 *        --------------------------------------
 *  0034  | Mount Count    |  Max Mount Count  |
 *        --------------------------------------
 *  0038  | Magic          |  State            |
 *        --------------------------------------
 *  003C  | Errors         |  Minor Rev        |
 *        --------------------------------------
 *  0040  |        Last Check                  |
 *        --------------------------------------
 *  0044  |        Check Interval              |
 *        --------------------------------------
 *  0048  |        Creator OS                  |
 *        --------------------------------------
 *  004C  |        Revision Level              | 
 *        --------------------------------------
 *  0050  | Rsrvd UID      |  Rsrvd GID        |  
 *        --------------------------------------
 *  0054  | Fisrt Inode    |  Inode Size       |  For EXT2_DYNAMIC_REV only from here
 *        --------------------------------------
 *  0058  |     Block Group Number of this SB  |  
 *        --------------------------------------
 *  005C  |     Feature Compatible             |  
 *        --------------------------------------
 *  0060` |     Feature Incompatible           |  
 *        --------------------------------------
 *  0064  |     Feature RO Compatible          |  
 *        --------------------------------------
 *  0068  |       Volume UUID                  |  
 *        --------------------------------------
 *  006C  |       Volume UUID                  |  
 *        --------------------------------------
 *  0070  |       Volume UUID                  |  
 *        --------------------------------------
 *  0074  |       Volume UUID                  |  
 *        --------------------------------------
 *  0078  |       Volume name                  |  
 *        --------------------------------------
 *  007C  |       Volume name                  |  
 *        --------------------------------------
 *  0080  |       Volume name                  |  
 *        --------------------------------------
 *  0084  |       Volume name                  |  
 *        --------------------------------------
 *  0088  |       Last mounted dir             |  
 *        --------------------------------------
 *                      :
 *        --------------------------------------
 *  00C4  |       Last mounted dir             |  
 *        --------------------------------------
 *  00C8  |       Algorithm Usage bitmap       |  
 *        --------------------------------------
 *  00CC  | pre blk|pre dir|    Padding        |  
 *        --------------------------------------
 *  00D0  |       Journal UUID                 |  
 *        --------------------------------------
 *                      :
 *        --------------------------------------
 *  00DC  |        Journal UUID                |
 *        --------------------------------------
 *  00E0  |        Journal Inode               |
 *        --------------------------------------
 *  00E4  |        Journal Device              |
 *        --------------------------------------
 *  00E8  |        Last Orphan                 |
 *        --------------------------------------
 *  00EC  |        Hash Seed                   |
 *        --------------------------------------
 *                      :
 *        --------------------------------------
 *  00F8  |        Hash Seed                   |
 *        --------------------------------------
 *  00FC  | HashVer|HashChar|	 Hash Word       |
 *        --------------------------------------
 *  0100  |     Default Mount Options          |
 *        --------------------------------------
 *  0104  |     First meta block               |
 *        --------------------------------------
 *  0108  |       Padding                      |
 *        --------------------------------------
 *                      :
 *        --------------------------------------
 *  03FC  |       Padding                      |
 *        --------------------------------------
 *
 * </pre>
 */

public class Ext2SuperBlock implements SuperBlock
{
  protected long inodeCount;
  protected long blockCount;      
  protected long reservedBlockCount;      
  protected long freeBlocks;     
  protected long freeInodes;     
  protected long firstDataBlock;     
  protected long logBlockSize;   // note: blockSize = 1024 << logBlockSize;
  protected long logFragSize;    // note: fragSize = 1024 << logFragSize;  (Can be negative!)
  protected long blocksPerGroup;
  protected long fragsPerGroup;
  protected long inodesPerGroup;
  protected long mountTime;
  protected long writeTime;
  protected int mountCount;
  protected int maxMountCount;
  protected int magic;          // defined to be 0xef53
  protected int state;
  protected int errors;
  protected int minorRevLevel;
  protected long lastCheck;
  protected long checkInterval;
  protected long creatorOS;
  protected long revLevel;
  protected int defaultResUID;
  protected int defaultResGID;

  /* The following fields are for EXT2_DYNAMIC_REV superblocks only */

  protected long firstInode;
  protected int inodeSize;
  protected int groupNumber;  // group number of this superblock
  protected long featureCompat;
  protected long featureInCompat;
  protected long featureROCompat;
  protected byte[] uniqueID;  // uuid
  protected byte[] label;     // volume name
  protected byte[] lastMountPoint;
  protected long algorithmUsageBitmap;  // for compression
  protected byte preallocBlocks;
  protected byte preallocDirBlocks;
  protected byte[] journalUniqueID;
  protected long journalInode;
  protected long journalDev;
  protected long lastOrphan;
  protected int[] hashSeed;
  protected byte defaultHashVersion;
  protected byte reservedCharPad;
  protected int reservedWordPad;
  protected long defaultMountOptions;
  protected long firstMetaBlockGroup;

  /* Constants */
  public final static int EXT2_GOOD_OLD_REV = 0;
  public final static int EXT2_DYNAMIC_REV = 1;
  public final static int EXT2_GOOD_OLD_INODE_SIZE = 128;

  public final static int EXT2_MAX_SUPPORTED_REV = EXT2_DYNAMIC_REV;

  public final static int EXT2_FEATURE_COMPAT_DIR_PREALLOC =     0x0001;
  public final static int  EXT2_FEATURE_COMPAT_IMAGIC_INODES =   0x0002;
  public final static int  EXT2_FEATURE_COMPAT_HAS_JOURNAL =     0x0004;
  public final static int  EXT2_FEATURE_COMPAT_EXT_ATTR =        0x0008;
  public final static int  EXT2_FEATURE_COMPAT_RESIZE_INODE =    0x0010;
  public final static int  EXT2_FEATURE_COMPAT_DIR_INDEX =       0x0020;

  public final static int  EXT2_FEATURE_RO_COMPAT_SPARSE_SUPER = 0x0001;
  public final static int  EXT2_FEATURE_RO_COMPAT_LARGE_FILE =   0x0002;
  public final static int  EXT2_FEATURE_RO_COMPAT_BTREE_DIR =    0x0004;

  public final static int  EXT2_FEATURE_INCOMPAT_COMPRESSION =   0x0001;
  public final static int  EXT2_FEATURE_INCOMPAT_FILETYPE =      0x0002;
  public final static int  EXT2_FEATURE_INCOMPAT_RECOVER =       0x0004; /* Needs recovery */
  public final static int  EXT2_FEATURE_INCOMPAT_JOURNAL_DEV =   0x0008; /* Journal device */
  public final static int  EXT2_FEATURE_INCOMPAT_META_BG =       0x0010;
 
  public final static int EXT2_SUPER_MAGIC = 0xef53;

  public final static int EXT2_MIN_BLOCK_SIZE = 1024;
  public final static int EXT2_MAX_BLOCK_SIZE = 4096;

  public Ext2SuperBlock()
  {
  }

  public int getDataSize() { return 0x400; }

  public long getInodeCount() { return inodeCount; }
  public void setInodeCount(long aInodeCount) { inodeCount=aInodeCount; }

  public long getBlockCount() { return blockCount; }
  public void setBlockCount(long aBlockCount) { blockCount=aBlockCount; }

  public long getReservedBlockCount() { return reservedBlockCount; }
  public void setReservedBlockCount(long aReservedBlockCount) { reservedBlockCount=aReservedBlockCount; }

  public long getFreeBlocks() { return freeBlocks; }
  public void setFreeBlocks(long aFreeBlocks) { freeBlocks=aFreeBlocks; }

  public long getFreeInodes() { return freeInodes; }
  public void setFreeInodes(long aFreeInodes) { freeInodes=aFreeInodes; }

  public long getFirstDataBlock() { return firstDataBlock; }
  public void setFirstDataBlock(long aFirstDataBlock) { firstDataBlock=aFirstDataBlock; }

  public long getLogBlockSize() { return logBlockSize; }
  public void setLogBlockSize(long aLogBlockSize) { logBlockSize=aLogBlockSize; }

  /** Gets the block size in bytes */
  public long getBlockSize()
  {
    return 1024 << getLogBlockSize();
  }

  public long getLogFragSize() { return logFragSize; }
  public void setLogFragSize(long aLogFragSize) { logFragSize=aLogFragSize; }

  /** Gets the frag size in bytes */
  public long getFragSize()
  {
    return 1024 << getLogBlockSize();
  }

  public long getBlocksPerGroup() { return blocksPerGroup; }
  public void setBlocksPerGroup(long aBlocksPerGroup) { blocksPerGroup=aBlocksPerGroup; }

  public long getFragsPerGroup() { return fragsPerGroup; }
  public void setFragsPerGroup(long aFragsPerGroup) { fragsPerGroup=aFragsPerGroup; }

  public long getInodesPerGroup() { return inodesPerGroup; }
  public void setInodesPerGroup(long aInodesPerGroup) { inodesPerGroup=aInodesPerGroup; }

  public long getMountTime() { return mountTime; }
  public void setMountTime(long aMountTime) { mountTime=aMountTime; }

  public long getWriteTime() { return writeTime; }
  public void setWriteTime(long aWriteTime) { writeTime=aWriteTime; }

  public int getMountCount() { return mountCount; }
  public void setMountCount(int aMountCount) { mountCount=aMountCount; }

  public int getMaxMountCount() { return maxMountCount; }
  public void setMaxMountCount(int aMaxMountCount) { maxMountCount=aMaxMountCount; }

  public int getMagic() { return magic; }
  public void setMagic(int aMagic) { magic=aMagic; }

  public int getState() { return state; }
  public void setState(int aState) { state=aState; }

  public int getErrors() { return errors; }
  public void setErrors(int aErrors) { errors=aErrors; }

  public int getMinorRevLevel() { return minorRevLevel; }
  public void setMinorRevLevel(int aMinorRevLevel) { minorRevLevel=aMinorRevLevel; }

  public long getLastCheck() { return lastCheck; }
  public void setLastCheck(long aLastCheck) { lastCheck=aLastCheck; }

  public long getCheckInterval() { return checkInterval; }
  public void setCheckInterval(long aCheckInterval) { checkInterval=aCheckInterval; }

  public long getCreatorOS() { return creatorOS; }
  public void setCreatorOS(long aCreatorOS) { creatorOS=aCreatorOS; }

  public long getRevLevel() { return revLevel; }
  public void setRevLevel(long aRevLevel) { revLevel=aRevLevel; }

  public int getDefaultResUID() { return defaultResUID; }
  public void setDefaultResUID(int aDefaultResUID) { defaultResUID=aDefaultResUID; }

  public int getDefaultResGID() { return defaultResGID; }
  public void setDefaultResGID(int aDefaultResGID) { defaultResGID=aDefaultResGID; }

  public long getFirstInode() { return firstInode; }
  public void setFirstInode(long aFirstInode) { firstInode=aFirstInode; }

  /** 
   * Get the size of an Inode.  Typically this is 128 bytes, but in theory it can 
   * be some other size.
   */
  public int getInodeSize() 
  { 
    if (revLevel == EXT2_GOOD_OLD_REV)
      return Ext2Inode.EXT2_GOOD_OLD_INODE_SIZE;
    return inodeSize; 
  }
  public void setInodeSize(int aInodeSize) { inodeSize=aInodeSize; }

  public int getGroupNumber() { return groupNumber; }
  public void setGroupNumber(int aGroupNumber) { groupNumber=aGroupNumber; }

  public long getFeatureCompat() { return featureCompat; }
  public void setFeatureCompat(long aFeatureCompat) { featureCompat=aFeatureCompat; }

  public long getFeatureInCompat() { return featureInCompat; }
  public void setFeatureInCompat(long aFeatureInCompat) { featureInCompat=aFeatureInCompat; }

  public long getFeatureROCompat() { return featureROCompat; }
  public void setFeatureROCompat(long aFeatureROCompat) { featureROCompat=aFeatureROCompat; }

  public byte[] getUniqueID() { return uniqueID; }
  public void setUniqueID(byte[] aUniqueID) { uniqueID=aUniqueID; }

  public byte[] getLabel() { return label; }
  public void setLabel(byte[] aLabel) { label=aLabel; }

  public byte[] getLastMountPoint() { return lastMountPoint; }
  public void setLastMountPoint(byte[] aLastMountPoint) { lastMountPoint=aLastMountPoint; }

  public long getAlgorithmUsageBitmap() { return algorithmUsageBitmap; }
  public void setAlgorithmUsageBitmap(long aAlgorithmUsageBitmap) { algorithmUsageBitmap=aAlgorithmUsageBitmap; }

  public byte getPreallocBlocks() { return preallocBlocks; }
  public void setPreallocBlocks(byte aPreallocBlocks) { preallocBlocks=aPreallocBlocks; }

  public byte getPreallocDirBlocks() { return preallocDirBlocks; }
  public void setPreallocDirBlocks(byte aPreallocDirBlocks) { preallocDirBlocks=aPreallocDirBlocks; }

  public byte[] getJournalUniqueID() { return journalUniqueID; }
  public void setJournalUniqueID(byte[] aJournalUniqueID) { journalUniqueID=aJournalUniqueID; }

  public long getJournalInode() { return journalInode; }
  public void setJournalInode(long aJournalInode) { journalInode=aJournalInode; }

  public long getJournalDev() { return journalDev; }
  public void setJournalDev(long aJournalDev) { journalDev=aJournalDev; }

  public long getLastOrphan() { return lastOrphan; }
  public void setLastOrphan(long aLastOrphan) { lastOrphan=aLastOrphan; }

  public int[] getHashSeed() { return hashSeed; }
  public void setHashSeed(int[] aHashSeed) { hashSeed=aHashSeed; }

  public byte getDefaultHashVersion() { return defaultHashVersion; }
  public void setDefaultHashVersion(byte aDefaultHashVersion) { defaultHashVersion=aDefaultHashVersion; }

  public byte getReservedCharPad() { return reservedCharPad; }
  public void setReservedCharPad(byte aReservedCharPad) { reservedCharPad=aReservedCharPad; }

  public int getReservedWordPad() { return reservedWordPad; }
  public void setReservedWordPad(int aReservedWordPad) { reservedWordPad=aReservedWordPad; }

  public long getDefaultMountOptions() { return defaultMountOptions; }
  public void setDefaultMountOptions(long aDefaultMountOptions) { defaultMountOptions=aDefaultMountOptions; }

  public long getFirstMetaBlockGroup() { return firstMetaBlockGroup; }
  public void setFirstMetaBlockGroup(long aFirstMetaBlockGroup) { firstMetaBlockGroup=aFirstMetaBlockGroup; }


  /** Load from a buffer containint the superblock.  We assume that the superblock
      begins at the specified offset within the buffer */
  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    /* Make sure we have enough buffer to do this.  We need 1024 bytes to read the
       superblock */

    if ( buffer.length < (offset+getDataSize()))
      throw new FsException("Insufficinet buffer to load super-block");

    setInodeCount(FsUtils.getU32(buffer, offset+0x0));
    setBlockCount(FsUtils.getU32(buffer, offset+0x4));
    setReservedBlockCount(FsUtils.getU32(buffer, offset+0x8));
    setFreeBlocks(FsUtils.getU32(buffer, offset+0xC));
    setFreeInodes(FsUtils.getU32(buffer, offset+0x10));
    setFirstDataBlock(FsUtils.getU32(buffer, offset+0x14));
    setLogBlockSize(FsUtils.getU32(buffer, offset+0x18));
    setLogFragSize(FsUtils.getS32(buffer, offset+0x1C));
    setBlocksPerGroup(FsUtils.getU32(buffer, offset+0x20));
    setFragsPerGroup(FsUtils.getU32(buffer, offset+0x24));
    setInodesPerGroup(FsUtils.getU32(buffer, offset+0x28));
    setMountTime(FsUtils.getU32(buffer, offset+0x2C));
    setWriteTime(FsUtils.getU32(buffer, offset+0x30));
    setMountCount(FsUtils.getU16(buffer, offset+0x34));
    setMaxMountCount(FsUtils.getS16(buffer, offset+0x36));
    setMagic(FsUtils.getU16(buffer, offset+0x38));
    setState(FsUtils.getU16(buffer, offset+0x3A));
    setErrors(FsUtils.getU16(buffer, offset+0x3C));
    setMinorRevLevel(FsUtils.getU16(buffer, offset+0x3E));
    setLastCheck(FsUtils.getU32(buffer, offset+0x40));
    setCheckInterval(FsUtils.getU32(buffer, offset+0x44));
    setCreatorOS(FsUtils.getU32(buffer, offset+0x48));
    setRevLevel(FsUtils.getU32(buffer, offset+0x4C));
    setDefaultResUID(FsUtils.getU16(buffer, offset+0x50));
    setDefaultResGID(FsUtils.getU16(buffer, offset+0x52));

    /* EXT2_DYNAMIC_REV starts here */
    if (revLevel >= EXT2_DYNAMIC_REV)
    {
      setFirstInode(FsUtils.getU32(buffer, offset+0x54));
      setInodeSize(FsUtils.getU16(buffer, offset+0x58));
      setGroupNumber(FsUtils.getU16(buffer, offset+0x5A));
      setFeatureCompat(FsUtils.getU32(buffer, offset+0x5C));
      setFeatureInCompat(FsUtils.getU32(buffer, offset+0x60));
      setFeatureROCompat(FsUtils.getU32(buffer, offset+0x64));

      uniqueID = new byte[16];
      System.arraycopy(buffer, offset+0x68, uniqueID, 0, 16);

      label = new byte[16];
      System.arraycopy(buffer, offset+0x78, label, 0, 16);

      lastMountPoint = new byte[64];
      System.arraycopy(buffer, offset+0x88, lastMountPoint, 0, 64);

      setAlgorithmUsageBitmap(FsUtils.getU32(buffer, offset+0xC8));
      setPreallocBlocks(buffer[offset+0xCC]);
      setPreallocDirBlocks(buffer[offset+0xCD]);

      journalUniqueID = new byte[16];
      System.arraycopy(buffer, offset+0xD0, journalUniqueID, 0, 16);

      setJournalInode(FsUtils.getU32(buffer, offset+0xE0));
      setJournalDev(FsUtils.getU32(buffer, offset+0xE4));
      setLastOrphan(FsUtils.getU32(buffer, offset+0xE8));

      /* Ok, we don't need to process the remaining data */
    }

    /* Check the magic field to make sure we have a valid superblock */
    try
    {
      if (magic != EXT2_SUPER_MAGIC)
        throw new FsException("Invalid magic.");

      if (revLevel < EXT2_GOOD_OLD_REV || revLevel > EXT2_DYNAMIC_REV)
        throw new FsException("Unsupported Version "+revLevel);

      if (revLevel >= EXT2_DYNAMIC_REV)
      {
        if ((featureInCompat & EXT2_FEATURE_INCOMPAT_COMPRESSION) != 0)
          throw new FsException("Compression not supported");

        if ((featureInCompat & EXT2_FEATURE_INCOMPAT_META_BG) != 0)
          throw new FsException("Meta Block not supported");
      }
    }
    catch (Exception ex)
    {
      FsException fex = new FsException("Error processing superblock.");
      fex.initCause(ex);
      throw fex;
    }
  }
}
