/*
 * $Id: FatDirEntry.java,v 1.8 2004/04/24 21:32:04 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FatDirEntry.java,v $
 * Revision 1.8  2004/04/24 21:32:04  wrossi
 * Support for lowercase short names.
 *
 * Revision 1.7  2004/04/18 22:11:54  wrossi
 * wip
 *
 * Revision 1.6  2004/04/18 20:50:49  wrossi
 * Added field for long filename.
 *
 * Revision 1.5  2004/04/16 20:45:25  wrossi
 * Hash function is working.
 *
 * Revision 1.4  2004/04/16 20:14:49  wrossi
 * Added method to read from buffer, compute hash, getShortName
 *
 * Revision 1.3  2004/04/15 20:39:10  wrossi
 * Filled in Inode interface.
 *
 * Revision 1.2  2004/04/15 11:01:18  wrossi
 * Work in progress.
 *
 * Revision 1.1  2004/04/14 20:42:30  wrossi
 * Added some calculations.
 *
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.Inode;
import rossi.fstools.fs.InodePtr;


/** 
 * Representation of a FAT directory entry.
 *
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |           Name                     |
 *        --------------------------------------
 *  0004  |           Name                     |
 *        --------------------------------------
 *  0008  |           Ext             |Attr    |
 *        --------------------------------------
 *  000C  | lcase | ctimems  | CTime           |
 *        --------------------------------------
 *  0010  |   CDate          |   ADate         |
 *        --------------------------------------
 *  0014  |   Start High     |   Time          |
 *        --------------------------------------
 *  0018  |   Date           |   Start         |
 *        --------------------------------------
 *  001C  |           Size                     |
 *        --------------------------------------
 *
 * </pre>
 */

public class FatDirEntry implements DiskStructure, Inode, InodePtr
{
  /** First 8 bytes of the short name. */
  protected byte[] name; 
  /** Three byte extension of the short name. */
  protected byte[] ext;
  /** Attributes. */
  protected byte attr;
  /** Lower case indicator for Windows NT. */
  protected byte lcase;
  /** Creation time milliseconds. */
  protected int ctimems;
  /** Creation time in msdos format. */
  protected int ctime;
  /** Creation date in msdos format. */
  protected int cdate;
  /** Access date in msdos format. */
  protected int adate;
  /** High 16 bits of the starting cluster for FAT32. */
  protected int starthi;
  /** Modification time. */
  protected int time;
  /** Modification date. */
  protected int date;
  /** Low 16 bits of starting cluster. */
  protected int start;
  /** File size. */
  protected long size;
  /** Hash of the short name. */
  protected byte hashValue;
  /** Long filename. */
  protected String longName;

  /** Linear day numbers of the respective 1sts in non-leap years.  Borrowed from Linux kernel fs/fat/misc.c */
  private final static int day_n[] = { 0,31,59,90,120,151,181,212,243,273,304,334,0,0,0,0 };

  /** No attribute bits. */
  public final static int ATTR_NONE    = 0;
  /** read-only attribute. */
  public final static int ATTR_RO      = 1;
  /** Hidden attribute. */
  public final static int ATTR_HIDDEN  = 2;
  /** System attribute. */
  public final static int ATTR_SYS     = 4;
  /** Volume attribute. */
  public final static int ATTR_VOLUME  = 8;
  /** Directory attribute. */
  public final static int ATTR_DIR     = 16;
  /** Arhive attribute. */
  public final static int ATTR_ARCH    = 32;
  /** Long filename indicator for VFAT.  This combination of attributes indicates the
   *  the entry is part of a long filename. 
   */
  public final static int ATTR_LFN     = 0xf;

  /** Base of shortname is in lowercase. */
  public final static int CASE_LOWER_BASE = 0x8;

  /** Ext of shortname is in lowercase. */
  public final static int CASE_LOWER_EXT = 0x10;

  /** Default constructor. */
  public FatDirEntry()
  {
  }

  public int getDataSize() { return 0x20; }

  /**
   * Get the short name prefix.
   * @return prefix.
   */
  public byte[] getNamePrefix() { return name; }
  public void setNamePrefix(byte[] aName) { name=aName; }

  /**
   * Get the short name extension.
   * @return extension
   */
  public byte[] getExt() { return ext; }
  public void setExt(byte[] aExt) { ext=aExt; }

  /**
   * Get the attributes for this file.  This gets mapped into file mode.
   * @return attributes
   */
  public byte getAttr() { return attr; }
  public void setAttr(byte aAttr) { attr=aAttr; }

  /**
   * Get the lowercase indicator.
   * @return lcase
   */
  public byte getLcase() { return lcase; }
  public void setLcase(byte aLcase) { lcase=aLcase; }

  /**
   * Get the creation time milliseconds.
   * @return int
   */
  public int getCtimems() { return ctimems; }
  public void setCtimems(int aCtimems) { ctimems=aCtimems; }

  /**
   * Get the creation time of day.
   * @return int
   */
  public int getCtime() { return ctime; }
  public void setCtime(int aCtime) { ctime=aCtime; }

  /**
   * Get the creation date.
   * @return int
   */
  public int getCdate() { return cdate; }
  public void setCdate(int aCdate) { cdate=aCdate; }

  /**
   * Get the last access date.
   * @return int
   */
  public int getAdate() { return adate; }
  public void setAdate(int aAdate) { adate=aAdate; }

  /**
   * Get the high 16 bits of the starting cluster for FAT32 only.
   * @return int
   */
  public int getStartHi() { return starthi; }
  public void setStartHi(int aStarthi) { starthi=aStarthi; }

  /**
   * Get the last modified time of day.
   * @return int
   */
  public int getTime() { return time; }
  public void setTime(int aTime) { time=aTime; }

  /**
   * Get the last modified date.
   * @return int
   */
  public int getDate() { return date; }
  public void setDate(int aDate) { date=aDate; }

  /**
   * Get the lower 16 bits of the starting cluster.
   * @return int
   */
  public int getStart() { return start; }
  public void setStart(int aStart) { start=aStart; }

  public long getSize() { return size; }
  public void setSize(long aSize) { size=aSize; }

  /* Inode interface */

  /** Encode the FAT attributes into the unix mode field. <p>
   *  The encoding goes as follows.   This is somewhat arbitrary. <b>
   *  <table>
   *  <tr><td> If attribute </td> <td> State </td> <td> Add mode bits </td> </tr>
   *  <tr><td> ATTR_DIR </td> <td> set </td> <td> Inode.DIRECORY_MODE </td> </tr>
   *  <tr><td> ATTR_DIR </td> <td> clear </td> <td> Inode.FILE_MODE </td> </tr>
   *  <tr><td> ATTR_RO </td> <td> clear </td> <td> ugo+w </td> </tr>
   *  <tr><td> ATTR_HIDDEN </td> <td> clear </td> <td> u+r </td> </tr>
   *  <tr><td> ATTR_SYS </td> <td> clear </td> <td> g+r </td> </tr>
   *  <tr><td> ATTR_ARCHIVE </td> <td> clear </td> <td> o+r </td> </tr>
   *  <tr><td> ATTR_VOLUME </td> <td> set </td> <td> o+x </td> </tr>
   *  </table>
   *  @return encoded attributes
   */
  public int getMode()
  {
    int mode = 0;
    int attr = getAttr();

    /* There are only files and directories in FAT. */
    if ( (attr & ATTR_DIR) != 0)
      mode |= Inode.DIRECTORY_MODE;
    else
      mode |= Inode.FILE_MODE;

    if ( (attr & ATTR_RO) == 0)  // if not readonly
      mode |= 0222;

    if ( (attr & ATTR_HIDDEN) == 0)  // if not hidden
      mode |= 0400;

    if ( (attr & ATTR_SYS) == 0)  // if not system
      mode |= 0040;

    if ( (attr & ATTR_ARCH) == 0)  // if not archive
      mode |= 0004;

    if ( (attr & ATTR_VOLUME) != 0) // if volume lable
      mode |= 0001;

    return mode;
  }

  
  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aMode the mode.
   */
  public void setMode(int aMode)
  {
    throw new UnsupportedOperationException("Not supported.");
  }

  public long getNumLinks()
  {
    /* Fat doesnt support hard links. */
    return 1;
  }

  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aNumLinks number of links.
   */
  public void setNumLinks(long aNumLinks)
  {
    throw new UnsupportedOperationException("Not supported.");
  }

  public long getUserId()
  {
    return 0;
  }

  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aUserId  user id
   */
  public void setUserId(long aUserId)
  {
    throw new UnsupportedOperationException("Not supported.");
  }

  public long getGroupId()
  {
    return 0;
  }

  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aGroupId  group id
   */
  public void setGroupId(long aGroupId)
  {
    throw new UnsupportedOperationException("Not supported.");
  }


  public long getAccessTime()
  {
    return date_dos2unix(0, getAdate());
  }

  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aAccessTime  accessTime
   */
  public void setAccessTime(long aAccessTime)
  {
    throw new UnsupportedOperationException("Not supported.");
  }
 

  public long getModifyTime()
  {
    return date_dos2unix(getTime(), getDate());
  }

  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aModifyTime  modifyTime
   */
  public void setModifyTime(long aModifyTime)
  {
    throw new UnsupportedOperationException("Not supported.");
  }

  public long getChangedTime()
  {
    return date_dos2unix(getCtime(), getCdate());
  }

  /**
   * Not supported.  Will throw UnsupportedOperationException if called.
   * @param aChangedTime  changedTime
   */
  public void setChangedTime(long aChangedTime)
  {
    throw new UnsupportedOperationException("Not supported.");
  }


  /** Convert a MS-DOS time/date pair to a UNIX date (seconds since 1 1 70).  Borrowed from the Linux kernel
   *  sources. fs/fat/misc.c 
   *  @param time msdos time
   *  @param date msdos date
   *  @return unix time
   */
  protected int date_dos2unix(int time, int date)
  {
    int month,year,secs;

    /* first subtract and mask after that... Otherwise, if
       date == 0, bad things happen */
    month = ((date >> 5) - 1) & 15;
    year = date >> 9;
    secs = (time & 31)*2+60*((time >> 5) & 63)+(time >> 11)*3600+86400*
        ((date & 31)-1+day_n[month]+(year/4)+year*365-((year & 3) == 0 &&
        month < 2 ? 1 : 0)+3653);
        /* days since 1.1.70 plus 80's leap day */
    //secs += sys_tz.tz_minuteswest*60;
    return secs;
  }


  /** Get the hash value of the short name. 
   * return hash value
   */
  public byte getHashValue() { return hashValue; }

  /** 
   * Get the long filename.  Returns null if no long filename.
   * @return long file name.
   */
  public String getLongName() { return longName; }
  public void setLongName(String aLongName) { longName=aLongName; }

  /** Return the long name if it exists, short name otherwise.
   * @return filename
   */
  public String getName()
  {
    String name = getLongName();
    if (name == null)
      name = getShortName();

    return name;
  }

  /** Compute the hash of the short name fields.  This is referenced by long file
   *  name entries. */
  private void computeHash()
  {
    hashValue = 0;

    for (int i=0; i<8; i++)
    {
      hashValue = (byte) ((((hashValue&0xfe) >> 1) | ( (hashValue & 0x1) << 7)) & 0xff);
      hashValue += name[i];
    }

    for (int i=0; i<3; i++)
    {
      hashValue = (byte) ((((hashValue&0xfe) >> 1) | ( (hashValue & 0x1) << 7)) & 0xff);
      hashValue += ext[i];
    }
  }

  /** Get the short filename. 
   *  @return short filename.
   */ 
  public String getShortName()
  {
    String shortName = null;
    String trimmedExt = null;

    try
    {
      shortName = new String(name, "ISO-8859-1").trim();
      trimmedExt = new String(ext, "ISO-8859-1").trim();
    }
    catch (Exception ex)
    {
    }

    if ( (getLcase() & CASE_LOWER_BASE) != 0)
      shortName = shortName.toLowerCase();

    if ( (getLcase() & CASE_LOWER_EXT) != 0)
      trimmedExt = trimmedExt.toLowerCase();

    if (trimmedExt.length() > 0)
      shortName = shortName+"."+trimmedExt;

    return shortName;
  }

  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    name = new byte[8];
    System.arraycopy(buffer, offset+0x0, name, 0, 8);

    ext = new byte[3];
    System.arraycopy(buffer, offset+0x8, ext, 0, 3);

    attr = buffer[0xb];
    lcase = buffer[0xc];
    ctimems = (int) buffer[0xd];
 
    setCtime(FsUtils.getU16(buffer, offset+0xe));
    setCdate(FsUtils.getU16(buffer, offset+0x10));
    setAdate(FsUtils.getU16(buffer, offset+0x12));
    setStartHi(FsUtils.getU16(buffer, offset+0x14));
    setTime(FsUtils.getU16(buffer, offset+0x16));
    setDate(FsUtils.getU16(buffer, offset+0x18));
    setStart(FsUtils.getU16(buffer, offset+0x1a));
    setSize(FsUtils.getU32(buffer, offset+0x1c));

    computeHash();
  }

}
