/*
 * $Id: FatDirectory.java,v 1.1 2004/04/21 20:20:13 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FatDirectory.java,v $
 * Revision 1.1  2004/04/21 20:20:13  wrossi
 * Directory implementation.
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.Directory;
import rossi.fstools.fs.File;
import rossi.fstools.fs.FsException;

import java.util.Map;
import java.util.HashMap;

import java.io.InputStream;
import java.io.IOException;

/** 
 * Represents an FAT directory. <p>
 * Direcoties on FAT are stored as files containing an array of DirEntries. <p>
 */
public class FatDirectory implements Directory
{
  private Map dirMap;
  private boolean atEOF = false;

  /**
   * Create an FatDirectory given a File.
   *
   * @param file   a File containting directory data.
   * @exception IOException
   * @exception FsException
   */
  public FatDirectory(File file) throws IOException, FsException
  {
    dirMap = new HashMap();
    load(file);
  }

  private void load(File file) throws IOException, FsException
  {
    InputStream in = file.getData();
    int direntrySize = new FatDirEntry().getDataSize();
    byte[] buffer = new byte[direntrySize]; 
    VFatProcessor vfatsm = new VFatProcessor();

    while (true)
    {
      int bytesRead = fillBuffer(in, buffer, 0);
      if (bytesRead < direntrySize)
        break;

      if (buffer[0] == 0x0)
        break;

      if ((buffer[0] & 0xff) == 0xe5)
        continue;

      if (buffer[0xb] == 0xf) // lfn entry
      {
        VFatDirEntry lfn = new VFatDirEntry();
        lfn.loadFromBuffer(buffer, 0);
        vfatsm.processLFNEntry(lfn);
      }
      else
      {
        FatDirEntry dir = new FatDirEntry();
        dir.loadFromBuffer(buffer, 0);
        vfatsm.processDirEntry(dir);

        dirMap.put(dir.getName(), dir);
      }
    }

    in.close();
  }

  /** Read in to fill the buffer.  */
  private int fillBuffer(InputStream in, byte[] buffer, int startoffset) throws IOException
  {
    int totalBytesRead = startoffset;
    int bytesToRead = buffer.length;

    if (atEOF)
      return 0;

    while (totalBytesRead < bytesToRead)
    {
      int bytesRead = in.read(buffer, totalBytesRead, bytesToRead-totalBytesRead);
      if (bytesRead == -1)
      {
        atEOF = true;
        return (totalBytesRead - startoffset);
      }
      totalBytesRead += bytesRead;
    }

    return (totalBytesRead - startoffset);
  }

  public Map getInodePtrMap()
  {
    return dirMap;
  }
}
