/*
 * $Id: FatFile.java,v 1.3 2004/04/22 23:36:06 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FatFile.java,v $
 * Revision 1.3  2004/04/22 23:36:06  wrossi
 * Bug fixes.
 *
 * Revision 1.2  2004/04/22 11:40:12  wrossi
 * Can extract a file from the root dir on FAT 12 now.
 *
 * Revision 1.1  2004/04/20 20:41:49  wrossi
 * File for the FAT file system
 *
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.File;
import rossi.fstools.io.BlockPointer;
import rossi.fstools.io.BlockReader;
import rossi.fstools.io.BlockPointerInputStream;

import java.util.List;
import java.util.ArrayList;

/**
 * Representation of a file.
 */

public class FatFile implements File
{
  protected List blkPointerList;
  protected List slackList;
  protected long size;
  protected long sizeRemaining;
  protected BlockReader reader;
  protected FatSuperBlock sb;

  /**
   * Create a file.
   *
   * @param sb   a FatSuperBlock
   * @param fileSize   The length of the file from its Inode.
   * @param reader   a BlockReader to read the disk.
   */
  public FatFile(FatSuperBlock sb, long fileSize, BlockReader reader)
  {
    blkPointerList = new ArrayList();
    slackList = new ArrayList();
    this.size = fileSize;
    this.reader = reader;
    this.sb = sb;
    sizeRemaining = size;
  }

  /**
   * Adds the specified cluster to the file.
   * @param cluster number of the block to be added.
   */
  public void add(int cluster)
  {
    BlockPointer bp;
    BlockPointer lastBp = null;
    int blocknum;
    int sectorSize;
    int clusterSize;
    int curSize;
    int lastend = 0;
    int lastendoffset = 0;
 
    sectorSize = (int) sb.getSectorSize();
    clusterSize = sectorSize * sb.getSectorsPerCluster();

    curSize = (sizeRemaining < clusterSize) ? (int)sizeRemaining : clusterSize;

    /* Compute the blocknum (sector address) for the given cluster */
    blocknum = sb.getFirstDataSector() + (cluster - 2) * sb.getSectorsPerCluster();

    if (blkPointerList.size() != 0)
    {
      lastBp = (BlockPointer) blkPointerList.get(blkPointerList.size() - 1);

      lastend = lastBp.getBlockNum() + ((lastBp.getOffset()+lastBp.getLength()) / lastBp.getBlockSize());
      lastendoffset = ((lastBp.getOffset()+lastBp.getLength()) % lastBp.getBlockSize());
    }

    if (lastBp != null && lastend == blocknum && lastendoffset == 0)
    {
      /* Add this block onto the last block pointer since it is a consecutive block */
      lastBp.setLength(lastBp.getLength() + curSize);
    }
    else
    {
      /* Create a new block pointer */

      bp = new BlockPointer();
      bp.setBlockNum((int) blocknum);
      bp.setOffset(0);
      bp.setLength(curSize);
      bp.setBlockSize(sectorSize);
      blkPointerList.add(bp);
    }

    /* record slack */
    if (curSize != clusterSize)
    {
      bp = new BlockPointer();
      bp.setBlockNum((int) blocknum + curSize/sectorSize);
      bp.setOffset(curSize%sectorSize);
      bp.setLength(clusterSize - curSize);
      bp.setBlockSize(sectorSize);
      slackList.add(bp);
    }

    sizeRemaining = sizeRemaining - curSize;
  }

  public BlockPointerInputStream getData()
  {
    return new BlockPointerInputStream(reader, blkPointerList);
  }

  public BlockPointerInputStream getSlack()
  {
    return new BlockPointerInputStream(reader, slackList);
  }
}
