/*
 * $Id: FatFileSystem.java,v 1.5 2004/04/24 21:58:16 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FatFileSystem.java,v $
 * Revision 1.5  2004/04/24 21:58:16  wrossi
 * Handle special case of zero length files.
 *
 * Revision 1.4  2004/04/22 23:36:06  wrossi
 * Bug fixes.
 *
 * Revision 1.3  2004/04/21 23:22:02  wrossi
 * Make root inode a directory, and avoid infinite loop in getClusters()/getFatBits()
 *
 * Revision 1.2  2004/04/21 22:20:10  wrossi
 * It compiles at least.
 *
 * Revision 1.1  2004/04/21 22:03:40  wrossi
 * Beginnings of the fat filesystem
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.FileSystem;
import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.InodePtr;
import rossi.fstools.fs.Inode;
import rossi.fstools.fs.File;
import rossi.fstools.fs.Directory;
import rossi.fstools.fs.FsObject;
import rossi.fstools.fs.FsException;

import rossi.fstools.io.BlockReader;

import java.io.IOException;

/** 
 * Implementation of the FAT 12/16/32 file systems.
 */
public class FatFileSystem implements FileSystem
{
  private BlockReader blockReader;
  private FatSuperBlock sb;
  private FatDirEntry rootDirEntry;
  private FatTable fatTable;
  private File rootDirFile;

  public SuperBlock open(BlockReader br) throws FsException, IOException
  {
    blockReader = br;

    br.setBlockSize(512);
    sb = new FatSuperBlock();

    sb.loadFromBuffer(br.getBlock(0), 0);

    br.setBlockSize((int) sb.getSectorSize());

    fatTable = new FatTable(sb, br);

    createRootInode();    // Create a fake inode for the root dir
    createRootDirFile();  // Create the file containing the root dir

    return sb;
  }

  private void createRootInode()
  {
    byte[] nameprefix, ext;

    nameprefix = new byte[8];
    ext = new byte[3];

    for (int i=0;i<nameprefix.length; i++)
      nameprefix[i] = 0x20;
    for (int i=0;i<ext.length; i++)
      ext[i] = 0x20;

    rootDirEntry = new FatDirEntry();
    rootDirEntry.setLongName("");
    rootDirEntry.setAttr((byte)FatDirEntry.ATTR_DIR);

    rootDirEntry.setStart(0);
    rootDirEntry.setStartHi(0);
  }

  private void createRootDirFile() throws FsException, IOException
  {
    if (sb.getFatBits() == 32)
    {
      rootDirFile = createFatFile((int)sb.getRootCluster(), Long.MAX_VALUE);
    }
    else  // Fat 12/16 case
    {
      rootDirFile = new FatRootDirFile(sb, blockReader);
    }
  }

  public void close() throws IOException
  {
    blockReader.close();
  }

  public InodePtr getRootDir()
  {
    return rootDirEntry;
  }

  /** Gets an inode.  In this case the ptr is the inode. */
  public Inode getInode(InodePtr ptr) throws FsException, IOException
  {
    return (Inode) ptr;
  }

  /** 
   *  Creates a file from an inode
   *
   *  @param inode the inode
   *  @return the file
   *  @exception IOException in case of an IO error
   *  @exception FSException in case of a FS format error
   */
  public File createFatFile(FatDirEntry inode) throws FsException, IOException
  {
    long fileSize;
    int startCluster;

    startCluster = inode.getStart();
    if (sb.getFatBits() == 32)
      startCluster = startCluster | (inode.getStartHi() << 16);

    fileSize = inode.getSize();

    /* Special case.  A request for cluster zero could mean a zero length file, or 
       it could mean the root directory.  The only way to resolve this is to look at the
       mode of the file.  
   
       For a directory, return the root directory.  For a file, return a new but 
       empty File object.  */
    if (startCluster == 0)
    {
      if ((inode.getMode() & Inode.FILETYPE_MASK) == Inode.DIRECTORY_MODE)
        return rootDirFile;
      else
        return new FatFile(sb, 0, blockReader);
    }

    /* Special case.  A request for a directory gets max filesize */
    if ((inode.getMode() & Inode.FILETYPE_MASK) == Inode.DIRECTORY_MODE)
      fileSize = Long.MAX_VALUE;

    return createFatFile(startCluster, fileSize);
  }


  private File createFatFile(int startCluster, long fileSize) throws FsException, IOException
  {
    FatFile file;
    int cluster = startCluster;

    file = new FatFile(sb, fileSize, blockReader);

    do
    {
      file.add(cluster);
      cluster = fatTable.lookup(cluster);
    } while (!fatTable.isEOF(cluster) && !fatTable.isBad(cluster));

    return file;
  }

  /**
   * Create a directory.
   * @param inode   a Ext2Inode
   * @return Directory
   * @exception FsException
   * @exception IOException
   */
  public Directory createFatDirectory(FatDirEntry inode) throws FsException, IOException
  {
    return new FatDirectory(createFatFile(inode));
  }

  public FsObject getObject(Inode inode) throws FsException, IOException
  {
    int fileType;

    fileType = inode.getMode() & Inode.FILETYPE_MASK;

    switch (fileType)
    {
      case Inode.DIRECTORY_MODE:  return createFatDirectory((FatDirEntry)inode);
      case Inode.FILE_MODE: return createFatFile((FatDirEntry)inode);    
    }

    throw new FsException("Unsupported file type "+fileType);
  }
}
