/*
 * $Id: FatSuperBlock.java,v 1.11 2004/04/22 23:36:06 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FatSuperBlock.java,v $
 * Revision 1.11  2004/04/22 23:36:06  wrossi
 * Bug fixes.
 *
 * Revision 1.10  2004/04/22 11:40:12  wrossi
 * Can extract a file from the root dir on FAT 12 now.
 *
 * Revision 1.9  2004/04/21 23:22:02  wrossi
 * Make root inode a directory, and avoid infinite loop in getClusters()/getFatBits()
 *
 * Revision 1.8  2004/04/20 20:42:56  wrossi
 * Getting FatTable to work.
 *
 * Revision 1.7  2004/04/19 20:12:56  wrossi
 * Fix calculation of number of clusters.
 *
 * Revision 1.6  2004/04/19 11:02:58  wrossi
 * Don't assume sectorSize = 512 bytes.
 *
 * Revision 1.5  2004/04/19 11:00:32  wrossi
 * Added tests to make sure superblock is valid.
 *
 * Revision 1.4  2004/04/14 20:42:30  wrossi
 * Added some calculations.
 *
 * Revision 1.3  2004/04/14 11:39:59  wrossi
 * Added getNumSectors() API
 *
 * Revision 1.2  2004/04/13 20:12:12  wrossi
 * Added javadoc, wrote loadFromBuffer()
 *
 * Revision 1.1  2004/04/13 11:08:13  wrossi
 * Start of the FAT file system
 *
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;

/** 
 * Representation of FAT superblock.
 *
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |    Bootstrap               | SysID |
 *        --------------------------------------
 *  0004  |         System ID                  |
 *        --------------------------------------
 *  0008  |         System ID          |SectSze|
 *        --------------------------------------
 *  000C  |SectSze |SecPerClus| Reserved Sects |
 *        --------------------------------------
 *  0010  |NumFats | Root Dir Entries  |sectors|
 *        --------------------------------------
 *  0014  |sectors | Media    | Fat Length     |
 *        --------------------------------------
 *  0018  |Sectors per Track  | Heads          |
 *        --------------------------------------
 *  001C  |        Hidden Sectors              |
 *        --------------------------------------
 *  0020  |        Total Sectors               |
 *        --------------------------------------
 *  0024  |        Fat32 length                |
 *        --------------------------------------
 *  0028  | Flags             |  Version       |
 *        --------------------------------------
 *  002C  |        Root cluster                |
 *        --------------------------------------
 *  0030  | Info Sector       |  backup boot   |
 *        --------------------------------------
 *  0034  |        Reserved                    |
 *        --------------------------------------
 *
 * </pre>
 */

public class FatSuperBlock implements SuperBlock
{
  protected byte[] bootStrap;
  protected String systemId;
  protected int sectorSize;
  protected int sectorsPerCluster;
  protected int reservedSectors;
  protected int numFATs;
  protected int rootDirEntries;
  protected int sectors;   // 16 bit count of sectors for FAT12 and very small FAT16
  protected byte media;  
  protected int fatLength; // sectors per fat
  protected int sectorsPerTrack;
  protected int heads;
  protected long hiddenSectors;
  protected long totalSectors;  // use this if sectors == 0.  32 bit value

  /* The following fields are valid for FAT32 only */

  protected long fat32Length;   // Length of the FAT32 table.
  protected int flags;
  protected int version;
  protected long rootCluster;
  protected int infoSector;
  protected int backupBootSector;

  /** Max number of clusters for FAT12. */
  public final static int MAX_FAT12 = 0xff4;

  /** Max number of clusters for FAT16. */
  public final static int MAX_FAT16 = 0xfff4;

  /** Max number of clusters for FAT32. */
  public final static int MAX_FAT32 = 0xffffff4;

  /** Size of a directory entry. */
  private final static int DIRENTRY_SIZE = (new FatDirEntry().getDataSize());

  /** Return the first 3 bytes of the filesystem.  This typically contains a 
   *  jump instruction to the bootstrap loader. 
   *  @return bootStrap instruction.
   */
  public byte[] getBootStrap() { return bootStrap; }
  public void setBootStrap(byte[] aBootStrap) { bootStrap=aBootStrap; }

  /**
   * Get the system identifier.
   * @return system id
   */
  public String getSystemId() { return systemId; }
  public void setSystemId(String aSystemId) { systemId=aSystemId; }

  /**
   * Sector size.  Should be 512 bytes.
   * @return sector size.
   */
  public int getSectorSize() { return sectorSize; }
  public void setSectorSize(int aSectorSize) { sectorSize=aSectorSize; }

  /**
   * Get number of sectors per cluster.  A cluster is the minimum allocation unit
   * for files.
   */
  public int getSectorsPerCluster() { return sectorsPerCluster; }
  public void setSectorsPerCluster(int aSectorsPerCluster) { sectorsPerCluster=aSectorsPerCluster; }

  /**
   * Get the number of reserved sectors.  The FAT tables begin this many 
   * sectors into the filesystem.
   * @return reservedSectors.
   */
  public int getReservedSectors() { return reservedSectors; }
  public void setReservedSectors(int aReservedSectors) { reservedSectors=aReservedSectors; }

  /**
   * Get the number of FAT tables.  Usually 2.
   * @return numFATs.
   */
  public int getNumFATs() { return numFATs; }
  public void setNumFATs(int aNumFATs) { numFATs=aNumFATs; }

  /**
   * Get the number of root directory entries.  This is only valid in FAT16 and FAT12, it 
   * will be zero in FAT32.
   */
  public int getRootDirEntries() { return rootDirEntries; }
  public void setRootDirEntries(int aRootDirEntries) { rootDirEntries=aRootDirEntries; }

  /**
   * Get the count of sectors in the FS.  Note that this is a 16 bit number and will be
   * set to zero if the FS contains more than 65535 sectors.  Unused in FAT32.
   * @see #getTotalSectors()
   * @return numSectors.
   */
  public int getSectors() { return sectors; }
  public void setSectors(int aSectors) { sectors=aSectors; }

  /**
   * Media descriptor byte.
   * @return media
   */
  public byte getMedia() { return media; }
  public void setMedia(byte aMedia) { media=aMedia; }

  /**
   * FAT length in sectors.  Not used in FAT32.
   * @see #getFat32Length()
   * @return fat length
   */
  public int getFatLength() { return fatLength; }
  public void setFatLength(int aFatLength) { fatLength=aFatLength; }

  /**
   * Number of sectors per track.
   * @return sectors per track.
   */
  public int getSectorsPerTrack() { return sectorsPerTrack; }
  public void setSectorsPerTrack(int aSectorsPerTrack) { sectorsPerTrack=aSectorsPerTrack; }

  /**
   * Number of heads.
   * @return heads
   */
  public int getHeads() { return heads; }
  public void setHeads(int aHeads) { heads=aHeads; }

  /**
   * Number of hidden sectors.   Unused.
   * @return hidden
   */
  public long getHiddenSectors() { return hiddenSectors; }
  public void setHiddenSectors(long aHiddenSectors) { hiddenSectors=aHiddenSectors; }

  /**
   * Total sectors in the filesystem.  This 32 bit value is only used if 
   * the 16 bit sectors value is zero.  
   * @see #getSectors()
   * @return long
   */
  public long getTotalSectors() { return totalSectors; }
  public void setTotalSectors(long aTotalSectors) { totalSectors=aTotalSectors; }

  /**
   * Get the length of the FAT in sectors for a FAT32 system.  This overrides getFatLength()
   * that is used in FAT12/16 systems.
   * @see #getFatLength()
   * @return fat length in sectors;
   */
  public long getFat32Length() { return fat32Length; }
  public void setFat32Length(long aFat32Length) { fat32Length=aFat32Length; }

  /**
   * Not sure what this is for.  Unused here.  Only valid for FAT32.
   * @return flags.
   */
  public int getFlags() { return flags; }
  public void setFlags(int aFlags) { flags=aFlags; }

  /**
   * Get the filesystem version.  Only valid for FAT32.
   * @return version
   */
  public int getVersion() { return version; }
  public void setVersion(int aVersion) { version=aVersion; }

  /**
   * Get the first cluster of the root directory.  This is only used for FAT32.  In FAT12
   * and FAT16, the root directory isn't allocated via the FAT table.  Rather it is of fixed
   * size and resides between the end of the FAT tables and the start of the data area.
   * @return root cluster
   */
  public long getRootCluster() { return rootCluster; }
  public void setRootCluster(long aRootCluster) { rootCluster=aRootCluster; }

  /**
   * Get the sector address of the FAT32 info block.
   * @return fat 32 info block location
   */
  public int getInfoSector() { return infoSector; }
  public void setInfoSector(int aInfoSector) { infoSector=aInfoSector; }

  /**
   * Get the sector address of the backup boot sector.  Only valid in FAT32.
   * @return backup boot sector address.
   */
  public int getBackupBootSector() { return backupBootSector; }
  public void setBackupBootSector(int aBackupBootSector) { backupBootSector=aBackupBootSector; }


  /** Get the number of sectors in the filesystem.  This handles figuring out which fields from
   *  the disk to use to arrive at the correct result.
   *  @return total number of sectors in the filesystem
   */
  public long getNumSectors()
  {
    long sectors;

    sectors = getSectors();
    if (sectors == 0)
      sectors = getTotalSectors();

    return sectors;
  }

  /**
   * Get the number of clusters.  The number of clusters is computed from the number
   * of sectors less the reserved sectors, the root directory sectors (if any), and the FAT sectors.
   * @return number of clusters.
   */
  public long getNumClusters()
  {
    long totalSectors;
    long dataAreaSectors;
    int fatSectors;
    int rootDirSectors;

    totalSectors = getNumSectors();
    fatSectors = (getFatLength() == 0 && getFat32Length() > 0)?(int)getFat32Length():(int)getFatLength();
    fatSectors *= getNumFATs();
    rootDirSectors = (getRootDirEntries() * DIRENTRY_SIZE / getSectorSize());
    dataAreaSectors = totalSectors - getReservedSectors() - rootDirSectors - fatSectors;

    return dataAreaSectors / getSectorsPerCluster();
  }

  /**
   * Get the number bits per entry in the FAT.
   * @return number of fat bits.
   */
  public int getFatBits()
  {
    if (getFatLength() == 0 && getFat32Length() > 0)
      return 32;

    if (getNumClusters() > MAX_FAT12)
      return 16;
    else 
      return 12;
  }

  /**
   * Get the sector address of the first data sector (cluster 2).
   */
  public int getFirstDataSector()
  {
    int fatLength;
    int rootDirLen;
    int dataArea;

    fatLength = (getFatBits() == 32)?(int)getFat32Length():(int)getFatLength();
    rootDirLen = getRootDirEntries() * DIRENTRY_SIZE / getSectorSize();

    dataArea = getReservedSectors() + getNumFATs()*fatLength + rootDirLen;
    return dataArea;
  }

  /** Default constructor. */
  public FatSuperBlock()
  {
  }

  public int getDataSize() { return 0x200; }


  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    bootStrap = new byte[3];
    System.arraycopy(buffer, offset+0x0, bootStrap, 0, 3);

    try
    {
      systemId = new String(buffer, offset+0x3, 8, "ISO-8859-1");
    }
    catch (Exception ex)
    {
      FsException fsex = new FsException(ex.getMessage());
      fsex.initCause(ex);
      throw fsex;
    }

    setSectorSize(FsUtils.getU16(buffer, offset+0xB));
    setSectorsPerCluster((int) buffer[0xD]);
    setReservedSectors(FsUtils.getU16(buffer, offset+0xE));
    setNumFATs((int) buffer[0x10]);
    setRootDirEntries(FsUtils.getU16(buffer, offset+0x11));
    setSectors(FsUtils.getU16(buffer, offset+0x13));
    setMedia(buffer[0x15]);
    setFatLength(FsUtils.getU16(buffer, offset+0x16));
    setSectorsPerTrack(FsUtils.getU16(buffer, offset+0x18));
    setHeads(FsUtils.getU16(buffer, offset+0x1A));
    setHiddenSectors(FsUtils.getU32(buffer, offset+0x1C));
    setTotalSectors(FsUtils.getU32(buffer, offset+0x20));
    setFat32Length(FsUtils.getU32(buffer, offset+0x24));
    setFlags(FsUtils.getU16(buffer, offset+0x28));
    setVersion(FsUtils.getU16(buffer, offset+0x2A));
    setRootCluster(FsUtils.getU32(buffer, offset+0x2C));
    setInfoSector(FsUtils.getU16(buffer, offset+0x30));
    setBackupBootSector(FsUtils.getU16(buffer, offset+0x32));

    /* Do some checks on some values. */
    if (getReservedSectors() == 0)
      throw new FsException("Invalid number of reserved sectors.\n");

    if (getNumFATs() == 0)
      throw new FsException("Invalid number of FATs.\n");

    if (getSectorsPerTrack() == 0)
      throw new FsException("Invalid sectors per track.\n");

    if (getHeads() == 0)
      throw new FsException("Invalid number of heads.\n");

    if (getHeads() == 0)
      throw new FsException("Invalid number of heads.\n");

    if (getSectorSize() == 0 || (getSectorSize() & (getSectorSize() - 1)) != 0 || getSectorSize() < 512 )
      throw new FsException("Invalid logical sector size.\n");

    if (getSectorsPerCluster() == 0 || (getSectorsPerCluster() & (getSectorsPerCluster() - 1)) != 0)
      throw new FsException("Invalid sectors per cluster.\n");

    // number of dir entries must fill an intergral number of sectors;
    int dirEntriesPerSector = getSectorSize() / DIRENTRY_SIZE;  
    if ((getRootDirEntries() % dirEntriesPerSector) != 0)
      throw new FsException("Invalid number of root directory entries.\n");
  }
}
