/*
 * $Id: FatTable.java,v 1.5 2004/04/24 20:57:36 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FatTable.java,v $
 * Revision 1.5  2004/04/24 20:57:36  wrossi
 * FAT12 fix
 *
 * Revision 1.4  2004/04/20 20:42:56  wrossi
 * Getting FatTable to work.
 *
 * Revision 1.3  2004/04/20 11:02:08  wrossi
 * Working on FAT12 handling.
 *
 * Revision 1.2  2004/04/19 23:50:51  wrossi
 * Work in progress...
 *
 * Revision 1.1  2004/04/19 20:40:34  wrossi
 * Class for reading the fat table.
 *
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.FsException;
import rossi.fstools.fs.FsUtils;

import rossi.fstools.io.BlockReader;

import java.io.IOException;

/** Class to lookup cluster pointers in the FAT table. */
public class FatTable
{
  /** Superblock for this fs. */
  private FatSuperBlock sb;
  /** Block reader for this fs. */
  private BlockReader blockReader;
  /** eof mark for this fs. */
  private int eof;
  /** bad block mark for this fs. */
  private int bad;
  /** Starting sector of the first FAT. */
  private int fatStart;

  private static int EOF_FAT12 = 0xff8;
  private static int EOF_FAT16 = 0xfff8;
  private static int EOF_FAT32 = 0x0ffffff8;

  private static int BAD_FAT12 = 0xff7;
  private static int BAD_FAT16 = 0xfff7;
  private static int BAD_FAT32 = 0x0ffffff7;

  /**
   * Create a new FatTable object for managing access to the FAT.
   *
   * @param sb   a FatSuperBlock for this filesystem
   * @param br   a BlockReader initialized to read in sb.getSectorSize() blocks.
   */
  public FatTable(FatSuperBlock sb, BlockReader br)
  {
    this.sb = sb;
    blockReader = br;

    if (sb.getFatBits() == 32)
    {
      eof = EOF_FAT32;
      bad = BAD_FAT32;
    }
    else
    {
      if (sb.getFatBits() == 16)
      {
        eof = EOF_FAT16;
        bad = BAD_FAT16;
      }
      else
      {
        eof = EOF_FAT12;
        bad = BAD_FAT12;
      }
    }

    fatStart = sb.getReservedSectors();
  }

  /**
   * Return true if the given cluster number is an EOF marker.
   *
   * @param cluster number
   * @return true if end of file mark.
   */
  public boolean isEOF(int cluster)
  {
    return ((cluster & eof) == eof);
  }

  /**
   * Return true if the given cluster number is bad.
   *
   * @param cluster number
   * @return true if marked bad.
   */
  public boolean isBad(int cluster)
  {
    return (cluster == bad);
  }

  /**
   * Return true if the given cluster number is free (unallocated).
   *
   * @param cluster number
   * @return true if free
   */
  public boolean isFree(int cluster)
  {
    return (cluster == 0);
  }

  /**
   * Lookup the value for the given cluster in the FAT table.
   *
   * @param cluster   a int
   * @return value contained in the FAT for the given cluster.
   * @exception FsException 
   * @exception IOException if an IO exception occurs.
   */
  public int lookup(int cluster) throws FsException, IOException
  {
    byte[] buffer;
    int sector;        // the number of the sector containing first byte of entry
    int sectorOffset;  // byte offset into the sector where the entry starts
    int tmpcluster;    // temp cluster number for a caclulation
    int result;

    /** Figure out what sector we are in.  For FAT12, we really want to know what sector
        the first part of the cluster pair is in.  */
    tmpcluster = (sb.getFatBits() == 12) ? (cluster & 0xfffffffe) : cluster;

    sector = (int) ((long) tmpcluster * sb.getFatBits()) / (sb.getSectorSize() * 8);
    sectorOffset = (int) (((long) tmpcluster * sb.getFatBits()) - (sector * sb.getSectorSize() * 8)) / sb.getFatBits();

    /* Read the sector containing the entry */
    buffer = blockReader.getBlock(fatStart + sector);

    switch (sb.getFatBits())
    {
      case 32:
        result = (int) FsUtils.getU32(buffer, sectorOffset*4);
        break;

      case 16:
        result = (int) FsUtils.getU16(buffer, sectorOffset*2);
        break;

      case 12:
        // This can get a bit messy since a fat entry can span a sector 
        // boundary
        int sectorAlignment = (int) ( (long) sector * sb.getSectorSize() ) % 3;
         
        sectorAlignment = (3 - sectorAlignment) % 3;

        int firstByte = sectorOffset / 2 * 3 + sectorAlignment;

        /* This is the spanning case.  Handle it by enlarging the buffer with the next sector. */
        if ((firstByte + 3) >= buffer.length)
        {
          byte[] newbuffer = new byte[buffer.length*2];
          /* copy buffer into new buffer */
          System.arraycopy(buffer, 0, newbuffer, 0, buffer.length);
          /* append into newbuffer the next sector */
          System.arraycopy(blockReader.getBlock(fatStart+sector+1), 0, newbuffer, buffer.length, buffer.length);

          buffer = newbuffer;
        }

        if ((cluster % 2) == 0)
        {
          result = (buffer[firstByte] & 0xff) | ((buffer[firstByte+1] & 0x0f) << 8);
        }
        else
        {
          result = ((buffer[firstByte+1] & 0xf0) >> 4) | ((buffer[firstByte+2] & 0xff) << 4);
        }
        break;

      default:
        throw new FsException("Invalid number of bits per fat entry.");
    }

    return result;
  }
}

