/*
 * $Id: VFatDirEntry.java,v 1.3 2004/04/18 22:11:54 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: VFatDirEntry.java,v $
 * Revision 1.3  2004/04/18 22:11:54  wrossi
 * wip
 *
 * Revision 1.2  2004/04/18 21:37:14  wrossi
 * Reading long name fragments works.
 *
 * Revision 1.1  2004/04/18 21:17:51  wrossi
 * Virtual fat directory entry.
 *
 * Revision 1.6  2004/04/18 20:50:49  wrossi
 *
 */

package rossi.fstools.fs.fatfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;


/** 
 * Representation of a VFAT directory entry.  This is where
 * long filenames are stored. 
 *
 * <pre>
 *  On disk the superblock looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |  Seq     |      Name               |
 *        --------------------------------------
 *  0004  |           Name                     |
 *        --------------------------------------
 *  0008  |           Name            |Attr    |
 *        --------------------------------------
 *  000C  | Resrvd| Hash     |   Name          |
 *        --------------------------------------
 *  0010  |           Name                     |
 *        --------------------------------------
 *  0014  |           Name                     |
 *        --------------------------------------
 *  0018  |    Name          |   Start         |
 *        --------------------------------------
 *  001C  |           Name                     |
 *        --------------------------------------
 *
 * </pre>
 */

public class VFatDirEntry implements DiskStructure
{
  /** Sequence number.  First part of long filename is numbered 1, 
   *  the last part has bit 6 set. */
  protected byte sequence;
  /** A fragment of the long filename. */
  protected String name; 
  /** Attributes. Must be 0xf. */
  protected byte attr;
  /** Hash value of the FatDirEntry this is associated with. */
  protected byte hashValue;
  /** Starting cluster */
  protected int start;

  /** Default constructor. */
  public VFatDirEntry()
  {
  }

  public int getDataSize() { return 0x20; }

  /** Get the sequence number.  First part of long filename is numbered 1, 
   *  the last part has bit 6 set.
   *  @return sequence number.
   */
  public byte getSequence() { return sequence; }
  public void setSequence(byte aSequence) { sequence=aSequence; }

  /** Fragment of the long name stored in this record. 
   *  @return long name fragment.
   */
  public String getName() { return name; }
  public void setName(String aName) { name=aName; }

  /** 
   * Attribute byte.  Must by 0xf.
   */
  public byte getAttr() { return attr; }
  public void setAttr(byte aAttr) { attr=aAttr; }

  /** 
   * Hash value of the FatDirEntry this long filename vfat record
   * is assocated with.
   * @return hash value.
   */
  public byte getHashValue() { return hashValue; }
  public void setHashValue(byte aHashValue) { hashValue=aHashValue; }


  /** Starting cluster.  Not used. 
   *  @return starting cluster.
   */
  public int getStart() { return start; }
  public void setStart(int aStart) { start=aStart; }

  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    int i;
    char namebuffer[];

    sequence = buffer[offset+0x0];

    namebuffer = new char[13];
    for (i=0; i<5; i++)
      namebuffer[i] = (char) ((buffer[offset+0x1+i*2]&0xff) | ((buffer[offset+0x1+i*2+1]&0xff) << 8));
    for (i=0; i<6; i++)
      namebuffer[i+5] = (char) ((buffer[offset+0xe+i*2]&0xff) | ((buffer[offset+0xe+i*2+1]&0xff) << 8));
    for (i=0; i<2; i++)
      namebuffer[i+11] = (char) ((buffer[offset+0x1c+i*2]&0xff) | ((buffer[offset+0x1c+i*2+1]&0xff) << 8));

    name = new String(namebuffer);

    /* handle null termination */
    int zeroidx = name.indexOf('\u0000');
    if (zeroidx > 0)
      name = name.substring(0, zeroidx);

    attr = buffer[0xb];
    if (attr != 0xf)
      throw new FsException("Invalid attribute byte.  0x"+Integer.toString(attr & 0xff, 16));

    hashValue = buffer[0xd];
    setStart(FsUtils.getU16(buffer, offset+0x1a));
  }

}


