/*
 * $Id: VFatProcessor.java,v 1.3 2004/04/18 23:03:45 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: VFatProcessor.java,v $
 * Revision 1.3  2004/04/18 23:03:45  wrossi
 * Removed printouts.
 *
 * Revision 1.2  2004/04/18 23:00:07  wrossi
 * Got VFat processor working.
 *
 * Revision 1.1  2004/04/18 22:07:02  wrossi
 * State machine for long filenames.
 *
 * Revision 1.2  2004/04/18 21:37:14  wrossi
 * Reading long name fragments works.
 *
 * Revision 1.1  2004/04/18 21:17:51  wrossi
 * Virtual fat directory entry.
 *
 * Revision 1.6  2004/04/18 20:50:49  wrossi
 *
 */

package rossi.fstools.fs.fatfs;

import java.util.Stack;

/** State machine to handle adding long filenames onto direcotry entries.  
 *  This processes a stream of FatDirEntry and VFatDirEntry records and 
 *  pulls the long filenames out of the VFatDirEntries, and adds them to 
 *  the matching FatDirEntrie's long filename field. <p>
 *  
 *  Normally, the longfile name entries occur in reverse sequence followed
 *  by the short filename, however it is possile for a non-vfat-aware OS
 *  to modify the disk,  this can result in DirEntries being inserted or
 *  deleted in this sequence. <p>
 */
public class VFatProcessor
{
  /** Stack of VFatDirEntries */
  private Stack lfnStack;
  /** Current hashcode. */
  private byte hashValue;
  /** Indicates if an LFN is available. */
  private boolean lfnReady;

  /** Default constructor. */
  public VFatProcessor()
  {
    lfnStack = new Stack();
    hashValue = 0;
    lfnReady = false;
  }

  public void processLFNEntry(VFatDirEntry vfat)
  {
    if ( (vfat.getSequence() & 0x40) != 0 )  // start of a new LFN
    {
      lfnStack.clear();
      lfnReady = false;
    }

    if ( (vfat.getSequence() & 0x3f) == 0x1)
    {
      hashValue = vfat.getHashValue();
      lfnReady = true;
    }

    lfnStack.push(vfat);
  }

  public void processDirEntry(FatDirEntry dir)
  {
    if (lfnReady && dir.getHashValue() == hashValue)
      dir.setLongName(getLongName());
  }

  private String getLongName()
  {
    StringBuffer sb = new StringBuffer();

    while (!lfnStack.empty())
    {
      sb.append(((VFatDirEntry)lfnStack.pop()).getName());
    }
    lfnReady = false;

    return sb.toString();
  }
}

