/*
 * $Id: DirectoryItem.java,v 1.4 2004/04/07 10:22:28 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: DirectoryItem.java,v $
 * Revision 1.4  2004/04/07 10:22:28  wrossi
 * Added documentation
 *
 * Revision 1.3  2003/11/10 12:59:27  wrossi
 * Can now read directory items
 *
 * Revision 1.2  2003/11/08 02:05:03  wrossi
 * Working on directory listing
 *
 * Revision 1.1  2003/11/06 23:35:01  wrossi
 * Initial revision
 *
 * Revision 1.1  2003/11/01 21:46:59  wrossi
 * Initial checkin
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a Directory.<p>
 *
 * An item is an abstract data holder within a block. <p>
 *
 * A directory has an array of directory entry items at the beginning, the names corresponding to
 * those entries are packed in at the end of the item. <p>
 * 
 */

public class DirectoryItem extends Item
{
  protected DirEntry entries[];

  /**
   * Create a new DirecotryItem and populate it from the supplied buffer.
   *
   * @param buffer   a byte[] containing the item data.
   * @param bufferOffset   a int containing the offset within the buffer of the beginning of the item.
   * @param block   The parent block containing this item.
   * @param header   an ItemHead -- The header for this item with the parent block.
   * @exception FsException
   */
  public DirectoryItem(byte[] buffer, int bufferOffset, LeafNode block, ItemHead header) throws FsException
  { 
    super(buffer, bufferOffset, block, header);
    load();   
  }

  /**
   * Get the number of directory entries from the Item header.
   * @return number of entries
   */
  public int numEntries()
  {
    return header.getFreeEntCt();
  }

  /**
   * Get the directory entries themselves.
   * @return array of directory entries
   */
  public DirEntry[] getEntries()
  {
    return entries;
  }

  /**
   * Parse the item and load the array of directory entries from the data with 
   * the item.
   */
  protected void load() throws FsException
  {
    int end;
    entries = new DirEntry[numEntries()];
    end = header.getItemLen();
    for (int i=0; i<numEntries(); i++)
    {
      entries[i] = new DirEntry();
      entries[i].loadFromBuffer(buffer, bufferOffset+header.getItemLocation(), i, end);
      end = entries[i].getLocation();
    }
  }
}
