/*
 * $Id: DiskChild.java,v 1.4 2004/04/08 23:45:12 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: DiskChild.java,v $
 * Revision 1.4  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.3  2004/04/07 10:54:28  wrossi
 * Adding javadoc
 *
 * Revision 1.2  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.1  2003/10/30 12:46:33  wrossi
 * Intra node pointer in B+Tree
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a DiskChild.  DiskChild is an block-to-block pointer
 * stored within InternalNode of the B+Tree in ReiserFS.
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Block Number                |
 *        --------------------------------------
 *  0004  |        Size       | Reserved       |
 *        --------------------------------------
 */

public class DiskChild implements DiskStructure
{
  protected long blockNumber;   // Block number of the child block
  protected int childSize;      // Number of used bytes within the child block -- not sure
                                // if this includes the block head and such.
  protected int reserved;       // reserved

  /**
   * Get the size of this structure.
   * @see DiskStructure#getDataSize()
   * @return size
   */
  public int getDataSize() { return 0x8; }

  /**
   * Get the block number of the child block that we are pointing to.
   * @return block number
   */
  public long getBlockNumber() { return blockNumber; }
  public void setBlockNumber(long aBlockNumber) { blockNumber=aBlockNumber; }

  /**
   * Get the number of bytes used within the child block.
   * @return child size.
   */
  public int getChildSize() { return childSize; }
  public void setChildSize(int aChildSize) { childSize=aChildSize; }

  public int getReserved() { return reserved; }
  public void setReserved(int aReserved) { reserved=aReserved; }


  /**
   * Load this structure from disk.  
   * @see DiskStructure#loadFromBuffer(byte[], int)
   * @param buffer[] buffer containgin data to be read
   * @param blkoffset  offset within buffer where data begins.
   * @exception FsException if a parse error occurs
   */
  public void loadFromBuffer(byte buffer[], int blkoffset) throws FsException
  {
    setBlockNumber(FsUtils.getU32(buffer, blkoffset+0x0));
    setChildSize(FsUtils.getU16(buffer, blkoffset+0x4));
    setReserved(FsUtils.getU16(buffer, blkoffset+0x6));
  }
}
