/*
 * $Id: FBlock.java,v 1.9 2004/04/08 23:45:12 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FBlock.java,v $
 * Revision 1.9  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.8  2004/04/07 10:54:28  wrossi
 * Adding javadoc
 *
 * Revision 1.7  2003/11/06 23:33:32  wrossi
 * Commented out loading of right Key.   doent seem to work.
 *
 * Revision 1.6  2003/11/05 17:32:53  wrossi
 * Added getBlockHeaderSize and createFromBuffer
 *
 * Revision 1.5  2003/11/01 21:42:52  wrossi
 * Added reference to parent block and superblock
 *
 * Revision 1.4  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.3  2003/10/30 12:48:24  wrossi
 * Added setters/getters and filled out loadFromBuffer() api
 *
 * Revision 1.2  2003/10/30 00:48:33  wrossi
 * Added loadFromBuffer() api
 *
 * Revision 1.1  2003/10/29 00:31:38  wrossi
 * Work in progress
 *
 * Revision 1.2  2003/10/27 11:57:33  wrossi
 * Added FsException and FsUtils.  Added code to ReiserSuperBlock to parse
 * a buffer and do some sanity checks.
 *
 * Revision 1.1  2003/10/22 23:20:11  wrossi
 * Initial checkin
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a formated block.  This is one of 2 types, either an 
 * InternalNode or a LeafNode.   This base class handles the block header.<p>
 *
 * Both InternalNode and Leaf node have a common header which is represented here.  <p>
 *
 * <pre>
 *  On disk the block header looks like this:
 *
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Level     |  Num Items      |
 *        --------------------------------------
 *  0004  |     Free Space   |  Reserved       |
 *        --------------------------------------
 *  0008  |            Right Key               |
 *        --------------------------------------
 *  000C  |            Right Key               |
 *        --------------------------------------
 *  0010  |            Right Key               |
 *        --------------------------------------
 *  0014  |            Right Key               |
 *        --------------------------------------
 * <pre>
 */


public class FBlock implements DiskStructure
{
  /* The fields below are on-disk entities associated with the block */
  protected int level;      // level in the tree.  1 indicates Leaf level.  Greater than one for Nodes
  protected int numItems;   // Number of items or keys in block
  protected int freeSpace;  // Amount of free space in block in bytes
  protected int reserved;   // Reserved.

  protected Key rightKey;   // Rightmost key -- obsolete, may be removed in future revs

  /* The fields below are for in-memory housekeeping */
  protected ReiserSuperBlock sb; // Used to get block size
  protected int blocknum;        // number of this block within the fs
  protected FBlock parent;       // Parent block

  /**
   * Create a new formated block.
   * @param sb   a ReiserSuperBlock
   * @param blocknum   a int
   */
  public FBlock(ReiserSuperBlock sb, int blocknum)
  {
    this.sb = sb;
    this.blocknum = blocknum;
  }


  /**
   * Get the size of the header for this formatted block.
   * @return block header size.
   */
  public int getBlockHeaderSize() { return 0x18; }

  /**
   * Get size of this structure.  
   * @see DiskStructure#getDataSize()
   * @return size
   */
  public int getDataSize() { return sb.getBlockSize(); }  

  /**
   * Get the superblock for this filesystem.
   * @return superblock
   */
  public ReiserSuperBlock getSuperBlock() { return sb; }
  public int getBlockNum() { return blocknum; }

  /** Get the parent of this block.  This is to allow walking back up the tree.
   * @return parent block.
   */
  public FBlock getParent() { return parent; }
  /** Set the parent of this block.  This is to allow walking back up the tree. */
  public void setParent(FBlock aParent) { parent=aParent; }

  /**
   * Get the level of this block in the B+tree.  A level of 1 indicates a Leaf block.  
   * Greater than one indicates a node block.  A level of zero indicates a free block.
   * @return level.
   */
  public int getLevel() { return level; }
  public void setLevel(int aLevel) { level=aLevel; }

  /**
   * Get the number of items or keys within this block.  Leaf blocks contain items,
   * node blocks contain keys.
   * @see LeafNode
   * @see InternalNode
   * @return int
   */
  public int getNumItems() { return numItems; }
  public void setNumItems(int aNumItems) { numItems=aNumItems; }

  /**
   * Get the amount of free space within this block.
   * @return number of bytes of free space.
   */
  public int getFreeSpace() { return freeSpace; }
  public void setFreeSpace(int aFreeSpace) { freeSpace=aFreeSpace; }

  public int getReserved() { return reserved; }
  public void setReserved(int aReserved) { reserved=aReserved; }

  /**
   * Get the right most (hightest in sort order) key contained within this block.<p>
   * @return the rightmost Key
   */
  public Key getRightKey() { return rightKey; }
  public void setRightKey(Key aRightKey) { rightKey=aRightKey; }

  /**
   * Create a formatted block instance.  This is a factory method that reads the supplied
   * buffer and creates the appropriate subclass.
   *
   * @param sb   a ReiserSuperBlock for this filesystem
   * @param blocknum   block number of the block to be processed.
   * @param buffer   a byte[] buffer containing the block
   * @param blkoffset   a int offset into the buffer to the start of the block
   * @return FBlock subclass -- either an InternalNode or LeafNode
   * @exception FsException
   */
  public static FBlock createFromBuffer(ReiserSuperBlock sb, int blocknum, 
         byte[] buffer, int blkoffset) throws FsException
  {
    FBlock blk;
    int level = FsUtils.getU16(buffer, blkoffset+0x0);

    if (level == 1)
      blk = new LeafNode(sb, blocknum);
    else
      blk = new InternalNode(sb, blocknum);

    blk.loadFromBuffer(buffer, blkoffset);
    return blk;
  }

  /**
   * Load this structure from disk.  
   * @see DiskStructure#loadFromBuffer(byte[], int)
   * @param buffer[] buffer containgin data to be read
   * @param blkoffset  offset within buffer where data begins.
   * @exception FsException if a parse error occurs
   */
  public void loadFromBuffer(byte buffer[], int offset) throws FsException
  {
    setLevel(FsUtils.getU16(buffer, offset+0x0));
    setNumItems(FsUtils.getU16(buffer, offset+0x2));
    setFreeSpace(FsUtils.getU16(buffer, offset+0x4));
    setReserved(FsUtils.getU16(buffer, offset+0x6));

    //setRightKey( Key.createKeyFromBuffer(buffer, offset+0x8) );
  }
}
