/*
 * $Id: IndirectItem.java,v 1.3 2004/04/08 23:45:12 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: IndirectItem.java,v $
 * Revision 1.3  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.2  2004/04/07 11:01:14  wrossi
 * Added javadocs
 *
 * Revision 1.1  2003/11/26 21:39:24  wrossi
 * Working on file extraction
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of an indirect item. <p>
 *
 * An indirect item is an array of 32 bit block pointers.   The number of pointers is
 * given by itemLength/4<p>
 *
 */

public class IndirectItem extends Item
{
  protected int ptrs[];

  /** 
   * Create an indirect item from the supplied buffer.
   * @param buffer the buffer to load data from.
   * @param bufferOffset offset within the buffer that data starts at
   * @param block leaf block containing this item.
   * @param header Item header for this item.
   */
  public IndirectItem(byte[] buffer, int bufferOffset, LeafNode block, ItemHead header) throws FsException
  { 
    super(buffer, bufferOffset, block, header);
    load();   
  }

  /**
   * Get the number of entries (pointers to blocks) contained within this item.
   * @return number of entries.
   */
  public int numEntries()
  {
    return header.getItemLen()/4;
  }

  /**
   * Get the entries (pointers to blocks) contained within this item.
   * @return int[]
   */
  public int[] getEntries()
  {
    return ptrs;
  }

  /**
   * Load the block pointer array from the buffer.
   * @exception FsException if a pase error occurs.
   */
  protected void load() throws FsException
  {
    int end;
    ptrs = new int[numEntries()];

    for (int i=0; i<numEntries(); i++)
    {
      ptrs[i] = (int) FsUtils.getU32(buffer, bufferOffset+header.getItemLocation()+i*4);
    }
  }
}
