/*
 * $Id: Item.java,v 1.5 2004/04/08 12:38:04 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Item.java,v $
 * Revision 1.5  2004/04/08 12:38:04  wrossi
 * Fix javadoc warning
 *
 * Revision 1.4  2004/04/07 19:47:40  wrossi
 * Adding javadoc
 *
 * Revision 1.3  2003/11/26 21:39:24  wrossi
 * Working on file extraction
 *
 * Revision 1.2  2003/11/04 12:10:14  wrossi
 * Added setters/getters
 *
 * Revision 1.1  2003/11/01 21:46:59  wrossi
 * Initial checkin
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

import rossi.fstools.io.BlockPointer;

/**
 * Representation of a Item.  <p>
 *
 * An item is an abstract data holder within a block.<p>
 * Items are the fundamental unit of storage in ReiserFS.  All data is packed into
 * items which are in turn packed into blocks.  Items have variable size and this is how
 * ReiserFS achives low overhead on small files.<p>
 * This class isnt typically used directly, rather on of the many subclasses is used.<p>
 */

public class Item
{
  protected byte[] buffer;   // Buffer that contains the block that holds the item
  protected int bufferOffset;  // Offset into buffer of block start
  protected LeafNode block;    // Reference to our parent block
  protected ItemHead header;   // ItemHeader

  /**
   * Create a new Item.
   *
   * @param buffer   a byte[] containing the item data
   * @param bufferOffset   a int specifying the offset within the buffer where the item starts.
   * @param block   a LeafNode containing this item.
   * @param header   a ItemHead - header for this item.
   */
  public Item(byte[] buffer, int bufferOffset, LeafNode block, ItemHead header)
  {
    this.buffer = buffer;
    this.bufferOffset = bufferOffset;
    this.block = block;
    this.header = header;
  }

  /**
   * Get the buffer holding this item's data.
   * @return byte[]
   */
  public byte[] getBuffer() { return buffer; }
  public void setBuffer(byte[] aBuffer) { buffer=aBuffer; }

  /**
   * Get the offset within the buffer where this item's data starts.
   * @return offset
   */
  public int getBufferOffset() { return bufferOffset; }
  public void setBufferOffset(int aBufferOffset) { bufferOffset=aBufferOffset; }

  /**
   * Get the LeafNode that contains this item.
   * @return LeafNode parent
   */
  public LeafNode getBlock() { return block; }
  public void setBlock(LeafNode aBlock) { block=aBlock; }

  /**
   * Get the item header for this item.
   * @return ItemHead
   */
  public ItemHead getHeader() { return header; }
  public void setHeader(ItemHead aHeader) { header=aHeader; }

  /** 
   * Gets a block pointer to the item data.  This is useful for locating
   * the item on disk.
   * @return BlockPointer
   */
  public BlockPointer getBlockPointer()
  {
    BlockPointer bp = new BlockPointer();
    bp.setBlockNum(block.getBlockNum());
    bp.setOffset(header.getItemLocation());
    bp.setLength(header.getItemLen());
    bp.setBlockSize(block.getSuperBlock().getBlockSize());

    return bp;
  }

  /** Get the raw data comprising this item. 
   *  This is a bit expensive since it does an array copy. 
   */
  public byte[] getRawData()
  {
    byte newbuf[];
 
    newbuf = new byte[header.getItemLen()];
    System.arraycopy(buffer, bufferOffset+header.getItemLocation(), 
                     newbuf, 0, header.getItemLen());
    return newbuf;
  }
}
