/*
 * $Id: ItemHead.java,v 1.6 2004/04/08 23:45:12 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: ItemHead.java,v $
 * Revision 1.6  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.5  2004/04/08 12:33:42  wrossi
 * Adding javadocs
 *
 * Revision 1.4  2004/04/07 19:47:40  wrossi
 * Adding javadoc
 *
 * Revision 1.3  2003/11/05 17:33:27  wrossi
 * Fixed key loading
 *
 * Revision 1.2  2003/11/01 21:46:22  wrossi
 * Filled in methods
 *
 * Revision 1.1  2003/10/31 12:51:31  wrossi
 * Initial rev
 *
 * Revision 1.1  2003/10/30 12:46:33  wrossi
 * Intra node pointer in B+Tree
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a ItemHead. <p>
 * Item Head is an intrablock pointer to an item.<p>
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Key                         |
 *        --------------------------------------
 *  0004  |        Key                         |
 *        --------------------------------------
 *  0008  |        Key                         |
 *        --------------------------------------
 *  000C  |        Key                         |
 *        --------------------------------------
 *  0010  | Free/EntCount    |  Item Length    |
 *        --------------------------------------
 *  0014  |  Item Location   |  Item Version   |
 *        --------------------------------------
 * </pre>
 */

public class ItemHead implements DiskStructure
{
  protected Key key;
  protected int freeEntCt;  // free space in last indirect block or entry ct for dirs
  protected int itemLen;
  protected int itemLocation;
  protected int itemVersion;

  public int getDataSize() { return 0x18; }

  /**
   * Get the Key of the item.  The key is used to associate directory entries to items.
   * @return the key
   */
  public Key getKey() { return key; }
  public void setKey(Key aKey) { key=aKey; }

  /**
   * Get the number of entries if this is a DirectoryItem, or the free space if this is a
   * indirect item.
   * @return free space or entry count
   */
  public int getFreeEntCt() { return freeEntCt; }
  public void setFreeEntCt(int aFreeEntCt) { freeEntCt=aFreeEntCt; }

  /**
   * Get the length of this Item in bytes.
   * @return item length.
   */
  public int getItemLen() { return itemLen; }
  public void setItemLen(int aItemLen) { itemLen=aItemLen; }

  /**
   * Location of the item within this block.
   * @return item location.
   */
  public int getItemLocation() { return itemLocation; }
  public void setItemLocation(int aItemLocation) { itemLocation=aItemLocation; }

  /**
   * Indicates the item version.  <p>
   * 0 for old items, 2 for new items.
   * @return item version.
   */
  public int getItemVersion() { return itemVersion; }
  public void setItemVersion(int aItemVersion) { itemVersion=aItemVersion; }

  public void loadFromBuffer(byte buffer[], int blkoffset) throws FsException
  {
    setItemVersion(FsUtils.getU16(buffer, blkoffset+0x16));

    Key k;

    if (getItemVersion() == Key.KEY_FORMAT_3_5)
    {
      k = (Key)new KeyV1();
      k.loadFromBuffer(buffer, blkoffset+0x0);
    }
    else
    {
      try
      {
        k = (Key)new KeyV2();
        k.loadFromBuffer(buffer, blkoffset+0x0);
      }
      catch (FsException ex)
      {
        k = (Key) new KeyV1();
        k.loadFromBuffer(buffer, blkoffset+0x0);
      }
    }

    setKey(k);
    setFreeEntCt(FsUtils.getU16(buffer, blkoffset+0x10));
    setItemLen(FsUtils.getU16(buffer, blkoffset+0x12));
    setItemLocation(FsUtils.getU16(buffer, blkoffset+0x14));
  }
}
