/*
 * $Id: KeyV2.java,v 1.5 2004/04/08 12:33:42 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: KeyV2.java,v $
 * Revision 1.5  2004/04/08 12:33:42  wrossi
 * Adding javadocs
 *
 * Revision 1.4  2003/11/20 19:36:40  wrossi
 * Performance improvement
 *
 * Revision 1.3  2003/11/05 17:31:08  wrossi
 * Fixed bug in extracting key type
 *
 * Revision 1.2  2003/10/31 12:51:02  wrossi
 * Moved endian into one central location
 *
 * Revision 1.1  2003/10/30 00:47:11  wrossi
 * Key implementation
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a format 3.6 key.  This version allows for larger offsets.<p>
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |        Parent Dir ID               |
 *        --------------------------------------
 *  0004  |        Object Id                   |
 *        --------------------------------------
 *  0008  |        Offset  (bits 0-31)         |
 *        --------------------------------------
 *  000C  |      Offset (bits 32-59)      |type|
 *        --------------------------------------
 * </pre>
 */

public class KeyV2 extends Key
{
  private final static FsException invalidex = new FsException("Invalid type for format 3.6 key");
  public void loadFromBuffer(byte buffer[], int blkoffset) throws FsException
  {
    setParentDirId(FsUtils.getU32(buffer, blkoffset+0x0));
    setObjectId(FsUtils.getU32(buffer, blkoffset+0x4));

    long lowoffset = FsUtils.getU32(buffer, blkoffset+0x8);
    long highoffset = FsUtils.getU32(buffer, blkoffset+0xC);

    int type = (int) ((highoffset & 0xf0000000) >> 28);
    
    if (type != TYPE_INDIRECT && type != TYPE_DIRECT && type != TYPE_DIRENTRY)
      throw invalidex;

    long off = (highoffset & 0x0fffffff) << 32;
    off |= lowoffset;
    
    setOffset(off);
    setType(type);
  }
}
