/*
 * $Id: LeafNode.java,v 1.7 2004/04/08 12:35:38 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: LeafNode.java,v $
 * Revision 1.7  2004/04/08 12:35:38  wrossi
 * Added javadocs
 *
 * Revision 1.6  2003/11/26 21:39:24  wrossi
 * Working on file extraction
 *
 * Revision 1.5  2003/11/12 21:29:31  wrossi
 * Working on stat data items.
 *
 * Revision 1.4  2003/11/08 02:05:03  wrossi
 * Working on directory listing
 *
 * Revision 1.3  2003/11/05 17:32:16  wrossi
 * Fixed loading of Keys
 *
 * Revision 1.2  2003/11/01 21:46:38  wrossi
 * Added block number
 *
 * Revision 1.1  2003/10/31 12:51:31  wrossi
 * Initial rev
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of an leaf node of the B+Tree.<p>
 *
 * This structure is the container for Items in the filesystem. <p>
 *
 * Following the block header (at offset 0x18), is an array of N ItemHeaders, followed by
 * some free space.  The from the end of the block working backwards is an array of
 * items.  The item headers need to be read first to determine where the item bodies are.
 */

public class LeafNode extends FBlock
{
  protected ItemHead[] itemHeaders;

  /**
   * Create a new leaf node
   * @param sb   a ReiserSuperBlock for this filesystem.
   * @param blocknum  number of this block
   */
  public LeafNode(ReiserSuperBlock sb, int blocknum)
  {
    super(sb, blocknum);
  }

  /**
   * Get an array of all the ItemHeaders in this leaf.
   * @return ItemHead[]
   */
  public ItemHead[] getItemHeaders() { return itemHeaders; }

  public void setItemHeaders(ItemHead[] aItemHeaders) { itemHeaders=aItemHeaders; }

  /**  
   * Returns the item matching the given key, or the next one greater. 
   * @return the requested Item
   */
  public Item getItem(byte buffer[], int offset, Key key) throws FsException
  {
    for (int i=0; i<itemHeaders.length; i++)
    {
      if (itemHeaders[i].getKey().compareTo(key) <= 0)
        return getItem(buffer, offset, itemHeaders[i]);
    }

    return null;
  }

  /** 
   * Gets an item from this block given the item header.  
   * @param buffer contains this block at offset.
   * @param offset points to the start of this block.
   * @return the requested item
   */
  public Item getItem(byte buffer[], int offset, ItemHead ih) throws FsException
  {
    Item itm;

    switch (ih.getKey().getType())
    {
      case Key.TYPE_DIRENTRY:
             itm = new DirectoryItem(buffer, offset, this, ih);
             return itm;
 
      case Key.TYPE_STAT_DATA:
             itm = StatDataItem.create(buffer, offset, this, ih);
             return itm;

      case Key.TYPE_INDIRECT:
             itm = new IndirectItem(buffer, offset, this, ih);
             return itm;

      case Key.TYPE_DIRECT:
             itm = new DirectItem(buffer, offset, this, ih);
             return itm;

      default:  throw new FsException("Unsupported item type: "+ih.getKey().getType());
    }
  }
  
  public void loadFromBuffer(byte buffer[], int blkoffset) throws FsException
  {
    super.loadFromBuffer(buffer, blkoffset);

    itemHeaders = new ItemHead[getNumItems()];
     
    int ihsize = (new ItemHead()).getDataSize();
    int offset = blkoffset + getBlockHeaderSize();
 
    for (int i=0; i<getNumItems(); i++)
    {
      ItemHead ih = new ItemHead();
      //System.out.println(offset+i*ihsize);
      ih.loadFromBuffer(buffer, offset+i*ihsize);
      itemHeaders[i] = ih;
    }
  }
}
