/*
 * $Id: ReiserFile.java,v 1.7 2004/04/08 23:45:12 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: ReiserFile.java,v $
 * Revision 1.7  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.6  2004/03/11 00:39:56  wrossi
 * Performance boost.  Allow a single BlockPointer to point to multiple
 * contigous blocks.  This reduces our memory footprint considerably
 *
 * Revision 1.5  2004/02/10 21:18:34  wrossi
 * Fixed slack for direct items
 *
 * Revision 1.4  2004/02/05 21:31:24  wrossi
 * Account for slack in direct items
 *
 * Revision 1.3  2004/01/23 21:16:02  wrossi
 * Expose BlockPointerInputStream to user
 *
 * Revision 1.2  2003/12/17 22:04:12  wrossi
 * Handle null padding on symlinks
 *
 * Revision 1.1  2003/11/26 21:39:24  wrossi
 * Working on file extraction
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.File;
import rossi.fstools.io.BlockPointer;
import rossi.fstools.io.BlockReader;
import rossi.fstools.io.BlockPointerInputStream;

import java.util.List;
import java.util.ArrayList;

/**
 * Representation of a file.  The file data is build up from DirectItems and IndirectItems.
 */

public class ReiserFile implements File
{
  protected List blkPointerList;
  protected List slackList;
  protected long size;
  protected long sizeRemaining;
  protected BlockReader reader;

  /**
   * Crate a new file.
   *
   * @param fileSize   the length of the file from the StatDataItem
   * @param reader   a BlockReader to read the physical disk with
   */
  public ReiserFile(long fileSize, BlockReader reader)
  {
    blkPointerList = new ArrayList();
    slackList = new ArrayList();
    this.size = fileSize;
    this.reader = reader;
    sizeRemaining = size;
  }

  /**
   * Adds an IndirectItem to this file.  Each block pointed to within the IndirectItem 
   * will be added in sequence to the list of blocks for this file.
   * @param ind  a IndirectItem
   */
  public void add(IndirectItem ind)
  {
    BlockPointer bp;
    BlockPointer lastBp = null;
    int entries[];
    int blksize;
    int curSize;
    int lastend = 0;
    int lastendoffset = 0;

    entries = ind.getEntries();
    blksize = ind.getBlock().getSuperBlock().getBlockSize();

    for (int i=0; i<entries.length; i++)
    {
      curSize = (sizeRemaining < blksize) ? (int)sizeRemaining : blksize;

      if (blkPointerList.size() != 0)
      {
        lastBp = (BlockPointer) blkPointerList.get(blkPointerList.size() - 1);

        lastend = lastBp.getBlockNum() + ((lastBp.getOffset()+lastBp.getLength()) / blksize);
        lastendoffset = ((lastBp.getOffset()+lastBp.getLength()) % blksize);
      }

      if (lastBp != null && lastend == entries[i] && lastendoffset == 0)
      {
        /* Add this indirect item block onto the last block pointer since it is
           a consecutive block */
        lastBp.setLength(lastBp.getLength() + curSize);
      }
      else
      {
        /* Create a new block pointer */

        bp = new BlockPointer();
        bp.setBlockNum(entries[i]);
        bp.setOffset(0);
        bp.setLength(curSize);
        bp.setBlockSize(blksize);
        blkPointerList.add(bp);
      }

      /* record slack */
      if (curSize != blksize)
      {
        bp = new BlockPointer();
        bp.setBlockNum(entries[i]);
        bp.setOffset(curSize);
        bp.setLength(blksize - curSize);
        bp.setBlockSize(blksize);
        slackList.add(bp);
      }

      sizeRemaining = sizeRemaining - curSize;
    }
  }

  /**
   * Add a DirectItem to the file.  Files can be composed of a combination of 
   * direct and indirect items.
   * @param itm   a DirectItem
   */
  public void add(DirectItem itm)
  {
    BlockPointer bpi = itm.getBlockPointer();
    BlockPointer bp;
    int blksize = bpi.getLength();
    int curSize;

    /* Have to go though some hoops because there **may** be some slack in a direct item */
    curSize = (sizeRemaining < blksize) ? (int)sizeRemaining : blksize;
    
    bp = new BlockPointer();  // copy the input pointer with adjusted size
    bp.setBlockNum(bpi.getBlockNum());
    bp.setOffset(bpi.getOffset());
    bp.setLength(curSize);
    bp.setBlockSize(bpi.getBlockSize());
    blkPointerList.add(bp);

    /* record slack */
    if (curSize != blksize)
    {
      bp = new BlockPointer();
      bp.setBlockNum(bpi.getBlockNum());
      bp.setOffset(bpi.getOffset() + curSize);
      bp.setLength(blksize - curSize);
      bp.setBlockSize(bpi.getBlockSize());
      slackList.add(bp);
    }

    sizeRemaining = sizeRemaining - curSize;
  }

  public BlockPointerInputStream getData()
  {
    return new BlockPointerInputStream(reader, blkPointerList);
  }

  public BlockPointerInputStream getSlack()
  {
    return new BlockPointerInputStream(reader, slackList);
  }
}
