/*
 * $Id: ReiserFileSystem.java,v 1.19 2004/04/08 23:45:12 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: ReiserFileSystem.java,v $
 * Revision 1.19  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.18  2004/02/28 20:49:16  wrossi
 * Added ability to close a filesystem.
 *
 * Revision 1.17  2003/12/08 21:41:06  wrossi
 * Added fifo and sockets
 *
 * Revision 1.16  2003/12/08 21:32:48  wrossi
 * Added device support
 *
 * Revision 1.15  2003/12/03 21:31:39  wrossi
 * Symlink support
 *
 * Revision 1.14  2003/11/30 16:23:24  wrossi
 * Working on file extraction
 *
 * Revision 1.13  2003/11/28 12:02:49  wrossi
 * Working on file extraction
 *
 * Revision 1.12  2003/11/26 21:39:24  wrossi
 * Working on file extraction
 *
 * Revision 1.11  2003/11/25 01:33:31  wrossi
 * Caching work
 *
 * Revision 1.10  2003/11/25 01:32:11  wrossi
 * Added node caching
 *
 * Revision 1.9  2003/11/20 11:53:56  wrossi
 * Removed printouts
 *
 * Revision 1.8  2003/11/19 21:12:30  wrossi
 * Reading directories works
 *
 * Revision 1.7  2003/11/18 21:40:25  wrossi
 * Adding dir read support
 *
 * Revision 1.6  2003/11/18 11:01:39  wrossi
 * Added generic fs interface
 *
 * Revision 1.5  2003/11/17 11:05:29  wrossi
 * Removed dead code.
 *
 * Revision 1.4  2003/11/17 11:02:42  wrossi
 * Working on being able to find next item
 *
 * Revision 1.3  2003/11/14 21:13:00  wrossi
 * Working on getting the next leaf
 *
 * Revision 1.2  2003/11/08 02:05:03  wrossi
 * Working on directory listing
 *
 * Revision 1.1  2003/11/06 23:35:01  wrossi
 * Initial revision
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.SuperBlock;
import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.FileSystem;
import rossi.fstools.fs.Inode;
import rossi.fstools.fs.InodePtr;
import rossi.fstools.fs.FsObject;
import rossi.fstools.fs.Directory;
import rossi.fstools.fs.File;
import rossi.fstools.fs.SymLink;
import rossi.fstools.fs.Device;
import rossi.fstools.io.BlockReader;
import rossi.fstools.io.BlockCache;

import java.io.IOException;

/**
 * Representation of the reiser file system.   Application code is to use this class.
 */

public class ReiserFileSystem implements FileSystem
{
  private BlockReader blockReader;
  private ReiserSuperBlock sb;
  private BlockCache cache;         // cache of parsed tree nodes

  public ReiserFileSystem()
  {
  }

  public SuperBlock open(BlockReader br) throws FsException, IOException
  {
    blockReader = br;

    cache = new BlockCache(20);    // Cache 20 blocks
    br.setBlockSize(4096);  // 4KB blocks 
    sb = new ReiserSuperBlock();

    sb.loadFromBuffer(br.getBlock(16), 0);

    br.setBlockSize(sb.getBlockSize());
    return sb;
  }

  public void close() throws IOException
  {
    blockReader.close();
  }

  /** Returns pointer to root directory. */
  public InodePtr getRootDir()
  {
    KeyV2 key = new KeyV2();
    key.setParentDirId(1);
    key.setObjectId(2);
    key.setOffset(0);
    key.setType(Key.TYPE_STAT_DATA);

    return (InodePtr) key;
  }

  /** Gets a tree node. */
  protected FBlock getNode(int blkno) throws IOException, FsException
  {
    FBlock nodeBlock;
    byte buffer[];

    /** Try to read from cache */
    nodeBlock =  (FBlock) cache.getBlockFromCache(blkno);

    /** If not it cache, go to disk */
    if (nodeBlock == null)
    {
      buffer = blockReader.getBlock( blkno );
      nodeBlock = FBlock.createFromBuffer(sb, blkno, buffer, 0);

      /* add block to cache */
      cache.addBlockToCache(blkno, nodeBlock);
    }

    if (nodeBlock.getBlockNum() != blkno)
      throw new FsException("Cache error");

    return nodeBlock;
  }

  /** Gets an inode given a pointer.   */
  public Inode getInode(InodePtr ptr) throws IOException, FsException
  {
    Key key = (Key) ptr;
    StatDataItem sd;

    sd = (StatDataItem) findNextItem(key);
    return (Inode) sd;
  }

  /**
   * This is temporarily public just for testing.
   * Find the first leaf node that may contain the given key -- if it exists.
   * @param key the Key to look for.
   * @return the leaf node containing the item pointed to by the key
   */
  public LeafNode findLeaf(Key key) throws FsException, IOException
  {
    FBlock currentBlock, nextBlock;
    int blkno;

    /* Start with the root block */
    currentBlock = getNode((int) sb.getRootBlock());

    while (currentBlock instanceof InternalNode)
    {
      InternalNode intnode = (InternalNode) currentBlock;
      DiskChild child;

      child = intnode.getPointerForKey(key);
      blkno = (int) child.getBlockNumber();
      //System.out.println("blkno -> "+blkno);
      nextBlock = getNode(blkno);
      nextBlock.setParent(currentBlock);
      currentBlock = nextBlock;
    }

    return (LeafNode) currentBlock;
  }
 
  /**
   * Gets the leftmost leaf of the given block if startblock points to an internal node.
   * If startblock points to a Leaf, just return it.
   *
   * @param parent   a FBlock which is the parent of startblock
   * @param startblock   a int pointing to the starting node/leaf block.
   * @return LeafNode the first leafnode within start block
   * @exception FsException if a parsing error occurs.
   * @exception IOException if an IO error occurs
   */
  public LeafNode findLeftMostLeaf(FBlock parent, int startblock) throws FsException, IOException
  {
    FBlock currentBlock, nextBlock;
    int blkno;

    /* Start with the start block */
    currentBlock = getNode(startblock);
    currentBlock.setParent(parent);

    while (currentBlock instanceof InternalNode)
    {
      InternalNode intnode = (InternalNode) currentBlock;
      DiskChild child;

      child = intnode.getPtrs()[0];
      blkno = (int) child.getBlockNumber();
      //System.out.println("down to blkno -> "+blkno);
      nextBlock = getNode(blkno);
      nextBlock.setParent(currentBlock);
      currentBlock = nextBlock;
    }

    //System.out.println("Leftmost leaf = "+currentBlock.getBlockNum());
    //System.out.println("Leftmost leaf = "+currentBlock);
    return (LeafNode) currentBlock;
  }
 
  /** 
   * Returns the next leaf following the given leaf.
   * @param current The current leaf
   * @return the leaf node just after (right of) the current leaf
   * @exception FsException if the next leaf cant be found.
   * @exception IOException if an IO error occurs.
   */
  public LeafNode nextLeaf(LeafNode current) throws FsException, IOException
  {
    InternalNode intnode;
    DiskChild ptrs[];
    int blockno;

    /* Walk up the tree until we find an internal node */
    blockno = current.getBlockNum();
    intnode = (InternalNode) current.getParent();

    while (true)
    {
      //System.out.println("Up to block "+blockno);
      ptrs = intnode.getPtrs();
      int rightblock = (int) ptrs[ptrs.length-1].getBlockNumber();
      if (blockno != rightblock)
        break;

      blockno = (int) intnode.getBlockNum();
      intnode = (InternalNode) intnode.getParent();
    }

    /* At this point blockno should point down the tree toward the current leaf,
       and intnode should be an Internal node which has childrent to the right of
       the one pointing to blkno.   Find the child node to the right of the one
       for the current branch  */

    for (int i=0; i<ptrs.length; i++)
    {
      int bn = (int) ptrs[i].getBlockNumber();
      if (bn == blockno)
      {
        //System.out.println("Look in block "+ptrs[i+1].getBlockNumber());
        return findLeftMostLeaf(intnode, (int) ptrs[i+1].getBlockNumber());
      }
    }
 
    throw new FsException("Error finding next leaf");
  }

  /**
   *  Finds the item whose key is equal to key if key exists in the tree.  Otherwise
   *  returns the next greater key.
   *  @param key
   *  @return Item
   *  @exception FsException if a parse error occurs
   *  @exception IOException if an IO error occurs
   */
  public Item findNextItem(Key key) throws FsException, IOException
  {
    LeafNode leaf = findLeaf(key);
    byte buffer[];
    Item itm;

    buffer = blockReader.getBlock(leaf.getBlockNum());

    itm = leaf.getItem(buffer, 0, key);

    if (itm == null)  // could be in the next leaf
    {
      leaf = nextLeaf(leaf);
      buffer = blockReader.getBlock(leaf.getBlockNum());
      itm = leaf.getItem(buffer, 0, key);
    }
    //System.out.println("Item = "+itm);
    //System.out.println("Item.len = "+itm.getHeader().getItemLen()+" "+itm.getHeader().getItemLocation());

    return itm;
  }

  /**
   * Create a Symlink from a StatDataItem.
   *
   * @param sdItem   a StatDataItem
   * @return SymLink
   * @exception FsException
   * @exception IOException
   */
  public SymLink createReiserSymLink(StatDataItem sdItem) throws FsException, IOException
  {
    return new ReiserSymLink((ReiserFile) createReiserFile(sdItem));
  }

  /** 
   * Creates a file for the given StatDataItem.  The information in the StatDataItem
   * is sufficent to create a Key to find the Direct and Indirect items that compose
   * the file.
   * @param sdItem   a StatDataItem
   * @return File
   * @exception FsException
   * @exception IOException
   */
  public File createReiserFile(StatDataItem sdItem) throws FsException, IOException
  {
    Key inodeKey, searchKey;
    ReiserFile file;
    long bytesLeft, fileSize;
    Item item;

    /* Search for the next indirect or direct item */
    searchKey = new KeyV2();
    inodeKey = sdItem.getHeader().getKey();   
    searchKey.setParentDirId(inodeKey.getParentDirId());
    searchKey.setObjectId(inodeKey.getObjectId());
    searchKey.setOffset(0);
    searchKey.setType(Key.TYPE_INDIRECT);
    bytesLeft = sdItem.getSize();
    fileSize = bytesLeft;

    //System.out.println("Searching for "+searchKey.getParentDirId()+" "+searchKey.getObjectId());

    file = new ReiserFile(fileSize, blockReader);
  
    while (bytesLeft > 0)
    {
      item = findNextItem(searchKey);

      /* Fell off the end of the object? */
      if (item.getHeader().getKey().getObjectId() != searchKey.getObjectId())
        throw new FsException("Unexpected end of file");

      if (item instanceof IndirectItem)
      {
        int entries = ((IndirectItem) item).numEntries();

        file.add( (IndirectItem) item );
        bytesLeft = bytesLeft - sb.getBlockSize()*entries;
        //System.out.println("Indirect item ("+item.getHeader().getKey().getOffset()+")"+ sb.getBlockSize()*entries);
      }

      if (item instanceof DirectItem)
      {
        file.add( (DirectItem) item );
        bytesLeft = bytesLeft - item.getHeader().getItemLen();
        //System.out.println("Direct item ("+item.getHeader().getKey().getOffset()+")"+ item.getHeader().getItemLen());
      }

      searchKey.setOffset(fileSize - bytesLeft);
    }

    return file;
  }

  /**
   * Create a Directory given the StatDataItem describing it.
   * @param sdItem   a StatDataItem
   * @return Directory
   * @exception FsException
   * @exception IOException
   */
  public Directory createReiserDirectory(StatDataItem sdItem) throws FsException, IOException
  {
    Key inodeKey, searchKey;
    DirectoryItem dirItem;
    ReiserDirectory directory;
    long bytesLeft;

    searchKey = new KeyV2();
    inodeKey = sdItem.getHeader().getKey();   
    searchKey.setParentDirId(inodeKey.getParentDirId());
    searchKey.setObjectId(inodeKey.getObjectId());
    searchKey.setOffset(0x1);
    searchKey.setType(Key.TYPE_DIRENTRY);
    bytesLeft = sdItem.getSize();
    directory = new ReiserDirectory();
  
    while (bytesLeft > 0)
    {
      dirItem = (DirectoryItem) findNextItem(searchKey);
      directory.add(dirItem);
      bytesLeft = bytesLeft - dirItem.getHeader().getItemLen();
      try
      {
        searchKey = (Key) dirItem.getHeader().getKey().clone();
      }
      catch (CloneNotSupportedException ex)
      {
        throw new FsException("key must be cloneable");
      }

      searchKey.setOffset(searchKey.getOffset()+1);
    }

    return directory;
  }

  /**
   * Create a Device from a StatDataItem.
   * @param sdItem   a StatDataItem
   * @return Device
   * @exception FsException
   * @exception IOException
   */
  public Device createReiserDevice(StatDataItem sdItem) throws FsException, IOException
  {
    return new ReiserDevice(sdItem);
  }

  /**
   * Get the file system object that is pointed to by the given Inode.
   *
   * @param inode   a Inode
   * @return FsObject
   * @exception FsException
   * @exception IOException
   */
  public FsObject getObject(Inode inode) throws FsException, IOException
  {
    StatDataItem sdItem;
    Key key;
    int fileType;

    sdItem = (StatDataItem) inode;
    key = sdItem.getHeader().getKey();

    fileType = inode.getMode() & Inode.FILETYPE_MASK;


    switch (fileType)
    {
      case Inode.DIRECTORY_MODE:  return createReiserDirectory(sdItem);
      case Inode.FILE_MODE: return createReiserFile(sdItem);    
      case Inode.SYMLINK_MODE: return createReiserSymLink(sdItem);    
      case Inode.BLOCKDEV_MODE: return createReiserDevice(sdItem);    
      case Inode.CHARDEV_MODE: return createReiserDevice(sdItem);    
      case Inode.SOCKET_MODE: return new ReiserSocket();
      case Inode.FIFO_MODE: return new ReiserFifo();
      default: break;
    }

    throw new FsException("Unsupported file type "+fileType);
  }
}
