/*
 * $Id: StatDataV1.java,v 1.7 2004/04/08 23:45:12 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: StatDataV1.java,v $
 * Revision 1.7  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.6  2003/11/12 21:29:31  wrossi
 * Working on stat data items.
 *
 * Revision 1.5  2003/11/12 21:14:31  wrossi
 * Created stat data item
 *
 * Revision 1.4  2003/11/12 11:05:49  wrossi
 * Added getRDev function
 *
 * Revision 1.3  2003/11/11 21:30:22  wrossi
 * Working on stat data
 *
 * Revision 1.2  2003/11/11 11:14:54  wrossi
 * Filled in setters/getters
 *
 * Revision 1.1  2003/11/11 00:56:06  wrossi
 * Adding stat data
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a Stat data v1.
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |      Mode         |  Num Links     |
 *        --------------------------------------
 *  0004  |      User ID      | Group Id       |
 *        --------------------------------------
 *  0008  |             File Size              |
 *        --------------------------------------
 *  000C  |            Access Time             |
 *        --------------------------------------
 *  0010  |            Modify Time             |
 *        --------------------------------------
 *  0014  |            Changed Time            |
 *        --------------------------------------
 *  0018  |            Rdev/Blocks             |
 *        --------------------------------------
 *  001C  |            FirstDirect             |
 *        --------------------------------------
 * </pre>
 */

public class StatDataV1 extends StatDataItem
{
  protected int mode;
  protected int numLinks;
  protected int userId;
  protected int groupId;
  protected long size;
  protected long accessTime;  // in seconds since jan 1, 1970?
  protected long modifyTime;  
  protected long changedTime;  
  protected long blocks;
  protected long firstDirect;

  public StatDataV1(byte[] buffer, int bufferOffset, LeafNode block, ItemHead header) throws FsException
  { 
    super(buffer, bufferOffset, block, header);
    load();   
  }

  public static int getDataSize() { return 0x20; }

  public int getMode() { return mode; }
  public void setMode(int aMode) { mode=aMode; }

  public long getNumLinks() { return numLinks; }
  public void setNumLinks(long aNumLinks) { numLinks=(int)aNumLinks; }

  public long getUserId() { return userId; }
  public void setUserId(long aUserId) { userId=(int) aUserId; }

  public long getGroupId() { return groupId; }
  public void setGroupId(long aGroupId) { groupId=(int) aGroupId; }

  public long getSize() { return size; }
  public void setSize(long aSize) { size=aSize; }

  public long getAccessTime() { return accessTime; }
  public void setAccessTime(long aAccessTime) { accessTime=aAccessTime; }

  public long getModifyTime() { return modifyTime; }
  public void setModifyTime(long aModifyTime) { modifyTime=aModifyTime; }

  public long getChangedTime() { return changedTime; }
  public void setChangedTime(long aChangedTime) { changedTime=aChangedTime; }

  public long getBlocks() { return blocks; }
  public void setBlocks(long aBlocks) { blocks=aBlocks; }

  public long getFirstDirect() { return firstDirect; }
  public void setFirstDirect(long aFirstDirect) { firstDirect=aFirstDirect; }

  public int getRDev() { return (int) (firstDirect & 0xffff); }

  public void load() throws FsException
  { 
    int blkoffset = bufferOffset+header.getItemLocation();

    setMode(FsUtils.getU16(buffer, blkoffset+0x0));
    setNumLinks(FsUtils.getU16(buffer, blkoffset+0x2));
    setUserId(FsUtils.getU16(buffer, blkoffset+0x4));
    setGroupId(FsUtils.getU16(buffer, blkoffset+0x6));
    setSize(FsUtils.getU32(buffer, blkoffset+0x8));
    setAccessTime(FsUtils.getU32(buffer, blkoffset+0xC));
    setModifyTime(FsUtils.getU32(buffer, blkoffset+0x10));
    setChangedTime(FsUtils.getU32(buffer, blkoffset+0x14));
    setBlocks(FsUtils.getU32(buffer, blkoffset+0x18));
    setFirstDirect(FsUtils.getU32(buffer, blkoffset+0x1C));
  }
}
