/*
 * $Id: StatDataV2.java,v 1.5 2004/04/08 23:45:12 wrossi Exp $
 
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: StatDataV2.java,v $
 * Revision 1.5  2004/04/08 23:45:12  wrossi
 * Working on javadoc
 *
 * Revision 1.4  2003/11/12 21:29:31  wrossi
 * Working on stat data items.
 *
 * Revision 1.3  2003/11/12 11:05:49  wrossi
 * Added getRDev function
 *
 * Revision 1.2  2003/11/11 21:30:22  wrossi
 * Working on stat data
 *
 * Revision 1.1  2003/11/11 00:56:06  wrossi
 * Adding stat data
 *
 *
 */

package rossi.fstools.fs.reiserfs;

import rossi.fstools.fs.FsUtils;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.DiskStructure;

/**
 * Representation of a Stat data v2.
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |      Mode         |  Attributes    |
 *        --------------------------------------
 *  0004  |             Num Links              |
 *        --------------------------------------
 *  0008  |             File Size              |
 *        --------------------------------------
 *  000C  |             File Size              |
 *        --------------------------------------
 *  0010  |             User Id                |
 *        --------------------------------------
 *  0014  |             Group Id               |
 *        --------------------------------------
 *  0018  |            Access Id               |
 *        --------------------------------------
 *  001C  |            Modify Time             |
 *        --------------------------------------
 *  0020  |            Changed Time            |
 *        --------------------------------------
 *  0024  |               Blocks               |
 *        --------------------------------------
 *  0028  |            Rdev/Generation         |
 *        --------------------------------------
 * </pre>
 */

public class StatDataV2 extends StatDataItem
{
  protected int mode;
  protected int attrs;
  protected long numLinks;
  protected long size;
  protected long userId;
  protected long groupId;
  protected long accessTime;  // in seconds since jan 1, 1970?
  protected long modifyTime;  
  protected long changedTime;  
  protected long blocks;
  protected long generation;  // not sure what this is.  A hashcode perhaps?

  public final static int IMMUTABLE = 0x00000010;
  public final static int SYNC      = 0x00000008;
  public final static int NOATIME   = 0x00000080;
  public final static int NODUMP    = 0x00000040;
  public final static int SECRM     = 0x00000001;
  public final static int UNRM      = 0x00000002;
  public final static int COMPR     = 0x00000004;
  public final static int NOTAIL    = 0x00008000;

  public StatDataV2(byte[] buffer, int bufferOffset, LeafNode block, ItemHead header) throws FsException
  { 
    super(buffer, bufferOffset, block, header);
    load();   
  }

  public static int getDataSize() { return 0x2C; }

  public int getMode() { return mode; }
  public void setMode(int aMode) { mode=aMode; }

  public int getAttrs() { return attrs; }
  public void setAttrs(int aAttrs) { attrs=aAttrs; }

  public long getNumLinks() { return numLinks; }
  public void setNumLinks(long aNumLinks) { numLinks=aNumLinks; }

  public long getSize() { return size; }
  public void setSize(long aSize) { size=aSize; }

  public long getUserId() { return userId; }
  public void setUserId(long aUserId) { userId=aUserId; }

  public long getGroupId() { return groupId; }
  public void setGroupId(long aGroupId) { groupId=aGroupId; }

  public long getAccessTime() { return accessTime; }
  public void setAccessTime(long aAccessTime) { accessTime=aAccessTime; }

  public long getModifyTime() { return modifyTime; }
  public void setModifyTime(long aModifyTime) { modifyTime=aModifyTime; }

  public long getChangedTime() { return changedTime; }
  public void setChangedTime(long aChangedTime) { changedTime=aChangedTime; }

  public long getBlocks() { return blocks; }
  public void setBlocks(long aBlocks) { blocks=aBlocks; }

  public long getGeneration() { return generation; }
  public void setGeneration(long aGeneration) { generation=aGeneration; }

  public int getRDev() { return (int) (generation & 0xffff); }

  public void load() throws FsException
  { 
    int blkoffset = bufferOffset+header.getItemLocation();

    setMode(FsUtils.getU16(buffer, blkoffset+0x0));
    setAttrs(FsUtils.getU16(buffer, blkoffset+0x2));
    setNumLinks(FsUtils.getU32(buffer, blkoffset+0x4));
    setSize(FsUtils.getU64(buffer, blkoffset+0x8));
    setUserId(FsUtils.getU32(buffer, blkoffset+0x10));
    setGroupId(FsUtils.getU32(buffer, blkoffset+0x14));
    setAccessTime(FsUtils.getU32(buffer, blkoffset+0x18));
    setModifyTime(FsUtils.getU32(buffer, blkoffset+0x1C));
    setChangedTime(FsUtils.getU32(buffer, blkoffset+0x20));
    setBlocks(FsUtils.getU32(buffer, blkoffset+0x24));
    setGeneration(FsUtils.getU32(buffer, blkoffset+0x28));
  }
}
