/*
 * $Id: Partition.java,v 1.2 2004/02/28 21:46:09 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: Partition.java,v $
 * Revision 1.2  2004/02/28 21:46:09  wrossi
 * Got partition parsing working
 *
 * Revision 1.1  2004/02/28 20:44:43  wrossi
 * Partition support
 *
 *
 */

package rossi.fstools.partition;

import rossi.fstools.fs.DiskStructure;
import rossi.fstools.fs.FsException;
import rossi.fstools.fs.FsUtils;

/*
 * Representation of a partition table entry
 *
 * <pre>
 *        31        24        16       8       0
 *        --------------------------------------
 *  0000  |  cyl    | sect   | head    | Boot  |
 *        --------------------------------------
 *  0004  | end cyl |end sect| endhead | system|
 *        --------------------------------------
 *  0008  |        Start Sector                |
 *        --------------------------------------
 *  000C  |        Number of Sectors           |
 *        --------------------------------------
 * </pre>
 */

public class Partition implements DiskStructure
{
  protected byte boot;           /* Boot indicator -- active flag */
  protected byte startHead;      /* Starting head                 */
  protected byte startSector;    /* Starting sector               */
  protected byte startCyl;       /* Starting cylinder             */
  protected byte system;         /* System indicator              */
  protected byte endHead;        /* End head                      */
  protected byte endSector;      /* End sector                    */
  protected byte endCyl;         /* End cylinder                  */
  protected long startLBA;       /* Starting Sector (LBA address) */
  protected long numLBASector;   /* Number of Sectors (LBA)       */

  public final static byte DOS_EXTENDED_PARTITION = 5;
  public final static byte LINUX_EXTENDED_PARTITION = (byte) 0x85;
  public final static byte WIN98_EXTENDED_PARTITION = 0x0f;

  public byte getBoot() { return boot; }
  public void setBoot(byte aBoot) { boot=aBoot; }

  public byte getStartHead() { return startHead; }
  public void setStartHead(byte aStartHead) { startHead=aStartHead; }

  public byte getStartSector() { return startSector; }
  public void setStartSector(byte aStartSector) { startSector=aStartSector; }

  public byte getStartCyl() { return startCyl; }
  public void setStartCyl(byte aStartCyl) { startCyl=aStartCyl; }

  public byte getSystem() { return system; }
  public void setSystem(byte aSystem) { system=aSystem; }

  public byte getEndHead() { return endHead; }
  public void setEndHead(byte aEndHead) { endHead=aEndHead; }

  public byte getEndSector() { return endSector; }
  public void setEndSector(byte aEndSector) { endSector=aEndSector; }

  public byte getEndCyl() { return endCyl; }
  public void setEndCyl(byte aEndCyl) { endCyl=aEndCyl; }

  public long getStartLBA() { return startLBA; }
  public void setStartLBA(long aStartLBA) { startLBA=aStartLBA; }

  public long getNumLBASector() { return numLBASector; }
  public void setNumLBASector(long aNumLBASector) { numLBASector=aNumLBASector; }

  public boolean isExtended()
  {
    if (system == DOS_EXTENDED_PARTITION ||
        system == LINUX_EXTENDED_PARTITION ||
        system == WIN98_EXTENDED_PARTITION)
      return true;
    return false;
  }

  /** Returns the size in bytes of the structure */
  public int getDataSize()
  {
    return 16;
  }

  /** Loads the object from a memory buffer */
  public void loadFromBuffer(byte[] buffer, int offset) throws FsException
  {
    setBoot(buffer[offset+0]);
    setStartHead(buffer[offset+1]);
    setStartSector(buffer[offset+2]);
    setStartCyl(buffer[offset+3]);
    setSystem(buffer[offset+4]);
    setEndHead(buffer[offset+5]);
    setEndSector(buffer[offset+6]);
    setEndCyl(buffer[offset+7]);

    setStartLBA(FsUtils.getU32(buffer, offset+0x8));
    setNumLBASector(FsUtils.getU32(buffer, offset+0xC));
  }

  public String toString()
  {
    StringBuffer sb = new StringBuffer();
    
    sb.append((getBoot()==0x80)?"* ":" ");
    sb.append("\t0x");
    sb.append(Integer.toHexString(getSystem()&0xff));
    sb.append("\t");
    /*
    sb.append(Integer.toString(getStartHead()));
    sb.append("\t");
    sb.append(Integer.toString(getStartSector()));
    sb.append("\t");
    sb.append(Integer.toString(getStartCyl()));
    sb.append("\t");
    sb.append(Integer.toString(getEndHead()));
    sb.append("\t");
    sb.append(Integer.toString(getEndSector()));
    sb.append("\t");
    sb.append(Integer.toString(getEndCyl()));
    sb.append("\t");
    */
    sb.append(Long.toString(getStartLBA()));
    sb.append("\t");
    sb.append(Long.toString(getNumLBASector()));

    return sb.toString();
  }
}
