/*
 * $Id: FileViewPanel.java,v 1.6 2004/02/10 21:27:16 wrossi Exp $
 *
 * (C) Copyright 2004 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: FileViewPanel.java,v $
 * Revision 1.6  2004/02/10 21:27:16  wrossi
 * Added slack display
 *
 * Revision 1.5  2004/02/08 19:19:22  wrossi
 * removed printouts
 *
 * Revision 1.4  2004/02/08 18:34:18  wrossi
 * Improved hex performance
 *
 * Revision 1.3  2004/01/28 13:48:22  wrossi
 * Added blocks display
 *
 * Revision 1.2  2004/01/27 11:01:28  wrossi
 * Have hex display sort of working
 *
 * Revision 1.1  2004/01/23 21:39:36  wrossi
 * Adding file viewer
 *
 *
 */

package rossi.fstoolsui.fstree;

import rossi.fstools.fs.File;
import rossi.fstools.io.BlockPointer;

//import rossi.metadata.FormatIdentification;

import java.awt.BorderLayout;

import java.util.Iterator;

import javax.swing.JTabbedPane;
import javax.swing.JScrollPane;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;

/** Panel to display some data about a file. */
public class FileViewPanel extends JPanel
{
  protected JLabel typeLabel;
  protected JLabel hexLabel;
  protected JLabel textLabel;
  protected JLabel blockLabel;
  protected JLabel slackLabel;
  protected JTabbedPane tabPane;

  private final static int MAX_BUFFER = 8192;
  private final static char hexDigits[] = {'0', '1', '2', '3', '4', '5', '6', '7', 
                                           '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

  public FileViewPanel()
  {
    initialize();
    performLayout();
  }

  protected void initialize()
  {
    tabPane = new JTabbedPane();
    typeLabel = new JLabel();
    typeLabel.setVerticalAlignment(SwingConstants.TOP);

    hexLabel = new JLabel();
    hexLabel.setVerticalAlignment(SwingConstants.TOP);

    textLabel = new JLabel();
    textLabel.setVerticalAlignment(SwingConstants.TOP);

    blockLabel = new JLabel();
    blockLabel.setVerticalAlignment(SwingConstants.TOP);

    slackLabel = new JLabel();
    slackLabel.setVerticalAlignment(SwingConstants.TOP);
  }

  protected void performLayout()
  {
    setLayout(new BorderLayout());

    //tabPane.add(new JScrollPane(typeLabel), "Type");
    tabPane.add(new JScrollPane(hexLabel), "Hex");
    tabPane.add(new JScrollPane(textLabel), "Text");
    tabPane.add(new JScrollPane(blockLabel), "Blocks");
    tabPane.add(new JScrollPane(slackLabel), "Slack");
    add(tabPane, "Center");
  }

  public void setFile(File f) 
  {
    byte buffer[];
    byte slack[];

    try
    {
      buffer = getBuffer(f.getData());
      slack = getBuffer(f.getSlack());
      //typeLabel.setText("<html><pre>"+FormatIdentification.identify(buffer)+"</pre>");
      hexLabel.setText("<html><pre>"+hexString(buffer)+"</pre>");

      textLabel.setText(bytesToText(buffer));
      blockLabel.setText(getBlocksText(f));
      slackLabel.setText("<html><pre>"+hexString(slack)+"</pre>");
    }
    catch (Exception ex)
    {
      hexLabel.setText(ex.getClass().getName()+": "+ex.getMessage());
      ex.printStackTrace();
    }
  }

  protected byte[] getBuffer(InputStream in) throws IOException
  {
    ByteArrayOutputStream bout = new ByteArrayOutputStream();
    byte[] buffer = new byte[4096];
    int count = 0;

    while (count < MAX_BUFFER)
    {
      int br;

      br = in.read(buffer);
      if (br == -1)
        break;

      bout.write(buffer, 0, br);
      count += br;
    }

    bout.close();
    return bout.toByteArray();
  }

  protected String getBlocksText(File f)
  {
    BlockPointer bp;
    StringBuffer sb = new StringBuffer(262144);

    sb.append("<html><table><tr><td>Block no.</td><td>Offset</td><td>Size</td></tr>");
    for (Iterator itr = f.getData().getBlocks(); itr.hasNext(); )
    {
      bp = (BlockPointer) itr.next();
      sb.append("<tr><td>");
      sb.append(Long.toString(bp.getBlockNum()));
      sb.append("</td><td>");
      sb.append(Integer.toString(bp.getOffset()));
      sb.append("</td><td>");
      sb.append(Integer.toString(bp.getLength()));
      sb.append("</td></tr>");
    }
    sb.append("</table></html>");
    return sb.toString();
  }

  protected String hexString(byte[] buffer)
  {
    StringBuffer sb = new StringBuffer(262144);

    for (int i=0; i<=buffer.length; i+=16)
    {
      formatHexLine(buffer, i, sb);
    }

    if (buffer.length == MAX_BUFFER)
      sb.append("\n\n**End of buffer, to see more extract the file**");
 
    return sb.toString();
  }

  protected String bytesToText(byte buffer[])
  {
    StringBuffer sb = new StringBuffer();
    sb.append("<html><pre>");
    for (int i=0; i<buffer.length; i++)
      formatByte(buffer[i], sb, true);
    sb.append("</pre>");

    if (buffer.length == MAX_BUFFER)
      sb.append("\n\n**End of buffer, to see more extract the file**");

    return sb.toString();
  }

  protected void formatByte(byte b, StringBuffer sb, boolean allowNewline)
  {
    if (b > 31 && b < 127)
    {
      switch ( (char) b )
      {
        case '&':
          sb.append("&amp;");
          break;
        case '<':
          sb.append("&lt;");
          break;
        case '>':
          sb.append("&gt;");
          break;
        default:
          sb.append((char) b);
          break;
      }
    }
    else
    {
      if (allowNewline && (b == 0xa || b == 0xd))
      {
        sb.append((char) b);
      }
      else
      {
        sb.append(".");
      }
    }
  }

  protected void intToHexString(StringBuffer sb, int hexNumber, int numDigits)
  {
    char buffer[] = new char[numDigits];   
    int x = hexNumber;

    for (int i=buffer.length-1; i>=0; i--)
    {
      buffer[i] = hexDigits[x & 0xf];
      x >>= 4;
    }

    sb.append(buffer);
  }

  protected void formatHexLine(byte[] buffer, int offset, StringBuffer sb)
  {
    intToHexString(sb, offset, 4);

    sb.append(" ");

    for (int x=0; x<2; x++)
    {
      for (int i=0; i<8; i++)
      {
        sb.append(" ");
        if ( (i+offset+x*8) < buffer.length)
        {
          int b = ((int) buffer[i+offset+x*8]) & 0xff;
          intToHexString(sb, b, 2);
        }
        else
        {
          sb.append("  ");
        }
      }
    }

    sb.append("  ");

    for (int i=0; i<16; i++)
    {
      if ((i+offset) >= buffer.length)
      {
        sb.append(" ");
        continue;
      }

      byte b = buffer[i+offset];
      formatByte(b, sb, false);
    }

    sb.append("\n");
  }

}
