/*
 * $Id: PartitionTableModel.java,v 1.4 2004/03/05 21:37:35 wrossi Exp $
 *
 * (C) Copyright 2003 Rossi Engineering, Inc.  All Rights Reserved
 *
 * $Log: PartitionTableModel.java,v $
 * Revision 1.4  2004/03/05 21:37:35  wrossi
 * Added partition types to partition dialog
 *
 * Revision 1.3  2004/03/04 21:16:58  wrossi
 * Working on partition table dialog
 *
 * Revision 1.2  2004/03/02 21:36:52  wrossi
 * Added getPartitionForRow()
 *
 * Revision 1.1  2004/03/02 21:17:46  wrossi
 * Working on partition dialog
 *
 * Revision 1.9  2004/03/01 12:02:46  wrossi
 * Added menu items and actions
 *
 */

package rossi.fstoolsui.fstree;

import rossi.fstools.partition.Partition;

import javax.swing.table.AbstractTableModel;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Properties;

import java.io.InputStream;

import java.text.DecimalFormat;

public class PartitionTableModel extends AbstractTableModel
{
  protected List partitionList;
  protected DecimalFormat nf = new DecimalFormat("###.#");
  protected Properties partTypes;
  
  /** Constructor.  Takes a list of InodeItem objects */
  public PartitionTableModel(List pList)
  {
    loadPartitions(pList);
    loadPartitionTypes();
  }

  public int getRowCount()
  {
    return partitionList.size();
  }

  public int getColumnCount()
  {
    return 3;
  }

  public String getColumnName(int col)
  {
    String colnames[] = {"#", "System", "Size"};

    return colnames[col];
  }

  public Class getColumnClass(int col)
  {
    return String.class;
  }

  public boolean isCellEditable(int row, int col)
  {
    return false;
  }

  public Object getValueAt(int row, int col)
  {
    NumberedPartition part = (NumberedPartition) partitionList.get(row);

    return cellFromPartition(part, col);
  }

  public Object cellFromPartition(NumberedPartition part, int col)
  {
    switch (col)
    {
      case 0: return Integer.toString(part.getOrdinal());
      case 1: return getPartitionTypeString(part.getPartition());
      case 2: return getPartSizeString(part.getPartition());
    }

    return "";
  }

  public void setValueAt(Object obj, int row, int col)
  {
  }

  public Partition getPartitionForRow(int row)
  { 
    NumberedPartition npart = (NumberedPartition) partitionList.get(row);
    return npart.getPartition();
  }

  public int getPartitionOrdinalForRow(int row)
  { 
    NumberedPartition npart = (NumberedPartition) partitionList.get(row);
    return npart.getOrdinal();
  }

  protected String getPartSizeString(Partition part)
  { 
    double size = part.getNumLBASector() * 512;

    size = size / 1000000.0;
    if (size < 1000.0)
      return nf.format(size)+" MB";

    size = size / 1000.0;
      return nf.format(size)+" GB";
  }

  protected String getPartitionTypeString(Partition part)
  { 
    String id = Integer.toHexString( part.getSystem() & 0xff );
    String desc = partTypes.getProperty(id);
    if (desc == null)
      return id;

    return id+": "+desc;
  }

  protected void loadPartitionTypes()
  {
    try
    {
      InputStream in = getClass().getClassLoader().getResourceAsStream("systypes.properties");
      partTypes = new Properties();
      partTypes.load(in);
      in.close();
    }
    catch (Exception ex)
    {
    }
  }

  protected void loadPartitions(List pList)
  {
    int ordinal = 0;
    partitionList = new ArrayList();
    for (Iterator itr=pList.iterator(); itr.hasNext(); )
    {
      Partition p = (Partition) itr.next();
      ordinal++;

      if (p.isExtended())  // don't show extended partitions
        continue;

      if (p.getNumLBASector() == 0)  // don't show zero length partitions
        continue;

      NumberedPartition npart = new NumberedPartition(ordinal, p);
      partitionList.add(npart);
    }
  }

  class NumberedPartition 
  {
    protected int ordinal;
    protected Partition part;

    public NumberedPartition(int ordinal, Partition p)
    {
      this.ordinal = ordinal;
      part = p;
    }

    public Partition getPartition() { return part; }
    public void setPartition(Partition aPart) { part=aPart; }

    public int getOrdinal() { return ordinal; }
    public void setOrdinal(int aOrdinal) { ordinal=aOrdinal; }
  }
}
