/******************************************************************************
 *
 * Copyright (c) 1998,99 by Mindbright Technology AB, Stockholm, Sweden.
 *                 www.mindbright.se, info@mindbright.se
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 *****************************************************************************
 * $Author: mats $
 * $Date: 1999/12/19 15:24:00 $
 * $Name: rel1-1-5 $
 *****************************************************************************/
package mindbright.ssh;

import java.io.*;
import java.net.*;
import java.util.Properties;
import java.util.Enumeration;
import java.util.NoSuchElementException; 
import java.util.Date; 

import mindbright.terminal.*;
import mindbright.security.*;

public final class SSHInteractiveClient extends SSHClient
  implements Runnable, SSHClientUser, SSHAuthenticator {

  public static final boolean   expires   = false;
  public static final boolean   licensed  = false;

  public static final String licenseMessage = "This copy of MindTerm is licensed to ";
  public static final String licensee       = "nobody";

  public static final long      validFrom = 945616918204L; // 991219/16:22
  public static final long      validTime = (33L * 24L * 60L * 60L * 1000L);

  public static boolean wantHelpInfo       = true;
  public static String  customStartMessage = null;

  public class KeepAliveThread extends Thread {
    int interval;
    public KeepAliveThread(int i) {
      super();
      interval = i;
    }
    public synchronized void setInterval(int i) {
      interval = i;
    }
    public void run() {
      int i;
      SSHPduOutputStream ignmsg;
      while(true) {
	try {
	  synchronized(this) {
	    i = interval;
	  }
	  sleep(1000 * i);
	  if(SSHInteractiveClient.this.controller != null) {
	      ignmsg = new SSHPduOutputStream(MSG_DEBUG, controller.sndCipher);
	      ignmsg.writeString("heartbeat");
	      controller.transmit(ignmsg);
	  }
	} catch (Exception e) {
	    // !!!
	}
      }
    }
  }

  static public class DumbConsoleThread implements Runnable {
    SSHChannelController controller;
    SSHStdIO             console;

    public DumbConsoleThread(SSHChannelController controller, SSHStdIO console) {
      this.controller = controller;
      this.console    = console;
    }

    public void run() {
      SSHPduOutputStream stdinPdu;
      String line;
      try {
	while(true) {
	  line = console.promptLine("", "", false);
	  stdinPdu = new SSHPduOutputStream(SSH.CMSG_STDIN_DATA, console.sndCipher);
	  stdinPdu.writeString(line + "\n");
	  controller.transmit(stdinPdu);
	  Thread.sleep(400);
	}
      } catch (SSHStdIO.CtrlDPressedException e) {
	controller.sendDisconnect("exit");
      } catch (Exception e) {
	controller.alert("Error in console-thread: " + e.toString());
      }
    }
  }

  public static String copyright() {
    return "Copyright (c) 1998,99 by Mindbright Technology AB, Stockholm, Sweden";
  }

  KeepAliveThread keepAliveThread;

  Thread        dumbConsoleThread;
  String        sshHomeDir;
  String        knownHosts;
  SSHRSAKeyFile keyFile;

  SSHMenuHandler menus;
  SSHStdIO       sshStdIO;

  public boolean quiet;
  boolean        initQuiet;

  static public final String PROPS_FILE_EXT  = ".mtp";
  static public final String GLOB_PROPS_FILE = "mindterm" + PROPS_FILE_EXT;
  static public final String DEF_IDFILE = "identity";

  static public final int PROP_NAME    = 0;
  static public final int PROP_VALUE   = 1;
  static public final int PROP_DESC    = 2;
  static public final int PROP_ALLOWED = 3;
  static public final Properties defaultProperties = new Properties();
  static public final String[][] defaultPropDesc = {
    { "server",   "",            "name of server to connect to", "" },
    { "realsrv",  "",            "real address of sshd if it is behind a firewall", "" },
    { "localhst", "0.0.0.0",     "address to use as localhost", "" },
    { "port",     String.valueOf(SSH.DEFAULTPORT),
                                "port on server to connect to", "" },
    { "usrname",  "",           "username to login as", "" },
    { "cipher",   SSH.getCipherName(SSH.CIPHER_DEFAULT),
                                "name of block cipher to use",
                                ("( " + SSH.listSupportedCiphers() + ")") },
    { "authtyp",  "passwd",      "method of authentication",
                                ("( " + SSH.listSupportedAuthTypes() + ")") },
    { "idfile",   DEF_IDFILE,    "name of file containing identity (rsa-key)", "" },
    { "display",  "localhost:0", "local display definition (i.e. <host>:<screen>)", "" },
    { "mtu",      "0",           "maximum packet size to use (0 means use default)",
                                "(4096 - 256k)" },
    { "escseq",   "~$",          "sequence of characters to type to enter local command-shell", "" },
    { "secrand",  "0",           "level of security in random-seed (for generating session-key)",
      "(0-2, 0=low and 2=high)" },
    { "alive",    "0",           "Connection keep-alive interval in seconds (0 means none)", "(0-600)" },
    { "x11fwd",   "false",       "indicates whether X11 display is forwarded or not", "(true/false)" },
    { "prvport",  "false",       "indicates whether to use a privileged port or not (locally)", "(true/false)" },
    { "forcpty",  "true",        "indicates whether to allocate a pty or not", "(true/false)" },
    { "remfwd",   "false",       "indicates whether we allow remote connects to local forwards", "(true/false)" },
    { "idhost",   "true",        "indicates whether to check hosts host-key in 'known_hosts'", "(true/false)" },
    { "portftp",  "false",       "indicates whether to enable ftp 'PORT' command support", "(true/false)" },
  };
  static {
    for(int i = 0; i < defaultPropDesc.length; i++)
      defaultProperties.put(defaultPropDesc[i][PROP_NAME], defaultPropDesc[i][PROP_VALUE]);
  }

  Properties props;
  protected String currentPropsFile;
  protected String currentPropsName;
  public boolean autoSaveProps;
  public boolean autoLoadProps;

  public Properties initTermProps;

  protected boolean propsChanged;

  // We store the password until logout to be able to clone the window
  //
  String password;
  String tisPassword;
  String rsaPassword;

  public SSHInteractiveClient(boolean quiet, boolean cmdsh,
			      boolean autoSaveProps, boolean autoLoadProps,
			      Properties initProps) {
    super(null, null); // !!!
    authenticator = this;
    user          = this;

    this.quiet     = quiet;
    this.initQuiet = quiet;

    currentPropsFile = null;

    setConsole(new SSHStdIO());
    sshStdIO = (SSHStdIO)console;
    sshStdIO.setClient(this);
    sshStdIO.enableCommandShell(cmdsh);

    this.knownHosts = KNOWN_HOSTS_FILE;

    this.autoSaveProps = autoSaveProps;
    this.autoLoadProps = autoLoadProps;

    if(initProps != null)
      setProperties(initProps, true);

    // Clear this here, since it should only be set after interaction with user
    //
    currentPropsFile = null;

    propsChanged     = false;
  }

  public SSHInteractiveClient(boolean quiet, boolean cmdsh,
			      boolean autoSaveProps, boolean autoLoadProps,
			      String propsFile) throws IOException {
    this(quiet, cmdsh, autoSaveProps, autoLoadProps, (Properties)null);
    setPropertyFileAndLoad(propsFile, true);
  }

  public SSHInteractiveClient(SSHInteractiveClient clone) {
    this(true, clone.sshStdIO.hasCommandShell(),
	 clone.autoSaveProps, clone.autoLoadProps, clone.props);
    this.sshHomeDir    = clone.sshHomeDir;
    this.password      = clone.password;
    this.tisPassword   = clone.tisPassword;
    this.rsaPassword   = clone.rsaPassword;
    this.currentPropsFile = clone.currentPropsFile;
    this.currentPropsName = clone.currentPropsName;
    this.activateTunnels  = false;

    this.wantHelpInfo       = clone.wantHelpInfo;
    this.customStartMessage = clone.customStartMessage;
  }

  public void setMenus(SSHMenuHandler menus) {
    this.menus = menus;
  }

  public void updateMenus() {
    if(menus != null)
      menus.update();
  }

  public void printCopyright() {
    console.println(copyright());

    if(licensed) {
	console.println(licenseMessage + licensee);
    }

    if(customStartMessage != null) {
	console.println(customStartMessage);
    }
  }

  void printHelpInfo() {
    if(!wantHelpInfo)
      return;

    if(sshHomeDir != null)
      console.println("MindTerm home: " + sshHomeDir);

    if(sshStdIO.hasCommandShell()) {
      console.println("\tpress <ctrl> + 'D' to enter local command-shell");
      if(isDumb())
	console.println("\t(...you might have to press ENTER also...)");
    }
    if(menus != null && menus.havePopupMenu) {
      console.println("\tpress <ctrl> + <mouse-" + menus.getPopupButton() + "> for main-menu");
    }
    console.println("");
  }

  boolean hasExpired() {
    boolean expired = false;
    long now = System.currentTimeMillis();

    if(licensed)
      return false;

    if(expires) {
      int daysRemaining = (int)((validTime - (now - validFrom)) / (1000L * 60L * 60L * 24L));
      if(daysRemaining <= 0) {
	console.println("This is a demo-version of MindTerm, it has expired!");
	console.println("Please go to http://www.mindbright.se/mindterm/ to get a copy");
	expired = true;
      } else {
	console.println("");
	console.println("This is a demo-version of MindTerm, it will expire in " + daysRemaining + " days");
	console.println("");
      }
    } else {
      int daysOld = (int)((now - validFrom) / (1000L * 60L * 60L * 24L));
      console.println("");
      console.println("This is a demo-version of MindTerm, it is " + daysOld + " days old.");
      console.println("Please go to http://www.mindbright.se/mindterm/");
      console.println("\tto check for new versions now and then");
      console.println("");
    }
    return expired;
  }

  void initRandomSeed() {
    console.print("Initializing random generator, please wait...");
    SSH.initSeedGenerator();
    console.println("done");
  }

  public void doSingleCommand(String commandLine, boolean background, long msTimeout)
    throws IOException {
    boolean haveDumbConsole = (wantPTY() && isDumb());

    initRandomSeed();
    console.println("");

    printHelpInfo();

    this.commandLine = commandLine;

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("TerminalEmulator");
	console.println("Full network access granted, can do tunneling and connect to any host");
      } catch (netscape.security.ForbiddenTargetException e) {
	console.println("Full network access denied, normal applet-security applies");
      }
      console.println("");
    }

    bootSSH(false);

    if(haveDumbConsole) {
      startDumbConsole();
    }

    if(background)
      startExitMonitor(msTimeout);
    else
      waitForExit(msTimeout);

    if(haveDumbConsole) {
      stopDumbConsole();
    }
  }

  public void run() {
    boolean doCommandShell;
    boolean gotExtMsg;

    initRandomSeed();

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("TerminalEmulator");
	console.println("Full network access granted, can do tunneling and connect to any host");
      } catch (netscape.security.ForbiddenTargetException e) {
	console.println("Full network access denied, normal applet-security applies");
      }
      console.println("");
    }

    if(hasExpired()) {
      while(true) {
	try {
	  Thread.sleep(100000);
	} catch (InterruptedException e) {
	}
      }
    }

    boolean keepRunning = true;
    while(keepRunning) {
      doCommandShell = false;
      gotExtMsg      = false;
      try {
	console.println("");
	printHelpInfo();

	// This starts a connection to the sshd and all the related stuff...
	//
	bootSSH(true);

	if(isDumb())
	  startDumbConsole();

	// Join main-receiver channel thread and wait for session to end
	//
	controller.waitForExit();

	if(isDumb())
	  stopDumbConsole();

	if(sshStdIO.isConnected()) {
	  // Server died on us without sending disconnect
	  sshStdIO.serverDisconnect("\n\r\n\rServer died or connection lost");
	}

	// !!! Wait for last session to close down entirely (i.e. so
	// disconnected gets a chance to be called...)
	Thread.sleep(1000);

      } catch(SSHStdIO.CtrlDPressedException e) {
	doCommandShell = true;
      } catch(SSHStdIO.SSHExternalMessage e) {
	gotExtMsg = true;
	console.println("");
	console.println(e.getMessage());
      } catch(UnknownHostException e) {
	console.println("Unknown host: " + getProperty("server"));
	clearServerSetting();
      } catch(FileNotFoundException e) {
	console.println("File not found: " + e.getMessage());
      } catch(Exception e) {
	String msg = e.getMessage();
	if(msg.trim().length() == 0)
	  msg = e.toString();
	console.println("");
	console.println("Error connecting to " + getProperty("server") + ", reason:");
	console.println("-> " + msg);
	if(SSH.DEBUGMORE) {
	  System.out.println("If an error occured, please send the below stack-trace to mats@mindbright.se");
	  e.printStackTrace();
	}
      } catch(ThreadDeath death) {
	if(controller != null)
	  controller.killAll();
	controller = null;
	if(SSH.bogusThread != null && SSH.bogusThread.isAlive())
	  bogusThread.stop();
	if(keepAliveThread != null && keepAliveThread.isAlive())
	  keepAliveThread.stop();
	throw death;
      }

      try {
	checkSave();
      } catch (IOException e) {
	alert("Error saving settings!");
      }

      password = null;
      tisPassword = null;
      rsaPassword = null;
      activateTunnels = true;
      currentPropsFile = null;

      // !!! How do we want this to behave?
      //
      if(!gotExtMsg)
	quiet = false;

      controller = null;

      TerminalWin t = getTerminalWin();
      if(t != null)
	t.setTitle(null);

      if(doCommandShell && sshStdIO.hasCommandShell()) {
	keepRunning = sshStdIO.commandShell.doCommandShell();
      }
    }
  }

  public void clearServerSetting() {
    setProperty("server", "");
    currentPropsFile = null;
    updateMenus();
  }

  public boolean isDumb() {
    return (console.getTerminal() == null);
  }

  public TerminalWin getTerminalWin() {
    Terminal term = console.getTerminal();
    if(term != null && term instanceof TerminalWin)
      return (TerminalWin)term;
    return null;
  }

  public void startDumbConsole() {
    Runnable dumbConsole = new DumbConsoleThread(controller, sshStdIO);
    dumbConsoleThread = new Thread(dumbConsole);
    dumbConsoleThread.start();
  }
  public void stopDumbConsole() {
    dumbConsoleThread.stop();
  }

  public String promptLine(String prompt, String defaultVal, boolean echoStar) throws IOException {
    return sshStdIO.promptLine(prompt, defaultVal, echoStar);
  }
  public void updateTitle() {
    sshStdIO.updateTitle();
  }

  public void setSSHHomeDir(String sshHomeDir) {
    if(sshHomeDir != null && !sshHomeDir.endsWith(File.separator))
      sshHomeDir += File.separator;

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    try {
      File sshDir = new File(sshHomeDir);
      if(!sshDir.exists()) {
	if(askConfirmation("MindTerm home-directory: '" + sshHomeDir +
			   "' does not exist, create it?", true)) {
	  try {
	    sshDir.mkdir();
	  } catch (Throwable t) {
	    alert("Could not create home-directory, file-operations disabled.");
	    sshHomeDir = null;
	  }
	} else {
	  report("No home-directory, file-operations disabled.");
	  sshHomeDir = null;
	}
      }
    /* !!! Does not work in Netscape?!
       else if(!sshDir.isDirectory()) {
	alert("Specified home-directory is not a directory, file-operations disabled.");
	sshHomeDir = null;
      }
    */
    } catch (Throwable t) {
      if(wantHelpInfo)
	report("Can't access local file-system, file-operations disabled.");
      sshHomeDir = null;
    }
    this.sshHomeDir = sshHomeDir;
    if(this.sshHomeDir == null) {
      autoSaveProps = false;
      autoLoadProps = false;
    }
    updateMenus();
  }

  public String getSSHHomeDir() {
    return sshHomeDir;
  }

  //
  // Methods delegated to Properties and other property-related methods
  //
  public void setProperties(Properties newProps, boolean merge) throws IllegalArgumentException,
    NoSuchElementException {
    String name, value;
    Enumeration enum;
    int i;
    Properties oldProps;

    oldProps = props;
    props    = new Properties(defaultProperties);

    if(merge && oldProps != null) {
      enum = oldProps.propertyNames();
      while(enum.hasMoreElements()) {
	name  = (String)enum.nextElement();
	value = oldProps.getProperty(name);
	props.put(name, value);
      }
    }

    enum = newProps.propertyNames();
    while(enum.hasMoreElements()) {
      name  = (String)enum.nextElement();
      value = newProps.getProperty(name);
      if(!isProperty(name))
	throw new NoSuchElementException("Unknown ssh-property '" + name + "'");
      props.put(name, value);
    }

    for(i = 0; i < defaultPropDesc.length; i++) {
      name  = defaultPropDesc[i][PROP_NAME];
      value = props.getProperty(name);
      setProperty(name, value);
    }
    i = 0;
    while((value = newProps.getProperty("local" + i)) != null) {
      setProperty("local" + i, value);
      i++;
    }
    i = 0;
    while((value = newProps.getProperty("remote" + i)) != null) {
      setProperty("remote" + i, value);
      i++;
    }
  }

  public void setPropertyFile(String fname) {
    if(fname.indexOf(File.separator) == -1 && sshHomeDir != null)
      fname = sshHomeDir + fname;
    if(!fname.endsWith(PROPS_FILE_EXT))
      fname = fname + PROPS_FILE_EXT;

    currentPropsFile = fname;
    int fsi = fname.lastIndexOf(File.separator);
    currentPropsName = fname.substring(fsi + 1, fname.length() - 4);
  }

  public void setPropertyFileAndLoad(String fname, boolean forceLoad) throws IOException {
    checkSave();

    if(fname == null || fname.length() == 0)
      throw new IOException("Illegal filename for property-file or file-operations disabled");

    String oldFile = currentPropsFile;

    setPropertyFile(fname);

    if(currentPropsFile.equals(oldFile) && !forceLoad) {
      return;
    }

    if(forceLoad || autoLoadProps) {
      try {
	loadProperties(currentPropsFile);
	quiet = initQuiet;
      } catch(FileNotFoundException e) {
	if(forceLoad) {
	  throw e;
	} else {
	  console.println("\n\rProperty file for " + fname + " not found, will be created");
	  propsChanged = true;
	}
      }
    }

    updateMenus();
  }

  public String getPropertyFile() {
    return currentPropsFile;
  }

  public Properties getProperties() {
    return props;
  }

  public boolean propsHaveChanged() {
    return (propsChanged ||
	    (getTerminalWin() != null ? getTerminalWin().propsChanged : false));
  }

  public boolean wantSave() {
    return (propsHaveChanged() && currentPropsFile != null && sshHomeDir != null);
  }

  public boolean canSaveAs() {
    if(getProperty("server").length() > 0)
      return true;
    return false;
  }

  public void saveCurrentProperties() throws IOException {
    if(propsHaveChanged() && currentPropsFile != null)
      saveProperties(currentPropsFile);
  }

  public void saveProperties(String fname) throws IOException {
    FileOutputStream f;
    TerminalWin      term      = getTerminalWin();
    Properties       termProps = (term != null ? term.getProperties() : null);

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    Properties saveProps = meltDefaults(props);
    
    f = new FileOutputStream(fname);
    saveProps.save(f, "MindTerm ssh-properties");
    if(termProps != null) {
      saveProps = meltDefaults(termProps);
      saveProps.save(f, "MindTerm terminal-properties");
    }
    f.close();

    propsChanged      = false;
    term.propsChanged = false;
    updateMenus();
  }

  final Properties meltDefaults(Properties propsToMelt) {
    Enumeration enum   = propsToMelt.propertyNames();
    Properties  melted = new Properties();
    while(enum.hasMoreElements()) {
      String name = (String)enum.nextElement();
      melted.put(name, propsToMelt.getProperty(name));
    }
    return melted;
  }

  public void loadProperties(String fname) throws IOException {
    FileInputStream f;
    TerminalWin term = getTerminalWin();

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    f = new FileInputStream(fname);
    Properties loadProps = new Properties();
    loadProps.load(f);

    Enumeration enum;
    String      name;

    Properties sshProps  = new Properties();
    Properties termProps = new Properties();

    enum = loadProps.propertyNames();
    while(enum.hasMoreElements()) {
      name = (String)enum.nextElement();
      if(isProperty(name))
	sshProps.put(name, loadProps.getProperty(name));
      else if(TerminalWin.isProperty(name))
	termProps.put(name, loadProps.getProperty(name));
      else
	console.println("Unknown property '" + name + "' found in file: " + fname);
    }

    clearAllForwards();
    setProperties(sshProps, false);

    if(term != null)
      term.setProperties(termProps, false);
    else
      initTermProps = termProps;

    // The properties just got loaded, they haven't changed yet...
    //
    if(term != null) term.propsChanged = false;
    propsChanged = false;
  }

  public final void checkSave() throws IOException {
    if(autoSaveProps && wantSave()) {
      saveProperties(currentPropsFile);
    }
  }

  public static boolean isHostPropertiesAvailable(String host, String dirName) {
    String[] aph = availablePropertyHosts(dirName);
    int i;
    if(aph == null)
      return false;
    for(i = 0; i < aph.length; i++)
      if(aph[i].equals(host))
	break;
    return (i < aph.length);
  }

  public static String[] availablePropertyHosts(String dirName) {
    String[] apf = availablePropertyFiles(dirName);
    if(apf == null)
      return null;
    String[] aph = new String[apf.length];
    for(int i = 0; i < apf.length; i++) {
      int pi = apf[i].lastIndexOf(PROPS_FILE_EXT);
      aph[i] = apf[i].substring(0, pi);
    }
    return aph;
  }

  public synchronized static String[] availablePropertyFiles(String dirName) {

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    File dir = new File(dirName);
    String[] list, plist;
    int  i, cnt = 0;

    /* !!! Does not work in Netscape?!
    if(!dir.isDirectory())
      return null;
    */

    list = dir.list();
    for(i = 0; i < list.length; i++) {
      if(!list[i].endsWith(PROPS_FILE_EXT)) {
	list[i] = null;
	cnt++;
      }
    }
    if(cnt == list.length)
      return null;
    plist = new String[list.length - cnt];
    cnt = 0;
    for(i = 0; i < list.length; i++) {
      if(list[i] != null)
	plist[cnt++] = list[i];
    }

    return plist;
  }

  public static boolean isProperty(String key) {
    return defaultProperties.containsKey(key) ||
	(key.indexOf("local") == 0) || (key.indexOf("remote") == 0);
  }

  public String getProperty(String key) {
    return props.getProperty(key);
  }

  public void setProperty(String key, String value) throws IllegalArgumentException,
    NoSuchElementException {
    boolean setChanged  = !(value.equals(getProperty(key)));
    //
    // Some sanity checks...
    //
    if(key.equals("cipher")) {
      if(SSH.getCipherType(value) == SSH.CIPHER_NOTSUPPORTED)
	throw new IllegalArgumentException("Cipher " + value + " not supported");
      //
    } else if(key.equals("authtyp")) {
      SSH.getAuthTypes(value);
      //
    } else if(key.equals("x11fwd")  || key.equals("prvport") ||
	      key.equals("forcpty") || key.equals("remfwd")  ||
	      key.equals("idhost")  || key.equals("portftp")) {
      if(!(value.equals("true") || value.equals("false")))
	throw new IllegalArgumentException("Value for " + key + " must be 'true' or 'false'");
      if(key.equals("remfwd")) {
	try {
	  SSHListenChannel.setAllowRemoteConnect((new Boolean(value)).booleanValue());
	} catch (Throwable t) {
	  // !!! Ignore if we don't have the SSHListenChannel class
	}
      } else if(key.equals("portftp")) {
	havePORTFtp = (new Boolean(value)).booleanValue();
	if(havePORTFtp && SSHProtocolPlugin.getPlugin("ftp") != null) {
	  SSHProtocolPlugin.getPlugin("ftp").initiate(this);
	}
      }
      //
    } else if(key.equals("port") || key.equals("mtu") || key.equals("secrand") || key.equals("alive")) {
      try {
	int val = Integer.valueOf(value).intValue();
	if(key.equals("port") && (val > 65535 || val < 0)) {
	  throw new IllegalArgumentException("Not a valid port number: " + value);
	} else if(key.equals("mtu") && val != 0 && (val > (256*1024) || val < 4096)) {
	  throw new IllegalArgumentException("Mtu must be between 4k and 256k");
	} else if(key.equals("alive")) {
	  if(val < 0 || val > 600)
	    throw new IllegalArgumentException("Alive interval must be 0-600");
	    setAliveInterval(val);
	} else if(key.equals("secrand")) {
	  if(val < 0 || val > 2)
	    throw new IllegalArgumentException("Secrand must be 0-2");
	  SecureRandom.secureLevel = val;
	}
	value = String.valueOf(val);
      } catch (NumberFormatException e) {
	throw new IllegalArgumentException("Value for " + key + " must be an integer");
      }
      //
    } else if(key.equals("server")) {
      if(isOpened) {
	throw new IllegalArgumentException("Server can only be set while not connected");
      } else {
	if(currentPropsFile == null && value.length() > 0)
	  setPropertyFile(value);
	updateMenus();
      }
      updateTitle();
    } else if(key.equals("realsrv")) {
      try {
	if(value != null && value.length() > 0)
	  setServerRealAddr(InetAddress.getByName(value));
	else
	  setServerRealAddr(null);
      } catch (UnknownHostException e) {
	throw new IllegalArgumentException("realsrv address must be a legal/known host-name");
      }
    } else if(key.equals("localhst")) {
      try {
	setLocalAddr(value);
      } catch (UnknownHostException e) {
	throw new IllegalArgumentException("localhost address must be a legal/known host-name");
      }
    } else if(key.equals("usrname")) {
      updateTitle();
    } else if(key.equals("display") || key.equals("escseq") || key.equals("idfile")) {
      // !!! Do nothing for these, arbitrary strings...
      //
    } else if(key.startsWith("local")) {
      int n = Integer.parseInt(key.substring(5));
      if(n > localForwards.size())
	throw new IllegalArgumentException("Port forwards must be given in unbroken sequence");
      if(value.startsWith("/general/"))
	value = value.substring(9);
      try {
	addLocalPortForward(value, true);
      } catch (IOException e) {
	throw new IllegalArgumentException("Error creating tunnel: " + e.getMessage());
      }
    } else if(key.startsWith("remote")) {
      try {
	int n = Integer.parseInt(key.substring(6));
	if(n > remoteForwards.size())
	  throw new IllegalArgumentException("Port forwards must be given in unbroken sequence");
	if(value.startsWith("/general/"))
	  value = value.substring(9);
	addRemotePortForward(value, true);
      } catch (Exception e) {
	throw new IllegalArgumentException("Not a valid port forward: " + key + " : " + value);
      }
    } else {
      throw new NoSuchElementException("Unknown ssh-property '" + key + "'");
    }

    if(!propsChanged)
      propsChanged = setChanged;

    if(setChanged) {
      updateMenus();
    }

    props.put(key, value);
  }

  public void removeLocalTunnelAt(int idx, boolean kill) {
    int i, sz = localForwards.size();
    props.remove("local" + idx);
    for(i = idx; i < sz - 1; i++) {
      props.put("local" + idx, props.get("local" + (idx + 1)));
      props.remove("local" + idx + 1);
    }
    propsChanged = true;
    if(kill) {
      LocalForward fwd = (LocalForward)localForwards.elementAt(idx);
      delLocalPortForward(fwd.localHost, fwd.localPort);
    } else {
      localForwards.removeElementAt(idx);
    }
  }

  public void removeRemoteTunnelAt(int idx) {
    int i, sz = remoteForwards.size();
    props.remove("remote" + idx);
    for(i = idx; i < sz - 1; i++) {
      props.put("remote" + idx, props.get("remote" + (idx + 1)));
      props.remove("remote" + idx + 1);
    }
    propsChanged = true;
    remoteForwards.removeElementAt(idx);
  }

  public void addLocalPortForward(String fwdSpec, boolean commit) throws IllegalArgumentException,
  IOException {
    int    localPort;
    String remoteHost;
    int    remotePort;
    int    d1, d2, d3;
    String tmp, plugin;
    String localHost = null;

    if(fwdSpec.charAt(0) == '/') {
      int i = fwdSpec.lastIndexOf('/');
      if(i == 0)
	throw new IllegalArgumentException("Invalid port forward spec. " + fwdSpec);
      plugin = fwdSpec.substring(1, i);
      fwdSpec = fwdSpec.substring(i + 1);
    } else
      plugin = "general";

    d1 = fwdSpec.indexOf(':');
    d2 = fwdSpec.lastIndexOf(':');
    if(d1 == d2)
      throw new IllegalArgumentException("Invalid port forward spec. " + fwdSpec);

    d3 = fwdSpec.indexOf(':', d1 + 1);

    if(d3 != d2) {
      localHost = fwdSpec.substring(0, d1);
      localPort = Integer.parseInt(fwdSpec.substring(d1 + 1, d3));
      remoteHost = fwdSpec.substring(d3 + 1, d2);
    } else {
      localPort = Integer.parseInt(fwdSpec.substring(0, d1));
      remoteHost = fwdSpec.substring(d1 + 1, d2);
    }

    tmp        = fwdSpec.substring(d2 + 1);
    remotePort = Integer.parseInt(tmp);
    if(commit) {
      if(localHost == null)
	addLocalPortForward(localPort, remoteHost, remotePort, plugin);
      else
	addLocalPortForward(localHost, localPort, remoteHost, remotePort, plugin);
    }
  }

  public void addRemotePortForward(String fwdSpec, boolean commit) throws IllegalArgumentException {
    int    remotePort;
    int    localPort;
    String localHost;
    int    d1, d2;
    String tmp, plugin;

    if(fwdSpec.charAt(0) == '/') {
      int i = fwdSpec.lastIndexOf('/');
      if(i == 0)
	throw new IllegalArgumentException("Invalid port forward spec.");
      plugin = fwdSpec.substring(1, i);
      fwdSpec = fwdSpec.substring(i + 1);
    } else
      plugin = "general";

    d1 = fwdSpec.indexOf(':');
    d2 = fwdSpec.lastIndexOf(':');
    if(d1 == d2)
      throw new IllegalArgumentException("Invalid port forward spec.");

    tmp        = fwdSpec.substring(0, d1);
    remotePort = Integer.parseInt(tmp);
    localHost  = fwdSpec.substring(d1 + 1, d2);
    tmp        = fwdSpec.substring(d2 + 1);
    localPort  = Integer.parseInt(tmp);
    if(commit) {
      addRemotePortForward(remotePort, localHost, localPort, plugin);
    }
  }

  public void setAliveInterval(int i) {
    if(i == 0) {
      if(keepAliveThread != null)
	keepAliveThread.stop();
      keepAliveThread = null;
    } else {
      if(keepAliveThread != null) {
        keepAliveThread.setInterval(i);
      } else {
	keepAliveThread = new KeepAliveThread(i);
	keepAliveThread.start();
      }
    }
  }

  public boolean isOpened() {
    return isOpened;
  }

  //
  // SSHAuthenticator interface
  //

  public String getUsername(SSHClientUser origin) throws IOException {
    String username = getProperty("usrname");
    if((commandLine == null && !quiet) || (username == null || username.equals(""))) {
      username = promptLine(getProperty("server") + " login: ", username, false);
      setProperty("usrname", username); // Changing the user-name does not save new properties...
    }
    return username;
  }

  public String getPassword(SSHClientUser origin) throws IOException {
    if(password == null)
      password = promptLine(getProperty("usrname") + "@" + getProperty("server") +
			    "'s password: ", "", true);
    return password;
  }

  public String getChallengeResponse(SSHClientUser origin, String challenge) throws IOException {
    if(tisPassword == null)
      tisPassword = promptLine(challenge, "", true);
    return tisPassword;
  }

  public int[] getAuthTypes(SSHClientUser origin) {
    return SSH.getAuthTypes(getProperty("authtyp"));
  }

  public int getCipher(SSHClientUser origin) {
    return SSH.getCipherType(getProperty("cipher"));
  }

  public SSHRSAKeyFile getIdentityFile(SSHClientUser origin) throws IOException {
    String idFile = getProperty("idfile");
    if(idFile.indexOf(File.separator) == -1) {
      idFile = sshHomeDir + idFile;
    }

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    keyFile = new SSHRSAKeyFile(idFile);
    return keyFile;
  }
  
  public String getIdentityPassword(SSHClientUser origin) throws IOException {
    if(rsaPassword == null)
      rsaPassword = promptLine("key-file '" + keyFile.getComment() + "' password: ", "", true);
    return rsaPassword;
  }

  public boolean verifyKnownHosts(RSAPublicKey hostPub) throws IOException {
    if(!Boolean.valueOf(getProperty("idhost")).booleanValue()) {
      return true;
    }

    if(sshHomeDir == null) {
      if(wantHelpInfo)
	report("File-operations disabled, server identity can't be verified");
      return true;
    }

    if(NETSCAPE_SECURITY_MODEL) {
      try {
	netscape.security.PrivilegeManager.enablePrivilege("UniversalFileAccess");
      } catch (netscape.security.ForbiddenTargetException e) {
	// !!!
      }
    }

    int     hostCheck;
    boolean confirm = true;
    String  fileName = sshHomeDir + knownHosts;
    File    tmpFile = new File(fileName);

    if(!tmpFile.exists()) {
      if(askConfirmation("File '"  + fileName + "' not found, create it?", true)) {
	FileOutputStream f = new FileOutputStream(tmpFile);
	f.close();
      } else {
	console.println("Verification of server key disabled in this session.");
	return true;
      }
    }

    SSHRSAPublicKeyFile file = new SSHRSAPublicKeyFile(fileName, true);

    if((hostCheck = file.checkPublic(hostPub.getN(), getProperty("server"))) ==
       SRV_HOSTKEY_KNOWN)
      return true;

    if(hostCheck == SRV_HOSTKEY_NEW) {
      console.println("Host key not found from the list of known hosts.");
      if(!askConfirmation("Do you want to add this host to your set of known hosts", false, true)) {
	console.println("Verification of server key disabled in this session.");
	return true;
      }
      confirm = true;
    } else {
      console.println("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
      console.println("@       WARNING: HOST IDENTIFICATION HAS CHANGED!         @");
      console.println("@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@");
      console.println("IT IS POSSIBLE THAT SOMEONE IS DOING SOMETHING NASTY,");
      console.println("ONLY PROCEED IF YOU KNOW WHAT YOU ARE DOING!");
      confirm = askConfirmation("Do you want to replace the identification of this host?", false, false);
      file.removePublic(getProperty("server"));
    }

    if(confirm) {
      file.addPublic(getProperty("server"), null, hostPub.getE(), hostPub.getN());
      tmpFile      = new File(fileName + ".tmp");
      File oldFile = new File(fileName);
      oldFile.renameTo(tmpFile);
      try {
	file.saveToFile(fileName);
      } catch (IOException e) {
	oldFile = new File(fileName);
	tmpFile.renameTo(oldFile);
	throw e;
      }
      tmpFile.delete();
    } else {
      return false;
    }

    return true;
  }

  //
  // SSHClientUser interface
  //

  public String getSrvHost() throws IOException {
    String host = currentPropsName;
    if((commandLine == null && !quiet) || (host == null || host.equals(""))) {
      do {
	host = promptLine("SSH-server: ", host, false);
	host = host.trim();
      } while ("".equals(host));

      // Set it in case of it's an unknown (i.e. .mtp-file does not exist)
      //
      props.put("server", host);
    }
    try {
      setPropertyFileAndLoad(host, false);
      updateMenus();
    } catch (Throwable e) {
      // !!!
    }
    host = getProperty("server");
    return host;
  }

  public int getSrvPort() {
    return Integer.valueOf(getProperty("port")).intValue();
  }

  public String getDisplay() {
    return getProperty("display");
  }

  public int getMaxPacketSz() {
    return Integer.valueOf(getProperty("mtu")).intValue();
  }

  public boolean wantX11Forward() {
    return Boolean.valueOf(getProperty("x11fwd")).booleanValue();
  }

  public boolean wantPrivileged() {
    return Boolean.valueOf(getProperty("prvport")).booleanValue();
  }

  public boolean wantPTY() {
    return Boolean.valueOf(getProperty("forcpty")).booleanValue();
  }

  public void open(SSHClient client) {
    updateMenus();
    updateTitle();
  }

  public void connected(SSHClient client) {
    updateMenus();
    updateTitle();
    if(wantHelpInfo) {
	console.println("Connected to server running " + srvVersionStr);
	if(sshStdIO.hasCommandShell())
	    console.println("(command-shell escape-sequence is '" + sshStdIO.commandShell.escapeString() + "')");
	console.println("");
    }
  }

  public void disconnected(SSHClient client, boolean graceful) {
    sshStdIO.breakPromptLine("Login aborted by user");
    srvVersionStr = null;
    updateMenus();
    updateTitle();
  }

  public void report(String msg) {
    console.println(msg);
    console.println("");
  }

  public void alert(String msg) {
    if(menus != null) {
      if(msg.length() < 35)
	menus.alertDialog(msg);
      else
	menus.textDialog("MindTerm - Alert", msg, 4, 38, true);
    } else {
      report(msg);
    }
  }

  public boolean askConfirmation(String message, boolean defAnswer) {
    boolean confirm = false;
    try {
      confirm = askConfirmation(message, true, defAnswer);
    } catch (IOException e) {
	// !!!
    }
    return confirm;
  }

  public boolean askConfirmation(String message, boolean preferDialog, boolean defAnswer) throws IOException {
    boolean confirm = false;
    if(menus != null && preferDialog) {
      confirm = menus.confirmDialog(message, defAnswer);
    } else {
      String answer = promptLine(message + (defAnswer ? " ([yes]/no) " : "(yes/[no]) "),
				 "", false);
      if(answer.equalsIgnoreCase("yes") || answer.equals("y")) {
	confirm = true;
      } else if(answer.equals("")) {
	confirm = defAnswer;
      }
    }
    return confirm;
  }

}
