/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdio.h>
 #include <stdlib.h>
 #include <string.h>
 #include <unistd.h>
 #include <netdb.h>
 #include <signal.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
 #include <sys/ioctl.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <netdb.h>
 #include <signal.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #define __EXTENSIONS__
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <netdb.h>
 #include <signal.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
int lcrzo_hs_init(const char *hs_src, lcrzo_hs hs_dest)
{ lcrzo_er(lcrzo_string_init_coretext(hs_src, LCRZO_HS_MAXBYTES, hs_dest));
  return(LCRZO_ERR_OK);
}
int lcrzo_ips_init(const char *ips_src, lcrzo_ips ips_dest)
{ lcrzo_ips ips;
  char *pc;
    
  if (ips_dest==NULL) pc=ips;
  else pc=ips_dest;

  lcrzo_er(lcrzo_string_init_coretext(ips_src, LCRZO_IPS_MAXBYTES, pc));

  /*try to convert it to see if it's OK*/
  lcrzo_er(lcrzo_ipa_init_ips(pc, NULL));

  return(LCRZO_ERR_OK);
}
int lcrzo_ipa_init(lcrzo_uint8 a, lcrzo_uint8 b,
		   lcrzo_uint8 c, lcrzo_uint8 d, lcrzo_ipa ipa_dest)
{ if (ipa_dest!=NULL)
  { ipa_dest[0]=a;
    ipa_dest[1]=b;
    ipa_dest[2]=c;
    ipa_dest[3]=d;
  }
  return(LCRZO_ERR_OK);
}
int lcrzo_ipl_init(lcrzo_uint8 a, lcrzo_uint8 b,
		 lcrzo_uint8 c, lcrzo_uint8 d, lcrzo_ipl *pipl_dest)
{ if (pipl_dest!=NULL) *pipl_dest= (a<<24) | (b<<16) | (c<<8) | d;
  return(LCRZO_ERR_OK);
}
int lcrzo_eths_init(const char *eths_src, lcrzo_eths eths_dest)
{ lcrzo_eths eths;
  char *pc;
    
  if (eths_dest==NULL) pc=eths;
  else pc=eths_dest;

  lcrzo_er(lcrzo_string_init_coretext(eths_src, LCRZO_ETHS_MAXBYTES, pc));

  /*try to convert it to see if it's OK*/
  lcrzo_er(lcrzo_etha_init_eths(pc, NULL));

  return(LCRZO_ERR_OK);
}
int lcrzo_etha_init(lcrzo_uint8 a, lcrzo_uint8 b, lcrzo_uint8 c, lcrzo_uint8 d,
		    lcrzo_uint8 e, lcrzo_uint8 f, lcrzo_etha etha_dest)
{ if (etha_dest!=NULL)
  { etha_dest[0]=a;
    etha_dest[1]=b;
    etha_dest[2]=c;
    etha_dest[3]=d;
    etha_dest[4]=e;
    etha_dest[5]=f;
  }
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
/*conversions ips-ipaipl et eths-etha*/

/*-------------------------------------------------------------*/
int lcrzo_ipa_init_ips(const lcrzo_ips ips, lcrzo_ipa ipa)
{ int i;
  char *pc, *pi;
  long l;
  lcrzo_ips ips_copie;
  
  lcrzo_er(lcrzo_string_init_coretext(ips, LCRZO_IPS_MAXBYTES, ips_copie));
  if ( ips_copie[strlen(ips_copie)-1]=='.' ) return(LCRZO_ERR_FMIP);
  pc=strtok(ips_copie, ".");
  if ( pc==NULL ) return(LCRZO_ERR_FMIP); 
  l=strtol(pc, &pi, 10);
  if ( (int)(*pi)!=0 ) return(LCRZO_ERR_FMIP); 
  if ( l>255 ) return(LCRZO_ERR_FMIP); 
  else { if (ipa!=NULL) ipa[0]=(lcrzo_int8)l; }
  for ( i=1 ; i<LCRZO_IPA_MAXBYTES ; i++ )
  { pc=strtok(NULL, ".");
    if ( pc==NULL ) return(LCRZO_ERR_FMIP); 
    l=strtol(pc, &pi, 10);
    if ( l>255 ) return(LCRZO_ERR_FMIP); 
    else { if (ipa!=NULL) ipa[i]=(lcrzo_int8)l; }
    if ( (int)(*pi)!=0 ) return(LCRZO_ERR_FMIP); 
  }
  pc=strtok(NULL, ".");
  if ( pc!=NULL ) return(LCRZO_ERR_FMIP); 
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
/* equivalent de inet_addr ou inet_aton*/
int lcrzo_ipl_init_ips(const lcrzo_ips ips, lcrzo_ipl *pipl)
{ int i;
  char *pc, *pi;
  long l;
  lcrzo_ips ips_copie;
  
  lcrzo_er(lcrzo_string_init_coretext(ips, LCRZO_IPS_MAXBYTES, ips_copie));
  if ( ips_copie[strlen(ips_copie)-1]=='.' ) return(LCRZO_ERR_FMIP); 
  pc=strtok(ips_copie, ".");
  if ( pc==NULL ) return(LCRZO_ERR_FMIP);
  l=strtol(pc, &pi, 10);
  if ( (int)(*pi)!=0 ) return(LCRZO_ERR_FMIP); 
  if ( l>255 ) return(LCRZO_ERR_FMIP);
  else { if (pipl!=NULL) *pipl=(lcrzo_int8)l; }
  for ( i=1 ; i<LCRZO_IPA_MAXBYTES ; i++ )
  { pc=strtok(NULL, ".");
    if ( pc==NULL ) return(LCRZO_ERR_FMIP); 
    l=strtol(pc, &pi, 10);
    if ( l>255 ) return(LCRZO_ERR_FMIP); 
    else
    { if (pipl!=NULL) *pipl= ((*pipl)<<8) | l;
    }
    if ( (int)(*pi)!=0 ) return(LCRZO_ERR_FMIP);
  }
  pc=strtok(NULL, ".");
  if ( pc!=NULL ) return(LCRZO_ERR_FMIP); 

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ips_init_ipa(const lcrzo_ipa ipa, lcrzo_ips ips)
{ int i;

  /*parameters verification*/
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  /*this function is always OK*/
  if (ips==NULL) return(LCRZO_ERR_OK);

  lcrzo_string_init_text("", LCRZO_IPS_MAXBYTES, ips);
  for (i=0;i<LCRZO_IPA_MAXBYTES-1;i++)
  { lcrzo_string_append_int(ipa[i], "%d", LCRZO_IPS_MAXBYTES, ips);
    lcrzo_string_append_char('.', 1, LCRZO_IPS_MAXBYTES, ips);
  }
  lcrzo_string_append_int(ipa[LCRZO_IPA_MAXBYTES-1], "%d",
		       LCRZO_IPS_MAXBYTES, ips);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_init_ipa(const lcrzo_ipa ipa, lcrzo_ipl *pipl)
{
  /*parameters verification*/
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  if (pipl!=NULL) *pipl= (ipa[0]<<24) | (ipa[1]<<16) | (ipa[2]<<8) | ipa[3];
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ips_init_ipl(lcrzo_ipl ipl, lcrzo_ips ips)
{
  /*this function is always OK*/
  if (ips==NULL) return(LCRZO_ERR_OK);

  lcrzo_string_init_int((lcrzo_uint8)(ipl>>24), "%d", LCRZO_IPS_MAXBYTES, ips);
  lcrzo_string_append_char('.', 1, LCRZO_IPS_MAXBYTES, ips);
  lcrzo_string_append_int((lcrzo_uint8)((ipl>>16)&0xFF), "%d",
		       LCRZO_IPS_MAXBYTES,ips);
  lcrzo_string_append_char('.', 1, LCRZO_IPS_MAXBYTES, ips);
  lcrzo_string_append_int((lcrzo_uint8)((ipl>>8)&0xFF), "%d",
		       LCRZO_IPS_MAXBYTES, ips);
  lcrzo_string_append_char('.', 1, LCRZO_IPS_MAXBYTES, ips);
  lcrzo_string_append_int((lcrzo_uint8)(ipl&0xFF), "%d", LCRZO_IPS_MAXBYTES, ips);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipa_init_ipl(lcrzo_ipl ipl, lcrzo_ipa ipa)
{ 
  if (ipa!=NULL)
  { ipa[0]=(lcrzo_uint8)(ipl>>24);
    ipa[1]=(lcrzo_uint8)((ipl>>16)&0xFF);
    ipa[2]=(lcrzo_uint8)((ipl>>8)&0xFF);
    ipa[3]=(lcrzo_uint8)(ipl&0xFF);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_etha_init_eths(const lcrzo_eths eths, lcrzo_etha etha)
{ int i;
  char *pc, *pi;
  long l;
  lcrzo_eths eths_copie;
  
  lcrzo_er(lcrzo_string_init_coretext(eths, LCRZO_ETHS_MAXBYTES, eths_copie));
  if ( eths_copie[strlen(eths_copie)-1]==':' ) return(LCRZO_ERR_FMETH); 
  pc=strtok(eths_copie, ":");
  if ( pc==NULL ) return(LCRZO_ERR_FMETH); 
  l=strtol(pc, &pi, 16);
  if ( (int)(*pi)!=0 ) return(LCRZO_ERR_FMETH); 
  if ( l>255 ) return(LCRZO_ERR_FMETH); 
  else { if (etha!=NULL) etha[0]=(lcrzo_int8)l; }
  for ( i=1 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
  { pc=strtok(NULL, ":");
    if ( pc==NULL ) return(LCRZO_ERR_FMETH); 
    l=strtol(pc, &pi, 16);
    if ( l>255 ) return(LCRZO_ERR_FMETH); 
    else { if (etha!=NULL) etha[i]=(lcrzo_int8)l; }
    if ( (int)(*pi)!=0 ) return(LCRZO_ERR_FMETH); 
  }
  pc=strtok(NULL, ":");
  if ( pc!=NULL ) return(LCRZO_ERR_FMETH); 

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_init_etha(const lcrzo_etha etha, lcrzo_eths eths)
{ int i;

  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  /*this function is always OK*/
  if (eths==NULL) return(LCRZO_ERR_OK);

  lcrzo_string_init_text("", LCRZO_ETHS_MAXBYTES, eths);
  for (i=0;i<LCRZO_ETHA_MAXBYTES-1;i++)
  { lcrzo_string_append_int(etha[i], "%02X", LCRZO_ETHS_MAXBYTES, eths);
    lcrzo_string_append_char(':', 1, LCRZO_ETHS_MAXBYTES, eths);
  }
  lcrzo_string_append_int(etha[LCRZO_ETHA_MAXBYTES-1], "%02X",
		       LCRZO_ETHS_MAXBYTES, eths);
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
/*conversions faisant appel a une fonction reseau*/

/*-------------------------------------------------------------*/
/*trouve la reponse en utilisant le resolver (local, dns, yp, etc.)*/
int lcrzo_priv_ipa_init_hs_resolver(const lcrzo_hs hs, lcrzo_ipa ipa);
int lcrzo_priv_ipa_init_hs_resolver(const lcrzo_hs hs, lcrzo_ipa ipa)
{ struct hostent *phe;
  int i;

  lcrzo_hs_verifbof(hs);
  phe=gethostbyname(hs);
  if ( phe!=NULL )
  { if (ipa!=NULL)
    { for (i=0;i<LCRZO_IPA_MAXBYTES;i++) ipa[i]=phe->h_addr_list[0][i];
    }
    lcrzo_err_purge();
    return(LCRZO_ERR_OK);
  }

  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
/*trouve la reponse en utilisant le resolver durant n secondes*/
void lcrzo_priv_ipa_init_hs_findufils(int num);
void lcrzo_priv_ipa_init_hs_findufils(int num) 
{ /*puts("alarme");*/
  _exit(0);
}
int lcrzo_priv_ipa_init_hs_resolvalarm(const lcrzo_hs hs, lcrzo_ipa ipa);
int lcrzo_priv_ipa_init_hs_resolvalarm(const lcrzo_hs hs, lcrzo_ipa ipa)
{ lcrzo_ipc ipc;
  lcrzo_ipa ipafork;
  int ret, pidsniff, status;

  /*initialisation du contexte d'IPC*/
  lcrzo_er(lcrzo_ipc_init(&ipc));

  /*processus de sniff*/
  pidsniff=fork();
  if (pidsniff<0)
  { lcrzo_er(lcrzo_ipc_close(ipc));
    return(LCRZO_ERR_OKUNRESOLVED);
  }
  if (pidsniff==0) /*on est dans le fils*/
  { /*Attention : on ne doit jamais quitter avec "return()" (ou lcrzo_er),
      mais avec "_exit()", sinon ce processus retourne dans
      "lcrzo__priv_ipa_init_hs_resolvalarm"*/
    /*on a lcrzo_global.resolver_timeout secondes pour obtenir le resultat*/
    alarm(lcrzo_global.resolver_timeout);
    signal(SIGALRM, lcrzo_priv_ipa_init_hs_findufils);
    /* we ask */
    ret=lcrzo_priv_ipa_init_hs_resolver(hs, ipafork);
    if ( ret==LCRZO_ERR_OK )
    { lcrzo_ipc_write_ipa(ipc, ipafork);
    }
    lcrzo_ipc_close(ipc);
    _exit(ret);
  }
  /*on attend la fin du fils*/
  waitpid(pidsniff, &status, 0);
  /*on lit ce qui est eventuellement dans le tunnel*/
  ret=lcrzo_ipc_read_ipa(ipc, LCRZO_FALSE, ipa);
  /*fermeture des contextes : les erreurs peuvent etre maintenant ignorees*/
  lcrzo_ipc_close(ipc);
  /*il y avait une reponse dans le tunnel*/
  if (ret==LCRZO_ERR_OK)
  { return(LCRZO_ERR_OK);
  }
  /*pas de reponse dans le tunnel*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
/*trouve la reponse en utilisant le resolver (local, dns, yp, etc.)*/
int lcrzo_priv_hs_init_ipa_resolver(const lcrzo_ipa ipa, lcrzo_hs hs);
int lcrzo_priv_hs_init_ipa_resolver(const lcrzo_ipa ipa, lcrzo_hs hs)
{ struct hostent *phe;

  phe=gethostbyaddr((const char *)ipa, LCRZO_IPA_MAXBYTES, AF_INET);
  if ( phe!=NULL )
  { lcrzo_hs_verifbof(phe->h_name);
    lcrzo_string_init_text(phe->h_name, LCRZO_HS_MAXBYTES, hs);
    lcrzo_err_purge();
    return(LCRZO_ERR_OK);
  }

  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
/*trouve la reponse en utilisant le resolver durant n secondes*/
void lcrzo_priv_hs_init_ipa_findufils(int num);
void lcrzo_priv_hs_init_ipa_findufils(int num) 
{ /*puts("alarme");*/
  _exit(0);
}
int lcrzo_priv_hs_init_ipa_resolvalarm(const lcrzo_ipa ipa, lcrzo_hs hs);
int lcrzo_priv_hs_init_ipa_resolvalarm(const lcrzo_ipa ipa, lcrzo_hs hs)
{ lcrzo_ipc ipc;
  lcrzo_hs hsfork;
  int ret, pidsniff, status;

  /*initialisation du contexte d'IPC*/
  lcrzo_er(lcrzo_ipc_init(&ipc));

  /*processus de sniff*/
  pidsniff=fork();
  if (pidsniff<0)
  { lcrzo_er(lcrzo_ipc_close(ipc));
    return(LCRZO_ERR_OKUNRESOLVED);
  }
  if (pidsniff==0) /*on est dans le fils*/
  { /*Attention : on ne doit jamais quitter avec "return()" (ou lcrzo_er),
      mais avec "_exit()", sinon ce processus retourne dans
      "lcrzo__priv_hs_init_ipa_resolvalarm"*/
    /*on a lcrzo_global.resolver_timeout secondes pour obtenir le resultat*/
    alarm(lcrzo_global.resolver_timeout);
    signal(SIGALRM, lcrzo_priv_hs_init_ipa_findufils);
    /* we ask */
    ret=lcrzo_priv_hs_init_ipa_resolver(ipa, hsfork);
    if ( ret==LCRZO_ERR_OK )
    { lcrzo_ipc_write_hs(ipc, hsfork);
    }
    lcrzo_ipc_close(ipc);
    _exit(ret);
  }
  /*on attend la fin du fils*/
  waitpid(pidsniff, &status, 0);
  /*on lit ce qui est eventuellement dans le tunnel*/
  ret=lcrzo_ipc_read_hs(ipc, LCRZO_FALSE, hs);
  /*fermeture des contextes : les erreurs peuvent etre maintenant ignorees*/
  lcrzo_ipc_close(ipc);
  /*il y avait une reponse dans le tunnel*/
  if (ret==LCRZO_ERR_OK)
  { return(LCRZO_ERR_OK);
  }
  /*pas de reponse dans le tunnel*/
  return(LCRZO_ERR_OKUNRESOLVED);
}


/*-------------------------------------------------------------*/
/* cette fonction envoie une requete arp sur le reseau, et sniffe
   la reponse*/
void lcrzo_priv_etha_init_ipa_findufils(int num);
void lcrzo_priv_etha_init_ipa_findufils(int num) 
{ /*puts("alarme");*/
  _exit(0);
}
int lcrzo_priv_etha_init_ipa__spoofsniff(const lcrzo_device device, 
				    const lcrzo_ipa ipa, lcrzo_etha etha);
int lcrzo_priv_global_get_intspoof_eth(const lcrzo_etha ethaforcomp,
				       lcrzo_etha etha);
int lcrzo_priv_global_get_intspoof_ip(const lcrzo_ipa ipaforcomp,
				      lcrzo_ipa ipa);
int lcrzo_priv_etha_init_ipa__spoofsniff(const lcrzo_device device, 
				    const lcrzo_ipa ipa, lcrzo_etha etha)
{ lcrzo_sniff cs;
  lcrzo_ipc ci;
  lcrzo_uint32 mtu;
  lcrzo_hdrleth hdrleth;
  lcrzo_hdrlarp hdrlarp, hdrlarprecu;
  int retour, pidsniff, status, i;
  lcrzo_spoof spoof;
  lcrzo_data pdata;
  lcrzo_int32 datasize;
  lcrzo_etha ethaforcomp;

  /*initialisation de l'entete ETH*/
  lcrzo_er(lcrzo_hdrleth_initdefault(&hdrleth));
  for ( i=0 ; i<LCRZO_ETHA_MAXBYTES ; i++ )
  { lcrzo_er(lcrzo_uint8_rand(1, 0xFE, &(ethaforcomp[i])));
  }
  lcrzo_er(lcrzo_priv_global_get_intspoof_eth(ethaforcomp, hdrleth.src));
  lcrzo_er(lcrzo_etha_init(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, hdrleth.dst));
  hdrleth.type=LCRZO_HDRLETH_TYPE_ARP;
  /*initialisation de l'entete ARP*/
  lcrzo_er(lcrzo_hdrlarp_initdefault(&hdrlarp));
  hdrlarp.op = 1;
  memcpy(hdrlarp.hw_src, hdrleth.src, LCRZO_ETHA_MAXBYTES);
  lcrzo_er(lcrzo_priv_global_get_intspoof_ip(ipa, hdrlarp.prot_src));
  lcrzo_er(lcrzo_etha_init(0,0,0,0,0,0, hdrlarp.hw_dst));
  memcpy(hdrlarp.prot_dst, ipa, LCRZO_IPA_MAXBYTES);
  /*initialisation du sniff*/
  lcrzo_er(lcrzo_mtu_init_device(device, &mtu));
  lcrzo_er(lcrzo_sniff_init(device, mtu, "", &cs));
  /*initialisation du contexte d'IPC*/
  retour=lcrzo_ipc_init(&ci);
  if (retour!=LCRZO_ERR_OK)
  { lcrzo_er(lcrzo_sniff_close(&cs));
    return(retour);
  }
  /*envoi de la requete arp*/
  lcrzo_er(lcrzo_spoof_init(&spoof));
  retour=lcrzo_spoof_etharp(&spoof, device, hdrleth, hdrlarp);
  if (retour!=LCRZO_ERR_OK)
  { lcrzo_er(lcrzo_sniff_close(&cs));
    lcrzo_er(lcrzo_ipc_close(ci));
    lcrzo_er(lcrzo_spoof_close(&spoof));
    return(LCRZO_ERR_OKUNRESOLVED);
  }
  lcrzo_er(lcrzo_spoof_close(&spoof));
  /*processus de sniff*/
  pidsniff=fork();
  if (pidsniff<0)
  { lcrzo_er(lcrzo_sniff_close(&cs));
    lcrzo_er(lcrzo_ipc_close(ci));
    return(LCRZO_ERR_OKUNRESOLVED);
  }
  if (pidsniff==0) /*on est dans le fils*/
  { /*Attention : on ne doit jamais quitter avec "return()" (ou lcrzo_er),
      mais avec "_exit()", sinon ce processus retourne dans
      "lcrzo_etha_init_ipa_spoofsniff"*/
    /*on a une seconde pour obtenir le resultat*/
    alarm(1);
    signal(SIGALRM, lcrzo_priv_etha_init_ipa_findufils);
    /*on sniffe le resultat*/
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_NORMAL,
			     &pdata, &datasize);
    while( retour==LCRZO_ERR_OK )
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*on voit si c'est la reponse attendue*/
      retour=lcrzo_packet_decode_etharp(pdata, datasize, 
					NULL, &hdrlarprecu);
      if ( retour==LCRZO_ERR_OK && hdrlarprecu.op==2 )
      { if (lcrzo_ipa_equal(hdrlarprecu.prot_src, ipa))
	{ /*on a trouve*/
          retour=lcrzo_ipc_write_etha(ci, hdrlarprecu.hw_src);
	  lcrzo_sniff_close(&cs);
	  lcrzo_ipc_close(ci);
          _exit(retour);
	}
      }
      lcrzo_data_free(pdata);
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_NORMAL,
			       &pdata, &datasize);
    }
    /*cet endroit n'est atteind que sur erreur de sniff*/
    lcrzo_sniff_close(&cs);
    lcrzo_ipc_close(ci);
    _exit(retour);
  }
  /*on attend la fin du fils*/
  waitpid(pidsniff, &status, 0);
  /*on lit ce qui est eventuellement dans le tunnel*/
  retour=lcrzo_ipc_read_etha(ci, 0, etha);
  /*fermeture des contextes : les erreurs peuvent etre maintenant ignorees*/
  lcrzo_sniff_close(&cs);
  lcrzo_ipc_close(ci);
  /*il y avait une reponse dans le tunnel*/
  if (retour==LCRZO_ERR_OK)
  { return(LCRZO_ERR_OK);
  }
  /*pas de reponse dans le tunnel*/
  return(LCRZO_ERR_OKUNRESOLVED);
}
int lcrzo_priv_etha_init_ipa_spoofsniff(const lcrzo_ipa ipa, lcrzo_etha etha);
int lcrzo_priv_etha_init_ipa_spoofsniff(const lcrzo_ipa ipa, lcrzo_etha etha)
{ lcrzo_device device;
  int retour;
  lcrzo_int32 count, i;
  lcrzo_etha localetha;
  lcrzo_ipl ipl;

  /*on spoofe et sniffe tous les devices*/
  lcrzo_er(lcrzo_device_count(LCRZO_DEVICE_TYPE_ONLYBOARDS, &count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_device_value_pos(LCRZO_DEVICE_TYPE_ONLYBOARDS, i, device));
    /*printf("demande sur le device=%s\n", device);*/
    retour=lcrzo_priv_etha_init_ipa__spoofsniff(device, ipa, localetha);
    if (retour==LCRZO_ERR_OK)
    { lcrzo_er(lcrzo_ipl_init_ipa(ipa, &ipl));
      lcrzo_er(lcrzo_conf_arp_add(device, localetha, ipl, LCRZO_FALSE));
      if (etha!=NULL) memcpy(etha, localetha, LCRZO_ETHA_MAXBYTES);
      return(LCRZO_ERR_OK);
    }
  }

  /*adresse non resolue*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_etha_init_ipa_conf(const lcrzo_ipa ipa, lcrzo_etha etha);
int lcrzo_priv_etha_init_ipa_conf(const lcrzo_ipa ipa, lcrzo_etha etha)
{ lcrzo_int32 count, i;
  lcrzo_ipl wantedipl, ipl;

  lcrzo_er(lcrzo_ipl_init_ipa(ipa, &wantedipl));
  lcrzo_er(lcrzo_conf_arp_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_arp_value_pos(i, NULL, etha, &ipl, NULL, NULL));
    if ( wantedipl == ipl )
    { return(LCRZO_ERR_OK);
    }
  } 
  /*adresse non resolue*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ipa_init_etha__spoofsniff(const lcrzo_device device, 
				    const lcrzo_etha etha, lcrzo_ipa ipa);
int lcrzo_priv_ipa_init_etha__spoofsniff(const lcrzo_device device, 
				    const lcrzo_etha etha, lcrzo_ipa ipa)
{ lcrzo_sniff cs;
  lcrzo_ipc ci;
  lcrzo_uint32 mtu;
  lcrzo_hdrleth hdrleth;
  lcrzo_hdrlarp hdrlarp, hdrlarprecu;
  int retour, pidsniff, status, i;
  lcrzo_spoof spoof;
  lcrzo_data pdata;
  lcrzo_int32 datasize;
  lcrzo_ipa ipaforcomp;

  /*initialisation de l'entete ETH*/
  lcrzo_er(lcrzo_hdrleth_initdefault(&hdrleth));
  lcrzo_er(lcrzo_priv_global_get_intspoof_eth(etha, hdrleth.src));
  lcrzo_er(lcrzo_etha_init(0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, hdrleth.dst));
  hdrleth.type=LCRZO_HDRLETH_TYPE_ARP;
  /*initialisation de l'entete RARP*/
  lcrzo_er(lcrzo_hdrlarp_initdefault(&hdrlarp));
  hdrlarp.op = 3;
  memcpy(hdrlarp.hw_src, hdrleth.src, LCRZO_ETHA_MAXBYTES);
  for ( i=0 ; i<LCRZO_IPA_MAXBYTES ; i++ )
  { lcrzo_er(lcrzo_uint8_rand(1, 0xFE, &(ipaforcomp[i])));
  }
  lcrzo_er(lcrzo_priv_global_get_intspoof_ip(ipaforcomp, hdrlarp.prot_src));
  lcrzo_er(lcrzo_ipa_init(0,0,0,0, hdrlarp.prot_dst));
  memcpy(hdrlarp.hw_dst, etha, LCRZO_ETHA_MAXBYTES);
  /*initialisation du sniff*/
  lcrzo_er(lcrzo_mtu_init_device(device, &mtu));
  lcrzo_er(lcrzo_sniff_init(device, mtu, "", &cs));
  /*initialisation du contexte d'IPC*/
  retour=lcrzo_ipc_init(&ci);
  if (retour!=LCRZO_ERR_OK)
  { lcrzo_er(lcrzo_sniff_close(&cs));
    return(retour);
  }
  /*envoi de la requete arp*/
  lcrzo_er(lcrzo_spoof_init(&spoof));
  retour=lcrzo_spoof_etharp(&spoof, device, hdrleth, hdrlarp);
  if (retour!=LCRZO_ERR_OK)
  { lcrzo_er(lcrzo_sniff_close(&cs));
    lcrzo_er(lcrzo_ipc_close(ci));
    lcrzo_er(lcrzo_spoof_close(&spoof));
    return(LCRZO_ERR_OKUNRESOLVED);
  }
  lcrzo_er(lcrzo_spoof_close(&spoof));
  /*processus de sniff*/
  pidsniff=fork();
  if (pidsniff<0)
  { lcrzo_er(lcrzo_sniff_close(&cs));
    lcrzo_er(lcrzo_ipc_close(ci));
    return(LCRZO_ERR_OKUNRESOLVED);
  }
  if (pidsniff==0) /*on est dans le fils*/
  { /*Attention : on ne doit jamais quitter avec "return()" (ou lcrzo_er),
      mais avec "_exit()", sinon ce processus retourne dans
      "lcrzo_etha_init_ipa_spoofsniff"*/
    /*on a une seconde pour obtenir le resultat*/
    alarm(1);
    signal(SIGALRM, lcrzo_priv_etha_init_ipa_findufils);
    /*on sniffe le resultat*/
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_NORMAL,
			     &pdata, &datasize);
    while( retour==LCRZO_ERR_OK )
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*on voit si c'est la reponse attendue*/
      retour=lcrzo_packet_decode_etharp(pdata, datasize, 
					NULL, &hdrlarprecu);
      if ( retour==LCRZO_ERR_OK && hdrlarprecu.op==4 )
      { if (lcrzo_ipa_equal(hdrlarprecu.hw_dst, etha))
	{ /*on a trouve*/
          retour=lcrzo_ipc_write_ipa(ci, hdrlarprecu.prot_src);
	  lcrzo_sniff_close(&cs);
	  lcrzo_ipc_close(ci);
          _exit(retour);
	}
      }
      lcrzo_data_free(pdata);
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_NORMAL,
			       &pdata, &datasize);
    }
    /*cet endroit n'est atteind que sur erreur de sniff*/
    lcrzo_sniff_close(&cs);
    lcrzo_ipc_close(ci);
    _exit(retour);
  }
  /*on attend la fin du fils*/
  waitpid(pidsniff, &status, 0);
  /*on lit ce qui est eventuellement dans le tunnel*/
  retour=lcrzo_ipc_read_ipa(ci, 0, ipa);
  /*fermeture des contextes : les erreurs peuvent etre maintenant ignorees*/
  lcrzo_sniff_close(&cs);
  lcrzo_ipc_close(ci);
  /*il y avait une reponse dans le tunnel*/
  if (retour==LCRZO_ERR_OK)
  { return(LCRZO_ERR_OK);
  }
  /*pas de reponse dans le tunnel*/
  return(LCRZO_ERR_OKUNRESOLVED);
}
int lcrzo_priv_ipa_init_etha_spoofsniff(const lcrzo_etha etha, lcrzo_ipa ipa);
int lcrzo_priv_ipa_init_etha_spoofsniff(const lcrzo_etha etha, lcrzo_ipa ipa)
{ lcrzo_device device;
  int retour;
  lcrzo_int32 count, i;
  lcrzo_ipa localipa;
  lcrzo_ipl ipl;

  lcrzo_er(lcrzo_device_count(LCRZO_DEVICE_TYPE_ONLYBOARDS, &count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_device_value_pos(LCRZO_DEVICE_TYPE_ONLYBOARDS, i, device));
    retour=lcrzo_priv_ipa_init_etha__spoofsniff(device, etha, localipa);
    if (retour==LCRZO_ERR_OK)
    { lcrzo_er(lcrzo_ipl_init_ipa(localipa, &ipl));
      lcrzo_er(lcrzo_conf_arp_add(device, etha, ipl, LCRZO_FALSE));
      if (ipa!=NULL) memcpy(ipa, localipa, LCRZO_IPA_MAXBYTES);
      return(LCRZO_ERR_OK);
    }
  }

  /*adresse non resolue*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ipa_init_etha_conf(const lcrzo_etha etha, lcrzo_ipa ipa);
int lcrzo_priv_ipa_init_etha_conf(const lcrzo_etha etha, lcrzo_ipa ipa)
{ lcrzo_int32 count, i;
  lcrzo_etha wantedetha;
  lcrzo_ipl ipl;

  lcrzo_er(lcrzo_conf_arp_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_arp_value_pos(i, NULL, wantedetha, &ipl, NULL, NULL));
    if ( lcrzo_etha_equal(etha, wantedetha) )
    { lcrzo_er(lcrzo_ipa_init_ipl(ipl, ipa));
      return(LCRZO_ERR_OK);
    }
  } 
  /*adresse non resolue*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

/*-------------------------------------------------------------*/
/*conversions faisant appel a des sous fonctions ci-dessus*/

/*-------------------------------------------------------------*/
/*les 4 fonctions ci-dessous vont tenter de resoudre la requete
  en utilisant successivement les differentes methodes.*/
int lcrzo_ipa_init_hs(const lcrzo_hs hs, lcrzo_ipa ipa)
{ int retour;
  lcrzo_hs hs_copie;

  lcrzo_er(lcrzo_string_init_coretext(hs, LCRZO_HS_MAXBYTES, hs_copie));

  /*on essaie ipa_init_ips*/
  retour=lcrzo_ipa_init_ips(hs_copie, ipa);
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*on interroge le resolver*/
  if ( lcrzo_global.resolver_timeout )
  { retour=lcrzo_priv_ipa_init_hs_resolvalarm(hs_copie, ipa);
  }
  else
  { retour=lcrzo_priv_ipa_init_hs_resolver(hs_copie, ipa);
  }
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*on n'a pas pu resoudre*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

int lcrzo_hs_init_ipa(const lcrzo_ipa ipa, lcrzo_hs hs)
{ int retour;

  /*parameters verification*/
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  /*on interroge le resolver*/
  if ( lcrzo_global.resolver_timeout )
  { retour=lcrzo_priv_hs_init_ipa_resolvalarm(ipa, hs);
  }
  else
  { retour=lcrzo_priv_hs_init_ipa_resolver(ipa, hs);
  }
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*on n'a pas pu resoudre*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

int lcrzo_etha_init_ipa(const lcrzo_ipa ipa, lcrzo_etha etha)
{ int retour;

  /*parameters verification*/
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  /*on interroge la conf locale*/
  retour=lcrzo_priv_etha_init_ipa_conf(ipa, etha);
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*we ask ioctl*/
  retour=lcrzo_sysdep_etha_init_ipa_ioctlarp(ipa, etha);
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*we ask /etc/ethers*/
  retour=lcrzo_sysdep_etha_init_ipa_etcethers(ipa, etha);
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*we ask again the configuration*/
  lcrzo_er(lcrzo_conf_set_forceupdate(1));
  retour=lcrzo_priv_etha_init_ipa_conf(ipa, etha);
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*on envoie une requete sur le reseau*/
  retour=lcrzo_priv_etha_init_ipa_spoofsniff(ipa, etha);
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*l'adresse n'a pas pu etre resolue*/
  return(LCRZO_ERR_OKUNRESOLVED);
}

int lcrzo_ipa_init_etha(const lcrzo_etha etha, lcrzo_ipa ipa)
{ int retour;

  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  /*on interroge la conf locale*/
  retour=lcrzo_priv_ipa_init_etha_conf(etha, ipa);
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*we ask ioctl*/
  retour=lcrzo_sysdep_ipa_init_etha_ioctlrarp(etha, ipa);
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*we ask /etc/ethers*/
  retour=lcrzo_sysdep_ipa_init_etha_etcethers(etha, ipa);
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*we ask again the configuration*/
  lcrzo_er(lcrzo_conf_set_forceupdate(1));
  retour=lcrzo_priv_ipa_init_etha_conf(etha, ipa);
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*on envoie une requete sur le reseau*/
  retour=lcrzo_priv_ipa_init_etha_spoofsniff(etha, ipa);
  lcrzo_err_purge();
  if (retour==LCRZO_ERR_OK) return(LCRZO_ERR_OK);
  /*l'adresse n'a pas pu etre resolue*/
  return(LCRZO_ERR_OKUNRESOLVED);
}


/*-------------------------------------------------------------*/
/*conversions faisant appel a des intermediaires*/

/*-------------------------------------------------------------*/
int lcrzo_ips_init_hs(const lcrzo_hs hs, lcrzo_ips ips)
{ lcrzo_ipa ipa;
  lcrzo_hs hs_copie;

  lcrzo_er(lcrzo_string_init_coretext(hs, LCRZO_HS_MAXBYTES, hs_copie));
  lcrzo_er(lcrzo_ipa_init_hs(hs_copie, ipa));
  lcrzo_er(lcrzo_ips_init_ipa(ipa, ips));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_init_hs(const lcrzo_hs hs, lcrzo_ipl *pipl)
{ lcrzo_ipa ipa;
  lcrzo_hs hs_copie;

  lcrzo_er(lcrzo_string_init_coretext(hs, LCRZO_HS_MAXBYTES, hs_copie));
  lcrzo_er(lcrzo_ipa_init_hs(hs_copie, ipa));
  lcrzo_er(lcrzo_ipl_init_ipa(ipa, pipl));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_init_hs(const lcrzo_hs hs, lcrzo_eths eths)
{ lcrzo_etha etha;
  lcrzo_hs hs_copie;

  lcrzo_er(lcrzo_string_init_coretext(hs, LCRZO_HS_MAXBYTES, hs_copie));
  lcrzo_er(lcrzo_etha_init_hs(hs_copie, etha));
  lcrzo_er(lcrzo_eths_init_etha(etha, eths));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_etha_init_hs(const lcrzo_hs hs, lcrzo_etha etha)
{ lcrzo_ipa ipa;
  lcrzo_hs hs_copie;

  lcrzo_er(lcrzo_string_init_coretext(hs, LCRZO_HS_MAXBYTES, hs_copie));
  lcrzo_er(lcrzo_ipa_init_hs(hs_copie, ipa));
  lcrzo_er(lcrzo_etha_init_ipa(ipa, etha));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hs_init_ips(const lcrzo_ips ips, lcrzo_hs hs)
{ lcrzo_etha ipa;
  lcrzo_ips ips_copie;

  lcrzo_er(lcrzo_string_init_coretext(ips, LCRZO_IPS_MAXBYTES, ips_copie));
  lcrzo_er(lcrzo_ipa_init_ips(ips_copie, ipa));
  lcrzo_er(lcrzo_hs_init_ipa(ipa, hs));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_init_ips(const lcrzo_ips ips, lcrzo_eths eths)
{ lcrzo_ipa ipa;
  lcrzo_ips ips_copie;

  lcrzo_er(lcrzo_string_init_coretext(ips, LCRZO_IPS_MAXBYTES, ips_copie));
  lcrzo_er(lcrzo_ipa_init_ips(ips_copie, ipa));
  lcrzo_er(lcrzo_eths_init_ipa(ipa, eths));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_etha_init_ips(const lcrzo_ips ips, lcrzo_etha etha)
{ lcrzo_etha ipa;
  lcrzo_ips ips_copie;

  lcrzo_er(lcrzo_string_init_coretext(ips, LCRZO_IPS_MAXBYTES, ips_copie));
  lcrzo_er(lcrzo_ipa_init_ips(ips_copie, ipa));
  lcrzo_er(lcrzo_etha_init_ipa(ipa, etha));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_init_ipa(const lcrzo_ipa ipa, lcrzo_eths eths)
{ lcrzo_etha etha;

  /*parameters verification*/
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_etha_init_ipa(ipa, etha));
  lcrzo_er(lcrzo_eths_init_etha(etha, eths));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hs_init_ipl(lcrzo_ipl ipl, lcrzo_hs hs)
{ lcrzo_ipa ipa;

  lcrzo_er(lcrzo_ipa_init_ipl(ipl, ipa));
  lcrzo_er(lcrzo_hs_init_ipa(ipa, hs));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_init_ipl(lcrzo_ipl ipl, lcrzo_eths eths)
{ lcrzo_ipa ipa;

  lcrzo_er(lcrzo_ipa_init_ipl(ipl, ipa));
  lcrzo_er(lcrzo_eths_init_ipa(ipa, eths));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_etha_init_ipl(lcrzo_ipl ipl, lcrzo_etha etha)
{ lcrzo_ipa ipa;

  lcrzo_er(lcrzo_ipa_init_ipl(ipl, ipa));
  lcrzo_er(lcrzo_etha_init_ipa(ipa, etha));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hs_init_eths(const lcrzo_eths eths, lcrzo_hs hs)
{ lcrzo_etha etha;
  lcrzo_eths eths_copie;

  lcrzo_er(lcrzo_string_init_coretext(eths, LCRZO_ETHS_MAXBYTES, eths_copie));
  lcrzo_er(lcrzo_etha_init_eths(eths_copie, etha));
  lcrzo_er(lcrzo_hs_init_etha(etha, hs));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ips_init_eths(const lcrzo_eths eths, lcrzo_ips ips)
{ lcrzo_etha etha;
  lcrzo_eths eths_copie;

  lcrzo_er(lcrzo_string_init_coretext(eths, LCRZO_ETHS_MAXBYTES, eths_copie));
  lcrzo_er(lcrzo_etha_init_eths(eths_copie, etha));
  lcrzo_er(lcrzo_ips_init_etha(etha, ips));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipa_init_eths(const lcrzo_eths eths, lcrzo_ipa ipa)
{ lcrzo_etha etha;
  lcrzo_eths eths_copie;

  lcrzo_er(lcrzo_string_init_coretext(eths, LCRZO_ETHS_MAXBYTES, eths_copie));
  lcrzo_er(lcrzo_etha_init_eths(eths_copie, etha));
  lcrzo_er(lcrzo_ipa_init_etha(etha, ipa));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_init_eths(const lcrzo_eths eths, lcrzo_ipl *pipl)
{ lcrzo_ipa ipa;
  lcrzo_eths eths_copie;

  lcrzo_er(lcrzo_string_init_coretext(eths, LCRZO_ETHS_MAXBYTES, eths_copie));
  lcrzo_er(lcrzo_ipa_init_eths(eths_copie, ipa));
  lcrzo_er(lcrzo_ipl_init_ipa(ipa, pipl));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hs_init_etha(const lcrzo_etha etha, lcrzo_hs hs)
{ lcrzo_ipa ipa;

  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipa_init_etha(etha, ipa));
  lcrzo_er(lcrzo_hs_init_ipa(etha, hs));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ips_init_etha(const lcrzo_etha etha, lcrzo_ips ips)
{ lcrzo_ipa ipa;

  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipa_init_etha(etha, ipa));
  lcrzo_er(lcrzo_ips_init_ipa(ipa, ips));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_init_etha(const lcrzo_etha etha, lcrzo_ipl *pipl)
{ lcrzo_ipa ipa;

  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipa_init_etha(etha, ipa));
  lcrzo_er(lcrzo_ipl_init_ipa(ipa, pipl));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_hs_equal(const lcrzo_hs hs1, const lcrzo_hs hs2)
{ int retour;
  lcrzo_hs h1, h2;
  lcrzo_ipl ipl1, ipl2;
  lcrzo_bool canconvert;

  /*parameters verification*/
  if (hs1==NULL) return(0);
  if (hs2==NULL) return(0);

  canconvert=0;  
  retour=lcrzo_string_init_coretext(hs1, LCRZO_HS_MAXBYTES, h1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_string_init_coretext(hs2, LCRZO_HS_MAXBYTES, h2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(hs1, hs2);
    if (!retour) return(1);
    else return(0);
  }

  canconvert=0;  
  retour=lcrzo_ipl_init_hs(h1, &ipl1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_ipl_init_hs(h2, &ipl2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(h1, h2);
    if (!retour) return(1);
    else return(0);
  }

  return(lcrzo_ipl_equal(ipl1, ipl2));
}

/*-------------------------------------------------------------*/
int lcrzo_hs_cmp(const lcrzo_hs hs1, const lcrzo_hs hs2)
{ int retour;
  lcrzo_hs h1, h2;
  lcrzo_ipl ipl1, ipl2;
  lcrzo_bool canconvert;

  /*parameters verification*/
  if (hs1==NULL) return(2);
  if (hs2==NULL) return(2);

  canconvert=0;  
  retour=lcrzo_string_init_coretext(hs1, LCRZO_HS_MAXBYTES, h1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_string_init_coretext(hs2, LCRZO_HS_MAXBYTES, h2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(hs1, hs2);
    if (retour<0) return(-1);
    else if (retour>0) return(+1);
    else return(0);
  }

  canconvert=0;  
  retour=lcrzo_ipl_init_hs(h1, &ipl1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_ipl_init_hs(h2, &ipl2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(h1, h2);
    if (retour<0) return(-1);
    else if (retour>0) return(+1);
    else return(0);
  }

  return(lcrzo_ipl_cmp(ipl1, ipl2));
}

/*-------------------------------------------------------------*/
int lcrzo_ips_equal(const lcrzo_ips ips1, const lcrzo_ips ips2)
{ int retour;
  lcrzo_ips ip1, ip2;
  lcrzo_ipl ipl1, ipl2;
  lcrzo_bool canconvert;

  /*parameters verification*/
  if (ips1==NULL) return(0);
  if (ips2==NULL) return(0);

  canconvert=0;  
  retour=lcrzo_string_init_coretext(ips1, LCRZO_IPS_MAXBYTES, ip1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_string_init_coretext(ips2, LCRZO_IPS_MAXBYTES, ip2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(ips1, ips2);
    if (!retour) return(1);
    else return(0);
  }

  canconvert=0;
  retour=lcrzo_ipl_init_ips(ip1, &ipl1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_ipl_init_ips(ip2, &ipl2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(ip1, ip2);
    if (!retour) return(1);
    else return(0);
  }

  return(lcrzo_ipl_equal(ipl1, ipl2));
}

/*-------------------------------------------------------------*/
int lcrzo_ips_cmp(const lcrzo_ips ips1, const lcrzo_ips ips2)
{ int retour;
  lcrzo_ips ip1, ip2;
  lcrzo_ipl ipl1, ipl2;
  lcrzo_bool canconvert;

  /*parameters verification*/
  if (ips1==NULL) return(2);
  if (ips2==NULL) return(2);

  canconvert=0;  
  retour=lcrzo_string_init_coretext(ips1, LCRZO_IPS_MAXBYTES, ip1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_string_init_coretext(ips2, LCRZO_IPS_MAXBYTES, ip2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(ips1, ips2);
    if (retour<0) return(-1);
    else if (retour>0) return(+1);
    else return(0);
  }

  canconvert=0;  
  retour=lcrzo_ipl_init_ips(ip1, &ipl1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_ipl_init_ips(ip2, &ipl2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(ip1, ip2);
    if (retour<0) return(-1);
    else if (retour>0) return(+1);
    else return(0);
  }

  return(lcrzo_ipl_cmp(ipl1, ipl2));
}

/*-------------------------------------------------------------*/
int lcrzo_ipa_equal(const lcrzo_ipa ipa1, const lcrzo_ipa ipa2)
{ 
  /*parameters verification*/
  if (ipa1==NULL) return(0);
  if (ipa2==NULL) return(0);
  return(!memcmp((const void*)ipa1, (const void*)ipa2, LCRZO_IPA_MAXBYTES));
}

/*-------------------------------------------------------------*/
int lcrzo_ipa_cmp(const lcrzo_ipa ipa1, const lcrzo_ipa ipa2)
{ 
  /*parameters verification*/
  if (ipa1==NULL) return(0);
  if (ipa2==NULL) return(0);

  if ( ipa1[0]<ipa2[0] ) return(-1);
  else if ( ipa1[0]>ipa2[0] ) return(+1);
  else
  { if ( ipa1[1]<ipa2[1] ) return(-1);
    else if ( ipa1[1]>ipa2[1] ) return(+1);
    else
    { if ( ipa1[2]<ipa2[2] ) return(-1);
      else if ( ipa1[2]>ipa2[2] ) return(+1);
      else
      { if ( ipa1[3]<ipa2[3] ) return(-1);
        else if ( ipa1[3]>ipa2[3] ) return(+1);
        else return(0);
      }
    }
  }
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_equal(lcrzo_ipl ipl1, lcrzo_ipl ipl2)
{
  if (ipl1==ipl2) return(1);
  else return(0);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_cmp(lcrzo_ipl ipl1, lcrzo_ipl ipl2)
{ if (ipl1<ipl2) return(-1);
  else if (ipl1>ipl2) return(+1);
  else return(0);
}

/*-------------------------------------------------------------*/
int lcrzo_eths_equal(const lcrzo_eths eths1, const lcrzo_eths eths2)
{ int retour;
  lcrzo_eths eth1, eth2;
  lcrzo_etha etha1, etha2;
  lcrzo_bool canconvert;

  /*parameters verification*/
  if (eths1==NULL) return(0);
  if (eths2==NULL) return(0);

  canconvert=0;  
  retour=lcrzo_string_init_coretext(eths1, LCRZO_ETHS_MAXBYTES, eth1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_string_init_coretext(eths2, LCRZO_ETHS_MAXBYTES, eth2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(eths1, eths2);
    if (!retour) return(1);
    else return(0);
  }

  canconvert=0;
  retour=lcrzo_etha_init_eths(eth1, etha1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_etha_init_eths(eth2, etha2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(eth1, eth2);
    if (!retour) return(1);
    else return(0);
  }

  return(lcrzo_etha_equal(etha1, etha2));
}

/*-------------------------------------------------------------*/
int lcrzo_eths_cmp(const lcrzo_eths eths1, const lcrzo_eths eths2)
{ int retour;
  lcrzo_eths eth1, eth2;
  lcrzo_etha etha1, etha2;
  lcrzo_bool canconvert;

  /*parameters verification*/
  if (eths1==NULL) return(2);
  if (eths2==NULL) return(2);

  canconvert=0;  
  retour=lcrzo_string_init_coretext(eths1, LCRZO_ETHS_MAXBYTES, eth1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_string_init_coretext(eths2, LCRZO_ETHS_MAXBYTES, eth2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(eths1, eths2);
    if (retour<0) return(-1);
    else if (retour>0) return(+1);
    else return(0);
  }

  canconvert=0;  
  retour=lcrzo_etha_init_eths(eth1, etha1);
  if (retour==LCRZO_ERR_OK)
  { retour=lcrzo_etha_init_eths(eth2, etha2);
    if (retour==LCRZO_ERR_OK) canconvert=1;
  }

  if (!canconvert)
  { lcrzo_err_purge();
    retour=strcmp(eth1, eth2);
    if (retour<0) return(-1);
    else if (retour>0) return(+1);
    else return(0);
  }

  return(lcrzo_etha_cmp(etha1, etha2));
}

/*-------------------------------------------------------------*/
int lcrzo_etha_equal(const lcrzo_etha etha1, const lcrzo_etha etha2)
{ 
  /*parameters verification*/
  if (etha1==NULL) return(0);
  if (etha2==NULL) return(0);
  return(!memcmp((const void*)etha1, (const void*)etha2, LCRZO_ETHA_MAXBYTES));
}

/*-------------------------------------------------------------*/
int lcrzo_etha_cmp(const lcrzo_etha etha1, const lcrzo_etha etha2)
{
  /*parameters verification*/
  if (etha1==NULL) return(2);
  if (etha2==NULL) return(2);

  if ( etha1[0]<etha2[0] ) return(-1);
  else if ( etha1[0]>etha2[0] ) return(+1);
  else
  { if ( etha1[1]<etha2[1] ) return(-1);
    else if ( etha1[1]>etha2[1] ) return(+1);
    else
    { if ( etha1[2]<etha2[2] ) return(-1);
      else if ( etha1[2]>etha2[2] ) return(+1);
      else
      { if ( etha1[3]<etha2[3] ) return(-1);
        else if ( etha1[3]>etha2[3] ) return(+1);
        else
        { if ( etha1[4]<etha2[4] ) return(-1);
          else if ( etha1[4]>etha2[4] ) return(+1);
          else
          { if ( etha1[5]<etha2[5] ) return(-1);
            else if ( etha1[5]>etha2[5] ) return(+1);
            else return(0);
          }
        }
      }
    }
  }
}

/*-------------------------------------------------------------*/
int lcrzo_hs_initdefault(lcrzo_hs hs)
{ if (hs!=NULL) hs[0]='\0';
  return(LCRZO_ERR_OK);
}
int lcrzo_ips_initdefault(lcrzo_ips ips)
{ if (ips!=NULL) ips[0]='\0';
  return(LCRZO_ERR_OK);
}
int lcrzo_ipa_initdefault(lcrzo_ipa ipa)
{ int i;
  if (ipa!=NULL)
  { for (i=0;i<LCRZO_IPA_MAXBYTES;i++) ipa[i]=0;
  }
  return(LCRZO_ERR_OK);
}
int lcrzo_ipl_initdefault(lcrzo_ipl *pipl)
{ if (pipl!=NULL) *pipl=0;
  return(LCRZO_ERR_OK);
}
int lcrzo_eths_initdefault(lcrzo_eths eths)
{ if (eths!=NULL) eths[0]='\0';
  return(LCRZO_ERR_OK);
}
int lcrzo_etha_initdefault(lcrzo_etha etha)
{ int i;
  if (etha!=NULL) 
  { for (i=0;i<LCRZO_ETHA_MAXBYTES;i++) etha[i]=0;
  }
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_ipa_stdin(const char *message,
		    const lcrzo_hs hsdefaut, lcrzo_ipa ipa)
{ char prompt;
  lcrzo_ipa ipadefaut;
  lcrzo_string usertext;
  int retour;

  /*parameters verification*/
  if (hsdefaut==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_hs_verifbof(hsdefaut);
  lcrzo_er(lcrzo_ipa_init_hs(hsdefaut, ipadefaut));

  prompt=':';
  while(1)
  { /*affichage du message*/
    if (message!=NULL) if (message[0]!='\0')
    { printf("%s [%s]%c ", message, hsdefaut, prompt);
    }
    prompt='>';
    /*saisie de la chaine de l'utilisateur*/
    lcrzo_er(lcrzo_priv_stdin(&usertext, NULL));
    /*si rien n'a ete entre, on met le defaut*/
    if (strlen(usertext)==0)
    { h_errno=0; /*si les tentatives precedentes etaient en erreur
                   on est oblige de mettre a zero ici car on ne
                   passe plus dans lcrzo_ipa_init_hs()*/
      lcrzo_string_free(usertext);
      if (ipa!=NULL) memcpy(ipa, ipadefaut, LCRZO_IPA_MAXBYTES);
      break;
    }
    /*conversion de hs en ipa*/
    retour=lcrzo_ipa_init_hs(usertext, ipa);
    lcrzo_string_free(usertext);
    if (retour==LCRZO_ERR_OK) break;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipl_stdin(const char *message,
		    const lcrzo_hs hsdefaut, lcrzo_ipl *pipl)
{ char prompt;
  lcrzo_ipl ipldefaut;
  lcrzo_string usertext;
  int retour;

  /*parameters verification*/
  if (hsdefaut==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_hs_verifbof(hsdefaut);
  lcrzo_er(lcrzo_ipl_init_hs(hsdefaut, &ipldefaut));

  prompt=':';
  while(1)
  { /*affichage du message*/
    if (message!=NULL) if (message[0]!='\0')
    { printf("%s [%s]%c ", message, hsdefaut, prompt);
    }
    prompt='>';
    /*saisie de la chaine de l'utilisateur*/
    lcrzo_er(lcrzo_priv_stdin(&usertext, NULL));
    /*si rien n'a ete entre, on met le defaut*/
    if (strlen(usertext)==0)
    { h_errno=0; /*si les tentatives precedentes etaient en erreur
                   on est oblige de mettre a zero ici car on ne
                   passe plus dans lcrzo_ipl_init_hs()*/
      lcrzo_string_free(usertext);
      if (pipl!=NULL) *pipl=ipldefaut;
      break;
    }
    /*conversion de hs en ipl*/
    retour=lcrzo_ipl_init_hs(usertext, pipl);
    lcrzo_string_free(usertext);
    if (retour==LCRZO_ERR_OK) break;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_etha_stdin(const char *message,
		     const lcrzo_eths ethsdefaut, lcrzo_etha etha)
{ char prompt;
  lcrzo_etha ethadefaut;
  lcrzo_string usertext;
  int retour;

  /*parameters verification*/
  if (ethsdefaut==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_eths_verifbof(ethsdefaut);
  lcrzo_er(lcrzo_etha_init_eths(ethsdefaut, ethadefaut));

  prompt=':';
  while(1)
  { /*affichage du message*/
    if (message!=NULL) if (message[0]!='\0')
    { printf("%s [%s]%c ", message, ethsdefaut, prompt);
    }
    prompt='>';
    /*saisie de la chaine de l'utilisateur*/
    lcrzo_er(lcrzo_priv_stdin(&usertext, NULL));
    /*si rien n'a ete entre, on met le defaut*/
    if (strlen(usertext)==0)
    { h_errno=0; /*si les tentatives precedentes etaient en erreur
                   on est oblige de mettre a zero ici car on ne
                   passe plus dans lcrzo_etha_init_hs()*/
      lcrzo_string_free(usertext);
      if (etha!=NULL) memcpy(etha, ethadefaut, LCRZO_ETHA_MAXBYTES);
      break;
    }
    /*conversion de eths en etha*/
    retour=lcrzo_etha_init_eths(usertext, etha);
    lcrzo_string_free(usertext);
    if (retour==LCRZO_ERR_OK) break;
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_str2int(char *str, lcrzo_uint32 maxi,
			int chainevideautorisee, lcrzo_uint32 valeursivide,
			int coderr, lcrzo_uint32 *pint);
int lcrzo_priv_str2int(char *str, lcrzo_uint32 maxi,
			int chainevideautorisee, lcrzo_uint32 valeursivide,
			int coderr, lcrzo_uint32 *pint)
{ char *pc;
  lcrzo_uint32 nb;

  if ( strlen(str)==0 )
  { if (chainevideautorisee)
    { *pint=valeursivide;
      return(LCRZO_ERR_OK);
    }
    else return(coderr);
  }
  nb=strtoul(str, &pc, 10);
  if ( *pc != '\0' ) return(coderr);
  if ( nb > maxi ) return(coderr);
  *pint=nb;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_etha_init_ethrange(const lcrzo_ethrange plage,
		       lcrzo_etha ethainf,
		       lcrzo_etha ethasup)
{ int poscarac, retour, type, i, doitplusavoirzero;
  lcrzo_uint32 nbbitsmasque;
  lcrzo_ethrange plagepurge, gauche, droite;
  char *pc;
  lcrzo_etha ethinf, ethsup, masque, ethinfandmasque;

  /*verifie la plage entree*/
  lcrzo_er(lcrzo_string_init_coretext(plage, LCRZO_ETHRANGE_MAXBYTES,
				   plagepurge));
  if (strlen(plagepurge)>LCRZO_ETHRANGE_MAXBYTES)
    return(LCRZO_ERR_PAOVERFLOW);

  /*cherche d'abord une adresse eth seule "a:b:c:d:e:f"*/
  retour=lcrzo_etha_init_eths(plagepurge, ethinf);
  if (retour==LCRZO_ERR_OK)
  { if (ethainf!=NULL) memcpy(ethainf, ethinf, LCRZO_ETHA_MAXBYTES);
    if (ethasup!=NULL) memcpy(ethasup, ethinf, LCRZO_ETHA_MAXBYTES);
    return(LCRZO_ERR_OK);
  }

  /*les autres plages doivent contenir -, / ou %*/
  type=0;
  poscarac=0; /*pour faire plaisir au compilateur*/
  pc=strchr(plagepurge, '-'); if (pc!=NULL) { type=1; poscarac=pc-plagepurge; }
  pc=strchr(plagepurge, '/'); if (pc!=NULL) { type=2; poscarac=pc-plagepurge; }
  pc=strchr(plagepurge, '%'); if (pc!=NULL) { type=3; poscarac=pc-plagepurge; }
  if (!type) return(LCRZO_ERR_FMETHRANGE);

  /*on scinde la chaine en 2*/
  lcrzo_er(lcrzo_string_init_text(plagepurge, LCRZO_ETHRANGE_MAXBYTES, gauche));
  gauche[poscarac]='\0';
  lcrzo_er(lcrzo_string_init_text(plagepurge+poscarac+1,
			     LCRZO_ETHRANGE_MAXBYTES, droite));

  /*on cherche maintenant une adresse de la forme "1:2:3:4:5:6-a:b:c:d:e:f"*/
  if ( type==1 )
  { retour=lcrzo_etha_init_eths(gauche, ethinf);
    if (retour==LCRZO_ERR_OK)
    { retour=lcrzo_etha_init_eths(droite, ethsup);
      if (retour==LCRZO_ERR_OK)
      { if (lcrzo_etha_cmp(ethinf, ethsup)==1)
	  return(LCRZO_ERR_PAINFHIGHERSUP);
        if (ethainf!=NULL) memcpy(ethainf, ethinf, LCRZO_ETHA_MAXBYTES);
	if (ethasup!=NULL) memcpy(ethasup, ethsup, LCRZO_ETHA_MAXBYTES);
	return(LCRZO_ERR_OK);
      }
    } 
    return(LCRZO_ERR_FMETHRANGE);
  }

  /*pour les autres types, gauche doit etre un eths, ou un fragment d'eths*/
  i=0;
  do
  { retour=lcrzo_etha_init_eths(gauche, ethinf);
    i++;
    lcrzo_er(lcrzo_string_append_char(':', 1,LCRZO_ETHRANGE_MAXBYTES, gauche));
    lcrzo_er(lcrzo_string_append_char('0', 1,LCRZO_ETHRANGE_MAXBYTES, gauche));
  } while ( retour!=LCRZO_ERR_OK && i<LCRZO_ETHA_MAXBYTES );
  if ( retour!=LCRZO_ERR_OK ) return(LCRZO_ERR_FMETHRANGE);
  /*ethinf contient maintenant la partie de gauche*/

  /*a droite, il doit y avoir soit un entier, soit un masque*/ 
  retour=lcrzo_priv_str2int(droite, 48, 0, 0,
			     LCRZO_ERR_FMETHRANGE, &nbbitsmasque);
  if (retour!=LCRZO_ERR_OK)
  { /*il doit donc y avoir un "FF:FF:0:0:0:0"*/
    retour=lcrzo_etha_init_eths(droite, ethsup);
    if (retour!=LCRZO_ERR_OK) return(LCRZO_ERR_FMETHRANGE);
    /*on verifie que ipsup est bien un masque*/
    nbbitsmasque=48;
    doitplusavoirzero=0;
    for ( i=0 ; i<48 ; i++ )
    { if ( !(ethsup[LCRZO_ETHA_MAXBYTES-1]&1) )
      { if (!doitplusavoirzero)
          nbbitsmasque--;
        else
	  return(LCRZO_ERR_FMETHRANGE);
      }
      else
      { doitplusavoirzero=1;
      }
      lcrzo_priv_etha_shr(ethsup, ethsup);
    }
  }
  /*nbbitsmasque contient maintenant le nb de bits du masque*/

  /*calcul du masque a partir de nbbitsmasque*/
  lcrzo_etha_init(0,0,0,0,0,0, masque);
  for ( i=0 ; i<48-(int)nbbitsmasque ; i++ )
  { lcrzo_priv_etha_shl(masque, masque);
    masque[LCRZO_ETHA_MAXBYTES-1]=(lcrzo_uint8)(masque[LCRZO_ETHA_MAXBYTES-1]
						| 1);
  }
  /*masque doit maintenant contenir 0b0000000...00000111111111111*/

  /*on verifie la coherence entre ethinf et nbbitsmasque*/
  lcrzo_priv_etha_and(ethinf, masque, ethinfandmasque);
  if ( ethinfandmasque[0] | ethinfandmasque[1] | ethinfandmasque[2] |
       ethinfandmasque[3] | ethinfandmasque[4] | ethinfandmasque[5] )
  { return(LCRZO_ERR_FMETHRANGE);
  }

  /*affecte le resultat*/
  lcrzo_priv_etha_or(ethinf, masque, ethsup);
  if (type==2)
  { if (ethainf!=NULL) memcpy(ethainf, ethinf, LCRZO_ETHA_MAXBYTES);
    if (ethasup!=NULL) memcpy(ethasup, ethsup, LCRZO_ETHA_MAXBYTES);
  }
  else 
  { /* xx%47 et xx%48 sont invalides*/
    if (nbbitsmasque>=47) return(LCRZO_ERR_FMETHRANGE);
    if (ethainf!=NULL) lcrzo_priv_etha_inc(ethinf, ethainf);
    if (ethasup!=NULL) lcrzo_priv_etha_dec(ethsup, ethasup);
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_ethrange_init_etha(lcrzo_etha ethainf,
		       lcrzo_etha ethasup,
		       lcrzo_ethrange plage)
{ lcrzo_etha masque, tmp, zero, masqueandethainf, notmasqueorethainf;
  lcrzo_etha ethainfmoinsun, ethasupplusun;
  int i;

  /*parameters verification*/
  if (ethainf==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethasup==NULL) return(LCRZO_ERR_PANULLPTR);

  if (lcrzo_etha_cmp(ethainf, ethasup)==1)
  { return(LCRZO_ERR_PAINFHIGHERSUP);
  }
  else if (lcrzo_etha_equal(ethainf, ethasup)) 
  { lcrzo_er(lcrzo_eths_init_etha(ethainf, plage));
  }
  else
  { /*we should never encounter an error, so we can stop now, if we do not
    need to compute "plage".*/
    if (plage==NULL) return(LCRZO_ERR_OK);
    /*on tente d'abord un affichage du type "x:x:x:x:x:x/y" */
    lcrzo_priv_etha_xor(ethainf, ethasup, tmp);
    lcrzo_etha_init(0,0,0,0,0,0, zero);
    lcrzo_etha_init(0,0,0,0,0,0, masque);
    i=48;
    while(!lcrzo_etha_equal(tmp, zero))
    { lcrzo_priv_etha_shr(tmp, tmp);
      lcrzo_priv_etha_shl(masque, masque);
      masque[LCRZO_ETHA_MAXBYTES-1]=(lcrzo_uint8)(masque[LCRZO_ETHA_MAXBYTES
							-1] | 1 );
      i--;
    }
    lcrzo_priv_etha_not(masque, masque);
    /*masque doit maintenant contenir 0b111111111111...1111110000*/
    lcrzo_priv_etha_and(masque, ethainf, masqueandethainf);
    lcrzo_priv_etha_not(masque, tmp);
    lcrzo_priv_etha_or(tmp, ethainf, notmasqueorethainf);
    if ( lcrzo_etha_equal(masqueandethainf, ethainf) &&
	 lcrzo_etha_equal(notmasqueorethainf, ethasup) )
    { /*il s'agit d'un intervalle correct*/
      lcrzo_er(lcrzo_eths_init_etha(ethainf, plage));
      lcrzo_er(lcrzo_string_append_char('/', 1,LCRZO_ETHRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_string_append_int(i, "%d", LCRZO_ETHRANGE_MAXBYTES, plage));
      return(LCRZO_ERR_OK);
    }
    lcrzo_priv_etha_dec(ethainf, ethainfmoinsun);
    lcrzo_priv_etha_inc(ethasup, ethasupplusun);
    if ( lcrzo_etha_equal(masqueandethainf, ethainfmoinsun) &&
	 lcrzo_etha_equal(notmasqueorethainf, ethasupplusun) )
    { /*il s'agit d'un intervalle correct, sans les broadcast*/
      lcrzo_er(lcrzo_eths_init_etha(ethainfmoinsun, plage));
      lcrzo_er(lcrzo_string_append_char('%', 1, LCRZO_ETHRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_string_append_int(i, "%d", LCRZO_ETHRANGE_MAXBYTES, plage));
    }
    else
    { lcrzo_er(lcrzo_eths_init_etha(ethainf, plage));
      lcrzo_er(lcrzo_string_append_char('-', 1, LCRZO_ETHRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_eths_init_etha(ethasup, plage+strlen(plage)));
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_ipl_init_iprange(const lcrzo_iprange plage,
		     lcrzo_ipl *piplinf,
		     lcrzo_ipl *piplsup)
{ int poscarac, retour, type, i, doitplusavoirzero;
  lcrzo_uint32 nbbitsmasque, masque;
  lcrzo_iprange plagepurge, gauche, droite;
  char *pc;
  lcrzo_ipl ipinf, ipsup;

  /*verifie la plage entree*/
  lcrzo_er(lcrzo_string_init_coretext(plage, LCRZO_IPRANGE_MAXBYTES, plagepurge));
  if (strlen(plagepurge)>LCRZO_IPRANGE_MAXBYTES)
    return(LCRZO_ERR_PAOVERFLOW);

  /*cherche d'abord une adresse IP seule "1.2.3.4"*/
  retour=lcrzo_ipl_init_ips(plagepurge, &ipinf);
  if (retour==LCRZO_ERR_OK)
  { if (piplinf!=NULL) *piplinf=ipinf;
    if (piplsup!=NULL) *piplsup=ipinf;
    return(LCRZO_ERR_OK);
  }

  /*les autres plages doivent contenir -, / ou %*/
  type=0;
  poscarac=0; /*pour faire plaisir au compilateur*/
  pc=strchr(plagepurge, '-'); if (pc!=NULL) { type=1; poscarac=pc-plagepurge; }
  pc=strchr(plagepurge, '/'); if (pc!=NULL) { type=2; poscarac=pc-plagepurge; }
  pc=strchr(plagepurge, '%'); if (pc!=NULL) { type=3; poscarac=pc-plagepurge; }
  if (!type) return(LCRZO_ERR_FMIPRANGE);

  /*on scinde la chaine en 2*/
  lcrzo_er(lcrzo_string_init_text(plagepurge, LCRZO_IPRANGE_MAXBYTES, gauche));
  gauche[poscarac]='\0';
  lcrzo_er(lcrzo_string_init_text(plagepurge+poscarac+1,
			     LCRZO_IPRANGE_MAXBYTES, droite));

  /*on cherche maintenant une adresse de la forme "1.2.3.4-5.6.7.8"*/
  if ( type==1 )
  { retour=lcrzo_ipl_init_ips(gauche, &ipinf);
    if (retour==LCRZO_ERR_OK)
    { retour=lcrzo_ipl_init_ips(droite, &ipsup);
      if (retour==LCRZO_ERR_OK)
      { if (ipsup<ipinf) return(LCRZO_ERR_PAINFHIGHERSUP);
        if (piplinf!=NULL) *piplinf=ipinf;
        if (piplsup!=NULL) *piplsup=ipsup;
	return(LCRZO_ERR_OK);
      }
    }
    return(LCRZO_ERR_FMIPRANGE);
  }

  /*pour les autres types, gauche doit etre un ips, ou un fragment d'ips*/
  i=0;
  do
  { retour=lcrzo_ipl_init_ips(gauche, &ipinf);
    i++;
    lcrzo_er(lcrzo_string_append_char('.', 1, LCRZO_IPRANGE_MAXBYTES, gauche));
    lcrzo_er(lcrzo_string_append_char('0', 1, LCRZO_IPRANGE_MAXBYTES, gauche));
  } while ( retour!=LCRZO_ERR_OK && i<=3 );
  if ( retour!=LCRZO_ERR_OK ) return(LCRZO_ERR_FMIPRANGE);
  /*ipinf contient maintenant la partie de gauche*/

  /*a droite, il doit y avoir soit un entier, soit un masque*/ 
  retour=lcrzo_priv_str2int(droite, 32, 0, 0,
			     LCRZO_ERR_FMIPRANGE, &nbbitsmasque);
  if (retour!=LCRZO_ERR_OK)
  { /*il doit donc y avoir un "255.255.0.0"*/
    retour=lcrzo_ipl_init_ips(droite, &ipsup);
    if (retour!=LCRZO_ERR_OK) return(LCRZO_ERR_FMIPRANGE);
    /*on verifie que ipsup est bien un masque*/
    nbbitsmasque=32;
    doitplusavoirzero=0;
    for ( i=0 ; i<32 ; i++ )
    { if ( !(ipsup&1) )
      { if (!doitplusavoirzero)
          nbbitsmasque--;
        else
	  return(LCRZO_ERR_FMIPRANGE);
      }
      else
      { doitplusavoirzero=1;
      }
      ipsup >>= 1;
    }
  }
  /*nbbitsmasque contient maintenant le nb de bits du masque*/

  /*calcul du masque a partir de nbbitsmasque*/
  masque=0;
  for ( i=0 ; i<32-(int)nbbitsmasque ; i++ )
  { masque=(masque<<1)|1;
  }
  /*masque doit maintenant contenir 0b0000000...00000111111111111*/

  /*on verifie la coherence entre ipinf et nbbitsmasque*/
  if (ipinf&masque) return(LCRZO_ERR_FMIPRANGE);

  /*affecte le resultat*/
  if (type==2)
  { if (piplinf!=NULL) *piplinf=ipinf;
    if (piplsup!=NULL) *piplsup=ipinf|masque;
  }
  else 
  { /* xx%31 et xx%32 sont invalides*/
    if (nbbitsmasque>=31) return(LCRZO_ERR_FMIPRANGE);
    if (piplinf!=NULL) *piplinf=ipinf+1;
    if (piplsup!=NULL) *piplsup=(ipinf|masque)-1;
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_iprange_init_ipl(lcrzo_ipl iplinf,
		     lcrzo_ipl iplsup,
		     lcrzo_iprange plage)
{ lcrzo_uint32 masque, tmp, i;

  if (iplinf>iplsup)
  { return(LCRZO_ERR_PAINFHIGHERSUP);
  }
  else if (iplinf==iplsup) 
  { lcrzo_er(lcrzo_ips_init_ipl(iplinf, plage));
  }
  else
  { /*we should never encounter an error, so we can stop now, if we do not
    need to compute "plage".*/
    if (plage==NULL) return(LCRZO_ERR_OK);
    /*on tente d'abord un affichage du type "x.x.x.x/y" */
    tmp = iplinf^iplsup;
    masque=0;
    i=32;
    while(tmp)
    { tmp>>=1;
      masque=(masque<<1)|1;
      i--;
    }
    masque = (~masque);
    /*masque doit maintenant contenir 0b111111111111...1111110000*/
    if ( (masque&iplinf)==iplinf &&
	 ((~masque)|iplinf)==iplsup )
    { /*il s'agit d'un intervalle correct*/
      lcrzo_er(lcrzo_ips_init_ipl(iplinf, plage));
      lcrzo_er(lcrzo_string_append_char('/', 1, LCRZO_IPRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_string_append_int(i, "%d", LCRZO_IPRANGE_MAXBYTES, plage));
    }
    else if ( (masque&iplinf)==(iplinf-1) &&
	      ((~masque)|iplinf)==(iplsup+1) )
    { /*il s'agit d'un intervalle correct, sans les broadcast*/
      lcrzo_er(lcrzo_ips_init_ipl(iplinf-1, plage));
      lcrzo_er(lcrzo_string_append_char('%', 1, LCRZO_IPRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_string_append_int(i, "%d", LCRZO_IPRANGE_MAXBYTES, plage));
    }
    else
    { lcrzo_er(lcrzo_ips_init_ipl(iplinf, plage));
      lcrzo_er(lcrzo_string_append_char('-', 1, LCRZO_IPRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_ips_init_ipl(iplsup, plage+strlen(plage)));
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_port_init_portrange(const lcrzo_portrange plage,
			 lcrzo_uint16 *pportinf,
			 lcrzo_uint16 *pportsup)
{ int posmoins;
  lcrzo_portrange plagepurge, inf, sup;
  char *pc;
  lcrzo_uint32 port;
  lcrzo_uint16 portinf, portsup;

  /*verifie la plage entree*/
  lcrzo_er(lcrzo_string_init_coretext(plage, LCRZO_PORTRANGE_MAXBYTES,
				   plagepurge));
  if (strlen(plagepurge)>LCRZO_PORTRANGE_MAXBYTES)
    return(LCRZO_ERR_PAOVERFLOW);

  /*trouve la position du '-'*/
  pc=strchr(plagepurge, '-');
  if (pc==NULL)
  { /*il ne doit y avoir qu'un port*/
    lcrzo_er(lcrzo_priv_str2int(plagepurge, 0xFFFF, 0, 0,
				 LCRZO_ERR_FMPORTRANGE, &port));
    portinf=(lcrzo_uint16)port;
    portsup=(lcrzo_uint16)port;
  }
  else
  { posmoins=pc-plagepurge;
    /*il y a un '-' dans la chaine*/
    /*on s'occupe du port inferieur*/  
    lcrzo_er(lcrzo_string_init_text(plagepurge, LCRZO_PORTRANGE_MAXBYTES, inf));
    inf[posmoins]='\0';
    lcrzo_er(lcrzo_priv_str2int(inf, 0xFFFF, 1, 1,
				 LCRZO_ERR_FMPORTRANGE, &port));
        
    portinf=(lcrzo_uint16)port;
    /*on s'occupe du port superieur*/
    lcrzo_er(lcrzo_string_init_text(plagepurge+posmoins+1,
			       LCRZO_PORTRANGE_MAXBYTES, sup));
    lcrzo_er(lcrzo_priv_str2int(sup, 0xFFFF, 1, 0xFFFF,
				 LCRZO_ERR_FMPORTRANGE, &port));
    portsup=(lcrzo_uint16)port;
    if (portinf>portsup) return(LCRZO_ERR_PAINFHIGHERSUP);
  }
  if (pportinf!=NULL) *pportinf=portinf;
  if (pportsup!=NULL) *pportsup=portsup;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_portrange_init_port(lcrzo_uint16 portinf,
			 lcrzo_uint16 portsup,
			 lcrzo_portrange plage)
{
  if (portinf>portsup) return(LCRZO_ERR_PAINFHIGHERSUP);
  else if (portinf==portsup)
  { lcrzo_er(lcrzo_string_init_int(portinf, "%d",
			       LCRZO_PORTRANGE_MAXBYTES, plage));
  }
  else
  { /*we should never encounter an error, so we can stop now, if we do not
    need to compute "plage".*/
    if (plage==NULL) return(LCRZO_ERR_OK);
    if (portsup==0xFFFF)
    { lcrzo_er(lcrzo_string_init_int(portinf, "%d",
				 LCRZO_PORTRANGE_MAXBYTES, plage));
      lcrzo_er(lcrzo_string_append_char('-', 1, LCRZO_PORTRANGE_MAXBYTES, plage));
    }
    else
    { if (portinf!=1)
      { lcrzo_er(lcrzo_string_init_int(portinf, "%d",
				   LCRZO_PORTRANGE_MAXBYTES,plage));
        lcrzo_er(lcrzo_string_append_char('-', 1, LCRZO_PORTRANGE_MAXBYTES, plage));
      }
      else
      { lcrzo_er(lcrzo_string_init_char('-', 1, LCRZO_PORTRANGE_MAXBYTES, plage));
      }
      lcrzo_er(lcrzo_string_append_int(portsup, "%d",
				   LCRZO_PORTRANGE_MAXBYTES,plage));
    }
  }

  return(LCRZO_ERR_OK);
}

