/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #define _GNU_SOURCE
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <errno.h>
 #include <signal.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
 #include <netinet/in.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <errno.h>
 #include <signal.h>
 #include <fcntl.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
 #include <netinet/in.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #define __EXTENSIONS__
 #include <stdlib.h>
 #include <stdio.h>
 #include <string.h>
 #include <unistd.h>
 #include <errno.h>
 #include <signal.h>
 #include <fcntl.h>
 #include <sys/types.h>
 #include <sys/socket.h>
 #include <sys/wait.h>
 #include <netinet/in.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
#define LCRZO_PRIV_TYPESOCK_REALCLIUDP 1
#define LCRZO_PRIV_TYPESOCK_REALCLITCP 2
#define LCRZO_PRIV_TYPESOCK_VIRTCLIUDP 3
#define LCRZO_PRIV_TYPESOCK_VIRTCLITCP 4
#define LCRZO_PRIV_TYPESOCK_REALSERUDP 5
#define LCRZO_PRIV_TYPESOCK_REALSERTCP 6
#define LCRZO_PRIV_TYPESOCK_VIRTSERUDP 7
#define LCRZO_PRIV_TYPESOCK_VIRTSERTCP 8
#define LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI 9
#define LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI 10
#define LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI 11

/***************************************************************/
/* Zone de fonctions privees */

/*-------------------------------------------------------------*/
int lcrzo_priv_initlesci(lcrzo_sock *psock);
int lcrzo_priv_initlesci(lcrzo_sock *psock)
{ int retour;
  
  lcrzo_er(lcrzo_ipc_init(&(psock->cisniff)));
  retour=lcrzo_ipc_init(&(psock->cispoof));
  if ( retour!=LCRZO_ERR_OK )
  { lcrzo_er(lcrzo_ipc_close(psock->cisniff));
    return(retour);
  }
  retour=lcrzo_ipc_init(&(psock->cioptions));
  if ( retour!=LCRZO_ERR_OK )
  { lcrzo_er(lcrzo_ipc_close(psock->cispoof));
    lcrzo_er(lcrzo_ipc_close(psock->cisniff));
    return(retour);
  }
  retour=lcrzo_ipc_init(&(psock->cicontrole1));
  if ( retour!=LCRZO_ERR_OK )
  { lcrzo_er(lcrzo_ipc_close(psock->cioptions));
    lcrzo_er(lcrzo_ipc_close(psock->cispoof));
    lcrzo_er(lcrzo_ipc_close(psock->cisniff));
    return(retour);
  }
  retour=lcrzo_ipc_init(&(psock->cicontrole2));
  if ( retour!=LCRZO_ERR_OK )
  { lcrzo_er(lcrzo_ipc_close(psock->cicontrole1));
    lcrzo_er(lcrzo_ipc_close(psock->cioptions));
    lcrzo_er(lcrzo_ipc_close(psock->cispoof));
    lcrzo_er(lcrzo_ipc_close(psock->cisniff));
    return(retour);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fermelesci(lcrzo_sock sock);
int lcrzo_priv_fermelesci(lcrzo_sock sock)
{ 
  lcrzo_ipc_close(sock.cioptions);
  lcrzo_ipc_close(sock.cicontrole2);
  lcrzo_ipc_close(sock.cicontrole1);
  lcrzo_ipc_close(sock.cisniff);
  lcrzo_ipc_close(sock.cispoof);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ajouteipopt(lcrzo_sock *psock, const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum);
int lcrzo_priv_ajouteipopt(lcrzo_sock *psock, const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum)
{
  if ( ipoptbytenum!=0 )
  { if (ipoptbytenum>LCRZO_IPOPT_MAXBYTES)
      return(LCRZO_ERR_SPOPTMAX10);
    psock->ipoptbytenum=ipoptbytenum;
    memcpy(psock->ipopt, ipopt, ipoptbytenum);
  }
  else
  { psock->ipoptbytenum=0;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fork(lcrzo_sock sock, int *ppid);
int lcrzo_priv_fork(lcrzo_sock sock, int *ppid)
{ 
  *ppid=fork();
  if (*ppid<0)
  { lcrzo_priv_fermelesci(sock);
    return(LCRZO_ERR_FUFORK);
  }
  /*Attention : on ne doit jamais quitter avec "return()" (ou lcrzo_er),
    mais avec "_exit()", sinon le processus retourne dans la fonction
    appelante du pere*/

  /*on initialise le generateur aleatoire du fils*/
  if (*ppid==0)
  { lcrzo_rand_init(0);
  }
 
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_processinitsniff(const lcrzo_device device,
				 lcrzo_sniff *pcs);
int lcrzo_priv_processinitsniff(const lcrzo_device device,
				 lcrzo_sniff *pcs)
{ int retour;
  lcrzo_uint32 mtu;

  retour=lcrzo_mtu_init_device(device, &mtu);
  if ( retour!=LCRZO_ERR_OK )
  { mtu=1500;
  }
  retour=lcrzo_sniff_init(device, mtu, "", pcs);
  if ( retour!=LCRZO_ERR_OK )
  { _exit(retour);
  }
    
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
/*processfermesniff*/
int lcrzo_priv_pfs(int retour, lcrzo_sniff *pcs);
int lcrzo_priv_pfs(int retour, lcrzo_sniff *pcs)
{ if ( retour!=LCRZO_ERR_OK )
  { lcrzo_sniff_close(pcs);
    _exit(retour);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
/*processexit*/
int lcrzo_priv_pe(int retour);
int lcrzo_priv_pe(int retour)
{ if ( retour!=LCRZO_ERR_OK )
  { _exit(retour);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_majinfossock(const lcrzo_device here_device,
			     const lcrzo_etha here_etha,
			     lcrzo_ipl here_ipl,
			     lcrzo_uint16 here_port,
			     const lcrzo_etha there_etha,
			     lcrzo_ipl there_ipl,
			     lcrzo_uint16 there_port,
			     lcrzo_sock *psock);
int lcrzo_priv_majinfossock(const lcrzo_device here_device,
			     const lcrzo_etha here_etha,
			     lcrzo_ipl here_ipl,
			     lcrzo_uint16 here_port,
			     const lcrzo_etha there_etha,
			     lcrzo_ipl there_ipl,
			     lcrzo_uint16 there_port,
			     lcrzo_sock *psock)
{
  lcrzo_er(lcrzo_device_init(here_device, psock->here_device));
  memcpy(psock->here_etha, here_etha, LCRZO_ETHA_MAXBYTES);
  psock->here_ipl=here_ipl;
  psock->here_port=here_port;
  memcpy(psock->there_etha, there_etha, LCRZO_ETHA_MAXBYTES);
  psock->there_ipl=there_ipl;
  psock->there_port=there_port;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
/*sendpaqipoptudp*/
int lcrzo_priv_spiou(lcrzo_sock *psock,
		      const lcrzo_device device,
		      const lcrzo_etha ethasrc,
		      const lcrzo_etha ethadst,
		      lcrzo_ipl iplsrc,
		      lcrzo_ipl ipldst,
		      lcrzo_uint16 portsrc,
		      lcrzo_uint16 portdst,
		      const lcrzo_uint8 *pdataudp,  
		      lcrzo_uint16 nboctdataudp);
int lcrzo_priv_spiou(lcrzo_sock *psock,
		      const lcrzo_device device,
		      const lcrzo_etha ethasrc,
		      const lcrzo_etha ethadst,
		      lcrzo_ipl iplsrc,
		      lcrzo_ipl ipldst,
		      lcrzo_uint16 portsrc,
		      lcrzo_uint16 portdst,
		      const lcrzo_uint8 *pdataudp,  
		      lcrzo_uint16 nboctdataudp)
{ lcrzo_hdrleth hdrleth;
  lcrzo_hdrlip  hdrlip;
  lcrzo_hdrludp hdrludp;
  int retour;
  lcrzo_ipopt ipopt;
  lcrzo_int32 ipoptsize;
  lcrzo_spoof spoof;

  /*initialisation de l'entete ETH*/
  lcrzo_er(lcrzo_hdrleth_initdefault(&hdrleth));
  memcpy(hdrleth.src, ethasrc, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrleth.dst, ethadst, LCRZO_ETHA_MAXBYTES);
  /*initialisation de l'entete IP*/
  lcrzo_er(lcrzo_hdrlip_initdefault(&hdrlip));
  hdrlip.saddr=iplsrc;
  hdrlip.daddr=ipldst;
  /*initialisation de l'entete UDP*/
  lcrzo_er(lcrzo_hdrludp_initdefault(&hdrludp));
  hdrludp.sport=portsrc;
  hdrludp.dport=portdst;

  /*mise a jour eventuelle des options IP*/
  retour=lcrzo_ipc_read_data(psock->cioptions, 0, LCRZO_IPOPT_MAXBYTES,
			     ipopt, &ipoptsize);
  while (retour==LCRZO_ERR_OK)
  { if (ipoptsize+psock->ipoptbytenum>LCRZO_IPOPT_MAXBYTES)
      return(LCRZO_ERR_SPOPTMAX10);
    memcpy(psock->ipopt+psock->ipoptbytenum, ipopt, ipoptsize);
    psock->ipoptbytenum=(lcrzo_uint8)(psock->ipoptbytenum+ipoptsize);
    retour=lcrzo_ipc_read_data(psock->cioptions, 0, LCRZO_IPOPT_MAXBYTES,
			       ipopt, &ipoptsize);
  }

  /*envoi d'un packet avec donnees*/
  lcrzo_er(lcrzo_spoof_init(&spoof));
  lcrzo_er(lcrzo_spoof_ethipoptudpdata(&spoof, device, hdrleth, hdrlip,
				       psock->ipopt, psock->ipoptbytenum,
				       hdrludp, pdataudp, nboctdataudp));
  lcrzo_er(lcrzo_spoof_close(&spoof));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
/*sendpaqipopttcp*/
int lcrzo_priv_spiot(lcrzo_sock *psock,
		      const lcrzo_device device,
		      const lcrzo_etha ethasrc,
		      const lcrzo_etha ethadst,
		      lcrzo_ipl iplsrc,
		      lcrzo_ipl ipldst,
		      lcrzo_uint16 portsrc,
		      lcrzo_uint16 portdst,
		      lcrzo_uint32 seqnum,
		      lcrzo_uint32 acknum,
		      lcrzo_bool bitsyn, lcrzo_bool bitack, lcrzo_bool bitrst,
		      const lcrzo_uint8 *pdatatcp,  
		      lcrzo_uint16 nboctdatatcp);
int lcrzo_priv_spiot(lcrzo_sock *psock,
		      const lcrzo_device device,
		      const lcrzo_etha ethasrc,
		      const lcrzo_etha ethadst,
		      lcrzo_ipl iplsrc,
		      lcrzo_ipl ipldst,
		      lcrzo_uint16 portsrc,
		      lcrzo_uint16 portdst,
		      lcrzo_uint32 seqnum,
		      lcrzo_uint32 acknum,
		      lcrzo_bool bitsyn, lcrzo_bool bitack, lcrzo_bool bitrst,
		      const lcrzo_uint8 *pdatatcp,  
		      lcrzo_uint16 nboctdatatcp)
{ lcrzo_hdrleth hdrleth;
  lcrzo_hdrlip  hdrlip;
  lcrzo_hdrltcp hdrltcp;
  int retour;
  lcrzo_ipopt ipopt;
  lcrzo_int32 ipoptsize;
  lcrzo_spoof spoof;

  /*initialisation de l'entete ETH*/
  lcrzo_er(lcrzo_hdrleth_initdefault(&hdrleth));
  memcpy(hdrleth.src, ethasrc, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrleth.dst, ethadst, LCRZO_ETHA_MAXBYTES);
  /*initialisation de l'entete IP*/
  lcrzo_er(lcrzo_hdrlip_initdefault(&hdrlip));
  hdrlip.saddr=iplsrc;
  hdrlip.daddr=ipldst;
  /*initialisation de l'entete TCP*/
  lcrzo_er(lcrzo_hdrltcp_initdefault(&hdrltcp));
  hdrltcp.sport=portsrc;
  hdrltcp.dport=portdst;
  hdrltcp.seqnum=seqnum;
  hdrltcp.acknum=acknum;
  hdrltcp.syn=(lcrzo_uint8)bitsyn;
  hdrltcp.ack=(lcrzo_uint8)bitack;
  hdrltcp.rst=(lcrzo_uint8)bitrst;

  /*mise a jour eventuelle des options IP*/
  retour=lcrzo_ipc_read_data(psock->cioptions, 0, LCRZO_IPOPT_MAXBYTES, 
			     ipopt, &ipoptsize);
  while (retour==LCRZO_ERR_OK)
  { if (ipoptsize+psock->ipoptbytenum>LCRZO_IPOPT_MAXBYTES)
      return(LCRZO_ERR_SPOPTMAX10);
    memcpy(psock->ipopt+psock->ipoptbytenum, ipopt, ipoptsize);
    psock->ipoptbytenum=(lcrzo_uint8)(psock->ipoptbytenum+ipoptsize);
    retour=lcrzo_ipc_read_data(psock->cioptions, 0, LCRZO_IPOPT_MAXBYTES,
			       ipopt, &ipoptsize);
  }

  /*envoi d'un packet avec donnees*/
  lcrzo_er(lcrzo_spoof_init(&spoof));
  lcrzo_er(lcrzo_spoof_ethipopttcpoptdata(&spoof, device, hdrleth,
					  hdrlip, psock->ipopt,
					  psock->ipoptbytenum,
					  hdrltcp, NULL, 0,
					  pdatatcp, nboctdatatcp));
  lcrzo_er(lcrzo_spoof_close(&spoof));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_myrecvfrom(int fd,
			     lcrzo_int32 maxdatasizetoread,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize,
			     struct sockaddr_in *psai);
int lcrzo_priv_fd_myrecvfrom(int fd,
			     lcrzo_int32 maxdatasizetoread,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize,
			     struct sockaddr_in *psai)
{ lcrzo_int32 nboct;
  lcrzo_data ptr;
  int sailength;

  if (maxdatasizetoread<0) return(LCRZO_ERR_PATOOLOW);

  if (maxdatasizetoread==0)
  { lcrzo_er(lcrzo_data_alloc(0, pdataout));
    if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  /*we read maxi 500 bytes*/
  if (maxdatasizetoread>500) maxdatasizetoread=500;

  /*read*/
  lcrzo_er(lcrzo_data_alloc(maxdatasizetoread, &ptr));
  sailength=sizeof(struct sockaddr_in);
  nboct=recvfrom(fd, ptr, maxdatasizetoread, 0, (struct sockaddr *)psai,
		 &sailength);
  /*une erreur s'est produite*/
  if (nboct==-1)
  { if (errno==EAGAIN)
    { /*annule l'erreur, c'est simplement que rien n'etait disponible*/
      errno=0;
      lcrzo_data_free(ptr);
      return(LCRZO_ERR_OKTEMPDATAEND);
    }
    else
    { /*c'etait une erreur fatale*/
      lcrzo_data_free(ptr);
      return(LCRZO_ERR_FURECVFROM);
    }
  }

  /*on a lu quelque chose*/
  if (pdataoutsize!=NULL) *pdataoutsize=nboct;

  /*if we read 0, we are at end of file*/
  if (nboct==0)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_OKDATAEND);
  }

  if (pdataout!=NULL)
  { lcrzo_er(lcrzo_data_realloc(nboct, &ptr));
    *pdataout=ptr; 
  }
  else
  { lcrzo_data_free(ptr);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_recvfrom(int fd,
			   lcrzo_bool beblocking,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize,
			   struct sockaddr_in *psai);
int lcrzo_priv_fd_recvfrom(int fd,
			   lcrzo_bool beblocking,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize,
			   struct sockaddr_in *psai)
{ lcrzo_data data;
  lcrzo_int32 datasize;
  int ret;
  
  if (dataoutmaxsize<0) return(LCRZO_ERR_PATOOLOW);
  if (dataoutmaxsize==0)
  { if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  lcrzo_er(lcrzo_fd_block_set(fd, beblocking));

  ret=lcrzo_priv_fd_myrecvfrom(fd, dataoutmaxsize, &data, &datasize, psai);
  if (ret==LCRZO_ERR_OK)
  { if (dataout!=NULL) memcpy(dataout, data, datasize);
    lcrzo_data_free(data);
    if (pdataoutsize!=NULL) *pdataoutsize=datasize;
  }

  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_recvfromm(int fd,
			    lcrzo_bool beblocking,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize,
			    struct sockaddr_in *psai);
int lcrzo_priv_fd_recvfromm(int fd,
			    lcrzo_bool beblocking,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize,
			    struct sockaddr_in *psai)
{ lcrzo_er(lcrzo_fd_block_set(fd, beblocking));
  return(lcrzo_priv_fd_myrecvfrom(fd, 0xFFFF, pdataout, pdataoutsize, psai));
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_recvfromm_fixed(int fd,
				  lcrzo_bool beblocking,
				  lcrzo_int32 datasizetoread,
				  lcrzo_data *pdataout,
				  struct sockaddr_in *psai);
int lcrzo_priv_fd_recvfromm_fixed(int fd,
				  lcrzo_bool beblocking,
				  lcrzo_int32 datasizetoread,
				  lcrzo_data *pdataout,
				  struct sockaddr_in *psai)
{ lcrzo_data ptr, tmpdata;
  lcrzo_int32 tmpdatasize, totalread, tmpdatasizetoread;
  int ret;


  if (datasizetoread<0) return(LCRZO_ERR_PATOOLOW);

  if (datasizetoread==0)
  { lcrzo_er(lcrzo_data_alloc(0, pdataout));
    return(LCRZO_ERR_OK);
  }
  lcrzo_er(lcrzo_fd_block_set(fd, beblocking));

  lcrzo_er(lcrzo_data_alloc(datasizetoread, &ptr));

  totalread=0;  
  while(1)
  { tmpdatasizetoread=datasizetoread-totalread;
    ret=lcrzo_priv_fd_myrecvfrom(fd, tmpdatasizetoread, 
				 &tmpdata, &tmpdatasize, psai);
    if (ret==LCRZO_ERR_OK)
    { /*concatene*/
      memcpy(ptr+totalread, tmpdata, tmpdatasize);
      lcrzo_data_free(tmpdata);
      totalread+=tmpdatasize;
      /*eventually quit*/
      if ( totalread==datasizetoread )
      { if (pdataout!=NULL) *pdataout=ptr;
        else lcrzo_data_free(ptr);
        return(LCRZO_ERR_OK);
      }
    }
    else if (ret==LCRZO_ERR_OKDATAEND)
    { /*no more data, so it's an error*/
      lcrzo_data_free(ptr);
      return(ret);
    }
    else if (ret==LCRZO_ERR_OKTEMPDATAEND)
    { if (!totalread)
      { /*we didn't read anathing, so we can quit now*/
	lcrzo_data_free(ptr);
        return(ret);
      }
    }
    else
    { lcrzo_data_free(ptr);
      return(ret);
    }
  }

  /*on ne doit jamais atteindre ce return*/
  return(LCRZO_ERR_IEINTERNALERROR);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_recvfrom_fixed(int fd,
				 lcrzo_bool beblocking,
				 lcrzo_int32 datasizetoread,
				 lcrzo_data dataout,
				 struct sockaddr_in *psai);
int lcrzo_priv_fd_recvfrom_fixed(int fd,
				 lcrzo_bool beblocking,
				 lcrzo_int32 datasizetoread,
				 lcrzo_data dataout,
				 struct sockaddr_in *psai)
{ lcrzo_data data;

  if (datasizetoread<0) return(LCRZO_ERR_PATOOLOW);
  if (datasizetoread==0) return(LCRZO_ERR_OK);
  /*get data*/
  lcrzo_er(lcrzo_priv_fd_recvfromm_fixed(fd, beblocking, datasizetoread, 
					 &data, psai));
  /*copy it*/
  memcpy(dataout, data, datasizetoread);
  /*free*/
  lcrzo_er(lcrzo_data_free(data));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_recvfromm_line(int fd,
				 lcrzo_bool beblocking,
				 lcrzo_data *pdataout,
				 lcrzo_int32 *pdataoutsize,
				 struct sockaddr_in *psai);
int lcrzo_priv_fd_recvfromm_line(int fd,
				 lcrzo_bool beblocking,
				 lcrzo_data *pdataout,
				 lcrzo_int32 *pdataoutsize,
				 struct sockaddr_in *psai)
{ lcrzo_data data, tmpdata;
  lcrzo_int32 datasize, tmpdatasize;
  int ret, c;
  
  lcrzo_er(lcrzo_fd_block_set(fd, beblocking));
  datasize=0;
  lcrzo_er(lcrzo_data_alloc(datasize, &data));
  while (1)
  { ret=lcrzo_priv_fd_myrecvfrom(fd, 1, &tmpdata, &tmpdatasize, psai);
    if (ret==LCRZO_ERR_OK)
    { c=tmpdata[0];
      lcrzo_er(lcrzo_data_free(tmpdata));
      /*ignore 0D*/
      if ( c == 0x0D ) {}
      else if ( c == 0x0A ) { break; }
      else
      { lcrzo_er(lcrzo_data_appendm_char(c, 1, datasize, &data, &datasize));
      }
    }
    else if (ret==LCRZO_ERR_OKTEMPDATAEND)
    { /*exit if we read nothing*/
      if (!datasize)
      { lcrzo_data_free(data);
        return(ret);
      }
    }
    else if (ret==LCRZO_ERR_OKDATAEND)
    { /*exit if we read nothing*/
      if (!datasize)
      { lcrzo_data_free(data);
        return(ret);
      }
      /* end of file is like end of line */
      break;
    }
    else
    { lcrzo_data_free(data);
      return(ret);
    }
  }

  /*on a lu quelque chose*/
  if (pdataout!=NULL)
  { *pdataout=data; 
  }
  else
  { lcrzo_data_free(data);
  }
  if (pdataoutsize!=NULL) *pdataoutsize=datasize;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_recvfrom_line(int fd,
				lcrzo_bool beblocking,
				lcrzo_int32 dataoutmaxsize,
				lcrzo_data dataout,
				lcrzo_int32 *pdataoutsize,
				struct sockaddr_in *psai);
int lcrzo_priv_fd_recvfrom_line(int fd,
				lcrzo_bool beblocking,
				lcrzo_int32 dataoutmaxsize,
				lcrzo_data dataout,
				lcrzo_int32 *pdataoutsize,
				struct sockaddr_in *psai)
{ lcrzo_data data;
  lcrzo_int32 datasize;
  int ret;
  
  if (dataoutmaxsize<0) return(LCRZO_ERR_PATOOLOW);
  if (dataoutmaxsize==0)
  { if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  lcrzo_er(lcrzo_priv_fd_recvfromm_line(fd, beblocking, &data, &datasize, 
					psai));
  ret=lcrzo_data_init_data(data, datasize,
			   dataoutmaxsize, dataout, pdataoutsize);
  lcrzo_data_free(data);
  return(ret);
}

/* Fin zone de fonctions privees */
/***************************************************************/

/*-------------------------------------------------------------*/
int lcrzo_sock_udpcli_real(lcrzo_ipl    ipl, 
			  lcrzo_uint16 port,
			  lcrzo_sock   *psock)
{ return(lcrzo_sock_udpcli_real2(0, ipl, 0, port, NULL, 0, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpcli_virt(const lcrzo_device device,
			   const lcrzo_etha ethaclient,
			   const lcrzo_etha ethaserveur,
			   const lcrzo_ipl iplclient,
			   const lcrzo_ipl iplserveur, 
			   lcrzo_uint16 portclient,
			   lcrzo_uint16 portserveur,
			   lcrzo_sock *psock)
{ return(lcrzo_sock_udpcli_virt2(device, ethaclient, ethaserveur,
				 iplclient, iplserveur,
				 portclient, portserveur,
				 NULL, 0,
				 psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpcli_real2(lcrzo_ipl    iplclient,
			    lcrzo_ipl    iplserveur, 
			    lcrzo_uint16 portclient,
			    lcrzo_uint16 portserveur,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock   *psock)
{ struct sockaddr_in cli;
  int retour, sock;

  /*parameters verification*/
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*creation de la socket*/
  sock=socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (sock<0) return(LCRZO_ERR_FUSOCKET);

  /*ajout eventuel des options IP*/
  if (ipoptbytenum)
  { retour = setsockopt(sock, IPPROTO_IP, IP_OPTIONS, ipopt, ipoptbytenum);
    if (retour)
      return(LCRZO_ERR_FUSETSOCKOPT);
  }

  /*le client specifie des infos locales*/
  if ( iplclient || portclient )
  { memset(&cli, 0, sizeof(cli));
    cli.sin_family = AF_INET;
    cli.sin_addr.s_addr=lcrzo_htonl(iplclient);
    cli.sin_port = lcrzo_htons(portclient);
    retour=bind(sock, (struct sockaddr *)&cli, sizeof(cli));
    if (retour<0) return(LCRZO_ERR_FUBIND);
  }

  /*connexion sur le serveur*/
  memset(&cli, 0, sizeof(cli));
  cli.sin_family = AF_INET;
  cli.sin_addr.s_addr=lcrzo_htonl(iplserveur);
  cli.sin_port = lcrzo_htons(portserveur);
  retour=connect(sock, (struct sockaddr *)&cli, sizeof(cli));
  if (retour<0) return(LCRZO_ERR_FUCONNECT);

  /*if the user only want to test, its ok, so we close*/
  if (psock==NULL)
  { retour=close(sock);
    if ( retour == -1 ) return(LCRZO_ERR_FUCLOSE);
    return(LCRZO_ERR_OK);
  }

  /*remplissage des caracteristiques de la socket*/
  psock->type=LCRZO_PRIV_TYPESOCK_REALCLIUDP;
  psock->socknum=sock;
  psock->beblocking=1;

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_device_initdefault(psock->here_device));
  lcrzo_er(lcrzo_etha_initdefault(psock->here_etha));
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  psock->there_ipl=iplserveur;
  psock->there_port=portserveur;
  psock->here_ipl=iplclient;
  psock->here_port=portclient;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
#define LCRZO_PRIV_READSIZE 400
int lcrzo_sock_udpcli_virt2(const lcrzo_device device,
			    const lcrzo_etha ethaclient,
			    const lcrzo_etha ethaserveur,
			    const lcrzo_ipl iplclient,
			    const lcrzo_ipl iplserveur, 
			    lcrzo_uint16 portclient,
			    lcrzo_uint16 portserveur,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock *psock)
{ int retour;
  lcrzo_uint16 nboctdata;
  lcrzo_sniff cs;
  lcrzo_data data;
  lcrzo_hdrleth hdrlethrecu;
  lcrzo_hdrlip hdrliprecu;
  lcrzo_hdrludp hdrludprecu;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize, arrsize;
  lcrzo_uint8 arr[LCRZO_PRIV_READSIZE];

  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaserveur==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaclient==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  /*initialisations*/
  psock->type=LCRZO_PRIV_TYPESOCK_VIRTCLIUDP;
  lcrzo_er(lcrzo_priv_initlesci(psock));
  lcrzo_er(lcrzo_priv_ajouteipopt(psock, ipopt, ipoptbytenum));

  /*processus de sniff*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidsniff)));
  if (psock->pidsniff==0) /*on est dans le fils*/
  { /*initialisation du sniff*/
    lcrzo_priv_processinitsniff(device, &cs);
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_IPREAS,
			     &sniffdata, &sniffdatasize);
    while(retour==LCRZO_ERR_OK)
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*repond aux requetes concernant la machine virtuelle*/
      if (lcrzo_global.cliser_virt_answeralive)
      { lcrzo_virtual_answer_alive(sniffdata, sniffdatasize,
				   device, ethaclient, iplclient);
      }
      /*fourni les packets au processus pere*/
      retour=lcrzo_packet_decodem_ethipoptudpdata(sniffdata, sniffdatasize,
						  &hdrlethrecu, &hdrliprecu,
						  NULL, NULL, &hdrludprecu,
						  &data, &nboctdata);
      lcrzo_data_free(sniffdata);
      if (retour==LCRZO_ERR_OK)
      { /*ne fournit que ceux correspondant a la communication en cours*/
        if ( lcrzo_priv_hdrleth_equal_addr(hdrlethrecu, ethaserveur,
					   ethaclient) &&
	     lcrzo_priv_hdrlip_equal_addr(hdrliprecu,
					  iplserveur, iplclient) &&
	     lcrzo_priv_hdrludp_equal_port(hdrludprecu,
					   portserveur, portclient) )
	{ lcrzo_priv_pfs(lcrzo_ipc_write_flow(psock->cisniff, data, nboctdata),
			  &cs);
	}
	lcrzo_data_free(data);
      }
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_IPREAS, 
			       &sniffdata, &sniffdatasize);
    }
    lcrzo_priv_pfs(4, &cs);
  }

  /*processus de spoof*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidspoof)));
  if (psock->pidspoof==0) /*on est dans le fils*/
  { while(1)
    { /*on envoie chaque bloc du flot*/
      lcrzo_priv_pe(lcrzo_ipc_read_flow(psock->cispoof, 1, LCRZO_PRIV_READSIZE,
					arr, &arrsize));
      lcrzo_priv_pe(lcrzo_priv_spiou(psock, device, ethaclient, ethaserveur,
				     iplclient, iplserveur, portclient, 
				     portserveur, arr, (lcrzo_uint16)arrsize));
    }
    /*jamais atteint*/
  }

  /*remplissage des informations connues*/
  lcrzo_er(lcrzo_priv_majinfossock(device, ethaclient, iplclient, portclient,
				   ethaserveur, iplserveur, portserveur,
				   psock));
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_tcpcli_real(lcrzo_ipl    ipl,
			  lcrzo_uint16 port, 
			  lcrzo_sock   *psock)
{ return(lcrzo_sock_tcpcli_real2(0, ipl, 0, port, NULL, 0, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpcli_virt(const lcrzo_device device,
			   const lcrzo_etha ethaclient,
			   const lcrzo_etha ethaserveur,
			   const lcrzo_ipl iplclient,
			   const lcrzo_ipl iplserveur, 
			   lcrzo_uint16 portclient,
			   lcrzo_uint16 portserveur,
			   lcrzo_sock *psock)
{ return(lcrzo_sock_tcpcli_virt2(device, ethaclient, ethaserveur,
				 iplclient, iplserveur, 
				 portclient, portserveur,
				 NULL, 0,
				 psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpcli_real2(lcrzo_ipl    iplclient,
			    lcrzo_ipl    iplserveur, 
			    lcrzo_uint16 portclient,
			    lcrzo_uint16 portserveur,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock   *psock)
{ struct sockaddr_in cli;
  int retour, sock;

  /*parameters verification*/
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*creation de la socket*/
  sock=socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
  if (sock<0) return(LCRZO_ERR_FUSOCKET);

  /*ajout eventuel des options IP*/
  if (ipoptbytenum)
  { retour = setsockopt(sock, IPPROTO_IP, IP_OPTIONS, ipopt, ipoptbytenum);
    if (retour)
      return(LCRZO_ERR_FUSETSOCKOPT);
  }

  /*le client specifie des infos locales*/
  if ( iplclient || portclient )
  { memset(&cli, 0, sizeof(cli));
    cli.sin_family = AF_INET;
    cli.sin_addr.s_addr=lcrzo_htonl(iplclient);
    cli.sin_port = lcrzo_htons(portclient);
    retour=bind(sock, (struct sockaddr *)&cli, sizeof(cli));
    if (retour<0) return(LCRZO_ERR_FUBIND);
  }

  /*connexion sur le serveur*/
  memset(&cli, 0, sizeof(cli));
  cli.sin_family = AF_INET;
  cli.sin_addr.s_addr=lcrzo_htonl(iplserveur);
  cli.sin_port = lcrzo_htons(portserveur);
  retour=connect(sock, (struct sockaddr *)&cli, sizeof(cli));
  if (retour<0) return(LCRZO_ERR_FUCONNECT);

  /*if the user only want to test, its ok, so we close*/
  if (psock==NULL)
  { retour=close(sock);
    if ( retour == -1 ) return(LCRZO_ERR_FUCLOSE);
    return(LCRZO_ERR_OK);
  }

  /*remplissage des caracteristiques de la socket*/
  psock->type=LCRZO_PRIV_TYPESOCK_REALCLITCP;
  psock->socknum=sock;
  psock->beblocking=1;

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_device_initdefault(psock->here_device));
  lcrzo_er(lcrzo_etha_initdefault(psock->here_etha));
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  psock->there_ipl=iplserveur;
  psock->there_port=portserveur;
  psock->here_ipl=iplclient;
  psock->here_port=portclient;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpcli_virt2(const lcrzo_device device,
			    const lcrzo_etha ethaclient,
			    const lcrzo_etha ethaserveur,
			    const lcrzo_ipl iplclient,
			    const lcrzo_ipl iplserveur, 
			    lcrzo_uint16 portclient,
			    lcrzo_uint16 portserveur,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock *psock)
{ int retour; 
  lcrzo_uint32 isn, dernieroctetrecu;
  lcrzo_uint16 nboctdata;
  lcrzo_sniff cs;
  lcrzo_data data;
  lcrzo_hdrleth hdrlethrecu;
  lcrzo_hdrlip hdrliprecu;
  lcrzo_hdrltcp hdrltcprecu;
  lcrzo_bool premierpacket;
  lcrzo_uint8 bitreset;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize, arrsize;
  lcrzo_uint8 arr[LCRZO_PRIV_READSIZE];

  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaserveur==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaclient==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  /*initialisations*/
  psock->type=LCRZO_PRIV_TYPESOCK_VIRTCLITCP;
  lcrzo_er(lcrzo_priv_initlesci(psock));
  lcrzo_er(lcrzo_priv_ajouteipopt(psock, ipopt, ipoptbytenum));
  lcrzo_er(lcrzo_uint32_rand(0, 0xFFFFFFFFu, &isn));

  /*processus de sniff*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidsniff)));
  if (psock->pidsniff==0) /*on est dans le fils*/
  { /*initialisation du sniff*/
    lcrzo_priv_processinitsniff(device, &cs);
    /*envoi du packet syn*/
    premierpacket=1;
    lcrzo_priv_pfs(lcrzo_priv_spiot(psock, device, ethaclient, ethaserveur, 
				      iplclient, iplserveur, portclient,
				      portserveur, isn, 0, 1,0,0,
				      NULL, 0), &cs);
    /*sniff des packets recus*/
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_TCPREOR,
			     &sniffdata, &sniffdatasize);
    while(retour==LCRZO_ERR_OK)
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*repond aux requetes concernant la machine virtuelle*/
      if (lcrzo_global.cliser_virt_answeralive)
      { lcrzo_virtual_answer_alive(sniffdata, sniffdatasize,
				   device, ethaclient, iplclient);
      }
      /*fourni les packets au processus pere*/
      retour=lcrzo_packet_decodem_ethipopttcpoptdata(sniffdata, sniffdatasize,
						     &hdrlethrecu, 
						     &hdrliprecu, NULL, NULL,
						     &hdrltcprecu, NULL, NULL, 
						     &data, &nboctdata);
      lcrzo_data_free(sniffdata);
      if (retour==LCRZO_ERR_OK)
      { /*resette les requetes ident*/
        if ( portclient != 113 ) /*sauf si cree un client sur ce port*/
        { if ( lcrzo_priv_hdrleth_equal_addr(hdrlethrecu,
					     ethaserveur, ethaclient) &&
	       lcrzo_priv_hdrlip_equal_addr(hdrliprecu,
					    iplserveur, iplclient) &&
               hdrltcprecu.dport==113 &&
               hdrltcprecu.syn==1 )
          { lcrzo_priv_spiot(psock, device, ethaclient, ethaserveur,iplclient,
			      iplserveur, portclient, hdrltcprecu.sport,
			      0, hdrltcprecu.seqnum+1, 0,1,1,
			      NULL, 0);
          }
	}
	/*ne fournit que ceux correspondant a la communication en cours*/
        if ( lcrzo_priv_hdrleth_equal_addr(hdrlethrecu, ethaserveur,
					   ethaclient) &&
	     lcrzo_priv_hdrlip_equal_addr(hdrliprecu, iplserveur, iplclient) &&
	     lcrzo_priv_hdrltcp_equal_port(hdrltcprecu,
					   portserveur, portclient) &&
             hdrltcprecu.ack==1 )
	{ /*si c'est le premier packet, on termine le handshake*/
          if (premierpacket)
          { dernieroctetrecu=hdrltcprecu.seqnum+1;
            premierpacket=0;
          }
	  else /*on envoie les donnees*/
	  { dernieroctetrecu=hdrltcprecu.seqnum+nboctdata;
  	    lcrzo_priv_pfs(lcrzo_ipc_write_flow(psock->cisniff, 
						data, nboctdata), &cs);
	  }
	  /*envoie le packet ack*/
	  lcrzo_priv_spiot(psock, device, ethaclient, ethaserveur, iplclient,
			    iplserveur, portclient, hdrltcprecu.sport,
			    hdrltcprecu.acknum, dernieroctetrecu,
			    0,1,0, NULL, 0);
	  /*informe par le biais du canal de controle*/
	  lcrzo_priv_pfs(lcrzo_ipc_write_uint32(psock->cicontrole1,
						dernieroctetrecu), &cs);
	}
	lcrzo_data_free(data);
      }
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_TCPREOR,
			      &sniffdata, &sniffdatasize);
    }
    lcrzo_priv_pfs(4, &cs);
  }

  /*attend que le handshake soit termine en bloquant jusqu'a recevoir
    des donnees dans le tunnel de controle*/
  lcrzo_er(lcrzo_ipc_read_uint32(psock->cicontrole1, 1, &dernieroctetrecu));

  /*processus de spoof*/
  isn+=1; /*car l'isn est incremente dans le handshake*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidspoof)));
  if (psock->pidspoof==0) /*on est dans le fils*/
  { while(1)
    { /*on regarde le numero de acknum (prend le dernier disponible)*/
      retour=lcrzo_ipc_read_uint32(psock->cicontrole1, 0, &dernieroctetrecu);
      while ( retour==LCRZO_ERR_OK )
      { retour=lcrzo_ipc_read_uint32(psock->cicontrole1, 0, &dernieroctetrecu);
      }
      /*on regarde les donnees a envoyer (par bloc de 400 octets)*/
      retour=lcrzo_ipc_read_flow(psock->cispoof, 0, LCRZO_PRIV_READSIZE,
				 arr, &arrsize);
      if (retour==LCRZO_ERR_OK)
      { lcrzo_priv_pe(lcrzo_priv_spiot(psock, device, ethaclient,
					 ethaserveur,
					 iplclient, iplserveur, portclient,
					 portserveur, isn, dernieroctetrecu,
					 0,1,0, arr, (lcrzo_uint16)arrsize));
        isn+=arrsize; /*si isn>0xffffffff passe automatiquement a zero*/
      }
      else
      { /*on regarde si un reset est demande (attend d'avoir tout envoye)*/
	bitreset=0;
	lcrzo_ipc_read_uint8(psock->cicontrole2, 0, &bitreset);
	if (bitreset)
        { lcrzo_priv_pe(lcrzo_priv_spiot(psock, device, ethaclient,
					   ethaserveur,
					   iplclient, iplserveur, portclient,
					   portserveur, isn, dernieroctetrecu,
					   0,1,1, NULL, 0));
	}
      }
    }
    /*jamais atteint*/
  }

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_priv_majinfossock(device, ethaclient, iplclient, portclient,
                                    ethaserveur, iplserveur, portserveur,
                                    psock));
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_udpser_real(lcrzo_uint16 port,
			  lcrzo_sock *psock)
{ return(lcrzo_sock_udpser_real2(0, port, NULL, 0, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpser_virt(const lcrzo_device device,
			  const lcrzo_etha ethaserveur,
			  const lcrzo_ipl iplserveur, 
			  lcrzo_uint16 portserveur,
			  lcrzo_sock *psock)
{ return(lcrzo_sock_udpser_virt2(device, ethaserveur,
				 iplserveur, portserveur,
				 NULL, 0, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpser_real2(lcrzo_ipl ipl,
			    lcrzo_uint16 port,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock *psock)
{ struct sockaddr_in sain;
  int retour, sock;

  /*parameters verification*/
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*creation de la socket*/
  sock=socket(AF_INET, SOCK_DGRAM, 0);
  if (sock<0) return(LCRZO_ERR_FUSOCKET);

  /*ajout eventuel des options IP*/
  if (ipoptbytenum)
  { retour = setsockopt(sock, IPPROTO_IP, IP_OPTIONS, ipopt, ipoptbytenum);
    if (retour)
      return(LCRZO_ERR_FUSETSOCKOPT);
  }

  /*ecoute sur le port*/
  memset(&sain, 0, sizeof(sain));
  sain.sin_family = AF_INET;
  if ( ipl==0 )
    sain.sin_addr.s_addr=lcrzo_htonl(INADDR_ANY);
  else
    sain.sin_addr.s_addr=lcrzo_htonl(ipl);
  sain.sin_port = lcrzo_htons(port);
  retour=bind(sock, (struct sockaddr *)&sain, sizeof(sain));
  if (retour<0) return(LCRZO_ERR_FUBIND);

  /*if the user only want to test, its ok, so we close*/
  if (psock==NULL)
  { retour=close(sock);
    if ( retour == -1 ) return(LCRZO_ERR_FUCLOSE);
    return(LCRZO_ERR_OK);
  }

  /*remplissage des caracteristiques de la socket*/
  psock->type=LCRZO_PRIV_TYPESOCK_REALSERUDP;
  psock->socknum=sock;
  psock->canwrite=0;
  psock->beblocking=1;

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_device_initdefault(psock->here_device));
  lcrzo_er(lcrzo_etha_initdefault(psock->here_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->here_ipl)));
  psock->here_port=port;
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->there_ipl)));
  psock->there_port=0;
   return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpser_virt2(const lcrzo_device device,
			    const lcrzo_etha ethaserveur,
			    const lcrzo_ipl iplserveur, 
			    lcrzo_uint16 portserveur,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock *psock)
{ int retour, dejaunclient;
  lcrzo_uint16 nboctdata;
  lcrzo_sniff cs;
  lcrzo_data data;
  lcrzo_hdrleth hdrlethrecu;
  lcrzo_hdrlip hdrliprecu;
  lcrzo_hdrludp hdrludprecu;
  lcrzo_etha ethaclient;
  lcrzo_uint16 portclient;
  lcrzo_ipl iplclient;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize, arrsize;
  lcrzo_uint8 arr[LCRZO_PRIV_READSIZE];

  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaserveur==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  /*initialisations*/
  psock->type=LCRZO_PRIV_TYPESOCK_VIRTSERUDP;
  lcrzo_er(lcrzo_priv_initlesci(psock));
  lcrzo_er(lcrzo_priv_ajouteipopt(psock, ipopt, ipoptbytenum));

  /*processus de sniff*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidsniff)));
  if (psock->pidsniff==0) /*on est dans le fils*/
  { /*initialisation du sniff*/
    lcrzo_priv_processinitsniff(device, &cs);
    dejaunclient=0;
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_IPREAS,
			     &sniffdata, &sniffdatasize);
    while(retour==LCRZO_ERR_OK)
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*repond aux requetes concernant la machine virtuelle*/
      if (lcrzo_global.cliser_virt_answeralive)
      { lcrzo_virtual_answer_alive(sniffdata, sniffdatasize,
				   device, ethaserveur, iplserveur);
      }
      /*fourni les packets au processus pere*/
      retour=lcrzo_packet_decodem_ethipoptudpdata(sniffdata, sniffdatasize,
						  &hdrlethrecu, &hdrliprecu,
						  NULL, NULL, &hdrludprecu,
						  &data, &nboctdata);
      lcrzo_data_free(sniffdata);
      if (retour==LCRZO_ERR_OK)
      { /*ne fournit que ceux correspondant a la communication en cours*/
        if ( lcrzo_etha_equal(hdrlethrecu.dst, ethaserveur) &&
	     hdrliprecu.daddr==iplserveur &&
	     hdrludprecu.dport==portserveur )
	{ if (!dejaunclient)
	  { /*le premier client a se connecter sera le seul reconnu*/
	    /*on sauvegarde ces informations*/
	    memcpy(ethaclient, hdrlethrecu.src, LCRZO_ETHA_MAXBYTES);
	    iplclient=hdrliprecu.saddr;
	    portclient=hdrludprecu.sport;
	    dejaunclient=1;
	    /*on l'ecrit deux fois car il y a deux lecteurs*/
            lcrzo_priv_pfs(lcrzo_ipc_write_ethaiplport(psock->cicontrole1,
						       ethaclient, iplclient,
						       portclient), &cs);
            lcrzo_priv_pfs(lcrzo_ipc_write_ethaiplport(psock->cicontrole1,
						       ethaclient, iplclient,
						       portclient), &cs);
	    /*on passe le flot*/
	    lcrzo_priv_pfs(lcrzo_ipc_write_flow(psock->cisniff, data,
						nboctdata), &cs);
	  }
	  else if ( lcrzo_etha_equal(hdrlethrecu.src, ethaclient) &&
		    hdrliprecu.saddr==iplclient &&
		    hdrludprecu.sport==portclient )
	  { /*on passe le flot*/
	    lcrzo_priv_pfs(lcrzo_ipc_write_flow(psock->cisniff, data,
						nboctdata), &cs);
	  }
	}
	lcrzo_data_free(data);
      }
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_IPREAS,
			       &sniffdata, &sniffdatasize);
    }
    lcrzo_priv_pfs(4, &cs);
  }

  /*processus de spoof*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidspoof)));
  if (psock->pidspoof==0) /*on est dans le fils*/
  { /*on attend d'abord de savoir qui est notre client*/
    lcrzo_priv_pe(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 1,
					    ethaclient, &iplclient,
					    &portclient));
    while(1)
    { /*on envoie chaque bloc du flot*/
      lcrzo_priv_pe(lcrzo_ipc_read_flow(psock->cispoof, 1, LCRZO_PRIV_READSIZE,
					arr, &arrsize));
      lcrzo_priv_pe(lcrzo_priv_spiou(psock, device, ethaserveur, ethaclient,
				     iplserveur, iplclient, portserveur, 
				     portclient, arr, (lcrzo_uint16)arrsize));
    }
    /*jamais atteint*/
  }

  /*remplissage des informations connues*/
  lcrzo_er(lcrzo_device_init(device, psock->here_device));
  memcpy(psock->here_etha, ethaserveur, LCRZO_ETHA_MAXBYTES);
  psock->here_ipl=iplserveur;
  psock->here_port=portserveur;
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->there_ipl)));
  psock->there_port=0;
  psock->canwrite=0;
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_udpmulser_real(lcrzo_uint16 port,
			      lcrzo_sock *psock)
{ return(lcrzo_sock_udpmulser_real2(0, port, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpmulser_virt(const lcrzo_device device,
			      const lcrzo_etha ethaserveur,
			      const lcrzo_ipl iplserveur, 
			      lcrzo_uint16 portserveur,
			      lcrzo_sock *psock)
{ int retour;
  lcrzo_uint16 nboctdata;
  lcrzo_sniff cs;
  lcrzo_data data;
  lcrzo_hdrleth hdrlethrecu;
  lcrzo_hdrlip hdrliprecu;
  lcrzo_hdrludp hdrludprecu;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize;

  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaserveur==NULL) return(LCRZO_ERR_PANULLPTR);

  /*initialisations*/
  psock->type=LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI;
  psock->ipoptbytenum=0;
  lcrzo_er(lcrzo_priv_initlesci(psock));

  /*processus de sniff*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidsniff)));
  if (psock->pidsniff==0) /*on est dans le fils*/
  { /*initialisation du sniff*/
    lcrzo_priv_processinitsniff(device, &cs);
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_IPREAS,
			     &sniffdata, &sniffdatasize);
    while(retour==LCRZO_ERR_OK)
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*repond aux requetes concernant la machine virtuelle*/
      if (lcrzo_global.cliser_virt_answeralive)
      { lcrzo_virtual_answer_alive(sniffdata, sniffdatasize,
				   device, ethaserveur, iplserveur);
      }
      /*fourni les packets au processus pere*/
      retour=lcrzo_packet_decodem_ethipoptudpdata(sniffdata, sniffdatasize,
						  &hdrlethrecu, &hdrliprecu,
						  NULL, NULL, &hdrludprecu,
						  &data, &nboctdata);
      lcrzo_data_free(sniffdata);
      if (retour==LCRZO_ERR_OK)
      { /*ne fournit que ceux destines au serveur*/
        if ( lcrzo_etha_equal(hdrlethrecu.dst, ethaserveur) &&
	     hdrliprecu.daddr==iplserveur &&
	     hdrludprecu.dport==portserveur )
	{ /*on passe l'information concernant le client courant*/
          lcrzo_priv_pfs(lcrzo_ipc_write_ethaiplport(psock->cicontrole1,
						     hdrlethrecu.src, 
						     hdrliprecu.saddr,
						     hdrludprecu.sport), &cs);
	  /*on passe le flot*/
	  lcrzo_priv_pfs(lcrzo_ipc_write_flow(psock->cisniff, data,
					      nboctdata), &cs);
	}
	lcrzo_data_free(data);
      }
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_IPREAS,
			       &sniffdata, &sniffdatasize);
    }
    lcrzo_priv_pfs(4, &cs);
  }

  /*ce type de serveur ne peut pas ecrire, donc pas de processus de spoof*/

  /*remplissage des informations connues*/
  lcrzo_er(lcrzo_device_init(device, psock->here_device));
  memcpy(psock->here_etha, ethaserveur, LCRZO_ETHA_MAXBYTES);
  psock->here_ipl=iplserveur;
  psock->here_port=portserveur;
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->there_ipl)));
  psock->there_port=0;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_udpmulser_real2(lcrzo_ipl ipl,
			       lcrzo_uint16 port,
			       lcrzo_sock *psock)
{ struct sockaddr_in sain;
  int retour, sock;

  /*creation de la socket*/
  sock=socket(AF_INET, SOCK_DGRAM, 0);
  if (sock<0) return(LCRZO_ERR_FUSOCKET);

  memset(&sain, 0, sizeof(sain));
  sain.sin_family = AF_INET;
  if ( ipl==0 )
    sain.sin_addr.s_addr=lcrzo_htonl(INADDR_ANY);
  else
    sain.sin_addr.s_addr=lcrzo_htonl(ipl);
  sain.sin_port = lcrzo_htons(port);
  retour=bind(sock, (struct sockaddr *)&sain, sizeof(sain));
  if (retour<0) return(LCRZO_ERR_FUBIND);

  /*if the user only want to test, its ok, so we close*/
  if (psock==NULL)
  { retour=close(sock);
    if ( retour == -1 ) return(LCRZO_ERR_FUCLOSE);
    return(LCRZO_ERR_OK);
  }

  /*remplissage des caracteristiques de la socket*/
  psock->type=LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI;
  psock->socknum=sock;
  psock->beblocking=1;

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_device_initdefault(psock->here_device));
  lcrzo_er(lcrzo_etha_initdefault(psock->here_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->here_ipl)));
  psock->here_port=port;
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->there_ipl)));
  psock->there_port=0;
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_tcpser_real(lcrzo_uint16 port,
			  lcrzo_sock *psock)
{ return(lcrzo_sock_tcpser_real2(0, port, NULL, 0, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpser_virt(const lcrzo_device device,
			  const lcrzo_etha ethaserveur,
			  const lcrzo_ipl iplserveur, 
			  lcrzo_uint16 portserveur,
			  lcrzo_sock *psock)
{ return(lcrzo_sock_tcpser_virt2(device, ethaserveur, iplserveur,
				 portserveur, NULL, 0, psock));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpser_real2(lcrzo_ipl ipl,
			    lcrzo_uint16 port,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock *psock)
{ struct sockaddr_in sain, sain2;
  int retour, soc, sizeof_sain2, un, sock;

  /*parameters verification*/
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*creation de la socket*/
  soc=socket(AF_INET, SOCK_STREAM, 0);
  if (soc<0) return(LCRZO_ERR_FUSOCKET);

  /*pour que l'adresse puisse etre immediatemment reutilisee*/
  un=1;
  retour=setsockopt(soc, SOL_SOCKET, SO_REUSEADDR, (void *)&un, sizeof(un));
  if (retour) return(LCRZO_ERR_FUSETSOCKOPT);

  /*ajout eventuel des options IP*/
  if (ipoptbytenum)
  { retour = setsockopt(soc, IPPROTO_IP, IP_OPTIONS, ipopt, ipoptbytenum);
    if (retour)
      return(LCRZO_ERR_FUSETSOCKOPT);
  }

  /*ecoute sur le port*/
  memset(&sain, 0, sizeof(sain));
  sain.sin_family = AF_INET;
  if ( ipl==0 )
    sain.sin_addr.s_addr=lcrzo_htonl(INADDR_ANY);
  else
    sain.sin_addr.s_addr=lcrzo_htonl(ipl);
  sain.sin_port = lcrzo_htons(port);
  retour=bind(soc, (struct sockaddr *)&sain, sizeof(sain));
  if (retour<0) return(LCRZO_ERR_FUBIND);

  retour=listen(soc, 1); /*maximum 1 connexion en parallele*/
  if (retour<0) return(LCRZO_ERR_FULISTEN);

  sizeof_sain2=sizeof(sain2);
  memset(&sain2, 0, sizeof_sain2);
  sock=accept(soc,  (struct sockaddr *)&sain2, &sizeof_sain2);
  if (sock<0) return(LCRZO_ERR_FUACCEPT);

  retour=close(soc);
  if (retour==-1) return(LCRZO_ERR_FUCLOSE);

  /*ajout eventuel des options IP*/
  if (ipoptbytenum)
  { retour = setsockopt(sock, IPPROTO_IP, IP_OPTIONS, ipopt, ipoptbytenum);
    if (retour)
      return(LCRZO_ERR_FUSETSOCKOPT);
  }

  /*if the user only want to test, its ok, so we close*/
  if (psock==NULL)
  { retour=close(sock);
    if ( retour == -1 ) return(LCRZO_ERR_FUCLOSE);
    return(LCRZO_ERR_OK);
  }

  /*remplissage des caracteristiques de la socket*/
  psock->type=LCRZO_PRIV_TYPESOCK_REALSERTCP;
  psock->socknum=sock;
  psock->beblocking=1;

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_device_initdefault(psock->here_device));
  lcrzo_er(lcrzo_etha_initdefault(psock->here_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->here_ipl)));
  psock->here_port=port;
  lcrzo_er(lcrzo_etha_initdefault(psock->there_etha));
  lcrzo_er(lcrzo_ipl_initdefault(&(psock->there_ipl)));
  psock->there_port=0;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpser_virt2(const lcrzo_device device,
			    const lcrzo_etha ethaserveur,
			    const lcrzo_ipl iplserveur, 
			    lcrzo_uint16 portserveur,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptbytenum,
			    lcrzo_sock *psock)
{ int retour;
  lcrzo_bool dejaunclient, packetisforthisconnection; 
  lcrzo_uint32 isn, dernieroctetrecu;
  lcrzo_uint16 nboctdata;
  lcrzo_sniff cs;
  lcrzo_data data;
  lcrzo_hdrleth hdrlethrecu;
  lcrzo_hdrlip hdrliprecu;
  lcrzo_hdrltcp hdrltcprecu;
  lcrzo_uint8 bitreset;
  lcrzo_etha ethaclient;
  lcrzo_uint16 portclient;
  lcrzo_ipl iplclient;
  lcrzo_data sniffdata;
  lcrzo_int32 sniffdatasize, arrsize;
  lcrzo_uint8 arr[LCRZO_PRIV_READSIZE];
   
  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ethaserveur==NULL) return(LCRZO_ERR_PANULLPTR);
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
 
  /*initialisations*/
  psock->type=LCRZO_PRIV_TYPESOCK_VIRTSERTCP;
  psock->ipoptbytenum=0;
  lcrzo_er(lcrzo_priv_initlesci(psock));
  lcrzo_er(lcrzo_priv_ajouteipopt(psock, ipopt, ipoptbytenum));
  lcrzo_er(lcrzo_uint32_rand(0, 0xFFFFFFFFu, &isn));

  /*processus de sniff*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidsniff)));
  if (psock->pidsniff==0) /*on est dans le fils*/
  { /*initialisation du sniff*/
    dejaunclient=0;
    lcrzo_priv_processinitsniff(device, &cs);
    /*sniff des packets recus*/
    retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_TCPREOR, 
			     &sniffdata, &sniffdatasize);
    while(retour==LCRZO_ERR_OK)
    { /*lcrzo_affpacket_eth(packet, nboct, lcrzo_paff_tabdump);*/
      /*repond aux requetes concernant la machine virtuelle*/
      if (lcrzo_global.cliser_virt_answeralive)
      { lcrzo_virtual_answer_alive(sniffdata, sniffdatasize,
				   device, ethaserveur, iplserveur);
      }
      /*fourni les packets au processus pere*/
      retour=lcrzo_packet_decodem_ethipopttcpoptdata(sniffdata, sniffdatasize,
						     &hdrlethrecu, 
						     &hdrliprecu, NULL, NULL,
						     &hdrltcprecu, NULL, NULL, 
						     &data, &nboctdata);
      lcrzo_data_free(sniffdata);      
      if (retour==LCRZO_ERR_OK)
      { /*ne fournit que ceux correspondant a la communication en cours*/
        if ( lcrzo_etha_equal(hdrlethrecu.dst, ethaserveur) &&
             lcrzo_ipl_equal(hdrliprecu.daddr, iplserveur) &&
             hdrltcprecu.dport==portserveur )
	{ /*si c'est le premier packet, on termine le handshake*/
	  if ( !dejaunclient && hdrltcprecu.syn )
          { dernieroctetrecu=hdrltcprecu.seqnum+1;
	    dejaunclient=1;
	    memcpy(ethaclient, hdrlethrecu.src, LCRZO_ETHA_MAXBYTES);
	    iplclient=hdrliprecu.saddr;
	    portclient=hdrltcprecu.sport;
	    /*informe qui est le client*/
            lcrzo_priv_pfs(lcrzo_ipc_write_ethaiplport(psock->cicontrole1,
						       ethaclient, iplclient,
						       portclient), &cs);
	    /*envoie le syn-ack*/
	    lcrzo_priv_spiot(psock, device, ethaserveur, ethaclient,
			      iplserveur, iplclient, portserveur, portclient, 
			      isn, dernieroctetrecu,
			      1,1,0, NULL, 0);
	    packetisforthisconnection=1;
	  }
	  else if ( lcrzo_etha_equal(hdrlethrecu.src, ethaclient) &&
		    hdrliprecu.saddr==iplclient &&
		    hdrltcprecu.sport==portclient )
	  { dernieroctetrecu=hdrltcprecu.seqnum+nboctdata;
	    /*envoie le packet ack*/
	    lcrzo_priv_spiot(psock, device, ethaserveur, ethaclient,
			      iplserveur, iplclient, portserveur, portclient, 
			      hdrltcprecu.acknum, dernieroctetrecu,
			      0,1,0, NULL, 0);
	    packetisforthisconnection=1;
	  }
	  else
	  { packetisforthisconnection=0;
	  }
	  if (packetisforthisconnection)
	  { /*on passe le flot*/
	    lcrzo_priv_pfs(lcrzo_ipc_write_flow(psock->cisniff, data,
						nboctdata), &cs);
	    /*informe par le biais du canal de controle*/
	    lcrzo_priv_pfs(lcrzo_ipc_write_uint32(psock->cicontrole1,
						  dernieroctetrecu), &cs);
	  }
	}
	lcrzo_data_free(data);
      }
      retour=lcrzo_sniff_nextm(&cs, 1, LCRZO_SNIFF_TYPE_TCPREOR,
			       &sniffdata, &sniffdatasize);
    }
    lcrzo_priv_pfs(4, &cs);
  }

  /*attend que le handshake soit termine en bloquant jusqu'a recevoir
    des donnees dans le tunnel de controle*/
  lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 1, ethaclient,
				    &iplclient, &portclient));

  /*processus de spoof*/
  isn+=1; /*car l'isn est incremente dans le handshake*/
  lcrzo_er(lcrzo_priv_fork(*psock, &(psock->pidspoof)));
  if (psock->pidspoof==0) /*on est dans le fils*/
  { while(1)
    { /*on regarde le numero de acknum (prend le dernier disponible)*/
      retour=lcrzo_ipc_read_uint32(psock->cicontrole1, 0, &dernieroctetrecu);
      while ( retour==LCRZO_ERR_OK )
      { retour=lcrzo_ipc_read_uint32(psock->cicontrole1, 0, &dernieroctetrecu);
      }
      /*on regarde les donnees a envoyer (par bloc de 400 octets)*/
      retour=lcrzo_ipc_read_flow(psock->cispoof, 0, LCRZO_PRIV_READSIZE,
				 arr, &arrsize);
      if (retour==LCRZO_ERR_OK)
      { lcrzo_priv_pe(lcrzo_priv_spiot(psock, device, ethaserveur,
					 ethaclient, iplserveur,
					 iplclient, portserveur, portclient, 
					 isn, dernieroctetrecu,
					 0,1,0, arr, (lcrzo_uint16)arrsize));
        isn+=arrsize; /*si isn>0xffffffff passe automatiquement a zero*/
      }
      else
      { /*on regarde si un reset est demande (attend d'avoir tout envoye)*/
	bitreset=0;
	lcrzo_ipc_read_uint8(psock->cicontrole2, 0, &bitreset);
	if (bitreset)
        { lcrzo_priv_pe(lcrzo_priv_spiot(psock, device, ethaserveur,
					   ethaclient, iplserveur,
					   iplclient, portserveur, portclient, 
					   isn, dernieroctetrecu,
					   0,1,1, NULL, 0));
	}
      }
    }
    /*jamais atteint*/
  }

  /*remplissage des informations connues et initialisation des autres*/
  lcrzo_er(lcrzo_priv_majinfossock(device, ethaserveur, iplserveur,
				    portserveur,
				    ethaclient, iplclient, portclient,
                                    psock));
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_tcpmulser_real(lcrzo_uint16 port,
			      int (*pfonc)(lcrzo_sock sock,
					   const void *pinfos),
			      const void *pinfos)
{ return(lcrzo_sock_tcpmulser_real2(0, port, NULL, 0, pfonc, pinfos));
}

/*-------------------------------------------------------------*/
int lcrzo_sock_tcpmulser_real2(lcrzo_ipl ipl,
			       lcrzo_uint16 port,
			       const lcrzo_ipopt ipopt,
			       lcrzo_uint8 ipoptbytenum,
			       int (*pfonc)(lcrzo_sock sock,
					    const void *pinfos),
			       const void *pinfos)
{ struct sockaddr_in sain, sain2;
  int retour, sockglo, sizeof_sain2, un, pid;
  lcrzo_sock sock;

  /*parameters verification*/
  if (ipopt==NULL && ipoptbytenum) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (pfonc==NULL) return(LCRZO_ERR_PANULLPTR);

  /*creation de la socket*/
  sockglo=socket(AF_INET, SOCK_STREAM, 0);
  if (sockglo<0) return(LCRZO_ERR_FUSOCKET);

  /*pour que l'adresse puisse etre immediatemment reutilisee*/
  un=1;
  retour=setsockopt(sockglo, SOL_SOCKET, SO_REUSEADDR,
		    (void *)&un, sizeof(un));
  if (retour) return(LCRZO_ERR_FUSETSOCKOPT);

  /*ajout eventuel des options IP*/
  if (ipoptbytenum)
  { retour = setsockopt(sockglo, IPPROTO_IP, IP_OPTIONS, ipopt, ipoptbytenum);
    if (retour)
      return(LCRZO_ERR_FUSETSOCKOPT);
  }

  /*ecoute sur le port*/
  memset(&sain, 0, sizeof(sain));
  sain.sin_family = AF_INET;
  if ( ipl==0 )
    sain.sin_addr.s_addr=lcrzo_htonl(INADDR_ANY);
  else
    sain.sin_addr.s_addr=lcrzo_htonl(ipl);
  sain.sin_port = lcrzo_htons(port);
  retour=bind(sockglo, (struct sockaddr *)&sain, sizeof(sain));
  if (retour<0) return(LCRZO_ERR_FUBIND);

  retour=listen(sockglo, lcrzo_global.sock_tcpser_backlog);
  if (retour<0) return(LCRZO_ERR_FULISTEN);

  /*on ecoute indefiniment*/
  while (1)
  { sizeof_sain2=sizeof(sain2);
    memset(&sain2, 0, sizeof_sain2);
    sock.socknum=accept(sockglo, (struct sockaddr *)&sain2, &sizeof_sain2);
    if (sock.socknum<0) return(LCRZO_ERR_FUACCEPT);
    pid=fork();
    if (pid<0) return(LCRZO_ERR_FUFORK);
    if (pid==0) /*on est dans le fils*/
    { /*on ferme la socket du pere*/
      retour=close(sockglo);
      if (retour==-1) return(LCRZO_ERR_FUCLOSE);
      /*ajout eventuel des options IP*/
      if (ipoptbytenum)
      { retour = setsockopt(sock.socknum, IPPROTO_IP, IP_OPTIONS,
			    ipopt, ipoptbytenum);
        if (retour) return(LCRZO_ERR_FUSETSOCKOPT);
      }
      /*remplissage des caracteristiques de la socket*/
      sock.type=LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI;
      sock.beblocking=1;
      /*remplissage des informations connues et initialisation des autres*/
      lcrzo_er(lcrzo_device_initdefault(sock.here_device));
      lcrzo_er(lcrzo_etha_initdefault(sock.here_etha));
      lcrzo_er(lcrzo_ipl_initdefault(&sock.here_ipl));
      sock.here_port=port;
      lcrzo_er(lcrzo_etha_initdefault(sock.there_etha));
      lcrzo_er(lcrzo_ipl_initdefault(&sock.there_ipl));
      sock.there_port=0;
      /*utilisation de sock*/
      retour=(*pfonc)(sock, pinfos);
      _exit(retour);
    }
    else /*on est dans le pere*/
    { retour=close(sock.socknum);
      if (retour==-1) return(LCRZO_ERR_FUCLOSE);
    }
  }
  /*jamais atteint, mais pour faire plaisir au compilateur*/
  return(LCRZO_ERR_IEINTERNALERROR);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_write(lcrzo_sock sock,
		     lcrzo_constdata donnees, 
		     lcrzo_int32 nboctdonnees)
{ 
  /*parameters verification*/
  if (donnees==NULL && nboctdonnees) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (nboctdonnees==0) return(LCRZO_ERR_OK);

  /*cas d'erreur*/
  switch(sock.type)
  { case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      return(LCRZO_ERR_BUUDPSERMULTINOWRITE);
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
      if (sock.canwrite==0)
	return(LCRZO_ERR_BUUDPSERREADBEFWRITE);
      break;
  }

  /*cas normaux*/
  switch(sock.type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      return(lcrzo_fd_write(sock.socknum, donnees, nboctdonnees));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      return(lcrzo_ipc_write_flow(sock.cispoof, donnees, nboctdonnees));
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_read(lcrzo_sock *psock,
		    lcrzo_bool etrebloquant,
		    lcrzo_int32 dataoutmaxsize,
		    lcrzo_data dataout,
		    lcrzo_int32 *pdataoutsize)
{ int retour;
  struct sockaddr_in sai;
          
  /*verification des parametres*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(etrebloquant);

  /*premiere lecture sur un serveur UDP : doit trouver le client*/
  if ( psock->canwrite==0 )
  { if ( psock->type==LCRZO_PRIV_TYPESOCK_REALSERUDP )
    { /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfrom(psock->socknum, etrebloquant,
				      dataoutmaxsize,
				      dataout, pdataoutsize, &sai));
      /*on se connecte maintenant sur le client afin de pouvoir y ecrire*/
      retour=connect(psock->socknum, (struct sockaddr *)&sai, sizeof(sai));
      if (retour<0) return(LCRZO_ERR_FUCONNECT);
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*on s'arrete here*/
      return(LCRZO_ERR_OK);
    }
    if ( psock->type==LCRZO_PRIV_TYPESOCK_VIRTSERUDP )
    { /*lecture des informations sur le client : si on n'est pas bloquant
         cette fonction retourne (donc psock->canwrite ne vaudra pas 1)*/
      lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, etrebloquant,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port)));
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*contrairement au cas ci-dessus, on continue la fonction*/
    }
  }

  /*lecture normale*/
  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      lcrzo_er(lcrzo_fd_read(psock->socknum, etrebloquant,
			     dataoutmaxsize, dataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_read_flow(psock->cisniff, etrebloquant,
				   dataoutmaxsize, dataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfrom(psock->socknum, etrebloquant,
				      dataoutmaxsize, dataout, pdataoutsize,
				      &sai));
      /*on met a jour les informations*/
      psock->there_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      psock->there_port=lcrzo_ntohs(sai.sin_port);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_ipc_read_flow(psock->cisniff, etrebloquant,
				   dataoutmaxsize, dataout, pdataoutsize));
      /*on met a jour les informations*/
      do
      { retour=lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 0,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port));
      }
      while ( retour==LCRZO_ERR_OK );
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_readm(lcrzo_sock *psock,
		     lcrzo_bool etrebloquant,
		     lcrzo_data *pdataout,
		     lcrzo_int32 *pdataoutsize)
{ int retour;
  struct sockaddr_in sai;
          
  /*verification des parametres*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(etrebloquant);

  /*premiere lecture sur un serveur UDP : doit trouver le client*/
  if ( psock->canwrite==0 )
  { if ( psock->type==LCRZO_PRIV_TYPESOCK_REALSERUDP )
    { /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfromm(psock->socknum, etrebloquant,
				       pdataout, pdataoutsize, &sai));
      /*on se connecte maintenant sur le client afin de pouvoir y ecrire*/
      retour=connect(psock->socknum, (struct sockaddr *)&sai, sizeof(sai));
      if (retour<0) return(LCRZO_ERR_FUCONNECT);
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*on s'arrete here*/
      return(LCRZO_ERR_OK);
    }
    if ( psock->type==LCRZO_PRIV_TYPESOCK_VIRTSERUDP )
    { /*lecture des informations sur le client : si on n'est pas bloquant
         cette fonction retourne (donc psock->canwrite ne vaudra pas 1)*/
      lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, etrebloquant,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port)));
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*contrairement au cas ci-dessus, on continue la fonction*/
    }
  }

  /*lecture normale*/
  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      lcrzo_er(lcrzo_fd_readm(psock->socknum, etrebloquant,
			      pdataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_readm_flow(psock->cisniff, etrebloquant,
				    pdataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfromm(psock->socknum, etrebloquant,
				       pdataout, pdataoutsize,
				       &sai));
      /*on met a jour les informations*/
      psock->there_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      psock->there_port=lcrzo_ntohs(sai.sin_port);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_ipc_readm_flow(psock->cisniff, etrebloquant,
				    pdataout, pdataoutsize));
      /*on met a jour les informations*/
      do
      { retour=lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 0,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port));
      }
      while ( retour==LCRZO_ERR_OK );
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_read_fixed(lcrzo_sock *psock,
			  lcrzo_bool beblocking,
			  lcrzo_int32 datasizetoread,
			  lcrzo_data dataout)
{ int retour;
  struct sockaddr_in sai;
          
  /*verification des parametres*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(beblocking);

  /*premiere lecture sur un serveur UDP : doit trouver le client*/
  if ( psock->canwrite==0 )
  { if ( psock->type==LCRZO_PRIV_TYPESOCK_REALSERUDP )
    { /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfrom_fixed(psock->socknum, beblocking,
					    datasizetoread,
					    dataout, &sai));
      /*on se connecte maintenant sur le client afin de pouvoir y ecrire*/
      retour=connect(psock->socknum, (struct sockaddr *)&sai, sizeof(sai));
      if (retour<0) return(LCRZO_ERR_FUCONNECT);
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*on s'arrete here*/
      return(LCRZO_ERR_OK);
    }
    if ( psock->type==LCRZO_PRIV_TYPESOCK_VIRTSERUDP )
    { /*lecture des informations sur le client : si on n'est pas bloquant
         cette fonction retourne (donc psock->canwrite ne vaudra pas 1)*/
      lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, beblocking,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port)));
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*contrairement au cas ci-dessus, on continue la fonction*/
    }
  }

  /*lecture normale*/
  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      lcrzo_er(lcrzo_fd_read_fixed(psock->socknum, beblocking,
				   datasizetoread, dataout));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_read_flow_fixed(psock->cisniff, beblocking,
					 datasizetoread, dataout));
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfrom_fixed(psock->socknum, beblocking,
					    datasizetoread, dataout, &sai));
      /*on met a jour les informations*/
      psock->there_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      psock->there_port=lcrzo_ntohs(sai.sin_port);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_ipc_read_flow_fixed(psock->cisniff, beblocking,
					 datasizetoread, dataout));
      /*on met a jour les informations*/
      do
      { retour=lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 0,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port));
      }
      while ( retour==LCRZO_ERR_OK );
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_readm_fixed(lcrzo_sock *psock,
			   lcrzo_bool beblocking,
			   lcrzo_int32 datasizetoread,
			   lcrzo_data *pdataout)
{ int retour;
  struct sockaddr_in sai;
          
  /*verification des parametres*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(beblocking);

  /*premiere lecture sur un serveur UDP : doit trouver le client*/
  if ( psock->canwrite==0 )
  { if ( psock->type==LCRZO_PRIV_TYPESOCK_REALSERUDP )
    { /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfromm_fixed(psock->socknum, beblocking,
					     datasizetoread, pdataout, &sai));
      /*on se connecte maintenant sur le client afin de pouvoir y ecrire*/
      retour=connect(psock->socknum, (struct sockaddr *)&sai, sizeof(sai));
      if (retour<0) return(LCRZO_ERR_FUCONNECT);
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*on s'arrete here*/
      return(LCRZO_ERR_OK);
    }
    if ( psock->type==LCRZO_PRIV_TYPESOCK_VIRTSERUDP )
    { /*lecture des informations sur le client : si on n'est pas bloquant
         cette fonction retourne (donc psock->canwrite ne vaudra pas 1)*/
      lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, beblocking,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port)));
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*contrairement au cas ci-dessus, on continue la fonction*/
    }
  }

  /*lecture normale*/
  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      lcrzo_er(lcrzo_fd_readm_fixed(psock->socknum, beblocking,
				    datasizetoread, pdataout));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_readm_flow_fixed(psock->cisniff, beblocking,
					  datasizetoread, pdataout));
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfromm_fixed(psock->socknum, beblocking,
					     datasizetoread, pdataout, 
					     &sai));
      /*on met a jour les informations*/
      psock->there_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      psock->there_port=lcrzo_ntohs(sai.sin_port);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_ipc_readm_flow_fixed(psock->cisniff, beblocking,
					  datasizetoread, pdataout));
      /*on met a jour les informations*/
      do
      { retour=lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 0,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port));
      }
      while ( retour==LCRZO_ERR_OK );
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_read_line(lcrzo_sock *psock,
			 lcrzo_bool beblocking,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize)
{ int retour;
  struct sockaddr_in sai;
          
  /*verification des parametres*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(beblocking);

  /*premiere lecture sur un serveur UDP : doit trouver le client*/
  if ( psock->canwrite==0 )
  { if ( psock->type==LCRZO_PRIV_TYPESOCK_REALSERUDP )
    { /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfrom_line(psock->socknum, beblocking,
					    dataoutmaxsize,
					    dataout, pdataoutsize, &sai));
      /*on se connecte maintenant sur le client afin de pouvoir y ecrire*/
      retour=connect(psock->socknum, (struct sockaddr *)&sai, sizeof(sai));
      if (retour<0) return(LCRZO_ERR_FUCONNECT);
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*on s'arrete here*/
      return(LCRZO_ERR_OK);
    }
    if ( psock->type==LCRZO_PRIV_TYPESOCK_VIRTSERUDP )
    { /*lecture des informations sur le client : si on n'est pas bloquant
         cette fonction retourne (donc psock->canwrite ne vaudra pas 1)*/
      lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, beblocking,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port)));
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*contrairement au cas ci-dessus, on continue la fonction*/
    }
  }

  /*lecture normale*/
  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      lcrzo_er(lcrzo_fd_read_line(psock->socknum, beblocking,
				   dataoutmaxsize, dataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_read_flow_line(psock->cisniff, beblocking,
					 dataoutmaxsize, dataout,
					 pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfrom_line(psock->socknum, beblocking,
					    dataoutmaxsize, dataout,
					    pdataoutsize, &sai));
      /*on met a jour les informations*/
      psock->there_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      psock->there_port=lcrzo_ntohs(sai.sin_port);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_ipc_read_flow_line(psock->cisniff, beblocking,
					 dataoutmaxsize, dataout, 
					 pdataoutsize));
      /*on met a jour les informations*/
      do
      { retour=lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 0,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port));
      }
      while ( retour==LCRZO_ERR_OK );
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_sock_readm_line(lcrzo_sock *psock,
			  lcrzo_bool beblocking,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize)
{ int retour;
  struct sockaddr_in sai;
          
  /*verification des parametres*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(beblocking);

  /*premiere lecture sur un serveur UDP : doit trouver le client*/
  if ( psock->canwrite==0 )
  { if ( psock->type==LCRZO_PRIV_TYPESOCK_REALSERUDP )
    { /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfromm_line(psock->socknum, beblocking,
					     pdataout, pdataoutsize, &sai));
      /*on se connecte maintenant sur le client afin de pouvoir y ecrire*/
      retour=connect(psock->socknum, (struct sockaddr *)&sai, sizeof(sai));
      if (retour<0) return(LCRZO_ERR_FUCONNECT);
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*on s'arrete here*/
      return(LCRZO_ERR_OK);
    }
    if ( psock->type==LCRZO_PRIV_TYPESOCK_VIRTSERUDP )
    { /*lecture des informations sur le client : si on n'est pas bloquant
         cette fonction retourne (donc psock->canwrite ne vaudra pas 1)*/
      lcrzo_er(lcrzo_ipc_read_ethaiplport(psock->cicontrole1, beblocking,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port)));
      /*maintenant, comme on connait le client, on pourra ecrire*/
      psock->canwrite=1;
      /*contrairement au cas ci-dessus, on continue la fonction*/
    }
  }

  /*lecture normale*/
  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      lcrzo_er(lcrzo_fd_readm_line(psock->socknum, beblocking,
				    pdataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_readm_flow_line(psock->cisniff, beblocking,
					  pdataout, pdataoutsize));
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_priv_fd_recvfromm_line(psock->socknum, beblocking,
					     pdataout, pdataoutsize,
					     &sai));
      /*on met a jour les informations*/
      psock->there_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      psock->there_port=lcrzo_ntohs(sai.sin_port);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      /*lecture des donnees presentes*/
      lcrzo_er(lcrzo_ipc_readm_flow_line(psock->cisniff, beblocking,
					  pdataout, pdataoutsize));
      /*on met a jour les informations*/
      do
      { retour=lcrzo_ipc_read_ethaiplport(psock->cicontrole1, 0,
					  psock->there_etha,
					  &(psock->there_ipl),
					  &(psock->there_port));
      }
      while ( retour==LCRZO_ERR_OK );
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_sock_close(lcrzo_sock sock)
{ int retour, status;

  /*socket reelles : fait un simple close*/
  switch(sock.type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      retour=close(sock.socknum);
      if (retour==-1) return(LCRZO_ERR_FUCLOSE);
      return(LCRZO_ERR_OK);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }

  /*maintenant, il ne reste que des sockets virtuelles*/
  sleep(1); /*pour laisser le temps aux processus d'envoyer les packets*/

  /*envoi du packet reset*/
  switch(sock.type)
  { case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      lcrzo_er(lcrzo_ipc_write_uint8(sock.cicontrole2, 1));
      sleep(1);
  }

  /*arret des processus*/
  retour=kill(sock.pidsniff, SIGTERM);
  if (retour==-1) return(LCRZO_ERR_FUKILL);
  retour=kill(sock.pidspoof, SIGTERM);
  if (retour==-1) return(LCRZO_ERR_FUKILL);
  retour=waitpid(sock.pidsniff, &status, 0);
  if (retour==-1) return(LCRZO_ERR_FUWAITPID);
  retour=waitpid(sock.pidspoof, &status, 0);
  if (retour==-1) return(LCRZO_ERR_FUWAITPID);

  /*fermeture des contextes d'IPC*/
  lcrzo_priv_fermelesci(sock);
  errno=0; /* because Solaris set it */
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_get_infos(lcrzo_sock sock,
			lcrzo_device here_device,
			lcrzo_etha here_etha,
			lcrzo_ipl *phere_ipl,
			lcrzo_uint16 *phere_port,
			lcrzo_etha there_etha,
			lcrzo_ipl *pthere_ipl,
			lcrzo_uint16 *pthere_port)
{ int retour, taillesa;
  struct sockaddr sa;
  struct sockaddr_in sai;

  switch(sock.type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      if (here_device!=NULL)
	lcrzo_er(lcrzo_device_initdefault(here_device));
      if (here_etha!=NULL)
	lcrzo_er(lcrzo_etha_initdefault(here_etha));
      if (there_etha!=NULL)
	lcrzo_er(lcrzo_etha_initdefault(there_etha));
      taillesa=sizeof(struct sockaddr);
      retour=getsockname(sock.socknum, &sa, &taillesa);
      if (retour==-1) return(LCRZO_ERR_FUGETSOCKNAME);
      memcpy(&sai, &sa, sizeof(struct sockaddr_in));
      if (phere_ipl!=NULL)
	*phere_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      if (phere_port!=NULL)
	*phere_port=lcrzo_ntohs(sai.sin_port);
      taillesa=sizeof(struct sockaddr);
      retour=getpeername(sock.socknum, &sa, &taillesa);
      if (retour==-1)
      { if (errno!=ENOTCONN) return(LCRZO_ERR_FUGETPEERNAME);
        /*personne n'est connecte*/
        errno=0;
        if (pthere_ipl!=NULL)
	  lcrzo_er(lcrzo_ipl_initdefault(pthere_ipl));
	if (pthere_port!=NULL)
	  *pthere_port=0;
      }
      else
      { memcpy(&sai, &sa, sizeof(struct sockaddr_in));
        if (pthere_ipl!=NULL)
	  *pthere_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
	if (pthere_port!=NULL)
	  *pthere_port=lcrzo_ntohs(sai.sin_port);
      }
      break;
    case LCRZO_PRIV_TYPESOCK_REALSERUDPMULTI:
      taillesa=sizeof(struct sockaddr);
      retour=getsockname(sock.socknum, &sa, &taillesa);
      if (retour==-1) return(LCRZO_ERR_FUGETSOCKNAME);
      memcpy(&sai, &sa, sizeof(struct sockaddr_in));
      if (phere_ipl!=NULL)
	*phere_ipl=lcrzo_ntohl(sai.sin_addr.s_addr);
      if (phere_port!=NULL)
	*phere_port=lcrzo_ntohs(sai.sin_port);
      if (here_device!=NULL)
	memcpy(here_device, sock.here_device, LCRZO_DEVICE_MAXBYTES);
      if (here_etha!=NULL)
	memcpy(here_etha, sock.here_etha, LCRZO_ETHA_MAXBYTES);
      if (there_etha!=NULL)
	memcpy(there_etha, sock.there_etha, LCRZO_ETHA_MAXBYTES);
      if (pthere_ipl!=NULL)
	*pthere_ipl=sock.there_ipl;
      if (pthere_port!=NULL)
	*pthere_port=sock.there_port;
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDPMULTI:
      if (here_device!=NULL)
	memcpy(here_device, sock.here_device, LCRZO_DEVICE_MAXBYTES);
      if (here_etha!=NULL)
	memcpy(here_etha, sock.here_etha, LCRZO_ETHA_MAXBYTES);
      if (phere_ipl!=NULL)
	*phere_ipl=sock.here_ipl;
      if (phere_port!=NULL)
	*phere_port=sock.here_port;
      if (there_etha!=NULL)
	memcpy(there_etha, sock.there_etha, LCRZO_ETHA_MAXBYTES);
      if (pthere_ipl!=NULL)
	*pthere_ipl=sock.there_ipl;
      if (pthere_port!=NULL)
	*pthere_port=sock.there_port;
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_fprint_infos(LCRZOFILE *pf,
			    lcrzo_sock sock)
{ lcrzo_device here_device;
  lcrzo_etha here_etha;
  lcrzo_ipl here_ipl;
  lcrzo_uint16 here_port;
  lcrzo_etha there_etha;
  lcrzo_ipl there_ipl;
  lcrzo_uint16 there_port;

  /*parameters verification*/
  if (pf==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_sock_get_infos(sock, here_device, here_etha, &here_ipl, 
			       &here_port,  there_etha, &there_ipl,
			       &there_port));
  lcrzo_er(lcrzo_device_fprint(pf, "Local : device=", here_device, ", "));
  lcrzo_er(lcrzo_etha_fprint(pf, "eth=", here_etha, ", "));
  lcrzo_er(lcrzo_ipl_fprint(pf, "ip=", here_ipl, ", "));
  fprintf(pf, "port=%d\n", here_port);
  lcrzo_er(lcrzo_etha_fprint(pf, "Distant : eth=", there_etha, ", "));
  lcrzo_er(lcrzo_ipl_fprint(pf, "ip=", there_ipl, ", "));
  fprintf(pf, "port=%d\n", there_port);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_sock_ipopt(lcrzo_sock *psock,
			  const lcrzo_ipopt opt, lcrzo_uint16 nboctopt);
int lcrzo_priv_sock_ipopt(lcrzo_sock *psock,
			  const lcrzo_ipopt opt, lcrzo_uint16 nboctopt)
{ int retour;

  if (opt==NULL && nboctopt) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (nboctopt==0)
    return(LCRZO_ERR_OK);

  switch(psock->type)
  { case LCRZO_PRIV_TYPESOCK_REALCLIUDP:
    case LCRZO_PRIV_TYPESOCK_REALCLITCP:
    case LCRZO_PRIV_TYPESOCK_REALSERUDP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCP:
    case LCRZO_PRIV_TYPESOCK_REALSERTCPMULTI:
      retour = setsockopt(psock->socknum, IPPROTO_IP, IP_OPTIONS,
			  opt, nboctopt);
      if (retour) return(LCRZO_ERR_FUSETSOCKOPT);
      break;
    case LCRZO_PRIV_TYPESOCK_VIRTCLIUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTCLITCP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERUDP:
    case LCRZO_PRIV_TYPESOCK_VIRTSERTCP:
      if (nboctopt+psock->ipoptbytenum>LCRZO_IPOPT_MAXBYTES)
	return(LCRZO_ERR_SPOPTMAX10);
      lcrzo_er(lcrzo_ipc_write_data(psock->cioptions, opt, nboctopt));
      break;
    default:
      return(LCRZO_ERR_IEINTERNALERROR);
  } 
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_noop(lcrzo_sock *psock)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_noop(opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_lsrr(lcrzo_sock *psock,
			 lcrzo_ipl ipl1,
			 lcrzo_ipl ipl2,
			 lcrzo_ipl ipl3,
			 lcrzo_ipl ipl4,
			 lcrzo_ipl ipl5,
			 lcrzo_ipl ipl6,
			 lcrzo_ipl ipl7,
			 lcrzo_ipl ipl8,
			 lcrzo_ipl ipl9,
			 lcrzo_int8 nbiplaffectes)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_lsrr(ipl1, ipl2, ipl3, ipl4, ipl5, ipl6,
				ipl7, ipl8, ipl9, nbiplaffectes,
				opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_ssrr(lcrzo_sock *psock,
			 lcrzo_ipl ipl1,
			 lcrzo_ipl ipl2,
			 lcrzo_ipl ipl3,
			 lcrzo_ipl ipl4,
			 lcrzo_ipl ipl5,
			 lcrzo_ipl ipl6,
			 lcrzo_ipl ipl7,
			 lcrzo_ipl ipl8,
			 lcrzo_ipl ipl9,
			 lcrzo_int8 nbiplaffectes)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_ssrr(ipl1, ipl2, ipl3, ipl4, ipl5, ipl6,
				ipl7, ipl8, ipl9, nbiplaffectes,
				opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_rr(lcrzo_sock *psock,
		       lcrzo_int8 nbroutesaenregistrer)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_rr(nbroutesaenregistrer, opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_time(lcrzo_sock *psock,
			 lcrzo_int8 nbtimestampsaenregistrer)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_time(nbtimestampsaenregistrer, 
				opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_timeip(lcrzo_sock *psock,
			   lcrzo_int8 nbtimestampsaenregistrer)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_timeip(nbtimestampsaenregistrer, 
				  opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_sock_ipopt_timeipp(lcrzo_sock *psock,
			    lcrzo_ipl ipl1,
			    lcrzo_ipl ipl2,
			    lcrzo_ipl ipl3,
			    lcrzo_ipl ipl4,
			    lcrzo_int8 nbiplaffectes)
{ lcrzo_ipopt opt;
  lcrzo_uint8 nboctopt;

  /*parameters verification*/
  if (psock==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipopt_init_timeipp(ipl1, ipl2, ipl3, ipl4, nbiplaffectes,
				   opt, &nboctopt));
  lcrzo_er(lcrzo_priv_sock_ipopt(psock, opt, nboctopt));
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_virtual_answer_arp(lcrzo_constdata packet,
			    lcrzo_uint32 nboct,
			    const lcrzo_device device,
			    const lcrzo_etha etha,
			    const lcrzo_ipl ipl)
{ int retour;
  lcrzo_hdrleth hdrlethrecu, hdrlethenvoye;
  lcrzo_hdrlarp hdrlarprecu, hdrlarpenvoye;
  lcrzo_ipa ipa;
  lcrzo_spoof spoof;

  /*parameters verification*/
  if (packet==NULL && nboct) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  /*tentative de decodage ARP*/
  retour=lcrzo_packet_decode_etharp(packet, nboct,
				   &hdrlethrecu, &hdrlarprecu);
  if (retour!=LCRZO_ERR_OK) return(LCRZO_ERR_OK);

  /*on n'est interesse que par les requetes arp*/
  if ( hdrlethrecu.type != LCRZO_HDRLETH_TYPE_ARP ) 
    return(LCRZO_ERR_OK);
  if ( hdrlarprecu.op != 1 ) return(LCRZO_ERR_OK);
  
  /*on n'est interesse que par les requetes de ipa*/
  lcrzo_er(lcrzo_ipa_init_ipl(ipl, ipa));
  if (!lcrzo_ipa_equal(hdrlarprecu.prot_dst, ipa)) return(LCRZO_ERR_OK);

  /*cree le nouveau entete ethernet*/
  lcrzo_er(lcrzo_hdrleth_initdefault(&hdrlethenvoye));
  memcpy(hdrlethenvoye.src, etha, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlethenvoye.dst, hdrlarprecu.hw_src, LCRZO_ETHA_MAXBYTES);
  hdrlethenvoye.type=LCRZO_HDRLETH_TYPE_ARP;
    
  /*cree le nouveau entete arp*/
  lcrzo_er(lcrzo_hdrlarp_initdefault(&hdrlarpenvoye));
  hdrlarpenvoye.op=2;
  memcpy(hdrlarpenvoye.hw_src, etha, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlarpenvoye.hw_dst, hdrlarprecu.hw_src, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlarpenvoye.prot_src, hdrlarprecu.prot_dst, LCRZO_IPA_MAXBYTES);
  memcpy(hdrlarpenvoye.prot_dst, hdrlarprecu.prot_src, LCRZO_IPA_MAXBYTES);
    
  /*envoi du packet*/
  lcrzo_er(lcrzo_spoof_init(&spoof));
  lcrzo_er(lcrzo_spoof_etharp(&spoof, device, hdrlethenvoye,
			      hdrlarpenvoye));
  lcrzo_er(lcrzo_spoof_close(&spoof));
  
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_virtual_answer_rarp(lcrzo_constdata packet,
			     lcrzo_uint32 nboct,
			     const lcrzo_device device,
			     const lcrzo_etha etha,
			     const lcrzo_ipl ipl)
{ int retour;
  lcrzo_hdrleth hdrlethrecu, hdrlethenvoye;
  lcrzo_hdrlarp hdrlarprecu, hdrlarpenvoye;
  lcrzo_ipa ipa;
  lcrzo_spoof spoof;

  /*parameters verification*/
  if (packet==NULL && nboct) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  /*tentative de decodage ARP*/
  retour=lcrzo_packet_decode_etharp(packet, nboct,
				   &hdrlethrecu, &hdrlarprecu);
  if (retour!=LCRZO_ERR_OK) return(LCRZO_ERR_OK);

  /*on n'est interesse que par les requetes rarp*/
  if ( hdrlethrecu.type != LCRZO_HDRLETH_TYPE_RARP )
    return(LCRZO_ERR_OK);
  if ( hdrlarprecu.op != 3 ) return(LCRZO_ERR_OK);
  
  /*on n'est interesse que par les requetes de etha*/
  if (!lcrzo_etha_equal(hdrlarprecu.hw_dst, etha)) return(LCRZO_ERR_OK);

  /*cree le nouveau entete ethernet*/
  lcrzo_er(lcrzo_hdrleth_initdefault(&hdrlethenvoye));
  memcpy(hdrlethenvoye.src, etha, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlethenvoye.dst, hdrlarprecu.hw_src, LCRZO_ETHA_MAXBYTES);
  hdrlethenvoye.type=LCRZO_HDRLETH_TYPE_RARP;
    
  /*cree le nouveau entete arp*/
  lcrzo_er(lcrzo_ipa_init_ipl(ipl, ipa));
  lcrzo_er(lcrzo_hdrlarp_initdefault(&hdrlarpenvoye));
  hdrlarpenvoye.op=4;
  memcpy(hdrlarpenvoye.hw_src, etha, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlarpenvoye.hw_dst, etha, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlarpenvoye.prot_src, ipa, LCRZO_IPA_MAXBYTES);
  memcpy(hdrlarpenvoye.prot_dst, ipa, LCRZO_IPA_MAXBYTES);
    
  /*envoi du packet*/
  lcrzo_er(lcrzo_spoof_init(&spoof));
  lcrzo_er(lcrzo_spoof_etharp(&spoof, device, hdrlethenvoye, hdrlarpenvoye));
  lcrzo_er(lcrzo_spoof_close(&spoof));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_virtual_answer_ping(lcrzo_constdata packet,
			     lcrzo_uint32 nboct,
			     const lcrzo_device device,
			     const lcrzo_etha etha,
			     const lcrzo_ipl ipl)
{ lcrzo_hdrleth hdrlethrecu, hdrlethenvoye;
  lcrzo_hdrlip  hdrliprecu, hdrlipenvoye;
  lcrzo_hdrlicmp hdrlicmprecu, hdrlicmpenvoye;
  lcrzo_data data;
  lcrzo_ipopt opt;
  lcrzo_uint16 nboctdata;
  lcrzo_uint8 nboctopt;
  lcrzo_spoof spoof;
  int retour;

  /*parameters verification*/
  if (packet==NULL && nboct) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  /*tentative de decodage ICMP*/
  retour=lcrzo_packet_decodem_ethipopticmpdata(packet, nboct,
					       &hdrlethrecu, &hdrliprecu,
					       opt, &nboctopt, &hdrlicmprecu,
					       &data, &nboctdata);
  if (retour!=LCRZO_ERR_OK) return(LCRZO_ERR_OK);

  /*on n'est interesse que par les echo request*/
  if ( hdrlicmprecu.type != 8 ) 
  { lcrzo_data_free(data);
    return(LCRZO_ERR_OK); 
  }
  
  /*on n'est interesse que par les requetes de etha et ipa*/
  if (!lcrzo_etha_equal(hdrlethrecu.dst, etha)) 
  { lcrzo_data_free(data);
    return(LCRZO_ERR_OK);
  }
  if (!lcrzo_ipl_equal(hdrliprecu.daddr, ipl))
  { lcrzo_data_free(data); 
    return(LCRZO_ERR_OK);
  }

  /*cree le nouveau entete ethernet*/
  lcrzo_efr(lcrzo_hdrleth_initdefault(&hdrlethenvoye), lcrzo_data_free(data));
  memcpy(hdrlethenvoye.src, etha, LCRZO_ETHA_MAXBYTES);
  memcpy(hdrlethenvoye.dst, hdrlethrecu.src, LCRZO_ETHA_MAXBYTES);
    
  /*cree le nouveau entete ip*/
  lcrzo_efr(lcrzo_hdrlip_initdefault(&hdrlipenvoye), lcrzo_data_free(data));
  hdrlipenvoye.saddr=hdrliprecu.daddr;
  hdrlipenvoye.daddr=hdrliprecu.saddr;

  /*cree le nouveau entete icmp*/
  lcrzo_efr(lcrzo_hdrlicmp_initdefault(&hdrlicmpenvoye),
	    lcrzo_data_free(data));
  hdrlicmpenvoye.type=0;

  /*envoi du packet*/
  lcrzo_efr(lcrzo_spoof_init(&spoof), lcrzo_data_free(data));
  lcrzo_efr(lcrzo_spoof_ethipopticmpdata(&spoof, device,
					 hdrlethenvoye,
					 hdrlipenvoye, opt, nboctopt,
					 hdrlicmpenvoye, 
					 data, nboctdata), 
	    lcrzo_data_free(data));
  lcrzo_data_free(data);
  lcrzo_er(lcrzo_spoof_close(&spoof));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_virtual_answer_alive(lcrzo_constdata packet,
				lcrzo_uint32 nboct,
				const lcrzo_device device,
				const lcrzo_etha etha,
				const lcrzo_ipl ipl)
{ 
  /*parameters verification*/
  if (packet==NULL && nboct) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_virtual_answer_arp(packet, nboct,
				    device, etha, ipl));
  lcrzo_er(lcrzo_virtual_answer_rarp(packet, nboct,
				     device, etha, ipl));
  lcrzo_er(lcrzo_virtual_answer_ping(packet, nboct,
				     device, etha, ipl));
  return(LCRZO_ERR_OK);
}
