/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_cliser module

  Functions herein allow to create clients and servers.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_cliser

  Les fonctions presentes dans ce module permettent de creer des
  clients ou des serveurs reels, ainsi que virtuels.
*/

/*---------------------------------------------------------------*/
/*us the definition of a lcrzo_list has to be public
   (because the compiler needs to know its size), 
   but you should not use its fields directly. */
/*fr la definition d'un lcrzo_list doit etre publique
   (car le compilateur doit connaitre sa taille), mais vous
   ne devriez pas utiliser directement ses champs. */
typedef struct
{ int type;
  int socknum;
  lcrzo_bool beblocking;
  int pidsniff; 
  int pidspoof;
  lcrzo_ipc cisniff;
  lcrzo_ipc cispoof;
  lcrzo_ipc cioptions;
  lcrzo_ipc cicontrole1;
  lcrzo_ipc cicontrole2;
  lcrzo_ipopt ipopt;
  lcrzo_uint8 ipoptbytenum;
  lcrzo_device here_device;
  lcrzo_etha here_etha;
  lcrzo_ipl here_ipl;
  lcrzo_uint16 here_port;
  lcrzo_etha there_etha;
  lcrzo_ipl there_ipl;
  lcrzo_uint16 there_port;
  lcrzo_bool canwrite;
} lcrzo_sock;

/*---------------------------------------------------------------*/
/*US************************************************************
 * Note about "real" and "virtual" :                           *
 *                                                             *
 * A real client/server uses IP address and Ethernet address   *
 * of the current computer (they are classical sockets).       *
 *                                                             *
 * A virtual client/server uses spoofed IP address and         *
 * Ethernet address. For example, a virtual tcp client behaves *
 * like this :                                                 *
 *  - spoof a syn packet                                       *
 *  - sniff the syn-ack from the server                        *
 *  - spoof a ack packet to terminate the handshake            *
 *  - then, it behaves exactly like a real client : we can use *
 *    the lcrzo_sock to read or write data on the network      *
 ***************************************************************/
/*FR************************************************************
 * Note sur "reel" et "virtuel" :                              *
 *                                                             *
 * Les clients/serveurs reels emploient les adresses ip et     *
 * ethernet de la machine courante (ce sont les sockets        *
 * classiques telles que tout le monde les connait).           *
 *                                                             *
 * Les clients/serveurs virtuels utilisent des adresses        *
 * ip/ethernet spoofees. En quelques mots, prenons l'exemple   *
 * d'un client virtuel tcp :                                   *
 *  - on spoofe un packet syn                                  *
 *  - on sniffe le reseau a la recherche du syn-ack            *
 *  - on spoofe un packet ack pour terminer le handshake       *
 *  - ensuite, on peut lire et ecrire des donnees, comme on le *
 *    ferait avec un client tcp reel (en utilisant le          *
 *    lcrzo_sock)                                              *
 ***************************************************************/


/*---------------------------------------------------------------*/
/*US                        UDP clients                          */
/*FR                        clients UDP                          */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_sock_udpcli_real
   Description :
     Initialize a real udp client
   Input parameter(s) :
     serveripl : IP address of the server where to connect
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_real
   Description :
     Initialise un client udp reel
   Parametre(s) d'entree :
     serveripl : adresse IP du serveur ou se connecter
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_real(lcrzo_ipl    serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock   *psock);

/*us Name : lcrzo_sock_udpcli_virt
   Description :
     Initialize a virtual udp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_virt
   Description :
     Initialise un client udp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresde Ethernet du serveur (ou routeur)
     clientipl : adresse IP a apsoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_virt(const lcrzo_device device,
			   const lcrzo_etha clientetha,
			   const lcrzo_etha serveretha,
			   const lcrzo_ipl clientipl,
			   const lcrzo_ipl serveripl, 
			   lcrzo_uint16 clientport,
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*us Name : lcrzo_sock_udpcli_real2
   Description :
     Initialize a real udp client
   Input parameter(s) :
     clientipl : local IP address to select (must be a configured
                 IP address of the current computer).
		 If 0, use the default value
     serveripl : IP address of the server where to connect
     clientport : local source port number to use.
                  If 0, the computer choose for us (a value
		  greater than 1024).
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_real2
   Description :
     Initialise un client udp reel
   Parametre(s) d'entree :
     clientipl : adresse IP locale a employer (cette adresse IP
                 doit etre configuree sur la mchine).
		 Si 0, la fonction utilise l'adresse par defaut
     serveripl : adresse IP du serveur ou se connecter
     client port : port source a utiliser
                   si 0, la machine choisit un numero disponible
		   et superieur a 1024
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_real2(lcrzo_ipl    clientipl,
			    lcrzo_ipl    serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock   *psock);

/*us Name : lcrzo_sock_udpcli_virt2
   Description :
     Initialize a virtual udp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpcli_virt2
   Description :
     Initialise un client udp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresde Ethernet du serveur (ou routeur)
     clientipl : adresse IP a apsoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpcli_virt2(const lcrzo_device device,
			    const lcrzo_etha clientetha,
			    const lcrzo_etha serveretha,
			    const lcrzo_ipl clientipl,
			    const lcrzo_ipl serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*---------------------------------------------------------------*/
/*US                        TCP clients                          */
/*FR                        clients TCP                          */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_sock_tcpcli_real
   Description :
     Initialize a real tcp client
   Input parameter(s) :
     serveripl : IP address of the server where to connect
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_real
   Description :
     Initialise un client tcp reel
   Parametre(s) d'entree :
     serveripl : adresse IP du serveur ou se connecter
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_real(lcrzo_ipl    serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock   *psock);

/*us Name : lcrzo_sock_tcpcli_virt
   Description :
     Initialize a virtual tcp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_virt
   Description :
     Initialise un client tcp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresde Ethernet du serveur (ou routeur)
     clientipl : adresse IP a apsoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_virt(const lcrzo_device device,
			   const lcrzo_etha clientetha,
			   const lcrzo_etha serveretha,
			   const lcrzo_ipl clientipl,
			   const lcrzo_ipl serveripl, 
			   lcrzo_uint16 clientport,
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*us Name : lcrzo_sock_tcpcli_real2
   Description :
     Initialize a real tcp client
   Input parameter(s) :
     clientipl : local IP address to select (must be a configured
                 IP address of the current computer).
		 If 0, use the default value
     serveripl : IP address of the server where to connect
     clientport : local source port number to use.
                  If 0, the computer choose for us (a value
		  greater than 1024).
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_real2
   Description :
     Initialise un client tcp reel
   Parametre(s) d'entree :
     clientipl : adresse IP locale a employer (cette adresse IP
                 doit etre configuree sur la mchine).
		 Si 0, la fonction utilise l'adresse par defaut
     serveripl : adresse IP du serveur ou se connecter
     clientport : port source a utiliser
                  si 0, la machine choisit un numero disponible
		  et superieur a 1024
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_real2(lcrzo_ipl    clientipl,
			    lcrzo_ipl    serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock   *psock);

/*us Name : lcrzo_sock_tcpcli_virt2
   Description :
     Initialize a virtual tcp client
   Input parameter(s) :
     device : device where to sniff and to spoof
     clientetha : Ethernet address to spoof
     serveretha : Ethernet address of the server (or gateway)
     clientipl : IP address to spoof
     serveripl : IP address of the server where to connect
     clientport : local port number
     serverport : port number of the server where to connect
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpcli_virt2
   Description :
     Initialise un client tcp virtuel
   Parametre(s) d'entree :
     device : device ou l'on sniffe et spoofe
     clientetha : adresse Ethernet a spoofer
     serveretha : adresde Ethernet du serveur (ou routeur)
     clientipl : adresse IP a apsoofer
     serveripl : adresse IP du serveur ou se connecter
     clientport : numero de port local
     serverport : port du serveur ou se connecter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpcli_virt2(const lcrzo_device device,
			    const lcrzo_etha clientetha,
			    const lcrzo_etha serveretha,
			    const lcrzo_ipl clientipl,
			    const lcrzo_ipl serveripl, 
			    lcrzo_uint16 clientport,
			    lcrzo_uint16 serverport,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*---------------------------------------------------------------*/
/*US                        UDP servers                          */
/*FR                        serveurs UDP                         */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_sock_udpser_real
   Description :
     Initialize a real udp server
   Input parameter(s) :
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_real
   Description :
     Initialise un serveur udp reel
   Parametre(s) d'entree :
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_real(lcrzo_uint16 port,
			   lcrzo_sock *psock);

/*us Name : lcrzo_sock_udpser_virt
   Description :
     Initialize a virtual udp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP addres on which we listen
     serverport : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_virt
   Description :
     Initialise un serveur udp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_virt(const lcrzo_device device,
			   const lcrzo_etha serveretha,
			   const lcrzo_ipl serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*us Name : lcrzo_sock_udpser_real2
   Description :
     Initialize a real udp server
   Input parameter(s) :
     ipl : listening IP address 
     port : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_real2
   Description :
     Initialise un serveur udp reel
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_real2(lcrzo_ipl ipl,
			    lcrzo_uint16 port,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);

/*us Name : lcrzo_sock_udpser_virt2
   Description :
     Initialize a virtual udp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP addres on which we listen
     serverport : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpser_virt2
   Description :
     Initialise un serveur udp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpser_virt2(const lcrzo_device device,
			    const lcrzo_etha serveretha,
			    const lcrzo_ipl serveripl, 
			    lcrzo_uint16 serverport,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*---------------------------------------------------------------*/
/*US                  UDP multiclients servers                   */
/*FR                  serveurs UDP multiclients                  */
/*---------------------------------------------------------------*/

/*US************************************************************
 * Note about UDP multiclients servers :                       *
 * We can only use them to read data sent by clients.          *
 * It's not possible to write data to a client.                *
 ***************************************************************/
/*FR************************************************************
 * Note sur les serveurs UDP multiclients :                    *
 * On ne peut que lire les donnees envoyees par les clients.   *
 * On ne peut pas ecrire des donnees vers les clients.         *
 ***************************************************************/

/*us Name : lcrzo_sock_udpmulser_real
   Description :
     Initialize a real multiclient udp server
   Input parameter(s) :
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpmulser_real
   Description :
     Initialise un serveur udp reel multiclient
   Parametre(s) d'entree :
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpmulser_real(lcrzo_uint16 port,
			      lcrzo_sock *psock);

/*us Name : lcrzo_sock_udpmulser_virt
   Description :
     Initialize a virtual udp multiclient server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP addres on which we listen
     serverport : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpmulser_virt
   Description :
     Initialise un serveur udp virtuel multiclients
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpmulser_virt(const lcrzo_device device,
			      const lcrzo_etha serveretha,
			      const lcrzo_ipl serveripl, 
			      lcrzo_uint16 serverport,
			      lcrzo_sock *psock);

/*us Name : lcrzo_sock_udpmulser_real2
   Description :
     Initialize a real multiclient udp server
   Input parameter(s) :
     ipl : listening IP address
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_udpmulser_real2
   Description :
     Initialise un serveur udp reel multiclient
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_udpmulser_real2(lcrzo_ipl ipl,
			       lcrzo_uint16 port,
			       lcrzo_sock *psock);


/*---------------------------------------------------------------*/
/*US                        TCP servers                          */
/*FR                        serveurs TCP                         */
/*---------------------------------------------------------------*/

/*US************************************************************
 * Note about TCP servers :                                    *
 * We return from the function only after a client has         *
 * connected.                                                  *
 ***************************************************************/
/*FR************************************************************
 * Note sur les serveurs TCP :                                 *
 * On ne quittte la fonction que lorsqu'un client s'est        *
 * connecte.                                                   *
 ***************************************************************/

/*us Name : lcrzo_sock_tcpser_real
   Description :
     Initialize a real tcp server
   Input parameter(s) :
     port : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_real
   Description :
     Initialise un serveur tcp reel
   Parametre(s) d'entree :
     port : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_real(lcrzo_uint16 port,
			   lcrzo_sock *psock);

/*us Name : lcrzo_sock_tcpser_virt
   Description :
     Initialize a virtual tcp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP addres on which we listen
     serverport : listening port
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_virt
   Description :
     Initialise un serveur tcp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_virt(const lcrzo_device device,
			   const lcrzo_etha serveretha,
			   const lcrzo_ipl serveripl, 
			   lcrzo_uint16 serverport,
			   lcrzo_sock *psock);

/*us Name : lcrzo_sock_tcpser_real2
   Description :
     Initialize a real tcp server
   Input parameter(s) :
     ipl : listennig IP address
     port : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_real2
   Description :
     Initialise un serveur tcp reel
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_real2(lcrzo_ipl ipl,
			    lcrzo_uint16 port,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);

/*us Name : lcrzo_sock_tcpser_virt2
   Description :
     Initialize a virtual tcp server
   Input parameter(s) :
     device : device where to sniff and spoof
     serveretha : spoofed Ethernet address on which we listen
     serveripl : spoofed IP addres on which we listen
     serverport : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
   Input/output parameter(s) :
   Output parameter(s) :
     *psock : lcrzo_sock initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_virt2
   Description :
     Initialise un serveur tcp virtuel
   Parametre(s) d'entree :
     device : device sur lequel sniffer et spoofer
     serveretha : adresse Ethernet spoofee sur laquelle on ecoute
     serveripl : adresse IP spoofee sur laquelle on ecoute
     serverport : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psock : lcrzo_sock initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpser_virt2(const lcrzo_device device,
			    const lcrzo_etha serveretha,
			    const lcrzo_ipl serveripl, 
			    lcrzo_uint16 serverport,
			    const lcrzo_ipopt ipopt,
			    lcrzo_uint8 ipoptsize,
			    lcrzo_sock *psock);


/*---------------------------------------------------------------*/
/*US                  TCP multiclients servers                   */
/*FR                  serveusr TCP multiclients                  */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_sock_tcpmulser_real
   Description :
     Deal with a real tcp multiclient server
   Input parameter(s) :
     port : listening port
     pfunc : memory address of the function which will be called
             for each client. For
             each call, the first parameter ('sock') will be set
             with socket, and the second parameter ('pinfos')
             will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_real
   Description :
     Gere un serveur tcp reel multiclient
   Parametre(s) d'entree :
     port : port sur lequel ecouter
     pfunc : adresse memoire de la fonction appelee pour chaque
             client. Lors de
             chaque appel, le premier parametre ('sock') est 
             affecte avec le socket courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpmulser_real(lcrzo_uint16 port,
			      int (*pfunc)(lcrzo_sock sock,
					   const void *pinfos),
			      const void *pinfos);

/*us Name : lcrzo_sock_tcpmulser_real2
   Description :
     Deal with a real tcp multiclient server
   Input parameter(s) :
     ipl : listening IP address
     port : listening port
     ipopt : IP options to add
     ipoptsize : size of ipopt
     pfunc : memory address of the function which will be called
             for each client. For
             each call, the first parameter ('sock') will be set
             with socket, and the second parameter ('pinfos')
             will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_tcpser_real2
   Description :
     Gere un serveur tcp reel multiclient
   Parametre(s) d'entree :
     ipl : adresse IP sur laquelle ecouter
     port : port sur lequel ecouter
     ipopt : options IP a employer
     ipoptsize : taille de ipopt
     pfunc : adresse memoire de la fonction appelee pour chaque
             client. Lors de
             chaque appel, le premier parametre ('sock') est 
             affecte avec le socket courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_tcpmulser_real2(lcrzo_ipl ipl,
			       lcrzo_uint16 port,
			       const lcrzo_ipopt ipopt,
			       lcrzo_uint8 ipoptsize,
			       int (*pfunc)(lcrzo_sock sock,
					    const void *pinfos),
			       const void *pinfos);


/*---------------------------------------------------------------*/
/*US                        IP options                           */
/*FR                        options IP                           */
/*---------------------------------------------------------------*/

/*us the folowing functions add IP options to a connection.
   Read help of the lcrzo_ipopt module for an explanation of
   each parameter */
/*fr les fonctions suivantes ajoutent des options IP a une
   connexion. Consultez l'aide du module lcrzo_ipopt pour 
   connaitre la signification des parametres*/

/* no operation */
int lcrzo_sock_ipopt_noop(lcrzo_sock *psock);

/* loose source routing */
int lcrzo_sock_ipopt_lsrr(lcrzo_sock *psock,
			  lcrzo_ipl ipl1,
			  lcrzo_ipl ipl2,
			  lcrzo_ipl ipl3,
			  lcrzo_ipl ipl4,
			  lcrzo_ipl ipl5,
			  lcrzo_ipl ipl6,
			  lcrzo_ipl ipl7,
			  lcrzo_ipl ipl8,
			  lcrzo_ipl ipl9,
			  lcrzo_int8 numberofsetipl);

/* strict source routing */
int lcrzo_sock_ipopt_ssrr(lcrzo_sock *psock,
			  lcrzo_ipl ipl1,
			  lcrzo_ipl ipl2,
			  lcrzo_ipl ipl3,
			  lcrzo_ipl ipl4,
			  lcrzo_ipl ipl5,
			  lcrzo_ipl ipl6,
			  lcrzo_ipl ipl7,
			  lcrzo_ipl ipl8,
			  lcrzo_ipl ipl9,
			  lcrzo_int8 numberofsetipl);

/* record route */
int lcrzo_sock_ipopt_rr(lcrzo_sock *psock,
			lcrzo_int8 numberofroutestosave);

/* timestamp */
int lcrzo_sock_ipopt_time(lcrzo_sock *psock,
			  lcrzo_int8 numberoftimestamptosave);
int lcrzo_sock_ipopt_timeip(lcrzo_sock *psock,
			    lcrzo_int8 numberoftimestamptosave);
int lcrzo_sock_ipopt_timeipp(lcrzo_sock *psock,
			     lcrzo_ipl ipl1,
			     lcrzo_ipl ipl2,
			     lcrzo_ipl ipl3,
			     lcrzo_ipl ipl4,
			     lcrzo_int8 numberofsetipl);


/*---------------------------------------------------------------*/
/*US                     reading and writing                     */
/*FR                     lecture et ecriture                     */
/*---------------------------------------------------------------*/
/*us Name : lcrzo_sock_write
   Description :
     Write data to a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock where the data is written
     data : data to write
     datasize : size of data
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_write
   Description :
     Ecrit des donnees dans un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont ecrites
     data : donnees a ecrire
     datasize : taille de data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_write(lcrzo_sock sock,
		     lcrzo_constdata data, 
		     lcrzo_int32 datasize);

/*us Name : lcrzo_sock_read
   Description :
     Read data from a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_sock_read
   Description :
     Lit des donnees depuis un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_sock_read(lcrzo_sock *psock,
		    lcrzo_bool beblocking,
		    lcrzo_int32 dataoutmaxsize,
		    lcrzo_data dataout,
		    lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_sock_readm
   Description :
     Read data from a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_sock_readm
   Description :
     Lit des donnees depuis un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_readm(lcrzo_sock *psock,
		     lcrzo_bool beblocking,
		     lcrzo_data *pdataout,
		     lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_sock_read_fixed
   Description :
     Read a fixed size data from a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     beblocking : if we want to block until receiving data
     datasizetoread : size to read
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_sock_read_fixed
   Description :
     Lit des donnees de taille constante depuis un lcrzo_sock.
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     datasizetoread : taille des donnees a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_read_fixed(lcrzo_sock *psock,
			  lcrzo_bool beblocking,
			  lcrzo_int32 datasizetoread,
			  lcrzo_data dataout);

/*us Name : lcrzo_sock_readm_fixed
   Description :
     Read a fixed size data from a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     beblocking : if we want to block until receiving data
     datasizetoread : size to read
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_sock_readm_fixed
   Description :
     Lit des donnees de taille constante depuis un lcrzo_sock.
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     datasizetoread : taille des donnees a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_readm_fixed(lcrzo_sock *psock,
			   lcrzo_bool beblocking,
			   lcrzo_int32 datasizetoread,
			   lcrzo_data *pdataout);

/*us Name : lcrzo_sock_read_line
   Description :
     Read data from a lcrzo_sock. We stop at first '\r' or '\n' (which
     is not stored in data).
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_sock_read_line
   Description :
     Lit des donnees depuis un lcrzo_sock. On s'arrete de lire
     au premier '\r' ou '\n' (qui n'est pas stocke dans
     *pdataout).
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_sock_read_line(lcrzo_sock *psock,
			 lcrzo_bool beblocking,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_sock_readm_line
   Description :
     Read data from a lcrzo_sock. We stop at first '\r' or '\n' (which
     is not stored in data).
   Input parameter(s) :
     sock : lcrzo_sock from where the datas are read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_sock_readm_line
   Description :
     Lit des donnees depuis un lcrzo_sock. On s'arrete de lire
     au premier '\r' ou '\n' (qui n'est pas stocke dans
     *pdataout).
   Parametre(s) d'entree :
     sock : lcrzo_sock ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_sock_readm_line(lcrzo_sock *psock,
			  lcrzo_bool beblocking,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*---------------------------------------------------------------*/
/*US                          closing                            */
/*FR                         fermeture                           */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_sock_close
   Description :
     Close a lcrzo_sock.
   Input parameter(s) :
     sock : lcrzo_sock to close
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_close
   Description :
     Ferme un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock a fermer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_close(lcrzo_sock sock);

/*---------------------------------------------------------------*/
/*                          informations                         */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_sock_get_infos
   Description :
     Get informations about a lcrzo_sock
   Input parameter(s) :
     sock : lcrzo_sock where to get info
   Input/output parameter(s) :
   Output parameter(s) :
     here_device : local device
     here_etha : local Ethernet address
     *phere_ipl : local IP address
     *phere_port : local port
     there_etha : remote Ethernet address
     *pthere_ipl : remote IP address
     *pthere_port : remote port
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_sock_get_infos
   Description :
     Obtient les informations d'un lcrzo_sock
   Parametre(s) d'entree :
     sock : lcrzo_sock sur lequel obtenir des informations
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     here_device : device local
     here_etha : adresse Ethernet locale
     *phere_ipl : adresse IP lcoale
     *phere_port : port local
     there_etha : adresse Ethernet distante
     *pthere_ipl : adresse IP distante
     *pthere_port : port distant
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_sock_get_infos(lcrzo_sock sock,
			 lcrzo_device here_device,
			 lcrzo_etha here_etha,
			 lcrzo_ipl *phere_ipl,
			 lcrzo_uint16 *phere_port,
			 lcrzo_etha there_etha,
			 lcrzo_ipl *pthere_ipl,
			 lcrzo_uint16 *pthere_port);

/*us print those informations */
/*fr ecrit ces informations */
int lcrzo_sock_fprint_infos(LCRZOFILE *pf,
			    lcrzo_sock sock);
/*us idem, but on stdout */
/*fr idem, mais sur stdout */
#define lcrzo_sock_print_infos(a) lcrzo_sock_fprint_infos(stdout, a)


/*---------------------------------------------------------------*/
/*US                     virtual answering                       */
/*FR                    reponses virtuelles                      */
/*---------------------------------------------------------------*/

/*us Name : lcrzo_virtual_answer_arp
   Description :
     Answer to an ARP request.
     If ethpacket contains an ARP request for ipl, this
     function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address of ipl
     ipl : IP address searched
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_arp
   Description :
     Repond a une requete ARP
     Si ethpacket contient une requete ARP pour ipl, cette 
     fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet de ipl
     ipl : adresse IP recherchee
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_arp(lcrzo_constdata ethpacket,
			     lcrzo_uint32 ethpacketsize,
			     const lcrzo_device device,
			     const lcrzo_etha etha,
			     const lcrzo_ipl ipl);

/*us Name : lcrzo_virtual_answer_rarp
   Description :
     Answer to a RARP request.
     If ethpacket contains a RARP request for etha, this
     function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address searched
     ipl : IP address of etha
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_rarp
   Description :
     Repond a une requete RARP
     Si ethpacket contient une requete RARP pour etha, cette 
     fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet recherchee
     ipl : adresse IP de etha
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_rarp(lcrzo_constdata ethpacket,
			      lcrzo_uint32 ethpacketsize,
			      const lcrzo_device device,
			      const lcrzo_etha etha,
			      const lcrzo_ipl ipl);

/*us Name : lcrzo_virtual_answer_ping
   Description :
     Answer to a ping request (icmp echo).
     If ethpacket contains a ping request for ipl, this
     function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address of ipl
     ipl : IP address searched
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_ping
   Description :
     Repond a une requete ping (icmp echo).
     Si ethpacket contient une requete ping pour ipl, cette 
     fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet de ipl
     ipl : adresse IP recherchee
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_ping(lcrzo_constdata ethpacket,
			      lcrzo_uint32 ethpacketsize,
			      const lcrzo_device device,
			      const lcrzo_etha etha,
			      const lcrzo_ipl ipl);

/*us Name : lcrzo_virtual_answer_alive
   Description :
     Answer to a arp/rarp/ping requests
     If ethpacket contains an arp/rarp/ping request
     for ipl, this function spoofs an answer on device.
   Input parameter(s) :
     ethpacket : packet to analyse
     ethpacketsize : size of ethpacket
     device : device where to spoof answer
     etha : Ethernet address of ipl
     ipl : IP address searched
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_virtual_answer_alive
   Description :
     Repond a une requete arp/rarpping.
     Si ethpacket contient une requete arp/rarp/ping
     pour ipl, cette fonction spoofe une reponse
   Parametre(s) d'entree :
     ethpacket : paquet a analyser
     ethpacketsize : taille de ethpacket
     device : device sur lequel spoofer les reponses
     etha : adresse Ethernet de ipl
     ipl : adresse IP recherchee
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_virtual_answer_alive(lcrzo_constdata ethpacket,
			       lcrzo_uint32 ethpacketsize,
			       const lcrzo_device device,
			       const lcrzo_etha etha,
			       const lcrzo_ipl ipl);
