/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdio.h>
 #include <stdlib.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #define __EXTENSIONS__
 #include <stdlib.h>
 #include <stdio.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*-------------------------------------------------------------*/
typedef struct {
  lcrzo_list list_devices;
  lcrzo_list list_arpperm;
  lcrzo_list list_arpcache;
  lcrzo_list list_routes;
  lcrzo_bool allowupdate;
  lcrzo_bool forceupdate;
  lcrzo_uint32 permentriestimeout;
  lcrzo_uint32 cacheentriestimeout;
} lcrzo_conf;
lcrzo_conf lcrzo_priv_conf;

/*-------------------------------------------------------------*/
typedef struct {
  lcrzo_device device;
  lcrzo_etha eth;
  lcrzo_ipl ip;
  lcrzo_ipl netmask;
  lcrzo_uint32 mtu;
  lcrzo_bool isup;
  lcrzo_bool isanalias;
} lcrzo_priv_conf_devices;

typedef struct {
  lcrzo_device device;
  lcrzo_etha eth;
  lcrzo_ipl ip;
  lcrzo_bool ispermanent;
  lcrzo_uint32 timeset;
} lcrzo_priv_conf_arp;

typedef struct {
  lcrzo_device device;
  lcrzo_ipl dest;
  lcrzo_ipl mask;
  lcrzo_ipl ipsource;
  lcrzo_ipl gateway;
  lcrzo_bool isup;
} lcrzo_priv_conf_routes;

/*-------------------------------------------------------------*/
int lcrzo_priv_conf_devices_cmp(const void *pelem, const void *pinfos);
int lcrzo_priv_conf_devices_cmp(const void *pelem, const void *pinfos)
{ const lcrzo_priv_conf_devices de1=*(const lcrzo_priv_conf_devices *)pelem;
  const lcrzo_priv_conf_devices de2=*(const lcrzo_priv_conf_devices *)pinfos;

  if ( lcrzo_device_equal(de1.device, de2.device) &&
       lcrzo_etha_equal(de1.eth, de2.eth) &&
       lcrzo_ipl_equal(de1.ip, de2.ip) )
  { return(1);
  }
  return(0);
}

int lcrzo_priv_conf_arp_cmp(const void *pelem, const void *pinfos);
int lcrzo_priv_conf_arp_cmp(const void *pelem, const void *pinfos)
{ const lcrzo_priv_conf_arp de1=*(const lcrzo_priv_conf_arp *)pelem;
  const lcrzo_priv_conf_arp de2=*(const lcrzo_priv_conf_arp *)pinfos;

  if ( lcrzo_device_equal(de1.device, de2.device) &&
       lcrzo_etha_equal(de1.eth, de2.eth) &&
       lcrzo_ipl_equal(de1.ip, de2.ip) )
  { return(1);
  }
  return(0);
}
int lcrzo_priv_conf_arp_cmpwdev(const void *pelem, const void *pinfos);
int lcrzo_priv_conf_arp_cmpwdev(const void *pelem, const void *pinfos)
{ const lcrzo_priv_conf_arp de1=*(const lcrzo_priv_conf_arp *)pelem;
  const lcrzo_priv_conf_arp de2=*(const lcrzo_priv_conf_arp *)pinfos;

  if ( lcrzo_etha_equal(de1.eth, de2.eth) &&
       lcrzo_ipl_equal(de1.ip, de2.ip) )
  { return(1);
  }
  return(0);
}
int lcrzo_priv_conf_arp_purge(const void *pelem, const void *pinfos);
int lcrzo_priv_conf_arp_purge(const void *pelem, const void *pinfos)
{ const lcrzo_priv_conf_arp de=*(const lcrzo_priv_conf_arp *)pelem;
  const lcrzo_uint32 purgeolderthan=*(const lcrzo_uint32 *)pinfos;

  if ( de.timeset<purgeolderthan )
  { return(1);
  }
  return(0);
}

int lcrzo_priv_conf_routes_cmp(const void *pelem, const void *pinfos);
int lcrzo_priv_conf_routes_cmp(const void *pelem, const void *pinfos)
{ const lcrzo_priv_conf_routes de1=*(const lcrzo_priv_conf_routes *)pelem;
  const lcrzo_priv_conf_routes de2=*(const lcrzo_priv_conf_routes *)pinfos;

  if ( lcrzo_ipl_equal(de1.dest, de2.dest) &&
       lcrzo_ipl_equal(de1.mask, de2.mask) )
  { return(1);
  }
  return(0);
}

int lcrzo_priv_conf_routes_sortcmp(const void *peleminf,
				   const void *pelemsup,
				   const void *pinfos);
int lcrzo_priv_conf_routes_sortcmp(const void *peleminf,
				   const void *pelemsup,
				   const void *pinfos)
{ const lcrzo_priv_conf_routes de1=*(const lcrzo_priv_conf_routes *)peleminf;
  const lcrzo_priv_conf_routes de2=*(const lcrzo_priv_conf_routes *)pelemsup;
  lcrzo_uint32 net1, net2;

  net1= ~de1.mask;
  net2= ~de2.mask;

  if ( net1==net2 ) return(0);
  else if ( net1>net2 ) return(+1);
  else return(-1);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_conf_eventuallyupdate(void);
int lcrzo_priv_conf_eventuallyinit(void);

/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
int lcrzo_conf_fprint(LCRZOFILE *pf)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());
  lcrzo_priv_conf.allowupdate=LCRZO_FALSE;
  fprintf(pf, "Devices\n");
  lcrzo_er(lcrzo_conf_devices_fprint(pf));
  fprintf(pf, "Arp\n");
  lcrzo_er(lcrzo_conf_arp_fprint(pf));
  fprintf(pf, "Routes\n");
  lcrzo_er(lcrzo_conf_routes_fprint(pf));
  lcrzo_priv_conf.allowupdate=LCRZO_TRUE;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_set_allowupdate(lcrzo_bool yes)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_bool_verifbof(yes);
  lcrzo_priv_conf.allowupdate=yes;
  return(LCRZO_ERR_OK);  
}

/*-------------------------------------------------------------*/
int lcrzo_conf_set_forceupdate(lcrzo_bool yes)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_bool_verifbof(yes);
  lcrzo_priv_conf.forceupdate=yes;
  return(LCRZO_ERR_OK);  
}

/*-------------------------------------------------------------*/
int lcrzo_conf_set_permentriestimeout(lcrzo_uint32 timeout)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_priv_conf.permentriestimeout=timeout;
  return(LCRZO_ERR_OK);  
}

/*-------------------------------------------------------------*/
int lcrzo_conf_set_cacheentriestimeout(lcrzo_uint32 timeout)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_priv_conf.cacheentriestimeout=timeout;
  return(LCRZO_ERR_OK);  
}

/*-------------------------------------------------------------*/
int lcrzo_conf_get_allowupdate(lcrzo_bool *pyes)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  if (pyes!=NULL)
  { *pyes=lcrzo_priv_conf.allowupdate;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_get_forceupdate(lcrzo_bool *pyes)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  if (pyes!=NULL)
  { *pyes=lcrzo_priv_conf.forceupdate;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_get_permentriestimeout(lcrzo_uint32 *ptimeout)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  if (ptimeout!=NULL)
  { *ptimeout=lcrzo_priv_conf.permentriestimeout;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_get_cacheentriestimeout(lcrzo_uint32 *ptimeout)
{ lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  if (ptimeout!=NULL)
  { *ptimeout=lcrzo_priv_conf.cacheentriestimeout;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_devices_add(const lcrzo_device device,
			   const lcrzo_etha etha,
			   lcrzo_ipl ipl,
			   lcrzo_ipl netmask,
			   lcrzo_uint32 mtu,
			   lcrzo_bool isup,
			   lcrzo_bool isanalias)
{ lcrzo_priv_conf_devices cd;
  lcrzo_int32 pos;
  int ret;
 
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  /* initialize the structure */
  lcrzo_er(lcrzo_device_init(device, cd.device));
  memcpy(cd.eth, etha, LCRZO_ETHA_MAXBYTES);
  cd.ip=ipl;
  cd.netmask=netmask;
  cd.mtu=mtu;
  cd.isup=isup;
  cd.isanalias=isanalias;

  /* search it in the list */
  ret=lcrzo_list_search_all(lcrzo_priv_conf.list_devices,
			    &lcrzo_priv_conf_devices_cmp,
			    &cd, &pos, NULL);
  if ( ret!=LCRZO_ERR_OKSEARCHNOTFOUND && ret!=LCRZO_ERR_OK )
    return(ret);

  if ( ret==LCRZO_ERR_OK )
  { /* we found it, so we replace it */
    lcrzo_er(lcrzo_list_replace_pos(&lcrzo_priv_conf.list_devices, pos, &cd));
  }
  else
  { /* we add it to the list */
    lcrzo_er(lcrzo_list_add_last(&lcrzo_priv_conf.list_devices, &cd));
  }
  /* add it in the ARP cache */
  lcrzo_er(lcrzo_conf_arp_add(device, etha, ipl, LCRZO_TRUE));
  /* add it in the routing table */
  lcrzo_er(lcrzo_conf_routes_add(device, ipl, 0xFFFFFFFFu, 0, 0, isup));
  lcrzo_er(lcrzo_conf_routes_add(device, ipl&netmask, netmask, ipl, 0, isup));
  
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_devices_del(const lcrzo_device device,
			   const lcrzo_etha etha,
			   lcrzo_ipl ipl)
{ lcrzo_priv_conf_devices cd;
 
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  
  /* initialize the structure */
  lcrzo_er(lcrzo_string_init_coretext(device, LCRZO_DEVICE_MAXBYTES, 
				      cd.device));
  memcpy(cd.eth, etha, LCRZO_ETHA_MAXBYTES);
  cd.ip=ipl;

  /* remove it from the list */
  lcrzo_er(lcrzo_list_remove_criteria_all(&lcrzo_priv_conf.list_devices, 
					  &lcrzo_priv_conf_devices_cmp,
					  &cd));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_devices_count(lcrzo_int32 *pcount)
{ 
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());

  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_devices, pcount));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_devices_value_pos(lcrzo_int32 pos,
				 lcrzo_device device,
				 lcrzo_etha etha,
				 lcrzo_ipl *pipl,
				 lcrzo_ipl *pnetmask,
				 lcrzo_uint32 *pmtu,
				 lcrzo_bool *pisup,
				 lcrzo_bool *pisanalias)
{ lcrzo_priv_conf_devices cd;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_devices, pos, &cd));
  lcrzo_er(lcrzo_device_init(cd.device, device));
  if ( etha!=NULL ) memcpy(etha, cd.eth, LCRZO_ETHA_MAXBYTES);
  if ( pipl!=NULL ) *pipl=cd.ip;
  if ( pnetmask!=NULL ) *pnetmask=cd.netmask;
  if ( pmtu!=NULL ) *pmtu=cd.mtu;
  if ( pisup!=NULL ) *pisup=cd.isup;
  if ( pisanalias!=NULL ) *pisanalias=cd.isanalias;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fprint_cstsized(LCRZOFILE *pf, 
			       const char *string, lcrzo_int32 size);
int lcrzo_priv_fprint_cstsized(LCRZOFILE *pf, 
			       const char *string, lcrzo_int32 size)
{ int spacenumber, i;

  fprintf(pf, "%s", string);
  spacenumber=size-strlen(string);
  if ( spacenumber<=0 ) spacenumber=1;
  for ( i=0 ; i<spacenumber ; i++ )
  { fputc(' ', pf);
  }
  return(LCRZO_ERR_OK);
}

int lcrzo_conf_devices_fprint(LCRZOFILE *pf)
{ lcrzo_int32 count, i;
  lcrzo_priv_conf_devices cd;
  lcrzo_ips ips;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());

  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_devices, &count));
  if (count)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG )
    { fprintf(pf, "  device  ethernet           ip              / netmask");
      fprintf(pf, "           mtu\n");
    }
    else
    { fprintf(pf, "  device  ethernet           ip              / netmask");
      fprintf(pf, "           mtu\n");
    }
  }
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_devices, i, &cd));
    fprintf(pf, "  ");
    lcrzo_priv_fprint_cstsized(pf, cd.device, 8);
    lcrzo_er(lcrzo_etha_fprint(pf, "", cd.eth, "  "));
    lcrzo_er(lcrzo_ips_init_ipl(cd.ip, ips));
    lcrzo_priv_fprint_cstsized(pf, ips, 16);
    fprintf(pf, "/ ");
    lcrzo_er(lcrzo_ips_init_ipl(cd.netmask, ips));
    lcrzo_priv_fprint_cstsized(pf, ips, 16);
    fprintf(pf, " %5lu  ", cd.mtu);
    if (cd.isup)
      fprintf(pf, "up");
    else
      fprintf(pf, "down");
    if (cd.isanalias)
      fprintf(pf, ", alias");
    fprintf(pf, "\n");
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_arp_add(const lcrzo_device device,
		       const lcrzo_etha etha,
		       lcrzo_ipl ipl,
		       lcrzo_bool ispermanent)
{ lcrzo_priv_conf_arp ca, permfoundca, cachefoundca;
  lcrzo_time currenttime;
  lcrzo_int32 permpos, cachepos;
  int permret, cacheret;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  /* initialize the structure */
  memcpy(ca.eth, etha, LCRZO_ETHA_MAXBYTES);
  lcrzo_er(lcrzo_device_init(device, ca.device));
  ca.ip=ipl;
  ca.ispermanent=ispermanent;
  lcrzo_er(lcrzo_time_init(&currenttime));
  ca.timeset=currenttime.sec;

  /* search it in the lists */
  permret=lcrzo_list_search_all(lcrzo_priv_conf.list_arpperm,
				&lcrzo_priv_conf_arp_cmpwdev,
				&ca, &permpos, &permfoundca);
  if ( permret!=LCRZO_ERR_OKSEARCHNOTFOUND && permret!=LCRZO_ERR_OK )
    return(permret);
  cacheret=lcrzo_list_search_all(lcrzo_priv_conf.list_arpcache,
				 &lcrzo_priv_conf_arp_cmpwdev,
				 &ca, &cachepos, &cachefoundca);
  if ( cacheret!=LCRZO_ERR_OKSEARCHNOTFOUND && cacheret!=LCRZO_ERR_OK )
    return(cacheret);

  /* if the user doesn't know the device, we look if we already know it */
  if ( strlen(ca.device)==0 )
  { if ( permret==LCRZO_ERR_OK && strlen(permfoundca.device)!=0 )
    { lcrzo_er(lcrzo_device_init(permfoundca.device, ca.device));
    }
    else if ( cacheret==LCRZO_ERR_OK && strlen(cachefoundca.device)!=0 )
    { lcrzo_er(lcrzo_device_init(cachefoundca.device, ca.device));
    }
  }

  if ( ispermanent )
  { if ( permret==LCRZO_ERR_OK ) 
    { /* we found it, so we replace it */
      lcrzo_er(lcrzo_list_replace_pos(&lcrzo_priv_conf.list_arpperm, 
				      permpos, &ca));
    }
    else
    { /* we add it to the list */
      lcrzo_er(lcrzo_list_add_last(&lcrzo_priv_conf.list_arpperm, &ca));
    }
    if ( cacheret==LCRZO_ERR_OK ) 
    { /* we remove it from the cache */
      lcrzo_er(lcrzo_list_remove_criteria_all(&lcrzo_priv_conf.list_arpcache, 
					      &lcrzo_priv_conf_arp_cmpwdev,
					      &ca));
    }
  }
  else
  { if ( cacheret==LCRZO_ERR_OK ) 
    { /* we found it, so we replace it */
      lcrzo_er(lcrzo_list_replace_pos(&lcrzo_priv_conf.list_arpcache,
				      cachepos, &ca));
    }
    else
    { /* we add it to the list */
      lcrzo_er(lcrzo_list_add_last(&lcrzo_priv_conf.list_arpcache, &ca));
    }
  }


  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_arp_del(const lcrzo_device device,
		       const lcrzo_etha etha,
		       lcrzo_ipl ipl)
{ lcrzo_priv_conf_arp ca;
 
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  /* initialize the structure */
  lcrzo_er(lcrzo_device_init(device, ca.device));
  memcpy(ca.eth, etha, LCRZO_ETHA_MAXBYTES);
  ca.ip=ipl;

  /* remove it from the list */
  lcrzo_er(lcrzo_list_remove_criteria_all(&lcrzo_priv_conf.list_arpperm, 
					  &lcrzo_priv_conf_arp_cmp,
					  &ca));
  lcrzo_er(lcrzo_list_remove_criteria_all(&lcrzo_priv_conf.list_arpcache, 
					  &lcrzo_priv_conf_arp_cmp,
					  &ca));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_arp_count(lcrzo_int32 *pcount)
{ lcrzo_int32 count;
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());

  if ( pcount!=NULL )
  { lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_arpperm, &count));
    *pcount = count;
    lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_arpcache, &count));
    *pcount += count;
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_arp_value_pos(lcrzo_int32 pos,
			     lcrzo_device device,
			     lcrzo_etha etha,
			     lcrzo_ipl *pipl,
			     lcrzo_bool *pispermanent,
			     lcrzo_uint32 *ptimeset)
{ lcrzo_priv_conf_arp ca;
  lcrzo_int32 permcount, cachecount;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  /* calculate the position */
  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_arpperm, &permcount));
  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_arpcache, &cachecount));
  if ( pos<0 )
  { pos=permcount+cachecount+1+pos;
    if ( pos<=0 ) return(LCRZO_ERR_PATOOLOW);
  }
 
  /* get the value from the correct list */
  if ( pos<=permcount )
  { lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_arpperm, pos, &ca));
  }
  else
  { pos-=permcount;
    lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_arpcache, pos, &ca));
  }
 
  /* set the output parameters */
  lcrzo_er(lcrzo_device_init(ca.device, device));
  if ( etha!=NULL ) memcpy(etha, ca.eth, LCRZO_ETHA_MAXBYTES);
  if ( pipl!=NULL ) *pipl=ca.ip;
  if ( pispermanent!=NULL ) *pispermanent=ca.ispermanent;
  if ( ptimeset!=NULL ) *ptimeset=ca.timeset;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_arp_fprint(LCRZOFILE *pf)
{ lcrzo_int32 count, i;
  lcrzo_time currenttime;
  lcrzo_ips ips;
  lcrzo_device device;
  lcrzo_etha etha;
  lcrzo_ipl ipl;
  lcrzo_bool ispermanent;
  lcrzo_uint32 timeset;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  lcrzo_er(lcrzo_conf_arp_count(&count));
  lcrzo_er(lcrzo_time_init(&currenttime));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_arp_value_pos(i, device, etha, &ipl,
				      &ispermanent, &timeset));
    fprintf(pf, "  ");
    lcrzo_priv_fprint_cstsized(pf, device, 8);
    lcrzo_er(lcrzo_etha_fprint(pf, "", etha, "  "));
    lcrzo_er(lcrzo_ips_init_ipl(ipl, ips));
    lcrzo_priv_fprint_cstsized(pf, ips, 16);
    if ( ispermanent ) fprintf(pf, "(permanent)\n");
    else fprintf(pf, "(-%lds)\n", currenttime.sec - timeset);
  }

  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_conf_routes_add(const lcrzo_device device,
			  lcrzo_ipl dest,
			  lcrzo_ipl mask,
			  lcrzo_ipl ipsource,
			  lcrzo_ipl gateway,
			  lcrzo_bool isup)
{ lcrzo_priv_conf_routes cr;
  lcrzo_int32 pos;
  int ret;

  lcrzo_bool_verifbof(isup);
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  /* initialize the structure */
  lcrzo_er(lcrzo_device_init(device, cr.device));
  cr.dest=dest;
  cr.mask=mask;
  cr.ipsource=ipsource;
  cr.gateway=gateway;
  cr.isup=isup;

  /* search it in the list */
  ret=lcrzo_list_search_all(lcrzo_priv_conf.list_routes,
			    &lcrzo_priv_conf_routes_cmp,
			    &cr, &pos, NULL);
  if ( ret!=LCRZO_ERR_OKSEARCHNOTFOUND && ret!=LCRZO_ERR_OK )
    return(ret);

  if ( ret==LCRZO_ERR_OK ) 
  { /* we found it, so we replace it */
    lcrzo_er(lcrzo_list_replace_pos(&lcrzo_priv_conf.list_routes, pos, &cr));
  }
  else
  { /* we add it to the list */
    lcrzo_er(lcrzo_list_add_last(&lcrzo_priv_conf.list_routes, &cr));
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_routes_del(lcrzo_ipl dest,
			  lcrzo_ipl mask)
{ lcrzo_priv_conf_routes cr;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  /* initialize the structure */
  cr.dest=dest;
  cr.mask=mask;

  /* remove it from the list */
  lcrzo_er(lcrzo_list_remove_criteria_all(&lcrzo_priv_conf.list_routes, 
					  &lcrzo_priv_conf_routes_cmp,
					  &cr));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_routes_count(lcrzo_int32 *pcount)
{
  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());

  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_routes, pcount));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_routes_value_pos(lcrzo_int32 pos,
				lcrzo_device device,
				lcrzo_ipl *pdest,
				lcrzo_ipl *pmask,
				lcrzo_ipl *pipsource,
				lcrzo_ipl *pgateway,
				lcrzo_bool *pisup)
{ lcrzo_priv_conf_routes cr;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());

  lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_routes, pos, &cr));
  lcrzo_er(lcrzo_device_init(cr.device, device));
  if ( pdest!=NULL ) *pdest=cr.dest;
  if ( pmask!=NULL ) *pmask=cr.mask;
  if ( pipsource!=NULL ) *pipsource=cr.ipsource;
  if ( pgateway!=NULL ) *pgateway=cr.gateway;
  if ( pisup!=NULL ) *pisup=cr.isup;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_conf_routes_fprint(LCRZOFILE *pf)
{ lcrzo_int32 count, i;
  lcrzo_priv_conf_routes cr;
  lcrzo_ips ips;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());

  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_routes, &count));
  if (count)
  { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG )
    { fprintf(pf, "  device  destination     / netmask");
      fprintf(pf, "         ip_source       routeur\n");
    }
    else
    { fprintf(pf, "  device  destination     / netmask");
      fprintf(pf, "         ip_source       gateway\n");
    }
  }
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_routes, i, &cr));
    fprintf(pf, "  ");
    lcrzo_priv_fprint_cstsized(pf, cr.device, 8);
    lcrzo_er(lcrzo_ips_init_ipl(cr.dest, ips));
    lcrzo_priv_fprint_cstsized(pf, ips, 16);
    fprintf(pf, "/ ");
    lcrzo_er(lcrzo_ips_init_ipl(cr.mask, ips));
    lcrzo_priv_fprint_cstsized(pf, ips, 16);
    if ( cr.ipsource==0 && cr.gateway==0 )
    { if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG )
      { fprintf(pf, "device_local                    ");
      }
      else
      { fprintf(pf, "local_device                    ");
      }
    }
    else if ( cr.gateway==0 )
    { lcrzo_er(lcrzo_ips_init_ipl(cr.ipsource, ips));
      lcrzo_priv_fprint_cstsized(pf, ips, 16);
      fprintf(pf, "                ");
    }
    else
    { lcrzo_er(lcrzo_ips_init_ipl(cr.ipsource, ips));
      lcrzo_priv_fprint_cstsized(pf, ips, 16);
      lcrzo_er(lcrzo_ips_init_ipl(cr.gateway, ips));
      lcrzo_priv_fprint_cstsized(pf, ips, 16);
    }
    if (cr.isup)
      fprintf(pf, "up");
    else
      fprintf(pf, "down");
    fprintf(pf, "\n");
  }
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
int lcrzo_priv_conf_verify_routes(void);
int lcrzo_priv_conf_verify_routes(void)
{ lcrzo_int32 count, i, cdcount, cdi;
  lcrzo_priv_conf_devices cd;
  lcrzo_priv_conf_routes cr;
  lcrzo_device device;
  lcrzo_bool needtoupdateip, needtoupdatedev;

  /* eventualy update gateway and device */
  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_routes, &count));
  for ( i=1 ; i<=count ; i++ )
  { needtoupdatedev=LCRZO_FALSE;
    needtoupdateip=LCRZO_FALSE;
    lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_routes, i, &cr));
    lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_devices, &cdcount));
    for ( cdi=1 ; cdi<=cdcount ; cdi++ )
    { lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_devices, cdi, &cd));
      if ( cr.ipsource!=0 )
      { /* need to set device */
        if ( cr.ipsource == cd.ip )
	{ /* ipsource was found in the device list */
          lcrzo_device_init(cd.device, device);
	  needtoupdatedev=LCRZO_TRUE;
  	  break;
	}
      }
      else
      { /* need to set ipsource and device */
        if ( cr.gateway!=0 )
	{ /* need to find the route to this gateway */
          if ( (cd.ip&cd.netmask) ==  (cr.gateway&cd.netmask) )
	  { cr.ipsource=cd.ip;
            lcrzo_device_init(cd.device, device);
	    needtoupdatedev=LCRZO_TRUE;
	    needtoupdateip=LCRZO_TRUE;
	    break;
  	  }
	}
	else
	{ /* need to find the route to ... */
	  if ( cd.ip == cr.dest )
	  { /* ... the device */
	    cr.ipsource=0;
	    lcrzo_device_init(cd.device, device);
	    needtoupdatedev=LCRZO_TRUE;
	    needtoupdateip=LCRZO_TRUE;
	    break;
	  }
	  if ( cd.netmask == cr.mask &&
	       (cd.ip&cd.netmask) == cr.dest )
	  { /* ... the lan */
            cr.ipsource=cd.ip;
            lcrzo_device_init(cd.device, device);
	    needtoupdatedev=LCRZO_TRUE;
	    needtoupdateip=LCRZO_TRUE;
	    break;
	  }
	}
      }
    }

    /* verify the device we want to set */
    if (needtoupdatedev)
    { /* if they are equal, do not update*/
      if ( lcrzo_device_equal(device, cr.device) )
      { needtoupdatedev=LCRZO_FALSE;
      }
      else
      { if ( strlen(device)==0 )
        { needtoupdatedev=LCRZO_FALSE;
        }
        else
	{ lcrzo_device_init(device, cr.device);
	}
      }
    }
 
    /* eventually update */
    if ( needtoupdatedev || needtoupdateip )
    { lcrzo_er(lcrzo_list_replace_pos(&lcrzo_priv_conf.list_routes, i, &cr));
    }
  }

  /* sort the routes : from "host routes" to "default route" */
  lcrzo_er(lcrzo_list_sort_all(&lcrzo_priv_conf.list_routes,
			       lcrzo_priv_conf_routes_sortcmp, 
			       NULL));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_conf_eventuallyinit(void)
{ static int firsttime=1;

  if (firsttime)
  { /* initialize the lists */
    lcrzo_er(lcrzo_list_init(&lcrzo_priv_conf.list_devices,
			     sizeof(lcrzo_priv_conf_devices), NULL));
    lcrzo_er(lcrzo_list_init(&lcrzo_priv_conf.list_arpperm,
			     sizeof(lcrzo_priv_conf_arp), NULL));
    lcrzo_er(lcrzo_list_init(&lcrzo_priv_conf.list_arpcache,
			     sizeof(lcrzo_priv_conf_arp), NULL));
    lcrzo_er(lcrzo_list_init(&lcrzo_priv_conf.list_routes,
			     sizeof(lcrzo_priv_conf_routes), NULL));
    lcrzo_priv_conf.allowupdate=1;
    lcrzo_priv_conf.forceupdate=1;
    lcrzo_priv_conf.permentriestimeout=5;
    lcrzo_priv_conf.cacheentriestimeout=30;
    firsttime=0; /*we have to set it before because eventuallyupdate
		   uses external functions */
    lcrzo_er(lcrzo_priv_conf_eventuallyupdate());
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_conf_eventuallyupdate(void)
{ static lcrzo_uint32 lastpermupdatetime=0;
  lcrzo_time currenttime;  
  lcrzo_uint32 removeolderthan;

  /* user doesn't want us to update the conf */
  if ( ! lcrzo_priv_conf.allowupdate )
  { return(LCRZO_ERR_OK);
  }

  /* check if we have to update permanent entries */
  lcrzo_er(lcrzo_time_init(&currenttime));
  if ( lcrzo_priv_conf.forceupdate || 
       ( lcrzo_priv_conf.permentriestimeout &&
	 ( lcrzo_priv_conf.permentriestimeout+lastpermupdatetime < 
	   currenttime.sec
	 ) 
       ) 
     )
  { /* remove all permanent entries */
    lcrzo_er(lcrzo_list_remove_all(&lcrzo_priv_conf.list_devices));
    lcrzo_er(lcrzo_list_remove_all(&lcrzo_priv_conf.list_arpperm));
    lcrzo_er(lcrzo_list_remove_all(&lcrzo_priv_conf.list_routes));
    /* set the new permament values*/
    { lcrzo_priv_conf.allowupdate=0;
      lcrzo_er(lcrzo_sysdep_conf_init_dev());
      lcrzo_er(lcrzo_sysdep_conf_init_arpcache());
      lcrzo_er(lcrzo_sysdep_conf_init_rarpcache());
      lcrzo_er(lcrzo_sysdep_conf_init_routes());
      /* We compute ipsource if it is != 0 (not oftently set by
	 system dependant functions)
         We eventually update device (Linux never set aliases, Solaris
         set local devices to "")
	 Moreover, sort the routes accordingly to the netmask (FreeBSD
	 doesn't do it) */
      lcrzo_er(lcrzo_priv_conf_verify_routes());
      lcrzo_priv_conf.allowupdate=1;
    }
    /* for next call */
    lastpermupdatetime=currenttime.sec;
  }

  /* check if we have to update cache entries */
  if ( lcrzo_priv_conf.forceupdate || lcrzo_priv_conf.cacheentriestimeout )
  { removeolderthan=currenttime.sec-lcrzo_priv_conf.cacheentriestimeout;
    lcrzo_er(lcrzo_list_remove_criteria_all(&lcrzo_priv_conf.list_arpcache, 
					    &lcrzo_priv_conf_arp_purge,
					    &removeolderthan));
  }

  /* the force update is effective only once */
  lcrzo_priv_conf.forceupdate=LCRZO_FALSE;

  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_conf_route_to_host(lcrzo_ipl hostipl,
			     lcrzo_device device,
			     lcrzo_etha srcetha,
			     lcrzo_etha destetha,
			     lcrzo_ipl *psrcipl,
			     lcrzo_ipl *prouteripl)
{ lcrzo_int32 count, i;
  lcrzo_priv_conf_routes cr;
  int ret;

  lcrzo_er(lcrzo_priv_conf_eventuallyinit());
  lcrzo_er(lcrzo_priv_conf_eventuallyupdate());

  lcrzo_er(lcrzo_list_count(lcrzo_priv_conf.list_routes, &count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_list_value_pos(lcrzo_priv_conf.list_routes, i, &cr));
    /* if this route matches */
    if ( cr.isup &&
	 ( (hostipl&cr.mask) == (cr.dest&cr.mask) ) )
    { /* set info we know */
      lcrzo_er(lcrzo_device_init(cr.device, device));
      lcrzo_er(lcrzo_etha_init_device(cr.device, srcetha));
      if (psrcipl!=NULL) *psrcipl=cr.ipsource;
      /* set more info */
      if ( cr.ipsource==0 && cr.gateway==0 )
      { /* it is a local device */
        return(LCRZO_ERR_OKROUTELOCALDEV);
      }
      else if ( cr.gateway==0 )
      { /* it is on the lan */
        if ( prouteripl!=NULL ) *prouteripl=0;
	ret=lcrzo_etha_init_ipl(hostipl, destetha);
        if ( ret==LCRZO_ERR_OK ) {}
	else if ( ret==LCRZO_ERR_OKUNRESOLVED ) 
        { return(LCRZO_ERR_OKROUTEHOSTUNREACH);
	}
	else return(ret);
      }
      else
      { /* it is after a router */
        if ( prouteripl!=NULL ) *prouteripl=cr.gateway;
	ret=lcrzo_etha_init_ipl(cr.gateway, destetha);
        if ( ret==LCRZO_ERR_OK ) {}
	else if ( ret==LCRZO_ERR_OKUNRESOLVED ) 
        { return(LCRZO_ERR_OKROUTEGWUNREACH);
	}
	else return(ret);
      }
      /* all the info are set */
      return(LCRZO_ERR_OK);
    }    
  }
  return(LCRZO_ERR_OKROUTENOTFOUND);
}





