/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_conf module

  Functions herein obtains configuration of the local computer.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_conf

  Les fonctions presentes dans ce module permettent d'obtenir des
informations sur la machine locale.
*/


/*---------------------------------------------------------------*/
/*US************************************************************
 * Note about lcrzo_conf :                                     *
 * Configuration obtained by this module is a copy of the      *
 * configuration of the local computer. Then the user can      *
 * modify it using "xx_add" and "xx_del" functions : the       *
 * configuration of lcrzo is modified, but not the             *
 * configuration of the computer.                              *
 * The initialization of the configuration is done             *
 * automagically on the first call of one of the folowing      *
 * functions.                                                  *
 * For examples the configuration of the local computer can    *
 * be (obtained with lcrzo_conf_print) :                       */
/*FR************************************************************
 * Note sur lcrzo_conf :                                       *
 * La configuration obtenue par ce module est une copie de la  *
 * configuration de la machine locale. L'utilisateur peut par  *
 * la suite la modifier avec les fonctions "xx_add" et         *
 * "xx_del" : la configuration de lcrzo est changee, mais pas  *
 * celle de l'ordinateur.                                      *
 * L'initialisation de la configuration a lieu lors du premier *
 * appel de l'une des fonctions suivantes.                     *
 * Par exemple la configuration de la machine locale peut etre *
 * (obtenu par lcrzo_conf_print) :                             */
/*
Devices
  device  ethernet           ip           / netmask        mtu
  lo      00:00:00:00:00:00  127.0.0.1    / 255.0.0.0      3924  up
  eth0    00:40:33:E0:C2:24  192.168.10.1 / 255.255.255.0  1500  up
  eth1    02:60:8C:A3:EA:AA  192.168.21.1 / 255.255.255.0  1500  up
  eth1:1  02:60:8C:A3:EA:AA  192.168.31.1 / 255.255.255.0  1500  up,alias
Arp
  lo      00:00:00:00:00:00  127.0.0.1       (permanent)
  eth0    00:40:33:E0:C2:24  192.168.10.1    (permanent)
  eth1    02:60:8C:A3:EA:AA  192.168.21.1    (permanent)
  eth1:1  02:60:8C:A3:EA:AA  192.168.31.1    (permanent)
Routes
  device  destination  / netmask         ip_source     gateway
  lo      127.0.0.1    / 255.255.255.255 local_device                up
  eth0    192.168.10.1 / 255.255.255.255 local_device                up
  eth1    192.168.21.1 / 255.255.255.255 local_device                up
  eth1:1  192.168.31.1 / 255.255.255.255 local_device                up
  eth1:1  192.168.50.3 / 255.255.255.255 192.168.31.1  192.168.31.3  up
  eth0    192.168.50.2 / 255.255.255.255 192.168.10.1  192.168.10.4  up
  eth0    192.168.10.0 / 255.255.255.0   192.168.10.1                up
  eth1:1  192.168.31.0 / 255.255.255.0   192.168.31.1                up
  eth1    192.168.21.0 / 255.255.255.0   192.168.21.1                up
  eth0    192.168.40.0 / 255.255.255.0   192.168.10.1  192.168.10.2  up
  lo      127.0.0.0    / 255.0.0.0       127.0.0.1                   up
 ***************************************************************/

/*-------------------------------------------------------------*/
/*us Name : lcrzo_conf_fprint
   Description :
     Print the configuration of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_fprint
   Description :
     Affiche la configuration de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_fprint(LCRZOFILE *pf);
#define lcrzo_conf_print() lcrzo_conf_fprint(stdout)


/*-------------------------------------------------------------*/
/*us Name : lcrzo_conf_devices_add
   Description :
     Add a device to the configuration of lcrzo.
   Input parameter(s) :
     device : device name
     etha : associated Ethernet address
     ipl : associated IP address
     netmask : associated netmask
     mtu : associated mtu
     isup : if this device is up
     isanalias : if this device is an alias
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_add
   Description :
     Ajoute un device a la configuration de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     etha : adresse Ethernet du device
     ipl : adresse IP du device
     netmask : masque reseau du device
     mtu : mtu du device
     isup : si le device est actif
     isanalias : si le device est un alias
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_add(const lcrzo_device device,
			   const lcrzo_etha etha,
			   lcrzo_ipl ipl,
			   lcrzo_ipl netmask,
			   lcrzo_uint32 mtu,
			   lcrzo_bool isup,
			   lcrzo_bool isanalias);

/*us Name : lcrzo_conf_devices_del
   Description :
     Delete a device from the configuration of lcrzo.
   Input parameter(s) :
     device : device name to delete
     etha : associated Ethernet address
     ipl : associated IP address
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_del
   Description :
     Enleve un device de la configuration de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     etha : adresse Ethernet du device
     ipl : adresse IP du device
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_del(const lcrzo_device device,
			   const lcrzo_etha etha,
			   lcrzo_ipl ipl);

/*us Name : lcrzo_conf_devices_count
   Description :
     Count the number of devices
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of devices
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_count
   Description :
     Compte le nombre de devices
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de devices
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_count(lcrzo_int32 *pcount);

/*US*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*us Name : lcrzo_conf_devices_value_pos
   Description :
     Get the device located at the wanted position
   Input parameter(s) :
     pos : position of the wanted device (cf. the above frame to
           know possible values)
   Input/output parameter(s) :
   Output parameter(s) :
     device : device name
     etha : associated Ethernet address
     *pipl : associated IP address
     *pnetmask : associated netmask
     *pmtu : associated mtu
     *pisup : if this device is up
     *pisanalias : if this device is an alias
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_value_pos
   Description :
     Fournit le device situe a la position voulue
   Parametre(s) d'entree :
     pos : position du device cherche (cf. le cadre
            ci-dessus pour connaitre les valeurs valides)
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : nom du device
     etha : adresse Ethernet du device
     *pipl : adresse IP du device
     *pnetmask : masque reseau du device
     *pmtu : mtu du device
     *pisup : si le device est actif
     *pisanalias : si le device est un alias
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_value_pos(lcrzo_int32 pos,
				 lcrzo_device device,
				 lcrzo_etha etha,
				 lcrzo_ipl *pipl,
				 lcrzo_ipl *pnetmask,
				 lcrzo_uint32 *pmtu,
				 lcrzo_bool *pisup,
				 lcrzo_bool *pisanalias);

/*us Name : lcrzo_conf_devices_fprint
   Description :
     Print the configuration of devices of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_devices_fprint
   Description :
     Affiche la configuration des devices de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_devices_fprint(LCRZOFILE *pf);
#define lcrzo_conf_devices_print() lcrzo_conf_devices_fprint(stdout)


/*-------------------------------------------------------------*/
/*us Name : lcrzo_conf_arp_add
   Description :
     Add an arp entry to the configuration of lcrzo.
   Input parameter(s) :
     device : device name
     etha : Ethernet address
     ipl : IP address
     ispermanent : if this arp entry is permanent
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_add
   Description :
     Ajoute une entree ARP a la configuration de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     etha : adresse Ethernet
     ipl : adresse IP
     ispermanent : si cette entree arp est permanente
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_add(const lcrzo_device device,
		       const lcrzo_etha etha,
		       lcrzo_ipl ipl,
		       lcrzo_bool ispermanent);

/*us Name : lcrzo_conf_arp_del
   Description :
     Delete an arp entry from the configuration of lcrzo.
   Input parameter(s) :
     device : device name
     etha : Ethernet address
     ipl : IP address
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_add
   Description :
     Enleve une entree ARP de la configuration de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     etha : adresse Ethernet
     ipl : adresse IP
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_del(const lcrzo_device device,
		       const lcrzo_etha etha,
		       lcrzo_ipl ipl);

/*us Name : lcrzo_conf_arp_count
   Description :
     Count the number of arp entries
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of entries
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_count
   Description :
     Compte le nombre d'entrees arp
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre d'entrees
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_count(lcrzo_int32 *pcount);

/*us Name : lcrzo_conf_arp_value_pos
   Description :
     Get the arp entry located at the wanted position
   Input parameter(s) :
     pos : position of the wanted arp entry
   Input/output parameter(s) :
   Output parameter(s) :
     device : device name
     etha : Ethernet address
     *pipl : IP address
     *pispermanent : if this arp entry is permanent
     *ptimeset : time to which this arp entry was set
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_value_pos
   Description :
     Fournit l'entree arp situee a la position voulue
   Parametre(s) d'entree :
     pos : position de l'entree arp voulue
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : nom de device
     etha : adresse Ethernet
     *pipl : adresse IP
     *pispermanent : si cette entree est permanente
     *ptimeset : heure a laquelle cette entree a ete ajoutee
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_value_pos(lcrzo_int32 pos,
			     lcrzo_device device,
			     lcrzo_etha etha,
			     lcrzo_ipl *pipl,
			     lcrzo_bool *pispermanent,
			     lcrzo_uint32 *ptimeset);

/*us Name : lcrzo_conf_arp_fprint
   Description :
     Print the arp configuration of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_arp_fprint
   Description :
     Affiche la configuration arp de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_arp_fprint(LCRZOFILE *pf);
#define lcrzo_conf_arp_print() lcrzo_conf_arp_fprint(stdout)


/*-------------------------------------------------------------*/
/*us Name : lcrzo_conf_route_add
   Description :
     Add a route to the configuration of lcrzo.
   Input parameter(s) :
     device : device name
     dest/mask : IP address/netmask
     ipsource : source IP address to use (if 0, local device)
     gateway : gateway to use (if 0, dest is on LAN)
     isup : if this route is up
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_route_add
   Description :
     Ajoute une route a la configuration de lcrzo.
   Parametre(s) d'entree :
     device : nom du device
     dest/mask : adresse IP/masque de la route voulue
     ipsource : adresse IP source a employer (si 0, device local)
     gateway : routeur a employer (si 0, dest est sur le LAN)
     isup : si cette route est active
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_add(const lcrzo_device device, 
			  lcrzo_ipl dest,
			  lcrzo_ipl mask,
			  lcrzo_ipl ipsource,
			  lcrzo_ipl gateway,
			  lcrzo_bool isup);

/*us Name : lcrzo_conf_routes_del
   Description :
     Delete a route from the configuration of lcrzo.
   Input parameter(s) :
     dest/mask : IP address/mask of the route
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_del
   Description :
     Enleve une route de la configuration de lcrzo.
   Parametre(s) d'entree :
     dest/mask : adresse IP/masque de la route
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_del(lcrzo_ipl dest,
			  lcrzo_ipl mask);

/*us Name : lcrzo_conf_routes_count
   Description :
     Count the number of routes
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of routes
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_count
   Description :
     Compte le nombre de routes
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nombre de routes
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_count(lcrzo_int32 *pcount);

/*us Name : lcrzo_conf_routes_value_pos
   Description :
     Get the route located at the wanted position
   Input parameter(s) :
     pos : position of the wanted route
   Input/output parameter(s) :
   Output parameter(s) :
     device : device name
     *pdest : IP address
     *pmask : netmask
     *pipsource : source IP address to use
     *pgateway : gateway to use
     *pisup : if this route is up
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_value_pos
   Description :
     Fournit la route situee a la position voulue
   Parametre(s) d'entree :
     pos : position de la route voulue
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : nom de device
     *pdest : adresse IP destination
     *pmask : masque destination
     *pipsource : adresse IP source a employer
     *pgateway : routeur a employer
     *pisup : si cette route est active
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_value_pos(lcrzo_int32 pos,
				lcrzo_device device,
				lcrzo_ipl *pdest,
				lcrzo_ipl *pmask,
				lcrzo_ipl *pipsource,
				lcrzo_ipl *pgateway,
				lcrzo_bool *pisup);

/*us Name : lcrzo_conf_routes_fprint
   Description :
     Print the routing configuration of lcrzo.
   Input parameter(s) :
     *pf : FILE where the function prints
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_conf_routes_fprint
   Description :
     Affiche la configuration de routage de lcrzo.
   Parametre(s) d'entree :
     *pf : FILE ou la fonction doit ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_conf_routes_fprint(LCRZOFILE *pf);
#define lcrzo_conf_routes_print() lcrzo_conf_routes_fprint(stdout)


/*-------------------------------------------------------------*/
/*US************************************************************
 * Note about the folowing functions :                         *
 * The default values are correct for most applications.       *
 * You should modify their parameters only if you know         *
 * what you are doing.                                         *
 ***************************************************************/
/*FR************************************************************
 * Note sur les fonctions suivantes :                          *
 * Les valeurs par defaut sont correctes pour la majorite des  *
 * applications.                                               *
 * Vous ne devriez modifier les parametres que si vous savez   *
 * ce que vous faites.                                         *
 ***************************************************************/

/*us to allow the update of the configuation */
/*fr pour permettre la mise a jour de la configuration */
int lcrzo_conf_set_allowupdate(lcrzo_bool yes);
int lcrzo_conf_get_allowupdate(lcrzo_bool *pyes);

/*us to force an update of the configuration */
/*fr pour forcer la mise a jour de la configuration */
int lcrzo_conf_set_forceupdate(lcrzo_bool yes);
int lcrzo_conf_get_forceupdate(lcrzo_bool *pyes);

/*us lifetime of permament entries before having to update them */
/*fr duree de vie des entrees permanentes au bout de laquelle
     celle-ci doivent etre mises a jour */
int lcrzo_conf_set_permentriestimeout(lcrzo_uint32 timeout);
int lcrzo_conf_get_permentriestimeout(lcrzo_uint32 *ptimeout);

/*us lifetime of cached entries */
/*fr duree de vie des entrees mises en cache */
int lcrzo_conf_set_cacheentriestimeout(lcrzo_uint32 timeout);
int lcrzo_conf_get_cacheentriestimeout(lcrzo_uint32 *ptimeout);


/*-------------------------------------------------------------*/
/*us Name : lcrzo_conf_route_to_host
   Description :
     Determine device, sources addresses and destination Ethernet
     address to use to send an IP packet to a host.
   Input parameter(s) :
     hostipl : host to reach
   Input/output parameter(s) :
   Output parameter(s) :
     device : device to use
     srcetha : Ethernet source address to use
     destetha : Ethernet destination address to use
     *psrcipl : IP source address to use
     *prouteripl : IP address of the router (or 0 if hostipl is
                   on the local LAN)
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKROUTELOCALDEV : hostipl is the address of a 
                                 local device
     LCRZO_ERR_OKROUTEHOSTUNREACH : hostipl in on LAN, but we 
                                    can't get its Ethernet address
     LCRZO_ERR_OKROUTEGWUNREACH : hostipl in behind a router, but we 
                                  can't get the Ethernet address of
                                  the router
     LCRZO_ERR_OKROUTENOTFOUND : the routing table doesn't contain
                                 a route to hostipl
*/
/*fr Nom : lcrzo_conf_route_to_host
   Description :
     Determine le device, les adresses source et l'adresse Ethernet
     destination a employer pour envoyer un paquet IP vers une machine.
   Parametre(s) d'entree :
     hostipl : machine a joindre
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     device : device a employer
     srcetha : adresse ethernet source a employer
     destetha : adresse ethernet destination a employer
     *psrcipl : adresse IP source a employer
     *prouteripl : adresse IP du routeur (ou 0 si hostipl est sur le 
                   LAN local)
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKROUTELOCALDEV : hostipl est l'adresse d'un device
                                 local
     LCRZO_ERR_OKROUTEHOSTUNREACH : hostipl est sur le LAN, mais on 
                                    n'a pas pu obtenir son adresse
                                    Ethernet
     LCRZO_ERR_OKROUTEGWUNREACH : hostipl est derriere un routeur, 
                                  mais on n'a pas pu obtenir l'adresse
                                  Ethernet du routeur
     LCRZO_ERR_OKROUTENOTFOUND : la table de routage ne contient pas 
                                 d'entree vers hostipl
*/
int lcrzo_conf_route_to_host(lcrzo_ipl hostipl,
			     lcrzo_device device,
			     lcrzo_etha srcetha,
			     lcrzo_etha destetha,
			     lcrzo_ipl *psrcipl,
			     lcrzo_ipl *prouteripl);

