/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_data module

  Functions herein allow to work on data arrays.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_data

  Les fonctions presentes dans ce module permettent gerer des
tableaux de donnees.
*/


/*-------------------------------------------------------------*/
/*us these typedefs are used for :
   lcrzo_data : an array of data
   lcrzo_constdata : an array of data which can not be modified
   lcrzo_string : a string 
   lcrzo_conststring : a string which can not be modified
*/
/*fr ces typedefs sont utilises pour :
   lcrzo_data : tableau de donnees
   lcrzo_constdata : tableau de donnees non modifiables
   lcrzo_string : chaine de caracteres
   lcrzo_conststring : chaine de caracteres non modifiables
*/
typedef lcrzo_uint8* lcrzo_data;
typedef const lcrzo_uint8* lcrzo_constdata;

typedef char* lcrzo_string;
typedef const char* lcrzo_conststring;

/*-------------------------------------------------------------*/
/*us these typedefs define 4 types of array which may be
   useful for programs */
/*fr ces typedefs definissent 4 types de tableau qui peuvent
   etre utiles dans les programmes */
#define LCRZO_ARRAY65K_MAXBYTES 66560
#define LCRZO_ARRAY64K_MAXBYTES 65536
#define LCRZO_ARRAY8K_MAXBYTES 8192
#define LCRZO_ARRAY2K_MAXBYTES 2048
typedef lcrzo_uint8 lcrzo_array65k[LCRZO_ARRAY65K_MAXBYTES+1];
typedef lcrzo_uint8 lcrzo_array64k[LCRZO_ARRAY64K_MAXBYTES+1];
typedef lcrzo_uint8 lcrzo_array8k[LCRZO_ARRAY8K_MAXBYTES+1];
typedef lcrzo_uint8 lcrzo_array2k[LCRZO_ARRAY2K_MAXBYTES+1];

/*-------------------------------------------------------------*/
/*US*******************************************
 * Note about sizes :                         *
 *   sizes are always in bytes                *
 **********************************************/
/*FR*******************************************
 * Note sur les tailles :                     *
 *   les tailles sont toujours en octet       *
 **********************************************/

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_alloc
   Description :
     Allocate a memory array.
   Input parameter(s) :
     allocsize : size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *pptr : pointer which will be malloced (so, the
             memory will have to be freed by the
	     user with 'lcrzo_data_free(*pdataout)' or
             'free(*pdataout)').
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_alloc
   Description :
     Alloue une zone de memoire.
   Parametre(s) d'entree :
     allocsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pptr : pointeur qui sera alloue (la memoire
             devra etre liberee par l'utilisateur 
	     avec 'lcrzo_data_free(*pdataout)' ou
             'free(*pdataout)').
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_alloc(lcrzo_int32 allocsize, lcrzo_data *pptr);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_realloc
   Description :
     Reallocate a memory array.
   Input parameter(s) :
     newallocsize : new size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *pptr : pointer which will be reaalloced (so, the
             memory will have to be freed by the
	     user with 'lcrzo_data_free(*pdataout)' or
             'free(*pdataout)').
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_realloc
   Description :
     Realloue une zone de memoire.
   Parametre(s) d'entree :
     newallocsize : nouvelle taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pptr : pointeur qui sera realloue (la memoire
             devra etre liberee par l'utilisateur 
	     avec 'lcrzo_data_free(*pdataout)' ou
             'free(*pdataout)').
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_realloc(lcrzo_int32 newallocsize, lcrzo_data *pptr);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_free
   Description :
     Free a memory array.
   Input parameter(s) :
     ptr : pointer to the memory to free
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_free
   Description :
     Libere une zone de memoire.
   Parametre(s) d'entree :
     ptr : pointeur sur la zone memoire a allouer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_free(lcrzo_data ptr);


/*-------------------------------------------------------------*/
/*US*******************************************
 * Note about indexes :                       *
 * Indexes permit to access range of data.    *
 * For example, if we have a 4 bytes data,    *
 * containing 'ABCD', we can represent it     *
 * like this :                                *
 *       +---+---+---+---+                    * 
 *       | A | B | C | D |                    *
 *       +---+---+---+---+                    *
 * The indexes are associated with the        *
 * corners :                                  *
 *       1   2   3   4   5                    *
 *      -5  -4  -3  -2  -1                    *
 * So :                                       *
 *   range [1,5]   is 'ABCD'                  * 
 *   range [2,3]   is 'B'                     *
 *   range [3,3]   is ''     (empty array)    *
 *   range [1,-1]  is 'ABCD'                  *
 *   range [2,-3]  is 'B'                     *
 *   range [3,1]   is 'BA'                    *
 *   range [-1,1]  is 'DCBA'                  *
 *   range [-2,-3] is 'C'                     *
 *   etc.                                     *
 * Note : 0 (zero) is an invalid index        *
 **********************************************/
/*FR*******************************************
 * Note sur les indices :                     *
 * Les indices permettent d'acceder a des     *
 * intervalles de donnees.                    *
 * Par exemple, si on a 4 octets de donnees   *
 * contenant 'ABCD', on peut les representer  *
 * ainsi :                                    *
 *       +---+---+---+---+                    * 
 *       | A | B | C | D |                    *
 *       +---+---+---+---+                    *
 * Les indices sont associes avec les         *
 * angles :                                   *
 *       1   2   3   4   5                    *
 *      -5  -4  -3  -2  -1                    *
 * Donc :                                     *
 *   intervalle [1,5]   est 'ABCD'            * 
 *   intervalle [2,3]   est 'B'               *
 *   intervalle [3,3]   est '' (tableau vide) *
 *   intervalle [1,-1]  est 'ABCD'            *
 *   intervalle [2,-3]  est 'B'               *
 *   intervalle [3,1]   est 'BA'              *
 *   intervalle [-1,1]  est 'DCBA'            *
 *   intervalle [-2,-3] est 'C'               *
 *   etc.                                     *
 * Note : l'indice 0 (zero) est invalide      *
 **********************************************/

/*-------------------------------------------------------------*/
/*US*******************************************
 * Note about mixed strings :                 *
 * A mixed string permits to represent data   *
 * as a clear form using hexadecimal and      *
 * text.                                      *
 * Hexadecimal is used without "0x" or "0h"   *
 * Text is included between apostrophes "'"   *
 * The character ' is ''                      *
 * For example :                              *
 *   'hello' : data "hello"                   *
 *   'a' 'b' : data "ab"                      *
 *   41 'b'  : data "Ab" (because 'A'==0x41)  *
 *   'man'00 : data "man" ending with 0x00    *
 *   'a''b'  : data "a'b"                     *
 * Real examples :                            *
 *  'hello' 0D 0A 'this is after a newline'   *
 *  'one' 09 'two' 0D 0A                      *
 **********************************************/
/*FR*******************************************
 * Note sur les chaines mixtes :              *
 * Une chaine mixte permet de representer des *
 * donnees de maniere claire en utilisant de  *
 * l'hexadecimal et du texte.                 *
 * L'hexa est utilise sans "0x" ou "0h"       *
 * Le texte est inclus entre apostrophes "'"  *
 * Le caractere ' est double ('')             *
 * Par exemple :                              *
 *   'hello' : donnee "hello"                 *
 *   'a' 'b' : donnee "ab"                    *
 *   41 'b'  : donnee "Ab" (car 'A'==0x41)    *
 *   'man'00 : donnee "man" terminant par 0x00*
 *   'a''b'  : donnee "a'b"                   *
 * Exemples reels :                           *
 *  'salut' 0D 0A 'ceci est apres le newline' *
 *  'un' 09 'deux' 0D 0A                      *
 **********************************************/

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_data
   Description :
     Initialize a data array, with data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with datain
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
				 data in dataout
*/
/*fr Nom : lcrzo_data_init_data
   Description :
     Initialise un tableau de donnees, avec un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra une copie de datain
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
				 donnees dans dataout
*/
int lcrzo_data_init_data(lcrzo_constdata datain,
			 lcrzo_int32 datainsize,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_data_initm_data
   Description :
     Allocate a memory array, and initialize it with data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
		 user with 'free(*pdataout)').
		 The allocated memory will be set with datain
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_initm_data
   Description :
     Alloue une zone de memoire et l'initialise avec un 
     tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
		 avec 'free(*pdataout)').
		 Cette memoire allouee recevra une copie de
		 datain
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_initm_data(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_data_append_data
   Description :
     Append at the end of a data array, with data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     dataoutcurrentsize : current size of dataout
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               datain will be appended to this array.
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
				 data in dataout
*/
/*fr Nom : lcrzo_data_append_data
   Description :
     Complete un tableau de donnees, avec un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     dataoutcurrentsize : taille courante de dataout
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               datain sera concatene a dataout
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
				 donnees dans dataout
*/
int lcrzo_data_append_data(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_int32 dataoutmaxsize,
			    lcrzo_data dataout,
			    lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_data_appendm_data
   Description :
     Reallocate a memory array, and append at the end, with
     data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     dataoutcurrentsize : current size of *pdataout
   Input/output parameter(s) :
     *pdataout : pointer which will be reallocated (so,
                 the memory will have to be freed by the
                 user with 'free(*pdataout)').
                 datain will be appended to this array.
   Output parameter(s) :
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_appendm_data
   Description :
     Realloue une zone de memoire et la complete avec un 
     tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     dataoutcurrentsize : taille courante de *pdataout
   Parametre(s) d'entree-sortie :
     *pdataout : pointeur qui sera realloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 datain sera concatene a *pdataout
   Parametre(s) de sortie :
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_appendm_data(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_range
   Description :
     Initialize a data array, with a range of data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     datainstartindex : start index of the range to copy
     datainendindex : end index of the range to copy
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with a range of datain
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_range
   Description :
     Initialise un tableau de donnees, avec un intervalle de
     tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     datainstartindex : indice de depart de l'intervalle a copier
     datainendindex : indice de fin de l'intervalle a copier
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra une copie de l'intervalle de
               datain
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_range(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_int32 datainstartindex,
			  lcrzo_int32 datainendindex,
			  lcrzo_int32 dataoutmaxsize,
			  lcrzo_data dataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_range, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_range, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_range(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_int32 datainstartindex,
			   lcrzo_int32 datainendindex,
			   lcrzo_data *pdataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_range, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_range, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_range(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_int32 datainstartindex,
			    lcrzo_int32 datainendindex,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_int32 dataoutmaxsize,
			    lcrzo_data dataout,
			    lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_range, but append */
/*fr idem lcrzo_data_initm_range, mais concatene */
int lcrzo_data_appendm_range(lcrzo_constdata datain,
			     lcrzo_int32 datainsize,
			     lcrzo_int32 datainstartindex,
			     lcrzo_int32 datainendindex,
			     lcrzo_int32 dataoutcurrentsize,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_fill
   Description :
     Initialize a data array, with data filling
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     fillingsize : size of the filling
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with a filling from datain
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_fill
   Description :
     Initialise un tableau de donnees, avec un remplissage de 
     donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     fillingsize : taille de remplissage
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra un remplissage base sur datain
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_fill(lcrzo_constdata datain,
                         lcrzo_int32 datainsize,
                         lcrzo_int32 fillingsize,
                         lcrzo_int32 dataoutmaxsize,
                         lcrzo_data dataout,
                         lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_fill, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_fill, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_fill(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_int32 fillingsize,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_fill, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_fill, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_fill(lcrzo_constdata datain,
                           lcrzo_int32 datainsize,
                           lcrzo_int32 fillingsize,
                           lcrzo_int32 dataoutcurrentsize,
                           lcrzo_int32 dataoutmaxsize,
                           lcrzo_data dataout,
                           lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_fill, but append */
/*fr idem lcrzo_data_initm_fill, mais concatene */
int lcrzo_data_appendm_fill(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_int32 fillingsize,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_c
   Description :
     Initialize a data array, with character filling
   Input parameter(s) :
     c : character to use for the filling
     fillingsize : size of the filling
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with a filling of c
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_c
   Description :
     Initialise un tableau de donnees, avec un remplissage de 
     caracteres
   Parametre(s) d'entree :
     c : caractere a utiliser pour le remplissage
     fillingsize : taille de remplissage
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra un remplissage de c
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_char(lcrzo_uint8 c,
			 lcrzo_int32 fillingsize,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_char, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_char, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_char(lcrzo_uint8 c,
			  lcrzo_int32 fillingsize,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_char, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_char, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_char(lcrzo_uint8 c,
			   lcrzo_int32 fillingsize,
			   lcrzo_int32 dataoutcurrentsize,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_char, but append */
/*fr idem lcrzo_data_initm_char, mais concatene */
int lcrzo_data_appendm_char(lcrzo_uint8 c,
			    lcrzo_int32 fillingsize,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_rand
   Description :
     Initialize a data array, with rand filling
   Input parameter(s) :
     minvalue : minimum value of the rand characters
     maxvalue : maximum value of the rand characters
     fillingsize : size of the filling
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with a filling of rand
	       characters between minvalue and maxvalue
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_rand
   Description :
     Initialise un tableau de donnees, avec un remplissage de 
     caracteres aleatoires
   Parametre(s) d'entree :
     minvalue : valeur minimale des caracteres aleatoires
     maxvalue : valeur maximale des caracteres aleatoires
     fillingsize : taille de remplissage
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra un remplissage de caracteres
	       aleatoires compris entre minvalue et maxvalue
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_rand(lcrzo_uint8 minvalue,
			 lcrzo_uint8 maxvalue,
			 lcrzo_int32 fillingsize,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_rand, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_rand, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_rand(lcrzo_uint8 minvalue,
			  lcrzo_uint8 maxvalue,
			  lcrzo_int32 fillingsize,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_rand, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_rand, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_rand(lcrzo_uint8 minvalue,
			   lcrzo_uint8 maxvalue,
			   lcrzo_int32 fillingsize,
			   lcrzo_int32 dataoutcurrentsize,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_rand, but append */
/*fr idem lcrzo_data_initm_rand, mais concatene */
int lcrzo_data_appendm_rand(lcrzo_uint8 minvalue,
			    lcrzo_uint8 maxvalue,
			    lcrzo_int32 fillingsize,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_int
   Description :
     Initialize a data array, with an integer
   Input parameter(s) :
     integer : integer to store
     format : format (printf) of the integer to store.
              This format must be :
                 "%[0+-.]{0,1}[1-9]{0,2}[l]{0,1}[douxX]"
              For example : "%d", "%3x", "%08lX", etc.
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the integer 
	       printed with the format 'format'
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_int
   Description :
     Initialise un tableau de donnees, avec un entier
   Parametre(s) d'entree :
     integer : entier a utiliser
     format : format (printf) a utiliser sur 'integer'
              Ce format doit respecter :
                 "%[0+-.]{0,1}[1-9]{0,2}[l]{0,1}[douxX]"
              Par exemple : "%d", "%3x", "%08lX", etc.
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra l'entier selon le format
	       'format'
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_int(lcrzo_int32 integer,
			const char * format,
			lcrzo_int32 dataoutmaxsize,
			lcrzo_data dataout,
			lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_int, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_int, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_int(lcrzo_int32 integer,
			 const char * format,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_int, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_int, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_int(lcrzo_int32 integer,
			  const char * format,
			  lcrzo_int32 dataoutcurrentsize,
			  lcrzo_int32 dataoutmaxsize,
			  lcrzo_data dataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_int, but append */
/*fr idem lcrzo_data_initm_int, mais concatene */
int lcrzo_data_appendm_int(lcrzo_int32 integer,
			   const char * format,
			   lcrzo_int32 dataoutcurrentsize,
			   lcrzo_data *pdataout,
			   lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_text
   Description :
     Initialize a data array, with an text string
   Input parameter(s) :
     text : text string to use
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the text
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_text
   Description :
     Initialise un tableau de donnees, avec du texte 
   Parametre(s) d'entree :
     text : texte a utiliser
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra le texte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_text(const char * text,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_text, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_text, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_text(const char * text,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_text, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_text, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_text(const char * text,
			   lcrzo_int32 dataoutcurrentsize,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_text, but append */
/*fr idem lcrzo_data_initm_text, mais concatene */
int lcrzo_data_appendm_text(const char * text,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_coretext
   Description :
     Initialize a data array, with an text string.
     This text string is cleaned before use : the spaces, tabs,
     and newline at the begining or the end of the text are removed
   Input parameter(s) :
     text : text string to clean, then to use
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the cleaned text
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_coretext
   Description :
     Initialise un tableau de donnees, avec du texte.
     Ce texte est purge avant emploi : les expaces, tabulations,
     et sauts de lignes en debut ou fin de chaine sont supprimes
   Parametre(s) d'entree :
     text : texte a purger, puis a utiliser
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra le texte purge
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_coretext(const char * text,
			     lcrzo_int32 dataoutmaxsize,
			     lcrzo_data dataout,
			     lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_coretext, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_coretext, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_coretext(const char * text,
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_coretext, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_coretext, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_coretext(const char * text,
			       lcrzo_int32 dataoutcurrentsize,
			       lcrzo_int32 dataoutmaxsize,
			       lcrzo_data dataout,
			       lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_coretext, but append */
/*fr idem lcrzo_data_initm_coretext, mais concatene */
int lcrzo_data_appendm_coretext(const char * text,
				lcrzo_int32 dataoutcurrentsize,
				lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_hexa
   Description :
     Initialize a data array, with data presented in hexa form
   Input parameter(s) :
     hexa : string in hexa ("45D2" means two bytes 0x45 0xD2) 
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the equivalent of
	       the hexa string
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_hexa
   Description :
     Initialise un tableau de donnees, avec des donnees
     representees sous forme hexadecimale
   Parametre(s) d'entree :
     hexa : chaine hexa ("45D2" signifie les 2 octets 0x45 0xD2)
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees equivalentes 
	       a la chaine hexa
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_hexa(const char * hexa,
			 lcrzo_int32 dataoutmaxsize,
			 lcrzo_data dataout,
			 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_hexa, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_hexa, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_hexa(const char * hexa,
			  lcrzo_data *pdataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_hexa, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_hexa, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_hexa(const char * hexa,
			   lcrzo_int32 dataoutcurrentsize,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_hexa, but append */
/*fr idem lcrzo_data_initm_hexa, mais concatene */
int lcrzo_data_appendm_hexa(const char * hexa,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_mixed
   Description :
     Initialize a data array, with data presented in mixed form
   Input parameter(s) :
     mixed : string in mixed (see frame at the top of the help 
             for this module describing what's a mixed string)
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the equivalent of
	       the mixed string
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_mixed
   Description :
     Initialise un tableau de donnees, avec des donnees
     representees sous forme mixte
   Parametre(s) d'entree :
     mixed : chaine mixte (cf. le cadre, en haut de l'aide
             de ce module, decrivant ce qu'est une chaine mixte)
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees equivalentes 
	       a la chaine mixte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_mixed(const char * mixed,
			  lcrzo_int32 dataoutmaxsize,
			  lcrzo_data dataout,
			  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_mixed, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_mixed, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_mixed(const char * mixed,
			   lcrzo_data *pdataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_mixed, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_mixed, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_mixed(const char * mixed,
			    lcrzo_int32 dataoutcurrentsize,
			    lcrzo_int32 dataoutmaxsize,
			    lcrzo_data dataout,
			    lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_mixed, but append */
/*fr idem lcrzo_data_initm_mixed, mais concatene */
int lcrzo_data_appendm_mixed(const char * mixed,
			     lcrzo_int32 dataoutcurrentsize,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_base64
   Description :
     Initialize a data array, with data presented in base64 form
   Input parameter(s) :
     base64 : string in base64
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the equivalent of
	       the base64 string
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_base64
   Description :
     Initialise un tableau de donnees, avec des donnees
     representees sous forme base64
   Parametre(s) d'entree :
     base64 : chaine en base64
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees equivalentes 
	       a la chaine base64
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_base64(const char * base64,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_base64, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_base64, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_base64(const char * base64,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_base64, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_base64, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_base64(const char * base64,
			     lcrzo_int32 dataoutcurrentsize,
			     lcrzo_int32 dataoutmaxsize,
			     lcrzo_data dataout,
			     lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_base64, but append */
/*fr idem lcrzo_data_initm_base64, mais concatene */
int lcrzo_data_appendm_base64(const char * base64,
			      lcrzo_int32 dataoutcurrentsize,
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_text_stdin
   Description :
     Initialize a data array, with an text string entered 
     by the user
   Input parameter(s) :
     message : message to print before asking user
     defaultext : text string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the text
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_text_stdin
   Description :
     Initialise un tableau de donnees, avec du texte entre
     par l'utilisateur
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du texte
     defaulttext : texte a utiliser si l'utilisateur n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra le texte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_text_stdin(const char * message,
			       const char * defaulttext,
			       lcrzo_int32 dataoutmaxsize,
			       lcrzo_data dataout,
			       lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_text_stdin, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_text_stdin, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_text_stdin(const char * message,
				const char * defaulttext,
				lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_text_stdin, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_text_stdin, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_text_stdin(const char * message,
				 const char * defaulttext,
				 lcrzo_int32 dataoutcurrentsize,
				 lcrzo_int32 dataoutmaxsize,
				 lcrzo_data dataout,
				 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_text_stdin, but append */
/*fr idem lcrzo_data_initm_text_stdin, mais concatene */
int lcrzo_data_appendm_text_stdin(const char * message,
				  const char * defaulttext,
				  lcrzo_int32 dataoutcurrentsize,
				  lcrzo_data *pdataout,
				  lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_coretext_stdin
   Description :
     Initialize a data array, with an text string entered 
     by the user. This text string is cleaned before 
     use : the spaces, tabs, and newline at the begining or
     the end of the text are removed
   Input parameter(s) :
     message : message to print before asking user
     defaulcoretext : text string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the text
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_coretext_stdin
   Description :
     Initialise un tableau de donnees, avec du texte entre
     par l'utilisateur.
     Ce texte est purge avant emploi : les expaces, tabulations,
     et sauts de lignes en debut ou fin de chaine sont supprimes
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du texte
     defaultcoretext : texte a utiliser si l'utilisateur
                       n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra le texte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_coretext_stdin(const char * message,
				   const char * defaultcoretext,
				   lcrzo_int32 dataoutmaxsize,
				   lcrzo_data dataout,
				   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_coretext_stdin, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_coretext_stdin, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_coretext_stdin(const char * message,
				    const char * defaultcoretext,
				    lcrzo_data *pdataout,
				    lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_coretext_stdin, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_coretext_stdin, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_coretext_stdin(const char * message,
				     const char * defaultcoretext,
				     lcrzo_int32 dataoutcurrentsize,
				     lcrzo_int32 dataoutmaxsize,
				     lcrzo_data dataout,
				     lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_coretext_stdin, but append */
/*fr idem lcrzo_data_initm_coretext_stdin, mais concatene */
int lcrzo_data_appendm_coretext_stdin(const char * message,
				      const char * defaultcoretext,
				      lcrzo_int32 dataoutcurrentsize,
				      lcrzo_data *pdataout,
				      lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_text_passwd
   Description :
     Initialize a data array, with an text string entered 
     by the user. This string is not echoed on the screen.
   Input parameter(s) :
     message : message to print before asking user
     defaultext : text string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the text
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_text_passwd
   Description :
     Initialise un tableau de donnees, avec du texte entre
     par l'utilisateur. Ce texte ne s'affiche pas a l'ecran.
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du texte
     defaulttext : texte a utiliser si l'utilisateur n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra le texte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_text_passwd(const char * message,
				const char * defaulttext,
				lcrzo_int32 dataoutmaxsize,
				lcrzo_data dataout,
				lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_text_passwd, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_text_passwd, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_text_passwd(const char * message,
				 const char * defaulttext,
				 lcrzo_data *pdataout,
				 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_text_passwd, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_text_passwd, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_text_passwd(const char * message,
				  const char * defaulttext,
				  lcrzo_int32 dataoutcurrentsize,
				  lcrzo_int32 dataoutmaxsize,
				  lcrzo_data dataout,
				  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_text_passwd, but append */
/*fr idem lcrzo_data_initm_text_passwd, mais concatene */
int lcrzo_data_appendm_text_passwd(const char * message,
				   const char * defaulttext,
				   lcrzo_int32 dataoutcurrentsize,
				   lcrzo_data *pdataout,
				   lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_coretext_passwd
   Description :
     Initialize a data array, with an text string entered 
     by the user. This text string is cleaned before 
     use : the spaces, tabs, and newline at the begining or
     the end of the text are removed. This text is no echoed
     ont the screen.
   Input parameter(s) :
     message : message to print before asking user
     defaulcoretext : text string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the text
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_coretext_passwd
   Description :
     Initialise un tableau de donnees, avec du texte entre
     par l'utilisateur.
     Ce texte est purge avant emploi : les expaces, tabulations,
     et sauts de lignes en debut ou fin de chaine sont supprimes.
     Ce texte ne s'affiche pas a l'ecran.
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du texte
     defaultcoretext : texte a utiliser si l'utilisateur
                       n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra le texte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_coretext_passwd(const char * message,
				    const char * defaultcoretext,
				    lcrzo_int32 dataoutmaxsize,
				    lcrzo_data dataout,
				    lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_coretext_passwd, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_coretext_passwd, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_coretext_passwd(const char * message,
				     const char * defaultcoretext,
				     lcrzo_data *pdataout,
				     lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_coretext_passwd, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_coretext_passwd, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_coretext_passwd(const char * message,
				      const char * defaultcoretext,
				      lcrzo_int32 dataoutcurrentsize,
				      lcrzo_int32 dataoutmaxsize,
				      lcrzo_data dataout,
				      lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_coretext_passwd, but append */
/*fr idem lcrzo_data_initm_coretext_passwd, mais concatene */
int lcrzo_data_appendm_coretext_passwd(const char * message,
				       const char * defaultcoretext,
				       lcrzo_int32 dataoutcurrentsize,
				       lcrzo_data *pdataout,
				       lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_hexa_stdin
   Description :
     Initialize a data array, with data presented 
     in hexa form, and entered by the user
   Input parameter(s) :
     message : message to print before asking user
     defaulhexa : hexa string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the equivalent of
               the hexa string
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_hexa_stdin
   Description :
     Initialise un tableau de donnees, avec des donnees
     representees sous forme hexadecimale, saisies
     par l'utilisateur
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer de l'hexa
     defaulthexa: hexa a utiliser si l'utilisateur n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees equivalentes 
               a la chaine hexa
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_hexa_stdin(const char * message,
			       const char * defaulthexa,
			       lcrzo_int32 dataoutmaxsize,
			       lcrzo_data dataout,
			       lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_hexa_stdin, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_hexa_stdin, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_hexa_stdin(const char * message,
				const char * defaulthexa,
				lcrzo_data *pdataout,
				lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_hexa_stdin, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_hexa_stdin, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_hexa_stdin(const char * message,
				 const char * defaulthexa,
				 lcrzo_int32 dataoutcurrentsize,
				 lcrzo_int32 dataoutmaxsize,
				 lcrzo_data dataout,
				 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_hexa_stdin, but append */
/*fr idem lcrzo_data_initm_hexa_stdin, mais concatene */
int lcrzo_data_appendm_hexa_stdin(const char * message,
				  const char * defaulthexa,
				  lcrzo_int32 dataoutcurrentsize,
				  lcrzo_data *pdataout,
				  lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_mixed_stdin
   Description :
     Initialize a data array, with data presented 
     in mixed form, and entered by the user (see frame at 
     the top of the help for this module describing what's
     a mixed string)
   Input parameter(s) :
     message : message to print before asking user
     defaulmixed : mixed string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the equivalent of
               the mixed string
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_mixed_stdin
   Description :
     Initialise un tableau de donnees, avec des donnees
     representees sous forme mixte, saisies  par l'utilisateur
     (cf. le  cadre, en haut de l'aide de ce module, decrivant ce 
     qu'est une chaine mixte)
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du mixte
     defaultmixed: chaine mixte a utiliser si l'utilisateur 
                   n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees equivalentes 
               a la chaine mixte
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_mixed_stdin(const char * message,
				const char * defaultmixed,
				lcrzo_int32 dataoutmaxsize,
				lcrzo_data dataout,
				lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_mixed_stdin, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_mixed_stdin, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_mixed_stdin(const char * message,
				 const char * defaultmixed,
				 lcrzo_data *pdataout,
				 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_mixed_stdin, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_mixed_stdin, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_mixed_stdin(const char * message,
				  const char * defaultmixed,
				  lcrzo_int32 dataoutcurrentsize,
				  lcrzo_int32 dataoutmaxsize,
				  lcrzo_data dataout,
				  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_mixed_stdin, but append */
/*fr idem lcrzo_data_initm_mixed_stdin, mais concatene */
int lcrzo_data_appendm_mixed_stdin(const char * message,
				   const char * defaultmixed,
				   lcrzo_int32 dataoutcurrentsize,
				   lcrzo_data *pdataout,
				   lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_init_base64_stdin
   Description :
     Initialize a data array, with data presented 
     in base64 form, and entered by the user
   Input parameter(s) :
     message : message to print before asking user
     defaulbase64 : base64 string to use if user enters nothing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the equivalent of
               the base64 string
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_init_base64_stdin
   Description :
     Initialise un tableau de donnees, avec des donnees
     representees sous forme base64, saisies par l'utilisateur
   Parametre(s) d'entree :
     message : message a afficher avant de demander a
               l'utilisateur l'entrer du base64
     defaultbase64: base64 a utiliser si l'utilisateur n'entre rien
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees equivalentes 
               a la chaine base64
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_init_base64_stdin(const char * message,
				 const char * defaultbase64,
				 lcrzo_int32 dataoutmaxsize,
				 lcrzo_data dataout,
				 lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_base64_stdin, but allocate memory. Read the
   help of lcrzo_data_initm_data for an explanation of the
   parameter used for allocation (pdataout)*/
/*fr idem lcrzo_data_init_base64_stdin, mais alloue de la memoire. Lisez
   l'aide de lcrzo_data_initm_data pour avoir une explication
   du parametre utilise lorsque l'on alloue (pdataout)*/
int lcrzo_data_initm_base64_stdin(const char * message,
				  const char * defaultbase64,
				  lcrzo_data *pdataout,
				  lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_init_base64_stdin, but append. Read the
   help of lcrzo_data_append_data for an explanation of the
   parameter used for appending (dataoutcurrentsize)*/
/*fr idem lcrzo_data_init_base64_stdin, mais concatene. Lisez l'aide
   de lcrzo_data_append_data pour avoir une explication du 
   parametre utilise lors d'une concatenation (dataoutcurrentsize)*/
int lcrzo_data_append_base64_stdin(const char * message,
				   const char * defaultbase64,
				   lcrzo_int32 dataoutcurrentsize,
				   lcrzo_int32 dataoutmaxsize,
				   lcrzo_data dataout,
				   lcrzo_int32 *pdataoutsize);

/*us idem lcrzo_data_initm_base64_stdin, but append */
/*fr idem lcrzo_data_initm_base64_stdin, mais concatene */
int lcrzo_data_appendm_base64_stdin(const char * message,
				    const char * defaultbase64,
				    lcrzo_int32 dataoutcurrentsize,
				    lcrzo_data *pdataout,
				    lcrzo_int32 *pdataoutsize);

/*-------------------------------------------------------------*/
/*us a lcrzo_regexp is an array of found positions during a
   regular expression search (do a "man regexp" if you don't
   know what is a "regular expression").
   The element at index 0 define the location of the 
   global expression.
   The elements at index 1, 2, etc. define the found string
   between '(' and ')'.
   The elements whose values are 0 define the end of the 
   found expression.
   For example, if the data is "houses" and the regular expression
   is "o(.(s))", then :
          +---+---+---+---+---+---+
	  | h | o | u | s | e | s |
	  +---+---+---+---+---+---+
	  1   2   3   4   5   6   7
	 -7  -6  -5  -4  -3  -2  -1
     'regexp[0]' indicates "ous", so :
          regexp[0].startindex_negative == -6
	  regexp[0].startindex_positive == 2
	  regexp[0].endindex_negative == -3
	  regexp[0].endindex_positive == 5
     'regexp[1]' indicates "us", so :
          regexp[0].startindex_negative == -5
	  regexp[0].startindex_positive == 3
	  regexp[0].endindex_negative == -3
	  regexp[0].endindex_positive == 5
     'regexp[2]' indicates "s", so :
          regexp[0].startindex_negative == -4
	  regexp[0].startindex_positive == 4
	  regexp[0].endindex_negative == -3
	  regexp[0].endindex_positive == 5
     all the regexp[i] with i>2 indicates the end, so :
          regexp[0].startindex_negative == 0
	  regexp[0].startindex_positive == 0
	  regexp[0].endindex_negative == 0
	  regexp[0].endindex_positive == 0
*/
/*fr un lcrzo_regexp est un tableau contenant les positions trouvees
   lors d'une recherche par expression reguliere (faites "man regexp"
   si vous ne savez pas ce qu'est une expression reguliere).
   L'element a l'indice 0 definit la place de l'expression globale
   trouvee.
   Les elements a l'indice 1, 2, etc. definissent les donnees 
   trouvees entre '(' et ')'.
   Les elements dont la valeur est 0 indiquent la fin.
   Par exemple, si les donnees sont "papier" et l'expression
   reguliere est "a(.(i))", alors :
          +---+---+---+---+---+---+
	  | p | a | p | i | e | r |
	  +---+---+---+---+---+---+
	  1   2   3   4   5   6   7
	 -7  -6  -5  -4  -3  -2  -1
     'regexp[0]' indique "api", donc :
          regexp[0].startindex_negative == -6
	  regexp[0].startindex_positive == 2
	  regexp[0].endindex_negative == -3
	  regexp[0].endindex_positive == 5
     'regexp[1]' indique "pi", donc :
          regexp[0].startindex_negative == -5
	  regexp[0].startindex_positive == 3
	  regexp[0].endindex_negative == -3
	  regexp[0].endindex_positive == 5
     'regexp[2]' indique "i", donc :
          regexp[0].startindex_negative == -4
	  regexp[0].startindex_positive == 4
	  regexp[0].endindex_negative == -3
	  regexp[0].endindex_positive == 5
     tous les regexp[i] avec i>2 indiquent la fin, donc :
          regexp[0].startindex_negative == 0
	  regexp[0].startindex_positive == 0
	  regexp[0].endindex_negative == 0
	  regexp[0].endindex_positive == 0
*/

/*us we cannot have more than 30 parenthesis. This should be 
   sufficient in all cases, but you may want to increase it*/
/*fr on ne peut pas avoir plus de 30 parentheses. Cela doit
   etre suffisant dans tous les cas, mais vous pouvez choisir
   une valeur plus grande.*/
#define LCRZO_REGEXP_MAXPAIRS 30
typedef struct 
{ lcrzo_int32 startindex_negative;
  lcrzo_int32 startindex_positive;
  lcrzo_int32 endindex_negative;
  lcrzo_int32 endindex_positive;
} lcrzo_regexp[LCRZO_REGEXP_MAXPAIRS+1];

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_search_char
   Description :
     Search a character in a data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we search c
                        else we search 'not c'
     datainstartindex : start index of the range to search
			The meaning of indexes is explained 
			at the top of the help for this module,
     datainendindex : end index of the range to search
     c : character to search
   Input/output parameter(s) :
     *pfoundindex_negative : index of the found character
                             (negative, because from the end)
     *pfoundindex_positive : index of the found character
                             (positive, because from the start)
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : the searched element was found
     LCRZO_ERR_OKSEARCHNOTFOUND : the searched element wasn't found
*/
/*fr Nom : lcrzo_data_search_char
   Description :
     Cherche un caractere dans un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche c
                        sinon, cherche un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
			au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     c : caractere a chercher
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfoundindex_negative : indice du caractere trouve
                             (negative, car compte depuis la fin)
     *pfoundindex_positive : indice du caractere trouve
                             (positive, car compte depuis le debut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : l'element cherche a ete trouve
     LCRZO_ERR_OKSEARCHNOTFOUND : l'element cherche n'a pas ete trouve
*/
int lcrzo_data_search_char(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_bool searchcomplement, 
			   lcrzo_int32 datainstartindex,
			   lcrzo_int32 datainendindex,
			   lcrzo_uint8 c,
			   lcrzo_int32 *pfoundindex_negative,
			   lcrzo_int32 *pfoundindex_positive);

/*us Name : lcrzo_data_search_nchar
   Description :
     Search a set of character in a data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we search the set
                        else we search a character not in the set
     datainstartindex : start index of the range to search
			The meaning of indexes is explained 
			at the top of the help for this module,
     datainendindex : end index of the range to search
     searchlist : array containing the set of characters to search
                  For example, use "aRt" to search for 'a', 'R' or 't'
     searchlistsize : size of this array
   Input/output parameter(s) :
     *pfoundindex_negative : index of the found character
                             (negative, because from the end)
     *pfoundindex_positive : index of the found character
                             (positive, because from the start)
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : the searched element was found
     LCRZO_ERR_OKSEARCHNOTFOUND : the searched element wasn't found
*/
/*fr Nom : lcrzo_data_search_nchar
   Description :
     Cherche un ensemble de caracteres dans un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche un caractere dans l'ensemble
                        sinon, cherche un caractere n'etant pas dans
			l'ensemble
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
			au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     searchlist : tableau contenant l'ensemble des caracteres a
                  chercher. Par exemple, utilisez "aRt" pour chercher
		  'a', 'R' ou 't'
     searchlistsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfoundindex_negative : indice du caractere trouve
                             (negative, car compte depuis la fin)
     *pfoundindex_positive : indice du caractere trouve
                             (positive, car compte depuis le debut)
   Valeurs de retour normales :
     LCRZO_ERR_OK : l'element cherche a ete trouve
     LCRZO_ERR_OKSEARCHNOTFOUND : l'element cherche n'a pas ete trouve
*/
int lcrzo_data_search_nchar(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_bool searchcomplement, 
			    lcrzo_int32 datainstartindex,
			    lcrzo_int32 datainendindex,
			    lcrzo_constdata searchlist,
			    lcrzo_int32 searchlistsize,
			    lcrzo_int32 *pfoundindex_negative,
			    lcrzo_int32 *pfoundindex_positive);

/*us Name : lcrzo_data_search_data
   Description :
     Search data in a data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we search searcheddata
                        else we search a data different from searcheddata
     datainstartindex : start index of the range to search
			The meaning of indexes is explained 
			at the top of the help for this module,
     datainendindex : end index of the range to search
     searcheddata : searched data
     searcheddatasize : size of the seached data
   Input/output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : the searched element was found
     LCRZO_ERR_OKSEARCHNOTFOUND : the searched element wasn't found
*/
/*fr Nom : lcrzo_data_search_data
   Description :
     Cherche des donnees dans un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche un caractere dans l'ensemble
                        sinon, cherche un caractere n'etant pas dans
			l'ensemble
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
			au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     searcheddata : donnees a chercher dans datain
     searcheddatasize : taille de searcheddata
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : l'element cherche a ete trouve
     LCRZO_ERR_OKSEARCHNOTFOUND : l'element cherche n'a pas ete trouve
*/
int lcrzo_data_search_data(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_bool searchcomplement, 
			   lcrzo_int32 datainstartindex,
			   lcrzo_int32 datainendindex,
			   lcrzo_constdata searcheddata,
			   lcrzo_int32 searcheddatasize,
			   lcrzo_int32 *pstartfoundindex_negative,
			   lcrzo_int32 *pstartfoundindex_positive,
			   lcrzo_int32 *pendfoundindex_negative,
			   lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_search_text
   Description :
     Search a string in a data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we search searcheddata
                        else we search a data different from searcheddata
     datainstartindex : start index of the range to search
			The meaning of indexes is explained 
			at the top of the help for this module,
     datainendindex : end index of the range to search
     searchedtext : searched string
   Input/output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : the searched element was found
     LCRZO_ERR_OKSEARCHNOTFOUND : the searched element wasn't found
*/
/*fr Nom : lcrzo_data_search_text
   Description :
     Cherche une chaine de caracteres dans un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche un caractere dans l'ensemble
                        sinon, cherche un caractere n'etant pas dans
			l'ensemble
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
			au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     searchedstring : chaine a chercher dans datain
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : l'element cherche a ete trouve
     LCRZO_ERR_OKSEARCHNOTFOUND : l'element cherche n'a pas ete trouve
*/
int lcrzo_data_search_text(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_bool searchcomplement, 
			   lcrzo_int32 datainstartindex,
			   lcrzo_int32 datainendindex,
			   lcrzo_conststring searchedtext,
			   lcrzo_int32 *pstartfoundindex_negative,
			   lcrzo_int32 *pstartfoundindex_positive,
			   lcrzo_int32 *pendfoundindex_negative,
			   lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_search_re
   Description :
     Search a regular expression in a data array
     ("man regexp", if you don't know what's a regular expression)
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we search searcheddata
                        else we search a data different from searcheddata
     datainstartindex : start index of the range to search
			The meaning of indexes is explained 
			at the top of the help for this module,
     datainendindex : end index of the range to search
     regularexpression : searched regular expression.
                         For example : "a.", "f[A-Z]", etc.
   Input/output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : the searched element was found
     LCRZO_ERR_OKSEARCHNOTFOUND : the searched element wasn't found
*/
/*fr Nom : lcrzo_data_search_re
   Description :
     Cherche une expression reguliere dans un tableau de donnees
     ("man regexp" pour savoir ce qu'est une expression reguliere)
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche un caractere dans l'ensemble
                        sinon, cherche un caractere n'etant pas dans
			l'ensemble
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
			au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     regularexpression : expression reguiliere a chercher dans datain
                         Par exemple : "a.", "f[A-Z]", etc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : l'element cherche a ete trouve
     LCRZO_ERR_OKSEARCHNOTFOUND : l'element cherche n'a pas ete trouve
*/
int lcrzo_data_search_re(lcrzo_constdata datain,
			 lcrzo_int32 datainsize,
			 lcrzo_bool searchcomplement, 
			 lcrzo_int32 datainstartindex,
			 lcrzo_int32 datainendindex,
			 lcrzo_conststring regularexpression,
			 lcrzo_int32 *pstartfoundindex_negative,
			 lcrzo_int32 *pstartfoundindex_positive,
			 lcrzo_int32 *pendfoundindex_negative,
			 lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_search_regexp
   Description :
     Search a regular expression in a data array
     ("man regexp", if you don't know what's a regular expression)
     This function is more complex than "lcrzo_data_search_re"
     because we return the complete lcrzo_regexp array.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we search searcheddata
                        else we search a data different from searcheddata
     datainstartindex : start index of the range to search
			The meaning of indexes is explained 
			at the top of the help for this module,
     datainendindex : end index of the range to search
     regularexpression : searched regular expression.
                         For example : "a(.)", "f([A-Z])", etc.
   Input/output parameter(s) :
     arrayfoundindex : lcrzo_regexp containing the indexes of
                       the found data
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : the searched element was found
     LCRZO_ERR_OKSEARCHNOTFOUND : the searched element wasn't found
*/
/*fr Nom : lcrzo_data_search_regexp
   Description :
     Cherche une expression reguliere dans un tableau de donnees
     ("man regexp" pour savoir ce qu'est une expression reguliere)
     Cette fonction est plus complexe que "lcrzo_data_search_re"
     car on retourne un tableau lcrzo_regexp complet.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche un caractere dans l'ensemble
                        sinon, cherche un caractere n'etant pas dans
			l'ensemble
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
			au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     regularexpression : expression reguiliere a chercher dans datain
                         Par exemple : "a.", "f[A-Z]", etc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     arrayfoundindex : lcrzo_regexp contenant les indices des
                        donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : l'element cherche a ete trouve
     LCRZO_ERR_OKSEARCHNOTFOUND : l'element cherche n'a pas ete trouve
*/
int lcrzo_data_search_regexp(lcrzo_constdata datain,
			     lcrzo_int32 datainsize,
			     lcrzo_bool searchcomplement, 
			     lcrzo_int32 datainstartindex,
			     lcrzo_int32 datainendindex,
			     lcrzo_conststring regularexpression,
			     lcrzo_regexp arrayfoundindex);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_token_char
   Description :
     This function can find data enclosed between delimiters.
     This function is to be used in a loop.
     The delimiters are characters.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, the delimiter is c c
                        else the delimiter is not c
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     c : character used as a delimiter
   Input/output parameter(s) :
     *pindexindicator : this is internally used.
                        it must be set to zero for the first call
   Output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Normal return values :
     LCRZO_ERR_OK : we found data in datain
     LCRZO_ERR_OKDATAEND : datainendindex was reached
   Example :
     p=0;
     while (1)
     { ret=lcrzo_data_token_char("badbebcebe", 10, 0, +1, -1,
                                 'b', &p, NULL, &inf, NULL, &sup);
       if (ret==LCRZO_ERR_OKDATAEND) break;
       else if (ret!=LCRZO_ERR_OK)   return(ret);
       /now, we can use inf and sup/
       ...
     }
     In this example, the loop will work on : "", "ad", "e", "ce", "e"
*/
/*fr Nom : lcrzo_data_token_char
   Description :
     Cette fonction trouve des donnes comprises entre des
     delimiteurs.
     Cette fonction a ete concue pour etre utilisee dans 
     une boucle.
     Les delimiteurs sont des caracteres.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche c
                        sinon, cherche un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     c : caractere utilise comme delimiteur
   Parametre(s) d'entree-sortie :
     *pindexindicator : cette variable est utilisee en interne.
                        Elle doit etre affecte a zero lors du
			premier appel de la fonction
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : on a trouve de donnees
     LCRZO_ERR_OKDATAEND : datainendindex a ete atteint
   Example :
     p=0;
     while (1)
     { ret=lcrzo_data_token_char("badbebcebe", 10, 0, +1, -1,
                                 'b', &p, NULL, &inf, NULL, &sup);
       if (ret==LCRZO_ERR_OKDATAEND) break;
       else if (ret!=LCRZO_ERR_OK)   return(ret);
       /maintenant, on peut utiliser inf et sup/
       ...
     }
     Dans cet exemple, la boucle va travailler sur
     "", "ad", "e", "ce" et "e".
*/
int lcrzo_data_token_char(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_bool searchcomplement, 
			  lcrzo_int32 datainstartindex,
			  lcrzo_int32 datainendindex,
			  lcrzo_uint8 c,
			  lcrzo_int32 *pindexindicator,
			  lcrzo_int32 *pstartfoundindex_negative,
			  lcrzo_int32 *pstartfoundindex_positive,
			  lcrzo_int32 *pendfoundindex_negative,
			  lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_token_nchar
   Description :
     This function can find data enclosed between delimiters.
     This function is to be used in a loop.
     The delimiter is a set of characters.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, the delimiter is c c
                        else the delimiter is not c
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     searchlist : array containing the set of characters used
                  as delimiters
     searchlistsize : size of this array
   Input/output parameter(s) :
     *pindexindicator : this is internally used.
                        it must be set to zero for the first call
   Output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Normal return values :
     LCRZO_ERR_OK : we found data in datain
     LCRZO_ERR_OKDATAEND : datainendindex was reached
   Example :
     see help for lcrzo_data_token_char
*/
/*fr Nom : lcrzo_data_token_nchar
   Description :
     Cette fonction trouve des donnes comprises entre des
     delimiteurs.
     Cette fonction a ete concue pour etre utilisee dans 
     une boucle.
     Le delimiteur est un ensemble de caracteres.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche c
                        sinon, cherche un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     searchlist : tableau contenant l'ensemble des caracteres a
                  utiliser comme delimiteur
     searchlistsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
     *pindexindicator : cette variable est utilisee en interne.
                        Elle doit etre affecte a zero lors du
			premier appel de la fonction
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : on a trouve de donnees
     LCRZO_ERR_OKDATAEND : datainendindex a ete atteint
   Exemple :
     regardez l'aide de lcrzo_data_token_char
*/
int lcrzo_data_token_nchar(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_bool searchcomplement, 
			   lcrzo_int32 datainstartindex,
			   lcrzo_int32 datainendindex,
			   lcrzo_constdata searchlist,
			   lcrzo_int32 searchlistsize,
			   lcrzo_int32 *pindexindicator,
			   lcrzo_int32 *pstartfoundindex_negative,
			   lcrzo_int32 *pstartfoundindex_positive,
			   lcrzo_int32 *pendfoundindex_negative,
			   lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_token_data
   Description :
     This function can find data enclosed between delimiters.
     This function is to be used in a loop.
     The delimiter is a data array.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, the delimiter is c c
                        else the delimiter is not c
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     searcheddata : data used as delimiter
     searcheddatasize : size of searcheddata
   Input/output parameter(s) :
     *pindexindicator : this is internally used.
                        it must be set to zero for the first call
   Output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Normal return values :
     LCRZO_ERR_OK : we found data in datain
     LCRZO_ERR_OKDATAEND : datainendindex was reached
   Example :
     see help for lcrzo_data_token_char
*/
/*fr Nom : lcrzo_data_token_data
   Description :
     Cette fonction trouve des donnes comprises entre des
     delimiteurs.
     Cette fonction a ete concue pour etre utilisee dans 
     une boucle.
     Le delimiteur est un tableau de donnees.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche c
                        sinon, cherche un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     searcheddata : donnees a utiliser comme delimiteur
     searcheddatasize : taille de searcheddata
   Parametre(s) d'entree-sortie :
     *pindexindicator : cette variable est utilisee en interne.
                        Elle doit etre affecte a zero lors du
			premier appel de la fonction
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : on a trouve de donnees
     LCRZO_ERR_OKDATAEND : datainendindex a ete atteint
   Exemple :
     regardez l'aide de lcrzo_data_token_char
*/
int lcrzo_data_token_data(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_bool searchcomplement, 
			  lcrzo_int32 datainstartindex,
			  lcrzo_int32 datainendindex,
			  lcrzo_constdata searcheddata,
			  lcrzo_int32 searcheddatasize,
			  lcrzo_int32 *pindexindicator,
			  lcrzo_int32 *pstartfoundindex_negative,
			  lcrzo_int32 *pstartfoundindex_positive,
			  lcrzo_int32 *pendfoundindex_negative,
			  lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_token_text
   Description :
     This function can find data enclosed between delimiters.
     This function is to be used in a loop.
     The delimiter is a string.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, the delimiter is c c
                        else the delimiter is not c
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     searchedtext : string used as delimiter
   Input/output parameter(s) :
     *pindexindicator : this is internally used.
                        it must be set to zero for the first call
   Output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Normal return values :
     LCRZO_ERR_OK : we found data in datain
     LCRZO_ERR_OKDATAEND : datainendindex was reached
   Example :
     see help for lcrzo_data_token_char
*/
/*fr Nom : lcrzo_data_token_text
   Description :
     Cette fonction trouve des donnes comprises entre des
     delimiteurs.
     Cette fonction a ete concue pour etre utilisee dans 
     une boucle.
     Le delimiteur est une chaine de caracteres.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche c
                        sinon, cherche un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     searchedtext : chaine a utiliser comme delimiteur
   Parametre(s) d'entree-sortie :
     *pindexindicator : cette variable est utilisee en interne.
                        Elle doit etre affecte a zero lors du
			premier appel de la fonction
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : on a trouve de donnees
     LCRZO_ERR_OKDATAEND : datainendindex a ete atteint
   Exemple :
     regardez l'aide de lcrzo_data_token_char
*/
int lcrzo_data_token_text(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_bool searchcomplement, 
			  lcrzo_int32 datainstartindex,
			  lcrzo_int32 datainendindex,
			  lcrzo_conststring searchedtext,
			  lcrzo_int32 *pindexindicator,
			  lcrzo_int32 *pstartfoundindex_negative,
			  lcrzo_int32 *pstartfoundindex_positive,
			  lcrzo_int32 *pendfoundindex_negative,
			  lcrzo_int32 *pendfoundindex_positive);

/*us Name : lcrzo_data_token_re
   Description :
     This function can find data enclosed between delimiters.
     This function is to be used in a loop.
     The delimiter is a regular expression.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, the delimiter is c c
                        else the delimiter is not c
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     regularexpression : regular expression used as delimiter
   Input/output parameter(s) :
     *pindexindicator : this is internally used.
                        it must be set to zero for the first call
   Output parameter(s) :
     *pstartfoundindex_negative : index of the start of the found data
                                  (negative, because from the end)
     *pstartfoundindex_positive : index of the start of the found data
                                  (positive, because from the start)
     *pendfoundindex_negative : index of the end of the found data
     *pendfoundindex_positive : index of the end of the found data
   Normal return values :
     LCRZO_ERR_OK : we found data in datain
     LCRZO_ERR_OKDATAEND : datainendindex was reached
   Example :
     see help for lcrzo_data_token_char
*/
/*fr Nom : lcrzo_data_token_re
   Description :
     Cette fonction trouve des donnes comprises entre des
     delimiteurs.
     Cette fonction a ete concue pour etre utilisee dans 
     une boucle.
     Le delimiteur est une expression reguliere.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, cherche c
                        sinon, cherche un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     regularexpression : expression reguliere a utiliser comme delimiteur
   Parametre(s) d'entree-sortie :
     *pindexindicator : cette variable est utilisee en interne.
                        Elle doit etre affecte a zero lors du
			premier appel de la fonction
   Parametre(s) de sortie :
     *pstartfoundindex_negative : indice du caractere du debut des
                                  donnees trouvees
				  (negative, car compte depuis la fin)
     *pstartfoundindex_positive : indice du caractere du debut des
                                  donnees trouvees
				  (positive, car compte depuis le debut)
     *pendfoundindex_negative : indice du caractere de fin des
                                donnees trouvees
     *pendfoundindex_positive : indice du caractere de fin des
                                donnees trouvees
   Valeurs de retour normales :
     LCRZO_ERR_OK : on a trouve de donnees
     LCRZO_ERR_OKDATAEND : datainendindex a ete atteint
   Exemple :
     regardez l'aide de lcrzo_data_token_char
*/
int lcrzo_data_token_re(lcrzo_constdata datain,
			lcrzo_int32 datainsize,
			lcrzo_bool searchcomplement, 
			lcrzo_int32 datainstartindex,
			lcrzo_int32 datainendindex,
			lcrzo_conststring regularexpression,
			lcrzo_int32 *pindexindicator,
			lcrzo_int32 *pstartfoundindex_negative,
			lcrzo_int32 *pstartfoundindex_positive,
			lcrzo_int32 *pendfoundindex_negative,
			lcrzo_int32 *pendfoundindex_positive);

/*-------------------------------------------------------------*/
/*us Name : lcrzo_data_replace_char
   Description :
     Replace a character in a data array
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we replace c
                        else we replace 'not c'
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     maxnumberofreplacement : maximum replacement to do 
                              a value zero means illimited
     c_toreplace : character to replace
     data_replacing : data to use for replacement
     data_replacingsize : size of data_replacing
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with datain
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_data_replace_char
   Description :
     Remplace un caractere dans un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, remplace c
                        sinon, remplace un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     maxnumberofreplacement : nombre maxi de remplacements a
                              effectuer. La valeur zero indique
			      qu'il n'y a pas de limite.
     c_toreplace : caractere a remplacer
     data_replacing : donnees a employer pour le remplacement
     data_replacingsize : taille de data_replacing
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra une copie de datain
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_data_replace_char(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_bool searchcomplement, 
			    lcrzo_int32 datainstartindex,
			    lcrzo_int32 datainendindex,
			    lcrzo_int32 maxnumberofreplacement,
			    lcrzo_uint8 c_toreplace,
			    lcrzo_constdata data_replacing,
			    lcrzo_int32 data_replacingsize,
			    lcrzo_int32 dataoutmaxsize,
			    lcrzo_data dataout,
			    lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_data_replacem_char
   Description :
     Replace a character in a data array.
     The outputdata is allocated.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     searchcomplement : if 0, we replace c
                        else we replace 'not c'
     datainstartindex : start index of the range to search
                        The meaning of indexes is explained 
                        at the top of the help for this module,
     datainendindex : end index of the range to search
     maxnumberofreplacement : maximum replacement to do 
                              a value zero means illimited
     c_toreplace : character to replace
     data_replacing : data to use for replacement
     data_replacingsize : size of data_replacing
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with datain
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_replacem_char
   Description :
     Remplace un caractere dans un tableau de donnees.
     Le tableau de sortie est alloue
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     serachcomplement : si 0, remplace c
                        sinon, remplace un caractere different de c
     datainstartindex : indice de depart de l'intervalle a chercher
                        La signification des 'indices' est donnee
                        au debut de l'aide de ce module.
     datainendindex : indice de fin de l'intervalle a chercher
     maxnumberofreplacement : nombre maxi de remplacements a
                              effectuer. La valeur zero indique
			      qu'il n'y a pas de limite.
     c_toreplace : caractere a remplacer
     data_replacing : donnees a employer pour le remplacement
     data_replacingsize : taille de data_replacing
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra une copie de
                 datain
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_replacem_char(lcrzo_constdata datain,
			     lcrzo_int32 datainsize,
			     lcrzo_bool searchcomplement, 
			     lcrzo_int32 datainstartindex,
			     lcrzo_int32 datainendindex,
			     lcrzo_int32 maxnumberofreplacement,
			     lcrzo_uint8 c_toreplace,
			     lcrzo_constdata data_replacing,
			     lcrzo_int32 data_replacingsize,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize);

/*us idem, but replace a set of characters */
/*fr idem, mais remplace un ensemble de caarcteres */
int lcrzo_data_replace_nchar(lcrzo_constdata datain,
			     lcrzo_int32 datainsize,
			     lcrzo_bool searchcomplement, 
			     lcrzo_int32 datainstartindex,
			     lcrzo_int32 datainendindex,
			     lcrzo_int32 maxnumberofreplacement,
			     lcrzo_constdata searchlist,
			     lcrzo_int32 searchlistsize,
			     lcrzo_constdata data_replacing,
			     lcrzo_int32 data_replacingsize,
			     lcrzo_int32 dataoutmaxsize,
			     lcrzo_data dataout,
			     lcrzo_int32 *pdataoutsize);
int lcrzo_data_replacem_nchar(lcrzo_constdata datain,
			      lcrzo_int32 datainsize,
			      lcrzo_bool searchcomplement, 
			      lcrzo_int32 datainstartindex,
			      lcrzo_int32 datainendindex,
			      lcrzo_int32 maxnumberofreplacement,
			      lcrzo_constdata searchlist,
			      lcrzo_int32 searchlistsize,
			      lcrzo_constdata data_replacing,
			      lcrzo_int32 data_replacingsize,
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize);

/*us idem, but replace data */
/*fr idem, mais remplace des donnees */
int lcrzo_data_replace_data(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_bool searchcomplement, 
			    lcrzo_int32 datainstartindex,
			    lcrzo_int32 datainendindex,
			    lcrzo_int32 maxnumberofreplacement,
			    lcrzo_constdata searcheddata,
			    lcrzo_int32 searcheddatasize,
			    lcrzo_constdata data_replacing,
			    lcrzo_int32 data_replacingsize,
			    lcrzo_int32 dataoutmaxsize,
			    lcrzo_data dataout,
			    lcrzo_int32 *pdataoutsize);
int lcrzo_data_replacem_data(lcrzo_constdata datain,
			     lcrzo_int32 datainsize,
			     lcrzo_bool searchcomplement, 
			     lcrzo_int32 datainstartindex,
			     lcrzo_int32 datainendindex,
			     lcrzo_int32 maxnumberofreplacement,
			     lcrzo_constdata searcheddata,
			     lcrzo_int32 searcheddatasize,
			     lcrzo_constdata data_replacing,
			     lcrzo_int32 data_replacingsize,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize);

/*us idem, but replace a string */
/*fr idem, mais remplace une chaine texte */
int lcrzo_data_replace_text(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_bool searchcomplement, 
			    lcrzo_int32 datainstartindex,
			    lcrzo_int32 datainendindex,
			    lcrzo_int32 maxnumberofreplacement,
			    lcrzo_conststring searchedtext,
			    lcrzo_constdata data_replacing,
			    lcrzo_int32 data_replacingsize,
			    lcrzo_int32 dataoutmaxsize,
			    lcrzo_data dataout,
			    lcrzo_int32 *pdataoutsize);
int lcrzo_data_replacem_text(lcrzo_constdata datain,
			     lcrzo_int32 datainsize,
			     lcrzo_bool searchcomplement, 
			     lcrzo_int32 datainstartindex,
			     lcrzo_int32 datainendindex,
			     lcrzo_int32 maxnumberofreplacement,
			     lcrzo_conststring searchedtext,
			     lcrzo_constdata data_replacing,
			     lcrzo_int32 data_replacingsize,
			     lcrzo_data *pdataout,
			     lcrzo_int32 *pdataoutsize);

/*us idem, but replace data matched by a regular expression */
/*fr idem, mais remplace les donnees correspondant a une 
   expression reguliere*/
int lcrzo_data_replace_re(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_bool searchcomplement, 
			  lcrzo_int32 datainstartindex,
			  lcrzo_int32 datainendindex,
			  lcrzo_int32 maxnumberofreplacement,
			  lcrzo_conststring regularexpression,
			  lcrzo_constdata data_replacing,
			  lcrzo_int32 data_replacingsize,
			  lcrzo_int32 dataoutmaxsize,
			  lcrzo_data dataout,
			  lcrzo_int32 *pdataoutsize);
int lcrzo_data_replacem_re(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_bool searchcomplement, 
			   lcrzo_int32 datainstartindex,
			   lcrzo_int32 datainendindex,
			   lcrzo_int32 maxnumberofreplacement,
			   lcrzo_conststring regularexpression,
			   lcrzo_constdata data_replacing,
			   lcrzo_int32 data_replacingsize,
			   lcrzo_data *pdataout,
			   lcrzo_int32 *pdataoutsize);

/*La 'chainederemplacement' peut contenir $1 a $9 pour indiquer
  les parametres matches dans l'expression ("$$" correspond au
  vrai caractere '$'). 
  Par exemple :
   expression="a(.)b",    chainederemplacement="A$1$1"
   expression="a(.)(b.)", chainederemplacement="A$2$1"
*/

int lcrzo_data_replace_regexp(lcrzo_constdata datain,
			      lcrzo_int32 datainsize,
			      lcrzo_bool searchcomplement, 
			      lcrzo_int32 datainstartindex,
			      lcrzo_int32 datainendindex,
			      lcrzo_int32 maxnumberofreplacement,
			      lcrzo_conststring regularexpression,
			      lcrzo_constdata data_replacing,
			      lcrzo_int32 data_replacingsize,
			      lcrzo_int32 dataoutmaxsize,
			      lcrzo_data dataout,
			      lcrzo_int32 *pdataoutsize);
int lcrzo_data_replacem_regexp(lcrzo_constdata datain,
			       lcrzo_int32 datainsize,
			       lcrzo_bool searchcomplement, 
			       lcrzo_int32 datainstartindex,
			       lcrzo_int32 datainendindex,
			       lcrzo_int32 maxnumberofreplacement,
			       lcrzo_conststring regularexpression,
			       lcrzo_constdata data_replacing,
			       lcrzo_int32 data_replacingsize,
			       lcrzo_data *pdataout,
			       lcrzo_int32 *pdataoutsize);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_hexa_init_data
   Description :
     Convert data to a string in hexa
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     hexaoutmaxsize : max size which can be stored in hexaout
   Input/output parameter(s) :
   Output parameter(s) :
     hexaout : output string
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : hexaoutmaxsize reached before
                                 being able to store all the needed
                                 data in hexaout
*/
/*fr Nom : lcrzo_hexa_init_data
   Description :
     Initialise une chaine hexa, avec un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     hexaoutmaxsize : taille maximale qui peut etre stockee
                      dans hexaout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     hexaout : chaine de sortie
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : hexaoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans hexaout
*/
int lcrzo_hexa_init_data(lcrzo_constdata datain,
			 lcrzo_int32 datainsize,
			 lcrzo_int32 hexaoutmaxsize,
			 lcrzo_string hexaout);

/*us Name : lcrzo_hexa_initm_data
   Description :
     Convert data to a string in hexa.
     This string is allocated.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *phexaout : output string. 
                 The memory will have to be freed by the
                 user with 'free(*phexaout)'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_hexa_initm_data
   Description :
     Initialise une chaine hexa, avec un tableau de donnees.
     La chaine hexa est allouee
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *phexaout : chaine de sortie
                 La memoire devra etre liberee par l'utilisateur 
                 avec 'free(*phexaaout)'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_hexa_initm_data(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_string *phexaout);

/*us Name : lcrzo_mixed_init_data
   Description :
     Convert data to a string in mixed
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     mixedoutmaxsize : max size which can be stored in mixedout
   Input/output parameter(s) :
   Output parameter(s) :
     mixedout : output string
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : mixedoutmaxsize reached before
                                 being able to store all the needed
                                 data in mixedout
*/
/*fr Nom : lcrzo_mixed_init_data
   Description :
     Initialise une chaine mixed, avec un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     mixedoutmaxsize : taille maximale qui peut etre stockee
                      dans mixedout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     mixedout : chaine de sortie
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : mixedoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans mixedout
*/
int lcrzo_mixed_init_data(lcrzo_constdata datain,
			  lcrzo_int32 datainsize,
			  lcrzo_int32 mixedoutmaxsize,
			  lcrzo_string mixedout);

/*us Name : lcrzo_mixed_initm_data
   Description :
     Convert data to a string in mixed.
     This string is allocated.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *pmixedout : output string. 
                 The memory will have to be freed by the
                 user with 'free(*pmixedout)'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_mixed_initm_data
   Description :
     Initialise une chaine mixed, avec un tableau de donnees.
     La chaine mixed est allouee
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pmixedout : chaine de sortie
                 La memoire devra etre liberee par l'utilisateur 
                 avec 'free(*pmixedaout)'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_mixed_initm_data(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_string *pmixedout);

/*us Name : lcrzo_base64_init_data
   Description :
     Convert data to a string in base64
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
     base64outmaxsize : max size which can be stored in base64out
   Input/output parameter(s) :
   Output parameter(s) :
     base64out : output string
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : base64outmaxsize reached before
                                 being able to store all the needed
                                 data in base64out
*/
/*fr Nom : lcrzo_base64_init_data
   Description :
     Initialise une chaine base64, avec un tableau de donnees
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
     base64outmaxsize : taille maximale qui peut etre stockee
                      dans base64out
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     base64out : chaine de sortie
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : base64outmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans base64out
*/
int lcrzo_base64_init_data(lcrzo_constdata datain,
			   lcrzo_int32 datainsize,
			   lcrzo_int32 base64outmaxsize,
			   lcrzo_string base64out);

/*us Name : lcrzo_base64_initm_data
   Description :
     Convert data to a string in base64.
     This string is allocated.
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *pbase64out : output string. 
                 The memory will have to be freed by the
                 user with 'free(*pbase64out)'
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_base64_initm_data
   Description :
     Initialise une chaine base64, avec un tableau de donnees.
     La chaine base64 est allouee
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pbase64out : chaine de sortie
                 La memoire devra etre liberee par l'utilisateur 
                 avec 'free(*pbase64aout)'
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_base64_initm_data(lcrzo_constdata datain,
			    lcrzo_int32 datainsize,
			    lcrzo_string *pbase64out);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_data_checksum
   Description :
     Compute the checksum of a data array 
   Input parameter(s) :
     datain : array containing input data
     datainsize : size of this array
   Input/output parameter(s) :
   Output parameter(s) :
     *pchecksum : checksum of datain.
                  Note : this checksum is in host byte order
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_data_checksum
   Description :
     Calcule le checksum d'un tableau de donnees.
   Parametre(s) d'entree :
     datain : tableau contenant les donnees d'entree
     datainsize : taille de ce tableau
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pchecksum : checksum de datain.
                  Note : ce checksum est dans l'ordre 'host'
		         (et non dans l'ordre 'network')
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_data_checksum(lcrzo_constdata data,
			lcrzo_int32 datasize, 
			lcrzo_uint16 *pchecksum);
