/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdlib.h>
 #include <stdio.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdlib.h>
 #include <stdio.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif


/*---------------------------------------------------------------*/
int lcrzo_device_init(const char *device_src, lcrzo_device device_dest)
{ lcrzo_er(lcrzo_string_init_coretext(device_src, LCRZO_DEVICE_MAXBYTES, 
				      device_dest));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_device_equal(const lcrzo_device device1, const lcrzo_device device2)
{ lcrzo_device dev1, dev2;
  int retour;

  retour=lcrzo_string_init_coretext(device1, LCRZO_DEVICE_MAXBYTES, dev1);
  if (retour!=LCRZO_ERR_OK) return(0);
  retour=lcrzo_string_init_coretext(device2, LCRZO_DEVICE_MAXBYTES, dev2);
  if (retour!=LCRZO_ERR_OK) return(0);
  if (!strcmp(dev1, dev2)) return(1);
  return(0);
}

/*---------------------------------------------------------------*/
int lcrzo_device_cmp(const lcrzo_device device1, const lcrzo_device device2)
{ lcrzo_device dev1, dev2;
  int retour;

  retour=lcrzo_string_init_coretext(device1, LCRZO_DEVICE_MAXBYTES, dev1);
  if (retour!=LCRZO_ERR_OK) return(2);
  retour=lcrzo_string_init_coretext(device2, LCRZO_DEVICE_MAXBYTES, dev2);
  if (retour!=LCRZO_ERR_OK) return(2);

  retour=strcmp(dev1, dev2);
  if (retour<0) return(-1);
  else if (retour>0) return(+1);
  else return(0);
}

/*---------------------------------------------------------------*/
int lcrzo_device_initdefault(lcrzo_device device)
{ if (device!=NULL) device[0]='\0';
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_mtu_init_device(const lcrzo_device device,
			  lcrzo_uint32 *pmtu)
{ lcrzo_int32 count, i;
  lcrzo_device founddevice, searcheddevice;

  /*verification du parametre*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_device_init(device, searcheddevice));
  /* search the device */
  lcrzo_er(lcrzo_conf_devices_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_devices_value_pos(i, founddevice, NULL, NULL, NULL,
					  pmtu, NULL, NULL));
    if ( lcrzo_device_equal(founddevice, searcheddevice) )
    { return(LCRZO_ERR_OK);
    }
  } 
  /* unknown device */
  return(LCRZO_ERR_SPUNKNOWNDEVICE);
}

/*---------------------------------------------------------------*/
int lcrzo_etha_init_device(const lcrzo_device device,
			   lcrzo_etha etha)
{ lcrzo_int32 count, i;
  lcrzo_device founddevice, searcheddevice;

  /*verification du parametre*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_device_init(device, searcheddevice));
  /* search the device */
  lcrzo_er(lcrzo_conf_devices_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_devices_value_pos(i, founddevice, etha, NULL, NULL,
					  NULL, NULL, NULL));
    if ( lcrzo_device_equal(founddevice, searcheddevice) )
    { return(LCRZO_ERR_OK);
    }
  } 
  /* unknown device */
  return(LCRZO_ERR_SPUNKNOWNDEVICE);
}

/*---------------------------------------------------------------*/
int lcrzo_ipl_init_device(const lcrzo_device device,
			  lcrzo_ipl *pipl,
			  lcrzo_ipl *pmask)
{ lcrzo_int32 count, i;
  lcrzo_device founddevice, searcheddevice;

  /*verification du parametre*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_device_init(device, searcheddevice));
  /* search the device */
  lcrzo_er(lcrzo_conf_devices_count(&count));
  for ( i=1 ; i<=count ; i++ )
    { lcrzo_er(lcrzo_conf_devices_value_pos(i, founddevice, NULL, pipl, pmask,
					    NULL, NULL, NULL));
    if ( lcrzo_device_equal(founddevice, searcheddevice) )
    { return(LCRZO_ERR_OK);
    }
  } 
  /* unknown device */
  return(LCRZO_ERR_SPUNKNOWNDEVICE);
}

/*---------------------------------------------------------------*/
int lcrzo_hs_init_device(const lcrzo_device device,
			 lcrzo_hs hs)
{ lcrzo_ipl ipl;

  lcrzo_er(lcrzo_ipl_init_device(device, &ipl, NULL));
  lcrzo_er(lcrzo_hs_init_ipl(ipl, hs));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_device_veriftype(int type);
int lcrzo_priv_device_veriftype(int type)
{ switch (type)
  { case LCRZO_DEVICE_TYPE_ALL:
    case LCRZO_DEVICE_TYPE_ONLYVALIDIP:
    case LCRZO_DEVICE_TYPE_ONLYBOARDS:
      break;
    default:
      if (lcrzo_global.language==LCRZO_GLOBAL_FRLANG)
	lcrzo_string_init_text("device type is not ALL, ONLYVALIDIP or ONLYBOARDS",
			       LCRZO_GLOERRMSG_MAXBYTES,
			       lcrzo_global.errmsglcrzo);
      else                         
	lcrzo_string_init_text("le type du device est pas ALL, ONLYVALIDIP ou ONLYBOARDS",
			       LCRZO_GLOERRMSG_MAXBYTES,
			       lcrzo_global.errmsglcrzo);
      return(LCRZO_ERR_SPUNKNOWNTYPE);
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_device_istype(const lcrzo_device device,
			int type,
			lcrzo_bool *pyesitistype)
{ lcrzo_device founddevice, searcheddevice;
  lcrzo_bool yes, isup, isanalias;
  lcrzo_int32 count, i;
  lcrzo_ipl ipl;

  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_device_init(device, searcheddevice));
  lcrzo_er(lcrzo_priv_device_veriftype(type));

  /* search the device */
  lcrzo_er(lcrzo_conf_devices_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_devices_value_pos(i, founddevice, NULL, &ipl, NULL,
					  NULL, &isup, &isanalias));
    if ( lcrzo_device_equal(founddevice, searcheddevice) ) break;
  } 
  if (i>count) return(LCRZO_ERR_SPUNKNOWNDEVICE);

  /*now, we test*/
  yes=0;
  switch (type)
  { case LCRZO_DEVICE_TYPE_ALL:
      /*loopback, down interfaces, up interfaces, down aliases, up aliases*/
      yes=1;
      break;
    case LCRZO_DEVICE_TYPE_ONLYVALIDIP:
      /*up interfaces, up aliases*/
      if ( isup && ipl!=0x7F000001 )
      { yes=1;
      }
      break;
    case LCRZO_DEVICE_TYPE_ONLYBOARDS:
      if ( isup && !isanalias && ipl!=0x7F000001 )
      { yes=1;
      }
      break;
  }

  if (pyesitistype!=NULL) *pyesitistype=yes;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_device_count(int type,
		       lcrzo_int32 *pcount)
{ lcrzo_bool isup, isanalias;
  lcrzo_int32 count, i, countall, countvalidip, countonlyboards;
  lcrzo_ipl ipl;

  /*parameters verification*/
  lcrzo_er(lcrzo_priv_device_veriftype(type));

  /* search the device */
  countall=0;
  countvalidip=0;
  countonlyboards=0;
  lcrzo_er(lcrzo_conf_devices_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_devices_value_pos(i, NULL, NULL, &ipl, NULL,
					  NULL, &isup, &isanalias));
    countall++;
    if ( isup && ipl!=0x7F000001 )
    { countvalidip++;
      if ( !isanalias )
      { countonlyboards++;
      }
    }
  } 

  if ( pcount!=NULL )
  { switch (type)
    { case LCRZO_DEVICE_TYPE_ALL:
        *pcount=countall;
        break;
      case LCRZO_DEVICE_TYPE_ONLYVALIDIP:
        *pcount=countvalidip;
        break;
      case LCRZO_DEVICE_TYPE_ONLYBOARDS:
        *pcount=countonlyboards;
        break;
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_device_value_pos(int type,
			   lcrzo_int32 pos,
			   lcrzo_device device)
{ lcrzo_int32 pospos, currentpos;
  lcrzo_bool isup, isanalias;
  lcrzo_int32 count, i;
  lcrzo_ipl ipl;

  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_device_count(type, &count));
  lcrzo_er(lcrzo_priv_manageelempos(count, pos, 0, NULL, &pospos, NULL,NULL));


  /* search the device */
  currentpos=0;
  lcrzo_er(lcrzo_conf_devices_count(&count));
  for ( i=1 ; i<=count ; i++ )
  { lcrzo_er(lcrzo_conf_devices_value_pos(i, device, NULL, &ipl, NULL,
					  NULL, &isup, &isanalias));
    switch (type)
    { case LCRZO_DEVICE_TYPE_ALL:
        currentpos++;
        break;
      case LCRZO_DEVICE_TYPE_ONLYVALIDIP:
        if ( isup && ipl!=0x7F000001 ) currentpos++;
        break;
      case LCRZO_DEVICE_TYPE_ONLYBOARDS:
        if ( isup && !isanalias && ipl!=0x7F000001 ) currentpos++;
        break;
    }
    if ( currentpos==pospos ) return(LCRZO_ERR_OK);
  }

  return(LCRZO_ERR_IEINTERNALERROR);
}

/*---------------------------------------------------------------*/
int lcrzo_device_value_first(int type,
			     lcrzo_device device)
{ return(lcrzo_device_value_pos(type, +1, device));
}

/*---------------------------------------------------------------*/
int lcrzo_device_value_last(int type,
			    lcrzo_device device)
{ return(lcrzo_device_value_pos(type, -1, device));
}

/*---------------------------------------------------------------*/
int lcrzo_device_loop_range(int type,
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos,
			    int (*pfonc)(lcrzo_device device,
					  const void *pinfos),
			    const void *pinfos)
{ lcrzo_int32 count, suppos, infpos, i;
  lcrzo_device device;

  /*parameters verification*/
  if (pfonc==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_er(lcrzo_device_count(type, &count));
  if ( count==0 && startpos==+1 && endpos==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrange(count, startpos, endpos,
				      NULL, &infpos, NULL, &suppos,
				      NULL,NULL,NULL,NULL));

  if (suppos>=infpos)
  { for ( i=infpos ; i<=suppos ; i++ )
    { lcrzo_er(lcrzo_device_value_pos(type, i, device));
      lcrzo_er(((*pfonc)(device, pinfos)));
    }
  }
  else
  { for ( i=infpos ; i>=suppos ; i-- )
    { lcrzo_er(lcrzo_device_value_pos(type, i, device));
      lcrzo_er(((*pfonc)(device, pinfos)));
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_device_loop_all(int type,
			  int (*pfonc)(lcrzo_device device,
					const void *pinfos),
			  const void *pinfos)
{ return(lcrzo_device_loop_range(type, +1, -1, pfonc, pinfos));
}

/*-------------------------------------------------------------*/
int lcrzo_device_stdin(const char *message,
		       const lcrzo_device devicedefaut, lcrzo_device device)
{ char prompt;
  lcrzo_string usertext;
  int retour;

  /*parameters verification*/
  if (devicedefaut==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_device_verifbof(devicedefaut);

  prompt=':';
  while(1)
  { /*affichage du message*/
    if (message!=NULL) if (message[0]!='\0')
    { printf("%s [%s]%c ", message, devicedefaut, prompt);
    }
    prompt='>';
    /*saisie de la chaine de l'utilisateur*/
    lcrzo_er(lcrzo_priv_stdin(&usertext, NULL));
    /*si rien n'a ete entre, on met le defaut*/
    if (strlen(usertext)==0)
    { lcrzo_string_free(usertext);
      lcrzo_er(lcrzo_device_init(devicedefaut, device));
      break;
    }
    /*purge des espaces et tabulations*/
    retour=lcrzo_device_init(usertext, device);
    if (retour!=LCRZO_ERR_OK) continue;
    lcrzo_string_free(usertext);
    break;
  }
  return(LCRZO_ERR_OK);
}
