/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #define _GNU_SOURCE
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <dirent.h>
 #include <sys/types.h>
 #include <sys/stat.h>      
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <dirent.h>
 #include <sys/types.h>
 #include <sys/stat.h>      
#elif defined LCRZODEF_SYSTEM_Solaris 
 #define __EXTENSIONS__
 #include <stdlib.h>
 #include <stdio.h>
 #include <unistd.h>
 #include <string.h>
 #include <ctype.h>
 #include <fcntl.h>
 #include <errno.h>
 #include <termios.h>
 #include <dirent.h>
 #include <sys/types.h>
 #include <sys/stat.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
typedef struct { 
  uid_t uid_user;
  uid_t uid_priv;
  gid_t gid_user;
  gid_t gid_priv;
} lcrzo_priv_guid;

/*---------------------------------------------------------------*/
int lcrzo_priv_privileges_loose(lcrzo_priv_guid *pguid);
int lcrzo_priv_privileges_loose(lcrzo_priv_guid *pguid)
{ int ret;

  /*parameters verification*/
  if (pguid==NULL) return(LCRZO_ERR_PANULLPTR);

  /*save the actual privileges*/
  pguid->uid_user=getuid();
  pguid->uid_priv=geteuid();
  pguid->gid_user=getgid();
  pguid->gid_priv=getegid();

  /*  printf("uid=%d euid=%d    gid=%d egid=%d\n", 
      getuid(), geteuid(), getgid(), getegid());*/

  /*loose the priv*/
  if (pguid->gid_user!=pguid->gid_priv)
  { ret=setegid(pguid->gid_user);
    if (ret==-1) return(LCRZO_ERR_FUSETEGID);
  }
  if (pguid->uid_user!=pguid->uid_priv)
  { ret=seteuid(pguid->uid_user);
    if (ret==-1) return(LCRZO_ERR_FUSETEUID);
  }

  /*  printf("uid=%d euid=%d    gid=%d egid=%d\n", 
      getuid(), geteuid(), getgid(), getegid());*/

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_privileges_restore(lcrzo_priv_guid guid);
int lcrzo_priv_privileges_restore(lcrzo_priv_guid guid)
{ int ret;

  /*restore the old priv*/
  if (guid.uid_user!=guid.uid_priv)
  { ret=seteuid(guid.uid_priv);
    if (ret==-1) return(LCRZO_ERR_FUSETEUID);
  }
  if (guid.gid_user!=guid.gid_priv)
  { ret=setegid(guid.gid_priv);
    if (ret==-1) return(LCRZO_ERR_FUSETEGID);
  }

  /*  printf("uid=%d euid=%d    gid=%d egid=%d\n", 
      getuid(), geteuid(), getgid(), getegid());*/

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
/*
void lcrzo_priv_stat_print(struct stat st);
void lcrzo_priv_stat_print(struct stat st)
{ printf("dev=%ld ino=%ld mode=%d nlink=%d uid=%d gid=%d size=%ld\n",
         (long)st.st_dev, st.st_ino, st.st_mode, st.st_nlink,
	 st.st_uid, st.st_gid, st.st_size);
}
*/

/*---------------------------------------------------------------*/
int lcrzo_priv_file_stat_get(const char *filename, 
			     struct stat *pst);
int lcrzo_priv_file_stat_get(const char *filename, 
			     struct stat *pst)
{ int ret;

  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);
  if (pst==NULL) return(LCRZO_ERR_PANULLPTR);
  ret=lstat(filename, pst);
  if (ret==-1) return(LCRZO_ERR_FULSTAT);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_fd_stat_get(int fd, 
			   struct stat *pst);
int lcrzo_priv_fd_stat_get(int fd, 
			   struct stat *pst)
{ int ret;

  if (pst==NULL) return(LCRZO_ERR_PANULLPTR);
  ret=fstat(fd, pst);
  if (ret==-1) return(LCRZO_ERR_FUFSTAT);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_forbidschange(struct stat st1, struct stat st2);
int lcrzo_priv_stat_forbidschange(struct stat st1, struct stat st2)
{
  if ( st1.st_dev!=st2.st_dev   || st1.st_ino!=st2.st_ino ||
       st1.st_mode!=st2.st_mode || st1.st_nlink!=st2.st_nlink ||
       st1.st_uid!=st2.st_uid   || st1.st_gid!=st2.st_gid ||
       st1.st_size!=st2.st_size )
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			   "File has changed":"Le fichier a change",
			   LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_forbidslink(struct stat st);
int lcrzo_priv_stat_forbidslink(struct stat st)
{ if (S_ISLNK(st.st_mode))
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			   "File is a symlink":
			   "Le fichier est un lien symbolique",
			   LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_forbidsnotreg(struct stat st);
int lcrzo_priv_stat_forbidsnotreg(struct stat st)
{ if ( !S_ISREG(st.st_mode) )
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			   "Not a regular file":
			   "Le fichier n'est pas regulier",
			   LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_type(struct stat st, 
			 int *ptype);
int lcrzo_priv_stat_type(struct stat st, 
			 int *ptype)
{
  /*set the correct value*/
  if (ptype!=NULL)
  { if (S_ISREG(st.st_mode)) *ptype=LCRZO_FILE_TYPE_REG;
    else if (S_ISDIR(st.st_mode)) *ptype=LCRZO_FILE_TYPE_DIR;
    else if (S_ISLNK(st.st_mode)) *ptype=LCRZO_FILE_TYPE_LINK;
    else if (S_ISLNK(st.st_mode)) *ptype=LCRZO_FILE_TYPE_LINK;
    else if (S_ISSOCK(st.st_mode)) *ptype=LCRZO_FILE_TYPE_SOCK;
    else *ptype=LCRZO_FILE_TYPE_OTHER;
  }
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_mode_get(struct stat st, 
			     int *pmode);
int lcrzo_priv_stat_mode_get(struct stat st, 
			     int *pmode)
{ 
  /*set the correct value*/
  if (pmode!=NULL)
  { *pmode=0;
    if (st.st_mode&S_IXOTH) *pmode|=LCRZO_FILE_MODE_XOTH;
    if (st.st_mode&S_IWOTH) *pmode|=LCRZO_FILE_MODE_WOTH;
    if (st.st_mode&S_IROTH) *pmode|=LCRZO_FILE_MODE_ROTH;
    if (st.st_mode&S_IXGRP) *pmode|=LCRZO_FILE_MODE_XGRP;
    if (st.st_mode&S_IWGRP) *pmode|=LCRZO_FILE_MODE_WGRP;
    if (st.st_mode&S_IRGRP) *pmode|=LCRZO_FILE_MODE_RGRP;
    if (st.st_mode&S_IXUSR) *pmode|=LCRZO_FILE_MODE_XUSR;
    if (st.st_mode&S_IWUSR) *pmode|=LCRZO_FILE_MODE_WUSR;
    if (st.st_mode&S_IRUSR) *pmode|=LCRZO_FILE_MODE_RUSR;
    if (st.st_mode&S_ISVTX) *pmode|=LCRZO_FILE_MODE_STIC;
    if (st.st_mode&S_ISGID) *pmode|=LCRZO_FILE_MODE_SGID;
    if (st.st_mode&S_ISUID) *pmode|=LCRZO_FILE_MODE_SUID;
  }  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_uid_get(struct stat st, 
			    int *puid);
int lcrzo_priv_stat_uid_get(struct stat st, 
			    int *puid)
{
  /*set the correct value*/
  if (puid!=NULL)
  { *puid=st.st_uid;
  }  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_gid_get(struct stat st, 
			    int *pgid);
int lcrzo_priv_stat_gid_get(struct stat st, 
			    int *pgid)
{
  /*set the correct value*/
  if (pgid!=NULL)
  { *pgid=st.st_gid;
  }  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_stat_size(struct stat st, 
			 lcrzo_int32 *psize);
int lcrzo_priv_stat_size(struct stat st, 
			 lcrzo_int32 *psize)
{
  /*set the correct value*/
  if (psize!=NULL)
  { *psize=st.st_size;
  }  
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_priv_fd_open_file(const char *filename, 
			    int *pfd);
int lcrzo_priv_fd_open_file(const char *filename, 
			    int *pfd)
{ int ret, fd;
  struct stat st1, st2;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*before openning, we verify if the file seems OK*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st1));
  lcrzo_er(lcrzo_priv_stat_forbidslink(st1));

  /*open*/
  fd=open(filename, O_RDONLY);
  if (fd==-1) return(LCRZO_ERR_FUOPEN);

  /*verify if the characteristics are the same*/
  lcrzo_efr(lcrzo_priv_fd_stat_get(fd, &st2), close(fd));
  lcrzo_efr(lcrzo_priv_stat_forbidschange(st1, st2), close(fd));

  if (pfd==NULL)
  { ret=close(fd);
    if (ret==-1) return(LCRZO_ERR_FUCLOSE);
  }
  else
  { *pfd=fd;
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_file_ensurealonetochange(const char *filename, 
					struct stat st,
					int uid);
int lcrzo_priv_file_ensurealonetochange(const char *filename,
					struct stat st,
					int uid)
{ int fd, ret;
  struct stat st2;

  /*must be owner*/
  if ( st.st_uid!=uid )
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			   "You do not own this file":
			   "Vous n'etes pas le proprietaire du fichier",
			   LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }

  /* verify mode */
  if ( (S_IWGRP&st.st_mode) || (S_IWOTH&st.st_mode) )
  { /*this file can be modified by other users, we change its mode*/

    /*open*/
    fd=open(filename, O_RDONLY);
    if (fd==-1) return(LCRZO_ERR_FUOPEN);

    /*verify if the characteristics are the same*/
    lcrzo_efr(lcrzo_priv_fd_stat_get(fd, &st2), close(fd));
    lcrzo_efr(lcrzo_priv_stat_forbidschange(st, st2), close(fd));
    ret=lcrzo_fd_mode_set(fd, LCRZO_FILE_MODE_WUSR|LCRZO_FILE_MODE_RUSR);
    close(fd);
    if ( ret!=LCRZO_ERR_OK )
    { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
			     "This file can be modified by other users":
			     "Ce fichier peut etre modifie par d'autres"
			     " utilisateurs",
			     LCRZO_GLOERRMSG_MAXBYTES,
			     lcrzo_global.errmsglcrzo);
      return(LCRZO_ERR_SPFILESECURITY);
    }
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_myread(int fd,
			 lcrzo_int32 maxdatasizetoread,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize);
int lcrzo_priv_fd_myread(int fd,
			 lcrzo_int32 maxdatasizetoread,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize)
{ lcrzo_int32 nboct;
  lcrzo_data ptr;

  if (maxdatasizetoread<0) return(LCRZO_ERR_PATOOLOW);

  if (maxdatasizetoread==0)
  { lcrzo_er(lcrzo_data_alloc(0, pdataout));
    if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  /*we read maxi 500 bytes*/
  if (maxdatasizetoread>500) maxdatasizetoread=500;

  /*read*/
  lcrzo_er(lcrzo_data_alloc(maxdatasizetoread, &ptr));
  nboct=read(fd, ptr, maxdatasizetoread);
  /*une erreur s'est produite*/
  if (nboct==-1)
  { if (errno==EAGAIN)
    { /*annule l'erreur, c'est simplement que rien n'etait disponible*/
      errno=0;
      lcrzo_data_free(ptr);
      return(LCRZO_ERR_OKTEMPDATAEND);
    }
    else
    { /*c'etait une erreur fatale*/
      lcrzo_data_free(ptr);
      return(LCRZO_ERR_FUREAD);
    }
  }

  /*on a lu quelque chose*/
  if (pdataoutsize!=NULL) *pdataoutsize=nboct;

  /*if we read 0, we are at end of file*/
  if (nboct==0)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_OKDATAEND);
  }

  if (pdataout!=NULL)
  { lcrzo_er(lcrzo_data_realloc(nboct, &ptr));
    *pdataout=ptr; 
  }
  else
  { lcrzo_data_free(ptr);
  }
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
int lcrzo_priv_fd_open_read(const char *filename,
			    int *pfd);
int lcrzo_priv_fd_open_read(const char *filename,
			    int *pfd)
{ int ret, fd;
  struct stat st1, st2;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*before openning, we verify if the file seems OK*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st1));
  lcrzo_er(lcrzo_priv_stat_forbidslink(st1));
  lcrzo_er(lcrzo_priv_stat_forbidsnotreg(st1));

  /*open*/
  fd=open(filename, O_RDONLY);
  if (fd==-1) return(LCRZO_ERR_FUOPEN);

  /*verify if the characteristics are the same*/
  lcrzo_efr(lcrzo_priv_fd_stat_get(fd, &st2), close(fd));
  lcrzo_efr(lcrzo_priv_stat_forbidschange(st1, st2), close(fd));

  /*eventually close the file descriptor*/
  if (pfd==NULL)
  { ret=close(fd);
    if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);
  }
  else *pfd=fd;

  return(LCRZO_ERR_OK);
}
 
/*-------------------------------------------------------------*/
int lcrzo_fd_open_read(const char *filename,
		       int *pfd)
{ int ret, retopen;
  lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  retopen=lcrzo_priv_fd_open_read(filename, pfd);

  /*restore the old priv*/
  ret=lcrzo_priv_privileges_restore(guid);
  if ( retopen==LCRZO_ERR_OK && ret!=LCRZO_ERR_OK )
  { if (pfd!=NULL) close(*pfd);
    return(ret);
  }

  return(retopen);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_open_read(const char *filename,
			   LCRZOFILE **ppstream)
{ int fd;
  FILE *pstream;

  lcrzo_er(lcrzo_fd_open_read(filename, &fd));
  pstream=fdopen(fd, "r");
  if (pstream==NULL)
  { close(fd);
    return(LCRZO_ERR_FUFDOPEN);
  }

  if (ppstream!=NULL)
  { *ppstream=pstream;
  }
  else
  { close(fd);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_open_write(const char *filename,
			     lcrzo_bool mustbeanexistentfile,
			     lcrzo_bool mustbeanewfile,
			     int *pfd);
int lcrzo_priv_fd_open_write(const char *filename,
			     lcrzo_bool mustbeanexistentfile,
			     lcrzo_bool mustbeanewfile,
			     int *pfd)
{ int ret, fd;
  struct stat st1, st2;
  lcrzo_bool fileexists;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(mustbeanewfile);
  lcrzo_bool_verifbof(mustbeanexistentfile);
  if (mustbeanewfile&&mustbeanexistentfile)
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		       "Cannot set mustbeanewfile and mustbeanexistentfile":
		      "Ne pas affecter mustbeanewfile et mustbeanexistentfile",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }

  /*before openning, we verify if the file seems OK*/
  ret=lcrzo_priv_file_stat_get(filename, &st1);
  if (ret!=LCRZO_ERR_OK)
  { /*unknown error*/
    if (errno!=ENOENT) return(LCRZO_ERR_FULSTAT);
    errno=0;
    fileexists=0;
  }
  else
  { fileexists=1;
    lcrzo_er(lcrzo_priv_stat_forbidslink(st1));
    lcrzo_er(lcrzo_priv_stat_forbidsnotreg(st1));
  }

  /*error cases*/
  if ( mustbeanewfile && fileexists )
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		       "We want a new file, but file exists":
		       "Nouveau fichier voulu, mais il existe deja",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }
  if ( mustbeanexistentfile && !fileexists )
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		       "We want an existent file, but file doesn't exists":
		       "Fichier existant voulu, mais il n'existe pas",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }
  
  /*open*/
  if (fileexists)
  { fd=open(filename, O_WRONLY&(~O_CREAT));
    if (fd==-1) return(LCRZO_ERR_FUOPEN);
    /*verify if the characteristics are the same*/
    lcrzo_efr(lcrzo_priv_fd_stat_get(fd, &st2), close(fd));
    lcrzo_efr(lcrzo_priv_stat_forbidschange(st1, st2), close(fd));
    /*truncate the file*/
    ret=ftruncate(fd, 0);
    if (ret==-1) { close(fd); return(LCRZO_ERR_FUFTRUNCATE); }
  }
  else
  { fd=open(filename, O_WRONLY|O_CREAT|O_EXCL, 0600);
    if (fd==-1) return(LCRZO_ERR_FUOPEN);
  }

  /*eventually close the file descriptor*/
  if (pfd==NULL)
  { ret=close(fd);
    if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);
  }
  else *pfd=fd;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_open_write(const char *filename,
			lcrzo_bool mustbeanexistentfile,
			lcrzo_bool mustbeanewfile,
			int *pfd)
{ int ret, retopen;
  lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(mustbeanewfile);
  lcrzo_bool_verifbof(mustbeanexistentfile);
  if (mustbeanewfile&&mustbeanexistentfile)
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		       "Cannot set mustbeanewfile and mustbeanexistentfile":
		      "Ne pas affecter mustbeanewfile et mustbeanexistentfile",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  retopen=lcrzo_priv_fd_open_write(filename, mustbeanexistentfile,
				   mustbeanewfile, pfd);

  /*restore the old priv*/
  ret=lcrzo_priv_privileges_restore(guid);
  if ( retopen==LCRZO_ERR_OK && ret!=LCRZO_ERR_OK )
  { if (pfd!=NULL) close(*pfd);
    return(ret);
  }

  return(retopen);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_open_write(const char *filename,
			    lcrzo_bool mustbeanexistentfile,
			    lcrzo_bool mustbeanewfile,
			    LCRZOFILE **ppstream)
{ int fd;
  FILE *pstream;

  lcrzo_er(lcrzo_fd_open_write(filename, mustbeanexistentfile,
			       mustbeanewfile, &fd));
  pstream=fdopen(fd, "w");
  if (pstream==NULL)
  { close(fd);
    return(LCRZO_ERR_FUFDOPEN);
  }

  if (ppstream!=NULL)
  { *ppstream=pstream;
  }
  else
  { close(fd);
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_fd_open_append(const char *filename,
			      lcrzo_bool mustbeanexistentfile,
			      int *pfd);
int lcrzo_priv_fd_open_append(const char *filename,
			      lcrzo_bool mustbeanexistentfile,
			      int *pfd)
{ int ret, fd;
  struct stat st1, st2;
  lcrzo_bool fileexists;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(mustbeanexistentfile);

  /*before openning, we verify if the file seems OK*/
  ret=lcrzo_priv_file_stat_get(filename, &st1);
  if (ret!=LCRZO_ERR_OK)
  { /*unknown error*/
    if (errno!=ENOENT) return(LCRZO_ERR_FULSTAT);
    errno=0;
    fileexists=0;
  }
  else
  { fileexists=1;
    lcrzo_er(lcrzo_priv_stat_forbidslink(st1));
    lcrzo_er(lcrzo_priv_stat_forbidsnotreg(st1));
  }

  /*error cases*/
  if ( mustbeanexistentfile && !fileexists )
  { lcrzo_string_init_text((lcrzo_global.language==LCRZO_GLOBAL_FRLANG)?
		       "We want an existent file, but file doesn't exists":
		       "Fichier existant voulu, mais il n'existe pas",
		       LCRZO_GLOERRMSG_MAXBYTES, lcrzo_global.errmsglcrzo);
    return(LCRZO_ERR_SPFILESECURITY);
  }
  
  /*open*/
  if (fileexists)
  { fd=open(filename, O_WRONLY|O_APPEND);
    if (fd==-1) return(LCRZO_ERR_FUOPEN);
    /*verify if the characteristics are the same*/
    lcrzo_efr(lcrzo_priv_fd_stat_get(fd, &st2), close(fd));
    lcrzo_efr(lcrzo_priv_stat_forbidschange(st1, st2), close(fd));
  }
  else
  { fd=open(filename, O_WRONLY|O_CREAT|O_EXCL, 0600);
    if (fd==-1) return(LCRZO_ERR_FUOPEN);
  }

  /*eventually close the file descriptor*/
  if (pfd==NULL)
  { ret=close(fd);
    if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);
  }
  else *pfd=fd;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_open_append(const char *filename,
			 lcrzo_bool mustbeanexistentfile,
			 int *pfd)
{ int ret, retopen;
  lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);
  lcrzo_bool_verifbof(mustbeanexistentfile);

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  retopen=lcrzo_priv_fd_open_append(filename, mustbeanexistentfile, pfd);

  /*restore the old priv*/
  ret=lcrzo_priv_privileges_restore(guid);
  if ( retopen==LCRZO_ERR_OK && ret!=LCRZO_ERR_OK )
  { if (pfd!=NULL) close(*pfd);
    return(ret);
  }

  return(retopen);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_open_append(const char *filename,
			     lcrzo_bool mustbeanexistentfile,
			     LCRZOFILE **ppstream)
{ int fd;
  FILE *pstream;

  lcrzo_er(lcrzo_fd_open_append(filename, mustbeanexistentfile, &fd));
  pstream=fdopen(fd, "a");
  if (pstream==NULL)
  { close(fd);
    return(LCRZO_ERR_FUFDOPEN);
  }

  if (ppstream!=NULL)
  { *ppstream=pstream;
  }
  else
  { close(fd);
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_close(int fd)
{ int ret;

  ret=close(fd);
  if ( ret == -1 ) return(LCRZO_ERR_FUCLOSE);

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_close(LCRZOFILE *pstream)
{ int ret;

  ret=fclose(pstream);
  if ( ret == -1 ) return(LCRZO_ERR_FUFCLOSE);

  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_fd_write(int fd,
		   lcrzo_constdata data,
		   lcrzo_int32 datasize)
{ int retour;
  lcrzo_int32 blocksize, i;
  
  /*parameters verification*/
  if (data==NULL && datasize) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (datasize<0) return(LCRZO_ERR_PATOOLOW);

  /*nothing to write*/
  if (data==NULL || datasize==0) return(LCRZO_ERR_OK);

  for ( i=0 ; i<datasize ; )
  { /*determine block size*/
    blocksize=datasize-i;
    if (blocksize>500) blocksize=500;
    /*write a block*/
    retour=write(fd, data+i, blocksize);
    if (retour==-1) return(LCRZO_ERR_FUWRITE);
    /*update the current position*/
    i+=retour;
  }

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_write(LCRZOFILE *pstream,
		       lcrzo_constdata data,
		       lcrzo_int32 datasize)
{ int retour;
  lcrzo_int32 blocksize, i;
  
  /*parameters verification*/
  if (data==NULL && datasize) return(LCRZO_ERR_SPNULLPTRSIZE);
  if (datasize<0) return(LCRZO_ERR_PATOOLOW);

  /*nothing to write*/
  if (data==NULL || datasize==0) return(LCRZO_ERR_OK);

  for ( i=0 ; i<datasize ; )
  { /*determine block size*/
    blocksize=datasize-i;
    if (blocksize>500) blocksize=500;
    /*write a block*/
    retour=fwrite(data+i, blocksize, 1, pstream);
    if (retour==-1) return(LCRZO_ERR_FUWRITE);
    /*update the current position*/
    i+=blocksize;
  }

  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_fd_read(int fd,
		  lcrzo_bool beblocking,
		  lcrzo_int32 dataoutmaxsize,
		  lcrzo_data dataout,
		  lcrzo_int32 *pdataoutsize)
{ lcrzo_data data;
  lcrzo_int32 datasize;
  int ret;
  
  if (dataoutmaxsize<0) return(LCRZO_ERR_PATOOLOW);
  if (dataoutmaxsize==0)
  { if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  lcrzo_er(lcrzo_fd_block_set(fd, beblocking));

  ret=lcrzo_priv_fd_myread(fd, dataoutmaxsize, &data, &datasize);
  if (ret==LCRZO_ERR_OK)
  { if (dataout!=NULL) memcpy(dataout, data, datasize);
    lcrzo_data_free(data);
    if (pdataoutsize!=NULL) *pdataoutsize=datasize;
  }

  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_read(LCRZOFILE *pstream,
		      lcrzo_bool beblocking,
		      lcrzo_int32 dataoutmaxsize,
		      lcrzo_data dataout,
		      lcrzo_int32 *pdataoutsize)
{ int ret;
  
  if (dataoutmaxsize<0) return(LCRZO_ERR_PATOOLOW);
  if (dataoutmaxsize==0)
  { if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  lcrzo_er(lcrzo_fd_block_set(fileno((FILE*)pstream), beblocking));

  /*end of file*/
  if (feof((FILE*)pstream)) return(LCRZO_ERR_OKDATAEND);
  /*get a char*/
  ret=fgetc((FILE*)pstream);
  if (ret==EOF)
  { if (beblocking) return(LCRZO_ERR_OKDATAEND);
    else return(LCRZO_ERR_OKTEMPDATAEND);
  }
  /*set values*/
  if (dataout!=NULL) dataout[0]=ret;
  if (pdataoutsize!=NULL) *pdataoutsize=1;

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_readm(int fd,
		   lcrzo_bool beblocking,
		   lcrzo_data *pdataout,
		   lcrzo_int32 *pdataoutsize)
{ lcrzo_er(lcrzo_fd_block_set(fd, beblocking));
  return(lcrzo_priv_fd_myread(fd, 0xFFFF, pdataout, pdataoutsize));
}

/*-------------------------------------------------------------*/
int lcrzo_stream_readm(LCRZOFILE *pstream,
		       lcrzo_bool beblocking,
		       lcrzo_data *pdataout,
		       lcrzo_int32 *pdataoutsize)
{ int ret;
  
  /*end of file*/
  lcrzo_er(lcrzo_fd_block_set(fileno((FILE*)pstream), beblocking));
  if (feof((FILE*)pstream)) return(LCRZO_ERR_OKDATAEND);
  /*get a char*/
  ret=fgetc((FILE*)pstream);
  if (ret==EOF)
  { if (beblocking) return(LCRZO_ERR_OKDATAEND);
    else return(LCRZO_ERR_OKTEMPDATAEND);
  }
  /*set values*/
  lcrzo_er(lcrzo_data_initm_char(ret, 1, pdataout, pdataoutsize));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_read_fixed(int fd,
			lcrzo_bool beblocking,
			lcrzo_int32 datasizetoread,
			lcrzo_data dataout)
{ lcrzo_data data;

  if (datasizetoread<0) return(LCRZO_ERR_PATOOLOW);
  if (datasizetoread==0) return(LCRZO_ERR_OK);
  /*get data*/
  lcrzo_er(lcrzo_fd_readm_fixed(fd, beblocking, datasizetoread, &data));
  /*copy it*/
  memcpy(dataout, data, datasizetoread);
  /*free*/
  lcrzo_er(lcrzo_data_free(data));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_read_fixed(LCRZOFILE *pstream,
			    lcrzo_bool beblocking,
			    lcrzo_int32 datasizetoread,
			    lcrzo_data dataout)
{ lcrzo_data data;

  if (datasizetoread<0) return(LCRZO_ERR_PATOOLOW);
  if (datasizetoread==0) return(LCRZO_ERR_OK);
  /*get data*/
  lcrzo_er(lcrzo_stream_readm_fixed(pstream, beblocking, datasizetoread, 
				    &data));
  /*copy it*/
  memcpy(dataout, data, datasizetoread);
  /*free*/
  lcrzo_er(lcrzo_data_free(data));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_readm_fixed(int fd,
			 lcrzo_bool beblocking,
			 lcrzo_int32 datasizetoread,
			 lcrzo_data *pdataout)
{ lcrzo_data ptr, tmpdata;
  lcrzo_int32 tmpdatasize, totalread, tmpdatasizetoread;
  int ret;


  if (datasizetoread<0) return(LCRZO_ERR_PATOOLOW);

  if (datasizetoread==0)
  { lcrzo_er(lcrzo_data_alloc(0, pdataout));
    return(LCRZO_ERR_OK);
  }
  lcrzo_er(lcrzo_fd_block_set(fd, beblocking));

  lcrzo_er(lcrzo_data_alloc(datasizetoread, &ptr));

  totalread=0;  
  while(1)
  { tmpdatasizetoread=datasizetoread-totalread;
    ret=lcrzo_priv_fd_myread(fd, tmpdatasizetoread, &tmpdata, &tmpdatasize);
    if (ret==LCRZO_ERR_OK)
    { /*concatene*/
      memcpy(ptr+totalread, tmpdata, tmpdatasize);
      lcrzo_data_free(tmpdata);
      totalread+=tmpdatasize;
      /*eventually quit*/
      if ( totalread==datasizetoread )
      { if (pdataout!=NULL) *pdataout=ptr;
        else lcrzo_data_free(ptr);
        return(LCRZO_ERR_OK);
      }
    }
    else if (ret==LCRZO_ERR_OKDATAEND)
    { /*no more data, so it's an error*/
      lcrzo_data_free(ptr);
      return(ret);
    }
    else if (ret==LCRZO_ERR_OKTEMPDATAEND)
    { if (!totalread)
      { /*we didn't read anathing, so we can quit now*/
	lcrzo_data_free(ptr);
        return(ret);
      }
    }
    else
    { lcrzo_data_free(ptr);
      return(ret);
    }
  }

  /*on ne doit jamais atteindre ce return*/
  return(LCRZO_ERR_IEINTERNALERROR);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_readm_fixed(LCRZOFILE *pstream,
			     lcrzo_bool beblocking,
			     lcrzo_int32 datasizetoread,
			     lcrzo_data *pdataout)
{ int ret;
  lcrzo_data ptr;

  if (datasizetoread<0) return(LCRZO_ERR_PATOOLOW);

  if (datasizetoread==0)
  { lcrzo_er(lcrzo_data_alloc(0, pdataout));
    return(LCRZO_ERR_OK);
  }
  lcrzo_er(lcrzo_fd_block_set(fileno((FILE*)pstream), beblocking));

  /*read*/
  lcrzo_er(lcrzo_data_alloc(datasizetoread, &ptr));
  ret=fread(ptr, datasizetoread, 1, (FILE*)pstream);

  /*une erreur s'est produite*/
  if (ret!=1)
  { /*if we read nothing, we are at end of file*/
    lcrzo_data_free(ptr);
    if (beblocking) return(LCRZO_ERR_OKDATAEND);
    else return(LCRZO_ERR_OKTEMPDATAEND);
  }

  /*on a lu quelque chose*/
  if (pdataout!=NULL)
  { lcrzo_er(lcrzo_data_realloc(datasizetoread, &ptr));
    *pdataout=ptr; 
  }
  else
  { lcrzo_data_free(ptr);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_read_line(int fd,
		       lcrzo_bool beblocking,
		       lcrzo_int32 dataoutmaxsize,
		       lcrzo_data dataout,
		       lcrzo_int32 *pdataoutsize)
{ lcrzo_data data;
  lcrzo_int32 datasize;
  int ret;
  
  if (dataoutmaxsize<0) return(LCRZO_ERR_PATOOLOW);
  if (dataoutmaxsize==0)
  { if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  lcrzo_er(lcrzo_fd_readm_line(fd, beblocking, &data, &datasize));
  ret=lcrzo_data_init_data(data, datasize,
			   dataoutmaxsize, dataout, pdataoutsize);
  lcrzo_data_free(data);
  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_read_line(LCRZOFILE *pstream,
			   lcrzo_bool beblocking,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize)
{ lcrzo_data data;
  lcrzo_int32 datasize;
  int ret;
  
  if (dataoutmaxsize<0) return(LCRZO_ERR_PATOOLOW);
  if (dataoutmaxsize==0)
  { if (pdataoutsize!=NULL) *pdataoutsize=0;
    return(LCRZO_ERR_OK);
  }

  lcrzo_er(lcrzo_stream_readm_line(pstream, beblocking, &data, &datasize));
  ret=lcrzo_data_init_data(data, datasize,
			   dataoutmaxsize, dataout, pdataoutsize);
  lcrzo_data_free(data);
  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_fd_readm_line(int fd,
			lcrzo_bool beblocking,
			lcrzo_data *pdataout,
			lcrzo_int32 *pdataoutsize)
{ lcrzo_data data, tmpdata;
  lcrzo_uint8 block[500];
  lcrzo_int32 datasize, tmpdatasize, blocksize;
  int ret, c;
  
  lcrzo_er(lcrzo_fd_block_set(fd, beblocking));
  datasize=0;
  lcrzo_er(lcrzo_data_alloc(datasize, &data));
  blocksize=0;
  while (1)
  { ret=lcrzo_priv_fd_myread(fd, 1, &tmpdata, &tmpdatasize);
    if (ret==LCRZO_ERR_OK)
    { c=tmpdata[0];
      lcrzo_er(lcrzo_data_free(tmpdata));
      /*ignore 0D*/
      if ( c == 0x0D ) {}
      else if ( c == 0x0A ) { break; }
      else
      { block[blocksize]=c;
        blocksize++;
        if (blocksize>499)
        { lcrzo_er(lcrzo_data_appendm_data(block, blocksize, 
					   datasize, &data, &datasize));
  	  blocksize=0;
	}
      }
    }
    else if (ret==LCRZO_ERR_OKTEMPDATAEND)
    { /*exit if we read nothing*/
      if (!datasize)
      { lcrzo_data_free(data);
        return(ret);
      }
    }
    else if (ret==LCRZO_ERR_OKDATAEND)
    { /*exit if we read nothing*/
      if (!datasize)
      { lcrzo_data_free(data);
        return(ret);
      }
      /* end of file is like end of line */
      break;
    }
    else
    { lcrzo_data_free(data);
      return(ret);
    }
  }
  /*append the end*/
  lcrzo_er(lcrzo_data_appendm_data(block, blocksize, 
				   datasize, &data, &datasize));

  /*on a lu quelque chose*/
  if (pdataout!=NULL)
  { *pdataout=data; 
  }
  else
  { lcrzo_data_free(data);
  }
  if (pdataoutsize!=NULL) *pdataoutsize=datasize;
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_stream_readm_line(LCRZOFILE *pstream,
			    lcrzo_bool beblocking,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize)
{ lcrzo_data data, tmpdata;
  lcrzo_int32 datasize, step, strlentmpdata;
  char *retfgets;
  
  if (feof((FILE*)pstream)) return(LCRZO_ERR_OKDATAEND);
  lcrzo_er(lcrzo_fd_block_set(fileno((FILE*)pstream), beblocking));

  datasize=0;
  lcrzo_er(lcrzo_data_alloc(datasize, &data));
  step=4;
  lcrzo_er(lcrzo_data_alloc(step, &tmpdata));
  while (1)
  { retfgets=fgets(tmpdata, step, (FILE*)pstream);
    /*an error occured*/
    if (retfgets==NULL)
    { lcrzo_er(lcrzo_data_free(tmpdata));
      lcrzo_er(lcrzo_data_free(data));
      if (feof((FILE*)pstream)) return(LCRZO_ERR_OKDATAEND);
      else
      { if ( !beblocking && errno==EAGAIN ) return(LCRZO_ERR_OKTEMPDATAEND);
        return(LCRZO_ERR_FUFGETS);
      }
    }
    /*set data*/
    strlentmpdata=strlen(tmpdata);
    lcrzo_er(lcrzo_data_appendm_data(tmpdata, strlentmpdata,
				     datasize, &data, &datasize));
    /*tmpdata was sufficiently big*/
    if ( strlentmpdata < step-1 )
    { break;  
    }
    if ( strlentmpdata > 0 )
    { if ( tmpdata[strlentmpdata-1] == '\n' ) break;  
    }
  }
  lcrzo_er(lcrzo_data_free(tmpdata));

  /*purge data*/
  while (datasize>0)
  { if ( data[datasize-1]==0x0D || data[datasize-1]==0x0A )
      datasize--;
    else
      break;
  }

  if (pdataout!=NULL)
  { *pdataout=data; 
  }
  else
  { lcrzo_data_free(data);
  }
  if (pdataoutsize!=NULL) *pdataoutsize=datasize;
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_file_type(const char *filename, 
		    int *ptype)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st));
  lcrzo_er(lcrzo_priv_stat_type(st, ptype));
 
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_type(int fd, 
		  int *ptype)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_fd_stat_get(fd, &st));
  lcrzo_er(lcrzo_priv_stat_type(st, ptype));
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_type(LCRZOFILE *pstream, 
		      int *ptype)
{ return(lcrzo_fd_type(fileno((FILE*)pstream), ptype));
}


/*---------------------------------------------------------------*/
int lcrzo_fd_flush(int fd)
{ int ret;
  ret=fsync(fd);
  if (ret<0) return(LCRZO_ERR_FUFSYNC);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_flush(LCRZOFILE *pstream)
{ int ret;
  ret=fflush((FILE*)pstream);
  if (ret<0) return(LCRZO_ERR_FUFFLUSH);
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_fd_block_set(int fd, 
		       lcrzo_bool beblocking)
{ int old_fc, new_fc, ret;

  /*verification des parametres*/
  lcrzo_bool_verifbof(beblocking);

  old_fc=fcntl(fd, F_GETFL, 0);
  if (old_fc<0) return(LCRZO_ERR_FUFCNTL);
  if (beblocking)
    new_fc = old_fc & (~O_NONBLOCK);
  else
    new_fc = old_fc | O_NONBLOCK;
  ret=fcntl(fd, F_SETFL, new_fc);
  if (ret<0) return(LCRZO_ERR_FUFCNTL);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_block_set(LCRZOFILE *pstream, 
			   lcrzo_bool beblocking)
{ return(lcrzo_fd_block_set(fileno((FILE*)pstream), beblocking));
}

/*---------------------------------------------------------------*/
int lcrzo_fd_block_get(int fd, 
		       lcrzo_bool *pbeblocking)
{ int fc;
  lcrzo_bool beblocking;

  fc=fcntl(fd, F_GETFL, 0);
  if (fc<0) return(LCRZO_ERR_FUFCNTL);
  if (fc&O_NONBLOCK)
  { beblocking=LCRZO_FALSE;
  }
  else
  { beblocking=LCRZO_TRUE;
  }

  if (pbeblocking!=NULL) *pbeblocking=beblocking;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_block_get(LCRZOFILE *pstream, 
			   lcrzo_bool *pbeblocking)
{ return(lcrzo_fd_block_get(fileno((FILE*)pstream), pbeblocking));
}


/*---------------------------------------------------------------*/
int lcrzo_fd_offset_set(int fd, 
			lcrzo_int32 offset)
{ int ret;
  if (offset<0) return(LCRZO_ERR_PATOOLOW);
  ret=lseek(fd, offset, SEEK_SET);
  if (ret<0) return(LCRZO_ERR_FULSEEK);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_offset_set(LCRZOFILE *pstream,
			    lcrzo_int32 offset)
{ int ret;
  if (offset<0) return(LCRZO_ERR_PATOOLOW);
  ret=fseek((FILE*)pstream, offset, SEEK_SET);
  if (ret<0) return(LCRZO_ERR_FUFSEEK);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_offset_get(int fd, 
			lcrzo_int32 *poffset)
{ int ret;

  ret=lseek(fd, 0, SEEK_CUR);
  if (ret<0) return(LCRZO_ERR_FULSEEK);

  if (poffset!=NULL) *poffset=ret;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_offset_get(LCRZOFILE *pstream,
			    lcrzo_int32 *poffset)
{ int ret;

  ret=ftell((FILE*)pstream);
  if (ret<0) return(LCRZO_ERR_FUFSEEK);

  if (poffset!=NULL) *poffset=ret;
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_file_mode_get(const char *filename, 
			int *pmode)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st));
  lcrzo_er(lcrzo_priv_stat_mode_get(st, pmode));
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_mode_get(int fd, 
		      int *pmode)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_fd_stat_get(fd, &st));
  lcrzo_er(lcrzo_priv_stat_mode_get(st, pmode));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_mode_get(LCRZOFILE *pstream, 
			  int *pmode)
{ return(lcrzo_fd_mode_get(fileno((FILE*)pstream), pmode));
}

/*---------------------------------------------------------------*/
int lcrzo_priv_fd_mode_set(int fd, 
			   int mode);
int lcrzo_priv_fd_mode_set(int fd, 
			   int mode)
{ int stmode, ret;

  stmode=0;
  if (mode&LCRZO_FILE_MODE_XOTH) stmode|=S_IXOTH;
  if (mode&LCRZO_FILE_MODE_WOTH) stmode|=S_IWOTH;
  if (mode&LCRZO_FILE_MODE_ROTH) stmode|=S_IROTH;
  if (mode&LCRZO_FILE_MODE_XGRP) stmode|=S_IXGRP;
  if (mode&LCRZO_FILE_MODE_WGRP) stmode|=S_IWGRP;
  if (mode&LCRZO_FILE_MODE_RGRP) stmode|=S_IRGRP;
  if (mode&LCRZO_FILE_MODE_XUSR) stmode|=S_IXUSR;
  if (mode&LCRZO_FILE_MODE_WUSR) stmode|=S_IWUSR;
  if (mode&LCRZO_FILE_MODE_RUSR) stmode|=S_IRUSR;
  if (mode&LCRZO_FILE_MODE_STIC) stmode|=S_ISVTX;
  if (mode&LCRZO_FILE_MODE_SGID) stmode|=S_ISGID;
  if (mode&LCRZO_FILE_MODE_SUID) stmode|=S_ISUID;

  ret=fchmod(fd, stmode);
  if (ret==-1) return(LCRZO_ERR_FUFCHMOD);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_file_mode_set(const char *filename, 
			     int mode);
int lcrzo_priv_file_mode_set(const char *filename, 
			     int mode)
{ int ret, fd, ret2;

  lcrzo_er(lcrzo_priv_fd_open_file(filename, &fd));
  ret2=lcrzo_priv_fd_mode_set(fd, mode);
  ret=close(fd);
  if (ret2==LCRZO_ERR_OK && ret==-1) return(LCRZO_ERR_FUCLOSE);
  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_file_mode_set(const char *filename, 
			int mode)
{ int ret;
  lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_file_mode_set(filename, mode);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_mode_set(int fd, 
		      int mode)
{ int ret;
  lcrzo_priv_guid guid;

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_fd_mode_set(fd, mode);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_mode_set(LCRZOFILE *pstream, 
			  int mode)
{ int ret;
  lcrzo_priv_guid guid;

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_fd_mode_set(fileno((FILE*)pstream), mode);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_file_uid_get(const char *filename, 
		       int *puid)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st));
  lcrzo_er(lcrzo_priv_stat_uid_get(st, puid));
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_uid_get(int fd, 
		     int *puid)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_fd_stat_get(fd, &st));
  lcrzo_er(lcrzo_priv_stat_uid_get(st, puid));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_uid_get(LCRZOFILE *pstream, 
			 int *puid)
{ return(lcrzo_fd_uid_get(fileno((FILE*)pstream), puid));
}

/*---------------------------------------------------------------*/
int lcrzo_priv_fd_uid_set(int fd, 
			  int uid);
int lcrzo_priv_fd_uid_set(int fd, 
			  int uid)
{ int ret;
  ret=fchown(fd, uid, -1);
  if (ret==-1) return(LCRZO_ERR_FUFCHOWN);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_file_uid_set(const char *filename, 
			     int uid);
int lcrzo_priv_file_uid_set(const char *filename, 
			     int uid)
{ int ret, fd;

  lcrzo_er(lcrzo_priv_fd_open_file(filename, &fd));
  lcrzo_er(lcrzo_priv_fd_uid_set(fd, uid));

  ret=close(fd);
  if (ret==-1) return(LCRZO_ERR_FUCLOSE);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_file_uid_set(const char *filename, 
		       int uid)
{ int ret;
  lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_file_uid_set(filename, uid);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_uid_set(int fd, 
		     int uid)
{ int ret;
  lcrzo_priv_guid guid;

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_fd_uid_set(fd, uid);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_uid_set(LCRZOFILE *pstream, 
			 int uid)
{ int ret;
  lcrzo_priv_guid guid;

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_fd_uid_set(fileno((FILE*)pstream), uid);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_file_gid_get(const char *filename, 
                       int *pgid)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st));
  lcrzo_er(lcrzo_priv_stat_gid_get(st, pgid));
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_gid_get(int fd, 
                     int *pgid)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_fd_stat_get(fd, &st));
  lcrzo_er(lcrzo_priv_stat_gid_get(st, pgid));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_gid_get(LCRZOFILE *pstream, 
			 int *pgid)
{ return(lcrzo_fd_gid_get(fileno((FILE*)pstream), pgid));
}


/*---------------------------------------------------------------*/
int lcrzo_priv_fd_gid_set(int fd, 
			  int gid);
int lcrzo_priv_fd_gid_set(int fd, 
			  int gid)
{ int ret;
  ret=fchown(fd, -1, gid);
  if (ret==-1) return(LCRZO_ERR_FUFCHOWN);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_file_gid_set(const char *filename, 
			     int gid);
int lcrzo_priv_file_gid_set(const char *filename, 
			     int gid)
{ int ret, fd;

  lcrzo_er(lcrzo_priv_fd_open_file(filename, &fd));
  lcrzo_er(lcrzo_priv_fd_gid_set(fd, gid));

  ret=close(fd);
  if (ret==-1) return(LCRZO_ERR_FUCLOSE);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_file_gid_set(const char *filename, 
		       int gid)
{ int ret;
  lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_file_gid_set(filename, gid);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_gid_set(int fd, 
		     int gid)
{ int ret;
  lcrzo_priv_guid guid;

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_fd_gid_set(fd, gid);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_gid_set(LCRZOFILE *pstream, 
			 int gid)
{ int ret;
  lcrzo_priv_guid guid;

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*open the file*/
  ret=lcrzo_priv_fd_gid_set(fileno((FILE*)pstream), gid);

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(ret);
}


/*---------------------------------------------------------------*/
int lcrzo_file_size(const char *filename, 
		    lcrzo_int32 *psize)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st));
  lcrzo_er(lcrzo_priv_stat_size(st, psize));
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_size(int fd, 
		  lcrzo_int32 *psize)
{ struct stat st;

  /*get the stat*/
  lcrzo_er(lcrzo_priv_fd_stat_get(fd, &st));
  lcrzo_er(lcrzo_priv_stat_size(st, psize));
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_size(LCRZOFILE *pstream, 
		      lcrzo_int32 *psize)
{ return(lcrzo_fd_size(fileno((FILE*)pstream), psize));
}


/*---------------------------------------------------------------*/
int lcrzo_file_exist(const char *filename,
		     lcrzo_bool *pyes)
{ int ret;
  struct stat st;
  lcrzo_bool fileexists;

  fileexists=LCRZO_FALSE;
  ret=lcrzo_priv_file_stat_get(filename, &st);
  if (ret==LCRZO_ERR_OK)
  { fileexists=LCRZO_TRUE;
  }

  if (pyes!=NULL)
  { *pyes=fileexists;
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_file_create(const char *filename)
{ return(lcrzo_fd_open_write(filename, LCRZO_FALSE, LCRZO_TRUE, NULL));
}

/*---------------------------------------------------------------*/
int lcrzo_priv_file_remove(const char *filename, int uid);
int lcrzo_priv_file_remove(const char *filename, int uid)
{ int ret;
  struct stat st1;

  /*before removing, we verify if the file seems OK*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st1));
  lcrzo_er(lcrzo_priv_stat_forbidslink(st1));
  lcrzo_er(lcrzo_priv_stat_forbidsnotreg(st1));
  lcrzo_er(lcrzo_priv_file_ensurealonetochange(filename, st1, uid));

  /* unlink the file */
  ret=unlink(filename);
  if ( ret == -1 ) return(LCRZO_ERR_FUUNLINK);
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_file_remove(const char *filename)
{ lcrzo_priv_guid guid;

  /*parameters verification*/
  if (filename==NULL) return(LCRZO_ERR_PANULLPTR);

  /*save the actual privileges*/
  lcrzo_er(lcrzo_priv_privileges_loose(&guid));

  /*unlink the file*/
  lcrzo_er(lcrzo_priv_file_remove(filename, guid.uid_user));

  /*restore the old priv*/
  lcrzo_er(lcrzo_priv_privileges_restore(guid));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_file_rename(const char *oldfilename, 
		      const char *newfilename)
{ lcrzo_er(lcrzo_file_copy(oldfilename, newfilename));
  return(lcrzo_file_remove(oldfilename));
}

/*---------------------------------------------------------------*/
int lcrzo_file_copy(const char *oldfilename, 
		    const char *newfilename)
{ int ret, fdin, fdout;
  lcrzo_data data;
  lcrzo_int32 datasize;

  /*open files*/
  lcrzo_er(lcrzo_fd_open_read(oldfilename, &fdin));
  lcrzo_efr(lcrzo_fd_open_write(newfilename, LCRZO_FALSE, LCRZO_TRUE, &fdout),
	    close(fdin));

  while(1)
  { ret=lcrzo_fd_readm(fdin, LCRZO_TRUE, &data, &datasize);
    if (ret==LCRZO_ERR_OKDATAEND)
    { break;
    }
    else if (ret==LCRZO_ERR_OK)
    { lcrzo_efr(lcrzo_fd_write(fdout, data, datasize), 
		close(fdin);close(fdout));
      lcrzo_data_free(data);
    }
    else
    { close(fdin); close(fdout); return(ret);
    }
  }
  lcrzo_er(lcrzo_fd_close(fdin));
  lcrzo_er(lcrzo_fd_close(fdout));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_copy(int fd, 
		  const char *newfilename)
{ int ret, fdout;
  lcrzo_data data;
  lcrzo_int32 datasize;

  /*open files*/
  lcrzo_er(lcrzo_fd_offset_set(fd, 0));
  lcrzo_er(lcrzo_fd_open_write(newfilename, LCRZO_FALSE, LCRZO_TRUE, &fdout));

  while(1)
  { ret=lcrzo_fd_readm(fd, LCRZO_TRUE, &data, &datasize);
    if (ret==LCRZO_ERR_OKDATAEND)
    { break;
    }
    else if (ret==LCRZO_ERR_OK)
    { lcrzo_efr(lcrzo_fd_write(fdout, data, datasize), close(fdout));
      lcrzo_data_free(data);
    }
    else
    { close(fdout); return(ret);
    }
  }
  lcrzo_er(lcrzo_fd_close(fdout));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_stream_copy(LCRZOFILE *pstream, 
		      const char *newfilename)
{ int ret, fdout;
  lcrzo_data data;
  lcrzo_int32 datasize;

  /*open files*/
  lcrzo_er(lcrzo_stream_offset_set(pstream, 0));
  lcrzo_er(lcrzo_fd_open_write(newfilename, LCRZO_FALSE, LCRZO_TRUE, &fdout));

  while(1)
  { ret=lcrzo_stream_readm(pstream, LCRZO_TRUE, &data, &datasize);
    if (ret==LCRZO_ERR_OKDATAEND)
    { break;
    }
    else if (ret==LCRZO_ERR_OK)
    { lcrzo_efr(lcrzo_fd_write(fdout, data, datasize), close(fdout));
      lcrzo_data_free(data);
    }
    else
    { close(fdout); return(ret);
    }
  }
  lcrzo_er(lcrzo_fd_close(fdout));

  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_file_truncate(const char *filename,
			lcrzo_int32 size)
{ int ret, fd;
  struct stat st1, st2;

  /*before removing, we verify if the file seems OK*/
  lcrzo_er(lcrzo_priv_file_stat_get(filename, &st1));
  lcrzo_er(lcrzo_priv_stat_forbidslink(st1));
  lcrzo_er(lcrzo_priv_stat_forbidsnotreg(st1));

  /*open*/
  fd=open(filename, O_RDWR);
  if (fd==-1) return(LCRZO_ERR_FUOPEN);

  /*verify if the characteristics are the same*/
  lcrzo_efr(lcrzo_priv_fd_stat_get(fd, &st2), close(fd));
  lcrzo_efr(lcrzo_priv_stat_forbidschange(st1, st2), close(fd));

  /* truncate the file */
  ret=ftruncate(fd, size);
  if ( ret == -1 ) return(LCRZO_ERR_FUFTRUNCATE);
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_fd_truncate(int fd, 
		      lcrzo_int32 size)
{ int ret;

  /* truncate the file */
  ret=ftruncate(fd, size);
  if ( ret == -1 ) return(LCRZO_ERR_FUFTRUNCATE);
  
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_file_read(const char *filename,
		    lcrzo_data *pdata,
		    lcrzo_int32 *pdatasize)
{ int ret, fd;
  lcrzo_data data, tmpdata;
  lcrzo_int32 datasize, tmpdatasize;

  datasize=0;
  lcrzo_er(lcrzo_data_alloc(datasize, &data));

  /*open file*/
  lcrzo_er(lcrzo_fd_open_read(filename, &fd));

  while(1)
  { ret=lcrzo_fd_readm(fd, LCRZO_TRUE, &tmpdata, &tmpdatasize);
    if (ret==LCRZO_ERR_OKDATAEND)
    { break;
    }
    else if (ret==LCRZO_ERR_OK)
    { ret=lcrzo_data_appendm_data(tmpdata, tmpdatasize,
				  datasize, &data, &datasize);
      lcrzo_data_free(tmpdata);
      if (ret!=LCRZO_ERR_OK) 
      { close(fd);
        lcrzo_data_free(data);
        return(ret);
      }
    }
    else
    { close(fd);
      lcrzo_data_free(data);
      return(ret);
    }
  }
  lcrzo_er(lcrzo_fd_close(fd));

  if (pdatasize!=NULL) *pdatasize=datasize;
  if (pdata!=NULL)
  { *pdata=data; 
  }
  else
  { lcrzo_data_free(data);
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_file_write(const char *filename,
		     lcrzo_bool mustbeanexistentfile,
		     lcrzo_bool mustbeanewfile,
		     lcrzo_constdata data,
		     lcrzo_int32 datasize)
{ int ret, ret2, fd;

  /*open file*/
  lcrzo_er(lcrzo_fd_open_write(filename, mustbeanexistentfile, mustbeanewfile,
			       &fd));
  ret=lcrzo_fd_write(fd, data, datasize);
  ret2=lcrzo_fd_close(fd);
  if ( ret==LCRZO_ERR_OK && ret2!=LCRZO_ERR_OK) return(ret2); 
  return(ret);
}

/*---------------------------------------------------------------*/
int lcrzo_file_append(const char *filename,
		      lcrzo_bool mustbeanexistentfile,
		      lcrzo_constdata data,
		      lcrzo_int32 datasize)
{ int ret, ret2, fd;

  /*open file*/
  lcrzo_er(lcrzo_fd_open_append(filename, mustbeanexistentfile, &fd));
  ret=lcrzo_fd_write(fd, data, datasize);
  ret2=lcrzo_fd_close(fd);
  if ( ret==LCRZO_ERR_OK && ret2!=LCRZO_ERR_OK) return(ret2); 
  return(ret);
}


/*---------------------------------------------------------------*/
int lcrzo_dir_mode_get(const char *dirname, 
		       int *pmode)
{ return(lcrzo_file_mode_get(dirname, pmode));
}

/*---------------------------------------------------------------*/
int lcrzo_dir_mode_set(const char *dirname, 
		       int mode)
{ return(lcrzo_file_mode_set(dirname, mode));
}

/*---------------------------------------------------------------*/
int lcrzo_dir_uid_get(const char *dirname, 
		      int *puid)
{ return(lcrzo_file_uid_get(dirname, puid));
}

/*---------------------------------------------------------------*/
int lcrzo_dir_uid_set(const char *dirname, 
		      int uid)
{ return(lcrzo_file_uid_set(dirname, uid));
}

/*---------------------------------------------------------------*/
int lcrzo_dir_gid_get(const char *dirname, 
		      int *pgid)
{ return(lcrzo_file_gid_get(dirname, pgid));
}

/*---------------------------------------------------------------*/
int lcrzo_dir_gid_set(const char *dirname, 
		      int gid)
{ return(lcrzo_file_gid_set(dirname, gid));
}

/*---------------------------------------------------------------*/
int lcrzo_dir_exist(const char *dirname,
		    lcrzo_bool *pyes)
{ int ret;
  struct stat st;
  lcrzo_bool fileexists;

  fileexists=LCRZO_FALSE;
  ret=lcrzo_priv_file_stat_get(dirname, &st);
  if (ret==LCRZO_ERR_OK)
  { if (S_ISDIR(st.st_mode))
    { fileexists=LCRZO_TRUE;
    }
  }

  if (pyes!=NULL)
  { *pyes=fileexists;
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_dir_create(const char *dirname)
{ lcrzo_priv_guid guid;
  int ret, ret2;

  lcrzo_er(lcrzo_priv_privileges_loose(&guid));
  ret=mkdir(dirname, 0600);
  ret2=lcrzo_priv_privileges_restore(guid);
  if (ret==-1) return(LCRZO_ERR_FUMKDIR);
  return(ret2);
}

/*---------------------------------------------------------------*/
int lcrzo_dir_remove(const char *dirname)
{ lcrzo_priv_guid guid;
  int ret, ret2;

  lcrzo_er(lcrzo_priv_privileges_loose(&guid));
  ret=rmdir(dirname);
  ret2=lcrzo_priv_privileges_restore(guid);
  if (ret==-1) return(LCRZO_ERR_FURMDIR);
  return(ret2);
}

/*---------------------------------------------------------------*/
int lcrzo_dir_rename(const char *olddirname, 
		     const char *newdirname)
{ lcrzo_priv_guid guid;
  int ret, ret2;

  lcrzo_er(lcrzo_priv_privileges_loose(&guid));
  ret=rename(olddirname, newdirname);
  ret2=lcrzo_priv_privileges_restore(guid);
  if (ret==-1) return(LCRZO_ERR_FURENAME);
  return(ret2);
}

/*---------------------------------------------------------------*/
int lcrzo_dir_count(const char *dirname,
		    lcrzo_int32 *pcount)
{ DIR *thisdir;
  struct dirent *file;
  int ret;
  lcrzo_string filename;
  lcrzo_int32 count;

  /*open the directory*/
  thisdir=opendir(dirname);
  if ( thisdir==NULL ) return(LCRZO_ERR_FUOPENDIR);

  file=readdir(thisdir);
  count=0;
  while (file!=NULL)
  { lcrzo_efr(lcrzo_string_initm_text(file->d_name, &filename),
	      closedir(thisdir););
    if ( strcmp(filename, ".") && strcmp(filename, "..") )
    { /*printf("%s\n", filename);*/
      count++;
    }
    lcrzo_string_free(filename);
    file=readdir(thisdir);
  }
  ret=closedir(thisdir);
 
  if (pcount!=NULL) *pcount=count;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_dir_value_pos(const char *dirname,
			lcrzo_int32 pos,
			lcrzo_string *pfilename)
{ DIR *thisdir;
  struct dirent *file;
  int ret;
  lcrzo_string filename;
  lcrzo_int32 count, pospos;

  lcrzo_er(lcrzo_dir_count(dirname, &count));
  lcrzo_er(lcrzo_priv_manageelempos(count, pos, 0, NULL, &pospos, NULL,NULL));

  /*open the directory*/
  thisdir=opendir(dirname);
  if ( thisdir==NULL ) return(LCRZO_ERR_FUOPENDIR);

  file=readdir(thisdir);
  count=0;
  while (file!=NULL)
  { lcrzo_efr(lcrzo_string_initm_text(file->d_name, &filename),
	      closedir(thisdir););
    if ( strcmp(filename, ".") && strcmp(filename, "..") )
    { /*printf("%s\n", filename);*/
      count++;
      if (count==pospos) 
      { if ( pfilename == NULL ) lcrzo_string_free(filename);
	else *pfilename=filename;
        break;
      }
    }
    lcrzo_string_free(filename);
    file=readdir(thisdir);
  }
  ret=closedir(thisdir);
 
  return(LCRZO_ERR_OK);
}






