/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_file module

  Functions herein allow to open/read/write/close files.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_file

  Les fonctions presentes dans ce module assurent la gestion
des fichiers.
*/

/*---------------------------------------------------------------*/
/*us the FILE type is necessary to write to a file. It's defined
   in stdio.h, but I do not want to include ".h" files in lcrzo.
   So, we define LCRZOFILE which has the same utility as FILE*/
/*fr le type FILE est normalement necessaire pour indiquer un fichier.
   Il est defini dans stdio.h, mais je n'ai pas envie de l'inclure
   pour les fichiers ".h" publics de lcrzo. D'un point de vue 
   fonctionnement, le LCRZOFILE ci-dessous est identique a FILE.*/
#define LCRZOFILE void

/*---------------------------------------------------------------*/
/*us Name : lcrzo_fd_open_read
   Description :
     Open a file for reading.
   Input parameter(s) :
     filename : name of the file to open
   Input/output parameter(s) :
   Output parameter(s) :
     *pfd : file descriptor which is openned
     *ppstream : FILE which is openned 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_open_read
   Description :
     Ouvre un fichier en lecture.
   Parametre(s) d'entree :
     filename : nom de fichier a ouvrir
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfd : descripteur de fichier ouvert
     *ppstream : FILE qui est ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_open_read(const char *filename,
		       int *pfd);
int lcrzo_stream_open_read(const char *filename,
			   LCRZOFILE **ppstream);

/*us Name : lcrzo_fd_open_write
   Description :
     Open a file for writing.
   Input parameter(s) :
     filename : name of the file to open
     mustbeanexistentfile : file must already exist
     mustbeanewfile : file musn't already exist
   Input/output parameter(s) :
   Output parameter(s) :
     *pfd : file descriptor which is openned
     *ppstream : FILE which is openned 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_open_write
   Description :
     Ouvre un fichier en ecriture.
   Parametre(s) d'entree :
     filename : nom de fichier a ouvrir
     mustbeanexistentfile : filename doit deja exister
     mustbeanewfile : filename ne doit pas deja exister
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfd : descripteur de fichier ouvert
     *ppstream : FILE qui est ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_open_write(const char *filename,
			lcrzo_bool mustbeanexistentfile,
			lcrzo_bool mustbeanewfile,
			int *pfd);
int lcrzo_stream_open_write(const char *filename,
			    lcrzo_bool mustbeanexistentfile,
			    lcrzo_bool mustbeanewfile,
			    LCRZOFILE **ppstream);

/*us Name : lcrzo_fd_open_append
   Description :
     Open a file for appending.
   Input parameter(s) :
     filename : name of the file to open
     mustbeanexistentfile : file must already exist
   Input/output parameter(s) :
   Output parameter(s) :
     *pfd : file descriptor which is openned
     *ppstream : FILE which is openned 
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_open_append
   Description :
     Ouvre un fichier en concatenation.
   Parametre(s) d'entree :
     filename : nom de fichier a ouvrir
     mustbeanexistentfile : filename doit deja exister
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfd : descripteur de fichier ouvert
     *ppstream : FILE qui est ouvert
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_open_append(const char *filename,
			 lcrzo_bool mustbeanexistentfile,
			 int *pfd);
int lcrzo_stream_open_append(const char *filename,
			     lcrzo_bool mustbeanexistentfile,
			     LCRZOFILE **ppstream);

/*us Name : lcrzo_fd_close
   Description :
     Close a file.
   Input parameter(s) :
     fd : file descriptor which is closed
     *pstream : FILE which is closed
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_close
   Description :
     Ferme un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier a fermer
     *pstream : FILE qui est ferme
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_close(int fd);
int lcrzo_stream_close(LCRZOFILE *pstream);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_fd_write
   Description :
     Write data in a file.
   Input parameter(s) :
     fd : file descriptor where to write
     *pstream : FILE where to write
     data : data to write
     datasize : size of data to write
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_stream_write
   Description :
     Ecrit des donnees dans un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier ou ecrire 
     *pstream : FILE ou ecrire
     data : donnees a ecrire
     datasize : taille des donnees a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_write(int fd,
		   lcrzo_constdata data,
		   lcrzo_int32 datasize);
int lcrzo_stream_write(LCRZOFILE *pstream,
		       lcrzo_constdata data,
		       lcrzo_int32 datasize);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_fd_read
   Description :
     Read data from a file.
   Input parameter(s) :
     fd : file descriptor where to read
     *pstream : FILE where to read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_stream_read
   Description :
     Lit des donnees depuis un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier ou lire 
     *pstream : FILE ou lire
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_fd_read(int fd,
		  lcrzo_bool beblocking,
		  lcrzo_int32 dataoutmaxsize,
		  lcrzo_data dataout,
		  lcrzo_int32 *pdataoutsize);
int lcrzo_stream_read(LCRZOFILE *pstream,
		      lcrzo_bool beblocking,
		      lcrzo_int32 dataoutmaxsize,
		      lcrzo_data dataout,
		      lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_fd_readm
   Description :
     Allocate a memory array, and initialize it with
     data read from a file.
   Input parameter(s) :
     fd : file descriptor where to read
     *pstream : FILE where to read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_stream_readm
   Description :
     Alloue une zone de memoire et l'initialise avec
     des donnees lues depuis un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier ou lire 
     *pstream : FILE ou lire
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_fd_readm(int fd,
		   lcrzo_bool beblocking,
		   lcrzo_data *pdataout,
		   lcrzo_int32 *pdataoutsize);
int lcrzo_stream_readm(LCRZOFILE *pstream,
		       lcrzo_bool beblocking,
		       lcrzo_data *pdataout,
		       lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_fd_read_fixed
   Description :
     Read a fixed size data from a file.
   Input parameter(s) :
     fd : file descriptor where to read
     *pstream : FILE where to read
     beblocking : if we want to block until receiving data
     datasizetoread : size to read
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_stream_read_fixed
   Description :
     Lit des donnees de taille constante depuis un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier ou lire 
     *pstream : FILE ou lire
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     datasizetoread : taille des donnees a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_fd_read_fixed(int fd,
			lcrzo_bool beblocking,
			lcrzo_int32 datasizetoread,
			lcrzo_data dataout);
int lcrzo_stream_read_fixed(LCRZOFILE *pstream,
			    lcrzo_bool beblocking,
			    lcrzo_int32 datasizetoread,
			    lcrzo_data dataout);

/*us Name : lcrzo_fd_readm_fixed
   Description :
     Allocate a memory array, and initialize it with
     data of fixed size read from a file.
   Input parameter(s) :
     fd : file descriptor where to read
     *pstream : FILE where to read
     beblocking : if we want to block until receiving data
     datasizetoread : size to read
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_stream_readm_fixed
   Description :
     Alloue une zone de memoire et l'initialise avec
     des donnees de taille constante lues depuis un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier ou lire 
     *pstream : FILE ou lire
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     datasizetoread : taille des donnees a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_fd_readm_fixed(int fd,
			 lcrzo_bool beblocking,
			 lcrzo_int32 datasizetoread,
			 lcrzo_data *pdataout);
int lcrzo_stream_readm_fixed(LCRZOFILE *pstream,
			     lcrzo_bool beblocking,
			     lcrzo_int32 datasizetoread,
			     lcrzo_data *pdataout);

/*us Name : lcrzo_fd_read_line
   Description :
     Read data from a file. We stop at first '\r' or '\n' (which
     is not stored in data).
   Input parameter(s) :
     fd : file descriptor where to read
     *pstream : FILE where to read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_stream_read_line
   Description :
     Lit des donnees depuis un fichier. On s'arrete de lire
     au premier '\r' ou '\n' (qui n'est pas stocke dans
     *pdataout).
   Parametre(s) d'entree :
     fd : descripteur de fichier ou lire 
     *pstream : FILE ou lire
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_fd_read_line(int fd,
		       lcrzo_bool beblocking,
		       lcrzo_int32 dataoutmaxsize,
		       lcrzo_data dataout,
		       lcrzo_int32 *pdataoutsize);
int lcrzo_stream_read_line(LCRZOFILE *pstream,
			   lcrzo_bool beblocking,
			   lcrzo_int32 dataoutmaxsize,
			   lcrzo_data dataout,
			   lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_fd_readm_line
   Description :
     Allocate a memory array, and initialize it with
     data read from a file. We stop at first '\r' or '\n' (which
     is not stored in data).
   Input parameter(s) :
     fd : file descriptor where to read
     *pstream : FILE where to read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
                               without reading data
*/
/*fr Nom : lcrzo_stream_readm_line
   Description :
     Alloue une zone de memoire et l'initialise avec
     des donnees lues depuis un fichier. On s'arrete de lire
     au premier '\r' ou '\n' (qui n'est pas stocke dans
     *pdataout).
   Parametre(s) d'entree :
     fd : descripteur de fichier ou lire 
     *pstream : FILE ou lire
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par *pdataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
                               retourne sans donnees.
*/
int lcrzo_fd_readm_line(int fd,
			lcrzo_bool beblocking,
			lcrzo_data *pdataout,
			lcrzo_int32 *pdataoutsize);
int lcrzo_stream_readm_line(LCRZOFILE *pstream,
			    lcrzo_bool beblocking,
			    lcrzo_data *pdataout,
			    lcrzo_int32 *pdataoutsize);

/*---------------------------------------------------------------*/
/*us file types */
/*fr types de fichiers */
/*us regular file *//*fr fichier normal */
#define LCRZO_FILE_TYPE_REG   1
/*us directory *//*fr repertoire */
#define LCRZO_FILE_TYPE_DIR   2
/*us link *//*fr lien */
#define LCRZO_FILE_TYPE_LINK  3
/*us unix socket *//*fr socket unix */
#define LCRZO_FILE_TYPE_SOCK  4
/*us other type *//*fr autre type */
#define LCRZO_FILE_TYPE_OTHER 5

/*us Name : lcrzo_xxx_type
   Description :
     Get type of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *ptype : type of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_type
   Description :
     Obtient le type du fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *ptype : type du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_type(const char *filename, 
		    int *ptype);
int lcrzo_fd_type(int fd, 
		  int *ptype);
int lcrzo_stream_type(LCRZOFILE *pstream, 
		      int *ptype);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_xxx_flush
   Description :
     Flush a file.
   Input parameter(s) :
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_flush
   Description :
     Force l'ecriture dans un fichier.
   Parametre(s) d'entree :
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_flush(int fd);
int lcrzo_stream_flush(LCRZOFILE *pstream);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_xxx_block_get
   Description :
     Check if access to a file has to be blocking.
   Input parameter(s) :
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *pbeblocking : if we want to block
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_block_get
   Description :
     Voir si les acces a un fichier doivent etre bloquants.
   Parametre(s) d'entree :
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pbeblocking : si on veut etre bloquant
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_block_get(int fd, 
		       lcrzo_bool *pbeblocking);
int lcrzo_stream_block_get(LCRZOFILE *pstream, 
			   lcrzo_bool *pbeblocking);

/*us Name : lcrzo_xxx_block_set
   Description :
     Decide if a file has to be blocking.
   Input parameter(s) :
     fd : file descriptor
     *pstream : FILE
     beblocking : if we want to block
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_block_set
   Description :
     Decide si les acces a un fichier doivent etre bloquants.
   Parametre(s) d'entree :
     fd : descripteur de fichier
     *pstream : FILE
     beblocking : si on veut etre bloquant
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_block_set(int fd, 
		       lcrzo_bool beblocking);
int lcrzo_stream_block_set(LCRZOFILE *pstream, 
			   lcrzo_bool beblocking);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_xxx_offset_get
   Description :
     Get offset of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *poffset : offset of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_offset_get
   Description :
     Obtient l'offset d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *poffset : offset du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_offset_get(int fd, 
			lcrzo_int32 *poffset);
int lcrzo_stream_offset_get(LCRZOFILE *pstream,
			    lcrzo_int32 *poffset);

/*us Name : lcrzo_xxx_offset_set
   Description :
     Set offset of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
     offset : offset of the file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_offset_set
   Description :
     Affecte l'offset d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
     offset : offset du fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_fd_offset_set(int fd, 
			lcrzo_int32 offset);
int lcrzo_stream_offset_set(LCRZOFILE *pstream,
			    lcrzo_int32 offset);

/*---------------------------------------------------------------*/
/*us access modes */
/*fr droits d'access */
/*us others can execute *//*fr autres peuvent executer */
#define LCRZO_FILE_MODE_XOTH 00001
/*us others can write *//*fr autres peuvent ecrite */
#define LCRZO_FILE_MODE_WOTH 00002
/*us others can read *//*fr autres peuvent lire */
#define LCRZO_FILE_MODE_ROTH 00004
/*us group can execute *//*fr groupe peut executer */
#define LCRZO_FILE_MODE_XGRP 00010
/*us group can write *//*fr groupe peut ecrire */
#define LCRZO_FILE_MODE_WGRP 00020
/*us group can read *//*fr groupe peut lire */
#define LCRZO_FILE_MODE_RGRP 00040
/*us user can execute *//*fr utilisateur peut executer */
#define LCRZO_FILE_MODE_XUSR 00100
/*us user can write *//*fr utilisateur peut ecrire */
#define LCRZO_FILE_MODE_WUSR 00200
/*us user can read *//*fr utilisateur peut lire */
#define LCRZO_FILE_MODE_RUSR 00400
/* sticky */
#define LCRZO_FILE_MODE_STIC 01000
/* setgid */
#define LCRZO_FILE_MODE_SGID 02000
/* setuid */
#define LCRZO_FILE_MODE_SUID 04000

/*us Name : lcrzo_xxx_mode_get
   Description :
     Get mode of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *pmode : mode of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_mode_get
   Description :
     Obtient le mode d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pmode : mode du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_mode_get(const char *filename, 
			int *pmode);
int lcrzo_fd_mode_get(int fd, 
		      int *pmode);
int lcrzo_stream_mode_get(LCRZOFILE *pstream, 
			  int *pmode);

/*us Name : lcrzo_xxx_mode_set
   Description :
     Set mode of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
     mode : mode of the file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_mode_set
   Description :
     Affecte le mode d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
     mode : mode du fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_mode_set(const char *filename, 
			int mode);
int lcrzo_fd_mode_set(int fd, 
		      int mode);
int lcrzo_stream_mode_set(LCRZOFILE *pstream, 
			  int mode);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_xxx_uid_get
   Description :
     Get uid of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *puid : uid of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_uid_get
   Description :
     Obtient le uid d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *puid : uid du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_uid_get(const char *filename, 
		       int *puid);
int lcrzo_fd_uid_get(int fd, 
		     int *puid);
int lcrzo_stream_uid_get(LCRZOFILE *pstream, 
			 int *puid);

/*us Name : lcrzo_xxx_uid_set
   Description :
     Set uid of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
     uid : uid of the file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_uid_set
   Description :
     Affecte le uid d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
     uid : uid du fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_uid_set(const char *filename, 
		       int uid);
int lcrzo_fd_uid_set(int fd, 
		     int uid);
int lcrzo_stream_uid_set(LCRZOFILE *pstream, 
			 int uid);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_xxx_gid_get
   Description :
     Get gid of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *pgid : gid of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_gid_get
   Description :
     Obtient le gid d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pgid : gid du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_gid_get(const char *filename, 
		       int *pgid);
int lcrzo_fd_gid_get(int fd, 
		     int *pgid);
int lcrzo_stream_gid_get(LCRZOFILE *pstream, 
			 int *pgid);

/*us Name : lcrzo_xxx_gid_set
   Description :
     Set gid of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
     gid : gid of the file
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_gid_set
   Description :
     Affecte le gid d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
     gid : gid du fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_gid_set(const char *filename, 
		       int gid);
int lcrzo_fd_gid_set(int fd, 
		     int gid);
int lcrzo_stream_gid_set(LCRZOFILE *pstream, 
			 int gid);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_xxx_size
   Description :
     Get size of a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     *pstream : FILE
   Input/output parameter(s) :
   Output parameter(s) :
     *psize : size of the file
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_xxx_size
   Description :
     Obtient la taille d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     *pstream : FILE
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *psize : taille du fichier
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_size(const char *filename, 
		    lcrzo_int32 *psize);
int lcrzo_fd_size(int fd, 
		  lcrzo_int32 *psize);
int lcrzo_stream_size(LCRZOFILE *pstream, 
		      lcrzo_int32 *psize);


/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_exist
   Description :
     Check if a file exists.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
     *pyes : if file exists
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_exist
   Description :
     Verifie si un fichier existe.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pyes : si le fichier existe
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_exist(const char *filename,
		     lcrzo_bool *pyes);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_create
   Description :
     Create an empty file.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_create
   Description :
     Cree un fichier vide.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_create(const char *filename);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_remove
   Description :
     Remove a file.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_remove
   Description :
     Efface un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_remove(const char *filename);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_rename
   Description :
     Rename a file.
   Input parameter(s) :
     *oldfilename : old filename
     *newfilename : new filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_rename
   Description :
     Efface un fichier.
   Parametre(s) d'entree :
     *oldfilename : ancien nom de fichier
     *newfilename : nouveau nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_rename(const char *oldfilename, 
		      const char *newfilename);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_copy
   Description :
     Copy a file.
   Input parameter(s) :
     *oldfilename : old filename
     *newfilename : new filename
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_copy
   Description :
     Copie un fichier.
   Parametre(s) d'entree :
     *oldfilename : ancien nom de fichier
     *newfilename : nouveau nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_copy(const char *oldfilename, 
		    const char *newfilename);
int lcrzo_fd_copy(int fd, 
		  const char *newfilename);
int lcrzo_stream_copy(LCRZOFILE *pstream, 
		      const char *newfilename);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_truncate
   Description :
     Truncate a file.
   Input parameter(s) :
     *filename : filename
     fd : file descriptor
     size : size to truncate
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_truncate
   Description :
     Tronque un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     fd : descripteur de fichier
     size : taille apres troncature
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_truncate(const char *filename,
			lcrzo_int32 size);
int lcrzo_fd_truncate(int fd, 
		      lcrzo_int32 size);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_file_read
   Description :
     Read all data from a file.
   Input parameter(s) :
     *filename : filename
   Input/output parameter(s) :
   Output parameter(s) :
     *pdata : pointer which will be malloced (so, the
              memory will have to be freed by the
              user with 'free(*pdata)').
              The allocated memory will be set with data
	      from the file
     *pdatasize : reached size of *pdata
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_read
   Description :
     Lit toutes les donnees d'un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdata : pointeur qui sera alloue (la memoire
              devra etre liberee par l'utilisateur 
              avec 'free(*pdata)').
              Cette memoire allouee recevra les donnees 
              du fichier
     *pdatasize : taille atteinte par *pdata
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_read(const char *filename,
		    lcrzo_data *pdata,
		    lcrzo_int32 *pdatasize);

/*us Name : lcrzo_file_write
   Description :
     Write data in a file.
   Input parameter(s) :
     *filename : filename
     mustbeanexistentfile : file must already exist
     mustbeanewfile : file musn't already exist
     data : data to write
     datasize : size of data to write
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_write
   Description :
     Ecrit des donnees dans un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     mustbeanexistentfile : filename doit deja exister
     mustbeanewfile : filename ne doit pas deja exister
     data : donnees a ecrire
     datasize : taille a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_write(const char *filename,
		     lcrzo_bool mustbeanexistentfile,
		     lcrzo_bool mustbeanewfile,
		     lcrzo_constdata data,
		     lcrzo_int32 datasize);

/*us Name : lcrzo_file_append
   Description :
     Append data in a file.
   Input parameter(s) :
     *filename : filename
     mustbeanexistentfile : file must already exist
     data : data to append
     datasize : size of data to append
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_file_append
   Description :
     Concatene des donnees dans un fichier.
   Parametre(s) d'entree :
     *filename : nom de fichier
     mustbeanexistentfile : filename doit deja exister
     data : donnees a ecrire
     datasize : taille a ecrire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_file_append(const char *filename,
		      lcrzo_bool mustbeanexistentfile,
		      lcrzo_constdata data,
		      lcrzo_int32 datasize);


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/*us the following functions have the same functionnalities
   like lcrzo_file_xxx, except they work on directories.*/
/*fr les fonctions suivantes ont les meme fonctionnalites que
   lcrzo_file_xxx, excepte qu'elles travaillent sur des
   repertoires.*/
#define LCRZO_DIR_MODE_XOTH 00001
#define LCRZO_DIR_MODE_WOTH 00002
#define LCRZO_DIR_MODE_ROTH 00004
#define LCRZO_DIR_MODE_XGRP 00010
#define LCRZO_DIR_MODE_WGRP 00020
#define LCRZO_DIR_MODE_RGRP 00040
#define LCRZO_DIR_MODE_XUSR 00100
#define LCRZO_DIR_MODE_WUSR 00200
#define LCRZO_DIR_MODE_RUSR 00400
#define LCRZO_DIR_MODE_STIC 01000
#define LCRZO_DIR_MODE_SGID 02000
#define LCRZO_DIR_MODE_SUID 04000
int lcrzo_dir_mode_get(const char *dirname, 
		       int *pmode);
int lcrzo_dir_mode_set(const char *dirname, 
		       int mode);

int lcrzo_dir_uid_get(const char *dirname, 
		      int *puid);
int lcrzo_dir_uid_set(const char *dirname, 
		      int uid);
int lcrzo_dir_gid_get(const char *dirname, 
		      int *pgid);
int lcrzo_dir_gid_set(const char *dirname, 
		      int gid);

int lcrzo_dir_exist(const char *dirname,
		    lcrzo_bool *pyes);

/*us Warning, these 3 functions are not secured against
   symlink races*/
/*fr Attention, ces trois fonctions ne sont pas sures face
   aux attaques par symlink*/
int lcrzo_dir_create(const char *dirname);
int lcrzo_dir_remove(const char *dirname);
int lcrzo_dir_rename(const char *olddirname, 
		     const char *newdirname);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_dir_count
   Description :
     Count the number of files in a directory.
   Input parameter(s) :
     *dirname : name of the directory
   Input/output parameter(s) :
   Output parameter(s) :
     *pcount : number of files
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_dir_count
   Description :
     Obtient le nombre de fichiers d'un repertoire
   Parametre(s) d'entree :
     *dirname : nom de repertoire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pcount : nom de fichiers du repertoire
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_dir_count(const char *dirname,
		    lcrzo_int32 *pcount);

/*US*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/
/*us Name : lcrzo_dir_value_pos
   Description :
     Get the name of the file situated at a given position.
   Input parameter(s) :
     *dirname : name of the directory
     pos : wanted position
   Input/output parameter(s) :
   Output parameter(s) :
     *pfilename : mallocated string containing the file
                  name at position pos
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_dir_value_pos
   Description :
     Obtient le nom du fichier situe a une position donnee. 
   Parametre(s) d'entree :
     *dirname : nom de repertoire
      pos : position du nom de fichier voulu
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pfilename : chaine alloue contenant le nom du fichier
                  situe a la position pos
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_dir_value_pos(const char *dirname,
			lcrzo_int32 pos,
			lcrzo_string *pfilename);






