/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"

#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdlib.h>
 #include <unistd.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD 
 #include <stdlib.h>
 #include <unistd.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdlib.h>
 #include <unistd.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*-------------------------------------------------------------*/
int lcrzo_ipc_init(lcrzo_ipc *pci)
{ int retour, tube[2];

  /*parameters verification*/
  if (pci==NULL) return(LCRZO_ERR_PANULLPTR);

  /*ouvre le tube*/
  retour=pipe(tube);
  if (retour<0) return(LCRZO_ERR_FUPIPE);
  /*affecte les elements de la structure*/
  pci->readpipe=tube[0];
  pci->writepipe=tube[1];
  /*pour que le canal de lecture ne soit pas bloquant*/
  lcrzo_er(lcrzo_fd_block_set(pci->readpipe, LCRZO_FALSE));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_close(lcrzo_ipc ci)
{ int retour;
  retour=close(ci.readpipe);
  if (retour==-1) return(LCRZO_ERR_FUCLOSE);
  retour=close(ci.writepipe);
  if (retour==-1) return(LCRZO_ERR_FUCLOSE);

  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
#define LCRZO_PRIV_IPCTYPE_BLOC     1
#define LCRZO_PRIV_IPCTYPE_FLOT     2
#define LCRZO_PRIV_IPCTYPE_FLOTFIN  3
#define LCRZO_PRIV_IPCTYPE_UINT8   10
#define LCRZO_PRIV_IPCTYPE_UINT16  11
#define LCRZO_PRIV_IPCTYPE_UINT32  12
#define LCRZO_PRIV_IPCTYPE_HS      20
#define LCRZO_PRIV_IPCTYPE_IPS     21
#define LCRZO_PRIV_IPCTYPE_IPA     22
#define LCRZO_PRIV_IPCTYPE_IPL     23
#define LCRZO_PRIV_IPCTYPE_ETHS    24
#define LCRZO_PRIV_IPCTYPE_ETHA    25
#define LCRZO_PRIV_IPCTYPE_DEVICE  26

/*-------------------------------------------------------------*/
int lcrzo_priv_ipc_writetlv(lcrzo_ipc ci, lcrzo_uint16 type,
			    lcrzo_constdata donnees,
			    lcrzo_int32 nboctdonnees);
int lcrzo_priv_ipc_writetlv(lcrzo_ipc ci, lcrzo_uint16 type,
			    lcrzo_constdata donnees,
			    lcrzo_int32 nboctdonnees)
{ lcrzo_uint8 typelongueur[6];

  /*parameters verification*/
  if (donnees==NULL && nboctdonnees) return(LCRZO_ERR_SPNULLPTRSIZE);

  /*preparation du type-longueur*/
  typelongueur[0]=(lcrzo_uint8)(type>>8);
  typelongueur[1]=(lcrzo_uint8)(type&0xFF);
  typelongueur[2]=(lcrzo_uint8)(nboctdonnees>>24);
  typelongueur[3]=(lcrzo_uint8)((nboctdonnees>>16)&0xFF);
  typelongueur[4]=(lcrzo_uint8)((nboctdonnees>>8)&0xFF);
  typelongueur[5]=(lcrzo_uint8)(nboctdonnees&0xFF);
  /*ecriture du type-longueur*/
  lcrzo_er(lcrzo_fd_write(ci.writepipe, typelongueur, 6));

  /*ecriture des donnees*/
  lcrzo_er(lcrzo_fd_write(ci.writepipe, donnees, nboctdonnees));

  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_priv_ipc_readtlv(lcrzo_ipc ci,
			   lcrzo_bool etrebloquant,
			   lcrzo_uint16 *ptype,
			   lcrzo_data *pdataout,
			   lcrzo_int32 *pdataoutsize);
int lcrzo_priv_ipc_readtlv(lcrzo_ipc ci,
			   lcrzo_bool etrebloquant,
			   lcrzo_uint16 *ptype,
			   lcrzo_data *pdataout,
			   lcrzo_int32 *pdataoutsize)
{ lcrzo_uint8 typelongueur[6];
  lcrzo_uint16 type;
  lcrzo_uint32 nboctdonnees;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*lecture de type-longueur*/
  lcrzo_er(lcrzo_fd_read_fixed(ci.readpipe, etrebloquant,
			       6, typelongueur));
  /*si on arrive ici, on a lu type-valeur, on doit donc continuer*/
  type = (lcrzo_uint16)( (typelongueur[0]<<8) | typelongueur[1] );
  nboctdonnees = (typelongueur[2]<<24) | (typelongueur[3]<<16) | 
                 (typelongueur[4]<<8) | typelongueur[5] ;

  /*lit les donnees*/
  if (nboctdonnees)
  { lcrzo_er(lcrzo_fd_readm_fixed(ci.readpipe, LCRZO_TRUE, nboctdonnees, 
				  pdataout));
  }

  if (ptype!=NULL) *ptype=type;
  if (pdataoutsize!=NULL) *pdataoutsize=nboctdonnees;
  return(LCRZO_ERR_OK);
}



/*-------------------------------------------------------------*/
int lcrzo_ipc_write_uint8(lcrzo_ipc ci, lcrzo_uint8 uint8)
{ lcrzo_uint8 donnees[2];
  lcrzo_uint32 nboctdonnees;

  /*preparation des donnees a ecrire*/
  donnees[0]=uint8;
  nboctdonnees=1;

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_UINT8,
				 donnees, nboctdonnees));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_uint8(lcrzo_ipc ci,
			 lcrzo_bool etrebloquant,
			 lcrzo_uint8 *puint8)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_UINT8)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees!=1 )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  /*conversion des donnees*/ 
  if (puint8!=NULL) *puint8 = ptr[0] ;
  lcrzo_data_free(ptr);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_uint16(lcrzo_ipc ci, lcrzo_uint16 uint16)
{ lcrzo_uint8 donnees[2];
  lcrzo_int32 nboctdonnees;

  /*preparation des donnees a ecrire*/
  donnees[0]=(lcrzo_uint8)( (uint16>>8)&0xFF );
  donnees[1]=(lcrzo_uint8)( uint16&0xFF );
  nboctdonnees=2;

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_UINT16,
				 donnees, nboctdonnees));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_uint16(lcrzo_ipc ci,
			 lcrzo_bool etrebloquant,
			 lcrzo_uint16 *puint16)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_UINT16)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees!=2 )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  /*conversion des donnees*/ 
  if (puint16!=NULL) *puint16 = (lcrzo_uint16)( (ptr[0]<<8) | ptr[1] );
  lcrzo_data_free(ptr);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_uint32(lcrzo_ipc ci, lcrzo_uint32 uint32)
{ lcrzo_uint8 donnees[4];
  lcrzo_int32 nboctdonnees;

  /*preparation des donnees a ecrire*/
  donnees[0]=(lcrzo_uint8)(uint32>>24);
  donnees[1]=(lcrzo_uint8)((uint32>>16)&0xFF);
  donnees[2]=(lcrzo_uint8)((uint32>>8)&0xFF);
  donnees[3]=(lcrzo_uint8)(uint32&0xFF);
  nboctdonnees=4;

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_UINT32,
				 donnees, nboctdonnees));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_uint32(lcrzo_ipc ci,
			 lcrzo_bool etrebloquant,
			 lcrzo_uint32 *puint32)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_UINT32)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees!=4 )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  /*conversion des donnees*/ 
  if (puint32!=NULL)
  { *puint32 = (ptr[0]<<24) | (ptr[1]<<16) | (ptr[2]<<8) | ptr[3] ;
  }
  lcrzo_data_free(ptr);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_hs(lcrzo_ipc ci, const lcrzo_hs hs)
{ lcrzo_hs hscopie;

  /*preparation des donnees a ecrire*/
  lcrzo_er(lcrzo_hs_init(hs, hscopie));

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_HS,
				 (const lcrzo_uint8 *)hscopie,
				 strlen(hscopie)));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_hs(lcrzo_ipc ci,
		    lcrzo_bool etrebloquant,
		    lcrzo_hs hs)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;
  int ret;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_HS)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees>LCRZO_HS_MAXBYTES )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  ptr[nboctdonnees]='\0';
  /*conversion des donnees*/ 
  ret=lcrzo_hs_init((const lcrzo_int8 *)ptr, hs);

  lcrzo_data_free(ptr);
  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_ips(lcrzo_ipc ci, const lcrzo_ips ips)
{ lcrzo_ips ipscopie;

  /*preparation des donnees a ecrire*/
  lcrzo_er(lcrzo_ips_init(ips, ipscopie));

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_IPS,
				 (const lcrzo_uint8 *)ipscopie,
				 strlen(ipscopie)));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_ips(lcrzo_ipc ci,
		     lcrzo_bool etrebloquant,
		     lcrzo_ips ips)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;
  int ret;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_IPS)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees>LCRZO_IPS_MAXBYTES )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  ptr[nboctdonnees]='\0';
  /*conversion des donnees*/ 
  ret=lcrzo_ips_init((const lcrzo_int8 *)ptr, ips);
  lcrzo_data_free(ptr);

  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_ipa(lcrzo_ipc ci, const lcrzo_ipa ipa)
{ 
  /*parameters verification*/
  if (ipa==NULL) return(LCRZO_ERR_PANULLPTR);

  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_IPA,
				 ipa, LCRZO_IPA_MAXBYTES));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_ipa(lcrzo_ipc ci,
		     lcrzo_bool etrebloquant,
		     lcrzo_ipa ipa)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_IPA)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees!=LCRZO_IPA_MAXBYTES )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  /*conversion des donnees*/ 
  if (ipa!=NULL) memcpy(ipa, ptr, LCRZO_IPA_MAXBYTES);
  lcrzo_data_free(ptr);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_ipl(lcrzo_ipc ci, lcrzo_ipl ipl)
{ return(lcrzo_ipc_write_uint32(ci, ipl));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_ipl(lcrzo_ipc ci,
		     lcrzo_bool etrebloquant,
		     lcrzo_ipl *pipl)
{ return(lcrzo_ipc_read_uint32(ci, etrebloquant, pipl));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_eths(lcrzo_ipc ci, const lcrzo_eths eths)
{ lcrzo_eths ethscopie;

  /*preparation des donnees a ecrire*/
  lcrzo_er(lcrzo_eths_init(eths, ethscopie));

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_ETHS,
				 (const lcrzo_uint8 *)ethscopie, 
				 strlen(ethscopie)));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_eths(lcrzo_ipc ci,
		     lcrzo_bool etrebloquant,
		     lcrzo_eths eths)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;
  int ret;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_ETHS)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees>LCRZO_ETHS_MAXBYTES )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  ptr[nboctdonnees]='\0';
  /*conversion des donnees*/ 
  ret=lcrzo_eths_init((const lcrzo_int8 *)ptr, eths);
  lcrzo_data_free(ptr);

  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_etha(lcrzo_ipc ci, const lcrzo_etha etha)
{
  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_ETHA,
				 etha, LCRZO_ETHA_MAXBYTES));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_etha(lcrzo_ipc ci,
		     lcrzo_bool etrebloquant,
		     lcrzo_etha etha)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_ETHA)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees!=LCRZO_ETHA_MAXBYTES )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  /*conversion des donnees*/ 
  if (etha!=NULL) memcpy(etha, ptr, LCRZO_ETHA_MAXBYTES);
  lcrzo_data_free(ptr);
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_device(lcrzo_ipc ci, const lcrzo_device device)
{ lcrzo_device devicecopie;

  /*preparation des donnees a ecrire*/
  lcrzo_er(lcrzo_device_init(device, devicecopie));

  /*ecriture*/
  return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_DEVICE,
				 (const lcrzo_uint8 *)devicecopie, 
				 strlen(devicecopie)));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_device(lcrzo_ipc ci,
			lcrzo_bool etrebloquant,
			lcrzo_device device)
{ lcrzo_data ptr;
  lcrzo_uint16 type;
  lcrzo_int32 nboctdonnees;
  int ret;

  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  &ptr, &nboctdonnees));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_DEVICE)
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  /*on verifie si la taille recue correspond*/
  if ( nboctdonnees>LCRZO_DEVICE_MAXBYTES )
  { lcrzo_data_free(ptr);
    return(LCRZO_ERR_IEIPCBADELEMSIZE);
  }
  ptr[nboctdonnees]='\0';
  /*conversion des donnees*/ 
  ret=lcrzo_device_init((const lcrzo_int8 *)ptr, device);
  lcrzo_data_free(ptr);

  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_deviceethaiplport(lcrzo_ipc ci,
				     const lcrzo_device device,
				     const lcrzo_etha etha,
				     lcrzo_ipl ipl,
				     lcrzo_uint16 port)
{
  /*parameters verification*/
  if (device==NULL) return(LCRZO_ERR_PANULLPTR);
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipc_write_device(ci, device));
  lcrzo_er(lcrzo_ipc_write_etha(ci, etha));
  lcrzo_er(lcrzo_ipc_write_ipl(ci, ipl));
  lcrzo_er(lcrzo_ipc_write_uint16(ci, port));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_deviceethaiplport(lcrzo_ipc ci, 
				   lcrzo_bool etrebloquant,
				   lcrzo_device device,
				   lcrzo_etha etha,
				   lcrzo_ipl *pipl,
				   lcrzo_uint16 *pport)
{
  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  lcrzo_er(lcrzo_ipc_read_device(ci, etrebloquant, device));
  lcrzo_er(lcrzo_ipc_read_etha(ci, 1, etha));
  lcrzo_er(lcrzo_ipc_read_ipl(ci, 1, pipl));
  lcrzo_er(lcrzo_ipc_read_uint16(ci, 1, pport));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_ethaiplport(lcrzo_ipc ci,
			       const lcrzo_etha etha,
			       lcrzo_ipl ipl,
			       lcrzo_uint16 port)
{
  /*parameters verification*/
  if (etha==NULL) return(LCRZO_ERR_PANULLPTR);

  lcrzo_er(lcrzo_ipc_write_etha(ci, etha));
  lcrzo_er(lcrzo_ipc_write_ipl(ci, ipl));
  lcrzo_er(lcrzo_ipc_write_uint16(ci, port));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_ethaiplport(lcrzo_ipc ci, 
			     lcrzo_bool etrebloquant,
			     lcrzo_etha etha,
			     lcrzo_ipl *pipl,
			     lcrzo_uint16 *pport)
{
  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  lcrzo_er(lcrzo_ipc_read_etha(ci, etrebloquant, etha));
  lcrzo_er(lcrzo_ipc_read_ipl(ci, 1, pipl));
  lcrzo_er(lcrzo_ipc_read_uint16(ci, 1, pport));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_iplport(lcrzo_ipc ci,
			   lcrzo_ipl ipl,
			   lcrzo_uint16 port)
{
  lcrzo_er(lcrzo_ipc_write_ipl(ci, ipl));
  lcrzo_er(lcrzo_ipc_write_uint16(ci, port));
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_iplport(lcrzo_ipc ci, 
			 lcrzo_bool etrebloquant,
			 lcrzo_ipl *pipl,
			 lcrzo_uint16 *pport)
{
  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  lcrzo_er(lcrzo_ipc_read_ipl(ci, etrebloquant, pipl));
  lcrzo_er(lcrzo_ipc_read_uint16(ci, 1, pport));
  return(LCRZO_ERR_OK);
}


/*-------------------------------------------------------------*/
int lcrzo_ipc_write_data(lcrzo_ipc ci,
			 lcrzo_constdata data,
			 lcrzo_int32 datasize)
{ return(lcrzo_priv_ipc_writetlv(ci, LCRZO_PRIV_IPCTYPE_BLOC,
				 data, datasize));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_data(lcrzo_ipc ci, 
			lcrzo_bool etrebloquant,
			lcrzo_int32 dataoutmaxsize,
			lcrzo_data dataout,
			lcrzo_int32 *pdataoutsize)
{ lcrzo_data ptr;
  lcrzo_int32 ptrsize;
  int ret;

  lcrzo_er(lcrzo_ipc_readm_data(ci, etrebloquant, &ptr, &ptrsize));
  ret=lcrzo_data_init_data(ptr, ptrsize, dataoutmaxsize, dataout,pdataoutsize);
  lcrzo_data_free(ptr);
  return(ret);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_readm_data(lcrzo_ipc ci, 
			 lcrzo_bool etrebloquant,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize)
{ lcrzo_uint16 type;
  
  /*verification des parametres*/
  lcrzo_bool_verifbof(etrebloquant);

  /*on lit le tlv suivant*/
  lcrzo_er(lcrzo_priv_ipc_readtlv(ci, etrebloquant, &type,
				  pdataout, pdataoutsize));
  /*on verifie si le type etait celui attendu*/
  if (type!=LCRZO_PRIV_IPCTYPE_BLOC)
  { if (pdataout!=NULL) lcrzo_data_free(*pdataout);
    return(LCRZO_ERR_BUIPCUNWANTEDTYPE);
  }
  return(LCRZO_ERR_OK);
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_write_flow(lcrzo_ipc ci,
			 lcrzo_constdata data,
			 lcrzo_int32 datasize)
{ return(lcrzo_fd_write(ci.writepipe, data, datasize));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_flow(lcrzo_ipc ci, 
			lcrzo_bool etrebloquant,
			lcrzo_int32 dataoutmaxsize,
			lcrzo_data dataout,
			lcrzo_int32 *pdataoutsize)
{ return(lcrzo_fd_read(ci.readpipe, etrebloquant,
		       dataoutmaxsize, dataout, pdataoutsize));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_readm_flow(lcrzo_ipc ci, 
			 lcrzo_bool etrebloquant,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize)
{ return(lcrzo_fd_readm(ci.readpipe, etrebloquant,
			pdataout, pdataoutsize));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_flow_fixed(lcrzo_ipc ci, 
			      lcrzo_bool etrebloquant,
			      lcrzo_int32 datasizetoread,
			      lcrzo_data dataout)
{ return(lcrzo_fd_read_fixed(ci.readpipe, etrebloquant, datasizetoread,
			     dataout));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_readm_flow_fixed(lcrzo_ipc ci, 
			       lcrzo_bool etrebloquant,
			       lcrzo_int32 datasizetoread,
			       lcrzo_data *pdataout)
{ return(lcrzo_fd_readm_fixed(ci.readpipe, etrebloquant, datasizetoread,
			      pdataout));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_read_flow_line(lcrzo_ipc ci, 
			     lcrzo_bool etrebloquant,
			     lcrzo_int32 dataoutmaxsize,
			     lcrzo_data dataout,
			     lcrzo_int32 *pdataoutsize)
{ return(lcrzo_fd_read_line(ci.readpipe, etrebloquant,
			    dataoutmaxsize, dataout, pdataoutsize));
}

/*-------------------------------------------------------------*/
int lcrzo_ipc_readm_flow_line(lcrzo_ipc ci, 
			      lcrzo_bool etrebloquant,
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize)
{ return(lcrzo_fd_readm_line(ci.readpipe, etrebloquant,
			     pdataout, pdataoutsize));
}







