/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_ipc module

  Functions herein allow two process to communicate.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_ipc

  Les fonctions presentes dans ce module permettent de communiquer
des donnees entres deux processus.
*/

/*US*********************************************************
 * Note about an ipc :                                      *
 * A lcrzo_ipc is used to transmit data between two         *
 * processes.                                               *
 * For example, the first process can write an integer into *
 * the "tube", and the other process will read it.          *
 ************************************************************/
/*FR*********************************************************
 * Note sur les ipc :                                       *
 * Un lcrzo_ipc est utilise pour transmettre des donnees    *
 * entre deux processus.                                    *
 * Par exemple, le premier processus peut ecrire un entier  *
 * dans le "tube", et l'autre processus peut le lire.       *
 ************************************************************/

/*---------------------------------------------------------------*/
/*us the definition of a lcrzo_ipc has to be public
   (because the compiler needs to know its size), 
   but you should not use its fields directly. */
/*fr la definition d'un lcrzo_ipc doit etre publique
   (car le compilateur doit connaitre sa taille), mais vous
   ne devriez pas utiliser directement ses champs. */
typedef struct
{ int readpipe;
  int writepipe;
} lcrzo_ipc;

/*---------------------------------------------------------------*/
/*us Name : lcrzo_ipc_init
   Description :
     Initialize a lcrzo_ipc.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *pipc : lcrzo_ipc initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_init
   Description :
     Initialise un lcrzo_ipc
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pipc : lcrzo_ipc initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_init(lcrzo_ipc *pipc);

/*us Name : lcrzo_ipc_close
   Description :
     Close a lcrzo_ipc.
   Input parameter(s) :
     ipc : lcrzo_ipc to close
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_close
   Description :
     Ferme un lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc a fermer
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_close(lcrzo_ipc ipc);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_ipc_write_uint8
   Description :
     Write an lcrzo_uint8 in a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc where the data is written
     uint8 : lcrzo_uint8 to write in ipc
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_write_uint8
   Description :
     Ecrit un lcrzo_uint8 dans un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont ecrites
     uint8 : lcrzo_uint8 a ecrire dans ipc
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_write_uint8(lcrzo_ipc ipc,
			  lcrzo_uint8 uint8);

/*us Name : lcrzo_ipc_read_uint8
   Description :
     Read an lcrzo_uint8 from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *puint8 : lcrzo_uint8 read from the ipc
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
*/
/*fr Nom : lcrzo_ipc_read_uint8
   Description :
     Lit un lcrzo_uint8 depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *puint8 : lcrzo_uint8 lu depuis ipc
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
			       retourne sans donnees.
*/
int lcrzo_ipc_read_uint8(lcrzo_ipc ipc, 
			 lcrzo_bool beblocking, 
			 lcrzo_uint8 *puint8);

/*us idem, but work on a lcrzo_uint16 */
/*fr idem, mais travaille sur un lcrzo_uint16 */
int lcrzo_ipc_write_uint16(lcrzo_ipc ipc,
			   lcrzo_uint16 uint16);
int lcrzo_ipc_read_uint16(lcrzo_ipc ipc, 
			  lcrzo_bool beblocking,
			  lcrzo_uint16 *puint16);

/*us idem, but work on a lcrzo_uint32 */
/*fr idem, mais travaille sur un lcrzo_uint32 */
int lcrzo_ipc_write_uint32(lcrzo_ipc ipc,
			   lcrzo_uint32 uint32);
int lcrzo_ipc_read_uint32(lcrzo_ipc ipc, 
			  lcrzo_bool beblocking,
			  lcrzo_uint32 *puint32);

/*us idem, but work on a lcrzo_hs */
/*fr idem, mais travaille sur un lcrzo_hs */
int lcrzo_ipc_write_hs(lcrzo_ipc ipc,
		       const lcrzo_hs hs);
int lcrzo_ipc_read_hs(lcrzo_ipc ipc, 
		      lcrzo_bool beblocking,
		      lcrzo_hs hs);

/*us idem, but work on a lcrzo_ips */
/*fr idem, mais travaille sur un lcrzo_ips */
int lcrzo_ipc_write_ips(lcrzo_ipc ipc,
			const lcrzo_ips ips);
int lcrzo_ipc_read_ips(lcrzo_ipc ipc, 
		       lcrzo_bool beblocking,
		       lcrzo_ips ips);

/*us idem, but work on a lcrzo_ipa */
/*fr idem, mais travaille sur un lcrzo_ipa */
int lcrzo_ipc_write_ipa(lcrzo_ipc ipc,
			const lcrzo_ipa ipa);
int lcrzo_ipc_read_ipa(lcrzo_ipc ipc, 
		       lcrzo_bool beblocking,
		       lcrzo_ipa ipa);

/*us idem, but work on a lcrzo_ipl */
/*fr idem, mais travaille sur un lcrzo_ipl */
int lcrzo_ipc_write_ipl(lcrzo_ipc ipc,
			lcrzo_ipl ipl);
int lcrzo_ipc_read_ipl(lcrzo_ipc ipc, 
		       lcrzo_bool beblocking,
		       lcrzo_ipl *pipl);

/*us idem, but work on a lcrzo_eths */
/*fr idem, mais travaille sur un lcrzo_eths */
int lcrzo_ipc_write_eths(lcrzo_ipc ipc,
			 const lcrzo_eths eths);
int lcrzo_ipc_read_eths(lcrzo_ipc ipc, 
			lcrzo_bool beblocking,
			lcrzo_eths eths);

/*us idem, but work on a lcrzo_etha */
/*fr idem, mais travaille sur un lcrzo_etha */
int lcrzo_ipc_write_etha(lcrzo_ipc ipc,
			 const lcrzo_etha etha);
int lcrzo_ipc_read_etha(lcrzo_ipc ipc, 
			lcrzo_bool beblocking,
			lcrzo_etha etha);

/*us idem, but work on a lcrzo_device */
/*fr idem, mais travaille sur un lcrzo_device */
int lcrzo_ipc_write_device(lcrzo_ipc ipc,
			   const lcrzo_device device);
int lcrzo_ipc_read_device(lcrzo_ipc ipc, 
			  lcrzo_bool beblocking,
			  lcrzo_device device);

/*---------------------------------------------------------------*/
/*us idem, but work on a bloc composed of
   device_name + Ethernet_address + IP_address + port_number */
/*fr idem, mais travaille sur un bloc compose de
   nom_de_device + adresse_Ethernet + adresse_IP + numero_de_port */
int lcrzo_ipc_write_deviceethaiplport(lcrzo_ipc ipc,
				      const lcrzo_device device,
				      const lcrzo_etha etha,
				      lcrzo_ipl ipl,
				      lcrzo_uint16 port);
int lcrzo_ipc_read_deviceethaiplport(lcrzo_ipc ipc, 
				     lcrzo_bool beblocking,
				     lcrzo_device device,
				     lcrzo_etha etha,
				     lcrzo_ipl *pipl,
				     lcrzo_uint16 *pport);

/*us idem, but work on a bloc composed of
   Ethernet_address + IP_address + port_number */
/*fr idem, mais travaille sur un bloc compose de
   adresse_Ethernet + adresse_IP + numero_de_port */
int lcrzo_ipc_write_ethaiplport(lcrzo_ipc ipc,
				const lcrzo_etha etha,
				lcrzo_ipl ipl,
				lcrzo_uint16 port);
int lcrzo_ipc_read_ethaiplport(lcrzo_ipc ipc, 
			       lcrzo_bool beblocking,
			       lcrzo_etha etha,
			       lcrzo_ipl *pipl,
			       lcrzo_uint16 *pport);

/*us idem, but work on a bloc composed of
   IP_address + port_number */
/*fr idem, mais travaille sur un bloc compose de
   adresse_IP + numero_de_port */
int lcrzo_ipc_write_iplport(lcrzo_ipc ipc,
			    lcrzo_ipl ipl,
			    lcrzo_uint16 port);
int lcrzo_ipc_read_iplport(lcrzo_ipc ipc, 
			   lcrzo_bool beblocking,
			   lcrzo_ipl *pipl,
			   lcrzo_uint16 *pport);

/*---------------------------------------------------------------*/
/*US*********************************************************
 * Note about lcrzo_ipc_xx_data and lcrzo_ipc_xx_flow :     *
 * The functions lcrzo_ipc_xx_data works on a bloc of data. *
 * The functions lcrzo_ipc_xx_flow works on a flow of data. *
 *                                                          *
 * Schematic example :                                      *
 *   lcrzo_ipc_write_data(ipc, "hello", 5);                 *
 *   lcrzo_ipc_write_data(ipc, "man", 3);                   *
 *   lcrzo_ipc_read_data(...);   reads "hello"              *
 *   lcrzo_ipc_read_data(...);   reads "man"                *
 *                                                          *
 *   lcrzo_ipc_write_flow(ipc, "hello", 5);                 *
 *   lcrzo_ipc_write_flow(ipc, "man", 3);                   *
 *   lcrzo_ipc_read_flow(...);   reads "helloman"           *
 ************************************************************/
/*FR*********************************************************
 * Note sur lcrzo_ipc_xx_data et lcrzo_ipc_xx_flow :        *
 * Les fonctions lcrzo_ipc_xx_data travaillent sur un bloc  *
 * de donnees.                                              *
 * Les fonctions lcrzo_ipc_xx_flow travaillent sur un flot  *
 * de donnees.                                              *
 *                                                          *
 * Exemple schematique :                                    *
 *   lcrzo_ipc_write_data(ipc, "hello", 5);                 *
 *   lcrzo_ipc_write_data(ipc, "man", 3);                   *
 *   lcrzo_ipc_read_data(...);   lit "hello"                *
 *   lcrzo_ipc_read_data(...);   lit "man"                  *
 *                                                          *
 *   lcrzo_ipc_write_flow(ipc, "hello", 5);                 *
 *   lcrzo_ipc_write_flow(ipc, "man", 3);                   *
 *   lcrzo_ipc_read_flow(...);   lit "helloman"             *
 ************************************************************/

/*---------------------------------------------------------------*/
/*us Name : lcrzo_ipc_write_data
   Description :
     Write a data block in a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc where the data is written
     data : data to write
     datasize : size of data
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_write_data
   Description :
     Ecrit un bloc de donnees dans un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont ecrites
     data : donnees a ecrire
     datasize : taille de data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_write_data(lcrzo_ipc ipc,
			 lcrzo_constdata data,
			 lcrzo_int32 datasize);

/*us Name : lcrzo_ipc_read_data
   Description :
     Read data from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_ipc_read_data
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
			       retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_read_data(lcrzo_ipc ipc, 
			lcrzo_bool beblocking,
			lcrzo_int32 dataoutmaxsize,
			lcrzo_data dataout,
			lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_ipc_readm_data
   Description :
     Read data from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
*/
/*fr Nom : lcrzo_ipc_readm_data
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
			       retourne sans donnees.
*/
int lcrzo_ipc_readm_data(lcrzo_ipc ipc, 
			 lcrzo_bool beblocking,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_ipc_write_flow
   Description :
     Write a data flow in a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc where the data is written
     data : data to write
     datasize : size of data
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_ipc_write_flow
   Description :
     Ecrit un flux de donnees dans un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont ecrites
     data : donnees a ecrire
     datasize : taille de data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_ipc_write_flow(lcrzo_ipc ipc,
			 lcrzo_constdata data,
			 lcrzo_int32 datasize);

/*us Name : lcrzo_ipc_read_flow
   Description :
     Read data from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_ipc_read_flow
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
			       retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_read_flow(lcrzo_ipc ipc, 
			lcrzo_bool beblocking,
			lcrzo_int32 dataoutmaxsize,
			lcrzo_data dataout,
			lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_ipc_readm_flow
   Description :
     Read data from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
*/
/*fr Nom : lcrzo_ipc_readm_flow
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_readm_flow(lcrzo_ipc ipc, 
			 lcrzo_bool beblocking,
			 lcrzo_data *pdataout,
			 lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_ipc_read_flow_fixed
   Description :
     Read data from a lcrzo_ipc tube. Data read has a fixed size.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
     datasizetoread : exact size to read
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_ipc_read_flow_fixed
   Description :
     Lit des donnees depuis un tube lcrzo_ipc.
     Les donnees lues ont une taille constante.
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     datasizetoread : taille exacte a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
			       retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_read_flow_fixed(lcrzo_ipc ipc, 
			      lcrzo_bool beblocking,
			      lcrzo_int32 datasizetoread,
			      lcrzo_data dataout);

/*us Name : lcrzo_ipc_readm_flow_fixed
   Description :
     Read data from a lcrzo_ipc tube. Data read has a fixed size.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
     datasizetoread : exact size to read
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
*/
/*fr Nom : lcrzo_ipc_readm_flow_fixed
   Description :
     Lit des donnees depuis un tube lcrzo_ipc
     Les donnees lues ont une taille constante.
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     datasizetoread : taille exacte a lire
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_readm_flow_fixed(lcrzo_ipc ipc, 
			       lcrzo_bool beblocking,
			       lcrzo_int32 datasizetoread,
			       lcrzo_data *pdataout);

/*us Name : lcrzo_ipc_read_flow_line
   Description :
     Read a line (till \r\n) from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
     dataoutmaxsize : max size which can be stored in dataout
   Input/output parameter(s) :
   Output parameter(s) :
     dataout : output array.
               This array will be set with the read data
     *pdataoutsize : reached size of dataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize reached before
                                 being able to store all the needed
                                 data in dataout
*/
/*fr Nom : lcrzo_ipc_read_flow_line
   Description :
     Lit une ligne (jusqu'a \r\n) depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans dataout
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     dataout : tableau de sortie.
               Ce tableau recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, et la function
                               n'a pas recu de donnees, donc on 
			       retourne sans donnees.
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_read_flow_line(lcrzo_ipc ipc, 
			     lcrzo_bool beblocking,
			     lcrzo_int32 dataoutmaxsize,
			     lcrzo_data dataout,
			     lcrzo_int32 *pdataoutsize);

/*us Name : lcrzo_ipc_readm_flow_line
   Description :
     Read a line (till \r\n) from a lcrzo_ipc tube.
   Input parameter(s) :
     ipc : lcrzo_ipc from where the datas are read
     beblocking : if we want to block until receiving data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdataout : pointer which will be malloced (so, the
                 memory will have to be freed by the
                 user with 'free(*pdataout)').
                 The allocated memory will be set with the read data
     *pdataoutsize : reached size of *pdataout
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKTEMPDATAEND : beblocking==0, and the function
                               received nothing, so it exited
			       without reading data
*/
/*fr Nom : lcrzo_ipc_readm_flow_line
   Description :
     Lit une ligne (jusqu'a \r\n) depuis un tube lcrzo_ipc
   Parametre(s) d'entree :
     ipc : lcrzo_ipc ou les donnees sont lues
     beblocking : si on veut bloquer jusqu'a recevoir des donnees
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdataout : pointeur qui sera alloue (la memoire
                 devra etre liberee par l'utilisateur 
                 avec 'free(*pdataout)').
                 Cette memoire allouee recevra les donnees lues
     *pdataoutsize : taille atteinte par dataout
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : dataoutmaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans dataout
*/
int lcrzo_ipc_readm_flow_line(lcrzo_ipc ipc, 
			      lcrzo_bool beblocking,
			      lcrzo_data *pdataout,
			      lcrzo_int32 *pdataoutsize);
