/*
		                  LCRZO
                             Network library
             Copyright(c) 1999,2000,2001, Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
  [my current email address is on the web servers]

                                  -----
  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details (http://www.gnu.org).

------------------------------------------------------------------------
*/

/* -- Note about comments and variable names --
   In versions 1.00 up to 2.08, lcrzo was only written in French (my
   mother tongue). Since version 3.00, lcrzo is written in English, but
   most of comments and variable names are still in French.
   In future versions, I'll try to translate everything into English,
   but as you may guess, that's a quite big work...
   So, for the moment, here are some word translation, if you want to
   be able to understand what's going on in the functions :
     afficher - print     ;  ajouter - add         ;  copie - copy
     dernier - last       ;  donnee - data         ;  ecrire - write
     entete - header      ;  envoye - sent         ;  ferme - close
     fichier - file       ;  lire - read           ;  masque - mask 
     nboct - byte number  ;  ouvre, ouvrir - open  ;  paq, paquet - packet
     plage - range        ;  premier - first       ;  recu - received
     reseau - network     ;  retour - return       ;  taille - size
     tunnel - pipe

*/

#include "lcrzo_priv.h"
#ifdef LCRZODEF_SYSTEM_Linux
 #include <stdlib.h>
#elif defined LCRZODEF_SYSTEM_FreeBSD
 #include <stdlib.h>
#elif defined LCRZODEF_SYSTEM_Solaris
 #include <stdlib.h>
#else
 #error "Traiter le cas de LCRZODEF_SYSTEM"
#endif

/*---------------------------------------------------------------*/
int lcrzo_priv_list_goto_listelem(lcrzo_list *pliste,
				  int direction, 
				  lcrzo_int32 distance,
				  lcrzo_bool stoponebefore,
				  lcrzo_listelem **pplistelem);
int lcrzo_priv_list_goto_listelem(lcrzo_list *pliste,
				  int direction, 
				  lcrzo_int32 distance,
				  lcrzo_bool stoponebefore,
				  lcrzo_listelem **pplistelem)
{ int i;

  lcrzo_bool_verifbof(stoponebefore);
  if (stoponebefore) i=1;
  else i=0;
  *pplistelem=(lcrzo_listelem *)pliste;
  /*si on parcourt la liste en croissant*/
  if (direction==LCRZO_PRIV_INCREASE)
  { /*on va jusqu'a l'element nous interessant*/
    while (i<distance)
    { *pplistelem=(lcrzo_listelem *)(*pplistelem)->pnext;
      i++;
    }
  }
  else
  { /*on va jusqu'a l'element nous interessant*/
    while (i<distance)
    { *pplistelem=(lcrzo_listelem *)(*pplistelem)->pprecedent;
      i++;
    }
  }
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_priv_list_remove_listelem(lcrzo_list *pliste,
				    lcrzo_listelem *plistelem);
int lcrzo_priv_list_remove_listelem(lcrzo_list *pliste,
				    lcrzo_listelem *plistelem)
{ lcrzo_listelem *pnext, *pprevious;

  /*unlink the element*/
  pnext = (lcrzo_listelem *)plistelem->pnext;
  pprevious = (lcrzo_listelem *)plistelem->pprecedent;
  pnext->pprecedent = (lcrzo_listelem *)pprevious;
  pprevious->pnext = (lcrzo_listelem *)pnext;

  /*free the memory*/
  if ( pliste->pfunc_erase != NULL ) /*dispose d'une procedure*/
    (*pliste->pfunc_erase)(plistelem->pelem);
  free(plistelem->pelem);
  free(plistelem);
  pliste->elemcount --;

  return(LCRZO_ERR_OK);
}
/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/


/*---------------------------------------------------------------*/
int lcrzo_list_init(lcrzo_list *pliste, lcrzo_uint32 sizeofelems,
		    void (*pfonceffaceelem)(void *pelem))
{
  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);

  pliste->pnext=(lcrzo_listelem *)pliste;
  pliste->pprecedent=(lcrzo_listelem *)pliste;
  pliste->sizeofelems=sizeofelems;
  pliste->elemcount=0;
  pliste->pfunc_erase=pfonceffaceelem;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_close(lcrzo_list *pliste)
{ return(lcrzo_list_remove_all(pliste));
}

/*---------------------------------------------------------------*/
int lcrzo_list_add_pos(lcrzo_list *pliste, lcrzo_int32 position,
		       const void *pelem)
{ lcrzo_int32 distance;
  int sens;
  lcrzo_listelem *plisteelem, *ptravail1, *ptravail2;
    
  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  if (pelem==NULL) return(LCRZO_ERR_PANULLPTR);
  /*liste pleine*/
  if (pliste->elemcount >= 0x7FFFFFFF) return(LCRZO_ERR_IELISTFULL);

  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, position, 1,
				     NULL, NULL, &sens, &distance));

  /*allocation de la memoire de mon element*/
  plisteelem=malloc(sizeof(lcrzo_listelem));
  if (plisteelem==NULL) return(LCRZO_ERR_FUMALLOC);
  plisteelem->pelem=malloc(pliste->sizeofelems);
  if (plisteelem->pelem==NULL) return(LCRZO_ERR_FUMALLOC);
  memcpy(plisteelem->pelem, pelem, pliste->sizeofelems);

  /*we go to the first element (stop one before)*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, sens, distance, 1,
					 &ptravail1));

  /*si on parcourt la liste en croissant*/
  if (sens==LCRZO_PRIV_INCREASE)
  { /*on insere notre element*/
    plisteelem->pnext = ptravail1->pnext;
    plisteelem->pprecedent = (lcrzo_listelem *)ptravail1;
    ptravail2 = (lcrzo_listelem *)ptravail1->pnext;
    ptravail2->pprecedent = (lcrzo_listelem *)plisteelem;
    ptravail1->pnext = (lcrzo_listelem *)plisteelem;
  }
  else
  { /*on insere notre element*/
    plisteelem->pprecedent = ptravail1->pprecedent;
    plisteelem->pnext = (lcrzo_listelem *)ptravail1;
    ptravail2 = (lcrzo_listelem *)ptravail1->pprecedent;
    ptravail2->pnext = (lcrzo_listelem *)plisteelem;
    ptravail1->pprecedent = (lcrzo_listelem *)plisteelem;
  }
  /*on a ajoute un element*/
  pliste->elemcount ++;

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_add_first(lcrzo_list *pliste, const void *pelem)
{ return(lcrzo_list_add_pos(pliste, +1, pelem));
}

/*---------------------------------------------------------------*/
int lcrzo_list_add_last(lcrzo_list *pliste, const void *pelem)
{ return(lcrzo_list_add_pos(pliste, -1, pelem));
}


/*---------------------------------------------------------------*/
int lcrzo_list_remove_pos(lcrzo_list *pliste, lcrzo_int32 position)
{ lcrzo_int32 distance;
  int sens;
  lcrzo_listelem *ptravail;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, position, 0,
				     NULL, NULL, &sens, &distance));

  /*we go to the element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, sens, distance, 0, 
					 &ptravail));

  /*on efface l'element*/
  lcrzo_er(lcrzo_priv_list_remove_listelem(pliste, ptravail));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_first(lcrzo_list *pliste)
{ return(lcrzo_list_remove_pos(pliste, +1));
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_last(lcrzo_list *pliste)
{ return(lcrzo_list_remove_pos(pliste, -1));
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_range(lcrzo_list *pliste, 
			    lcrzo_int32 posdebut, lcrzo_int32 posfin)
{ lcrzo_int32 infdistance, supdistance, i;
  int infdirection, supdirection;
  lcrzo_listelem *ptravail;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  if ( pliste->elemcount==0 && posdebut==+1 && posfin==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrangefast(pliste->elemcount, posdebut, posfin,
					  NULL, NULL, NULL, NULL,
					  &infdirection, &infdistance,
					  &supdirection, &supdistance));

  /*we go to the first element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, infdirection, infdistance, 0,
					 &ptravail));

  /*remove the range*/
  i=0;
  if (supdirection==LCRZO_PRIV_INCREASE)
  { ptravail = (lcrzo_listelem *)ptravail->pprecedent;
    while (i<=supdistance)
    { lcrzo_er(lcrzo_priv_list_remove_listelem(pliste, ptravail->pnext));
      i++;
    }
  }
  else
  { ptravail = (lcrzo_listelem *)ptravail->pnext;
    while (i<=supdistance)
    { lcrzo_er(lcrzo_priv_list_remove_listelem(pliste, ptravail->pprecedent));
      i++;
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_all(lcrzo_list *pliste)
{ lcrzo_er(lcrzo_list_remove_range(pliste, 1, -1));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_duplicate_range(lcrzo_list *pliste,
				      lcrzo_int32 posdebut,
				      lcrzo_int32 posfin,
				      int (*pfonccmpelem)(const void *peleminf,
							  const void *pelemsup,
							  const void *pinfos),
				      const void *pinfos)
{ lcrzo_int32 infdistance, supdistance, i, j, numremoved;
  int infdirection, supdirection;
  lcrzo_listelem *ptravail, *ptravail2;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  if (pfonccmpelem==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  if ( pliste->elemcount==0 && posdebut==+1 && posfin==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrangefast(pliste->elemcount, posdebut, posfin,
					  NULL, NULL, NULL, NULL,
					  &infdirection, &infdistance,
					  &supdirection, &supdistance));

  /*we go to the first element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, infdirection, infdistance, 0,
					 &ptravail));

  /*remove the elements in the range*/
  numremoved=0;
  if (supdirection==LCRZO_PRIV_INCREASE)
  { ptravail = (lcrzo_listelem *)ptravail->pprecedent;
    for ( i=0 ; i<=supdistance-numremoved ; i++ )
    { ptravail = (lcrzo_listelem *)ptravail->pnext;
      ptravail2 = (lcrzo_listelem *)ptravail;
      for ( j=i+1 ; j<=supdistance-numremoved ; j++ )
      { if ( !(*pfonccmpelem)(ptravail->pelem, ptravail2->pnext->pelem,
			      pinfos) )
        { lcrzo_er(lcrzo_priv_list_remove_listelem(pliste, ptravail2->pnext));
	  numremoved++;
	}
	else
	{ ptravail2 = (lcrzo_listelem *)ptravail2->pnext;
        }
      }
    }
  }
  else
  { ptravail = (lcrzo_listelem *)ptravail->pnext;
    for ( i=0 ; i<=supdistance-numremoved ; i++ )
    { ptravail = (lcrzo_listelem *)ptravail->pprecedent;
      ptravail2 = (lcrzo_listelem *)ptravail;
      for ( j=i+1 ; j<=supdistance-numremoved ; j++ )
      { /*for debug printf("%ld, %ld : %d %d\n", i, j,
	       *(int*)(ptravail->pelem),
	       *(int*)(ptravail2->pprecedent->pelem)
	       );*/
	if ( !(*pfonccmpelem)(ptravail->pelem, ptravail2->pprecedent->pelem,
			      pinfos) )
        { lcrzo_er(lcrzo_priv_list_remove_listelem(pliste,
						   ptravail2->pprecedent));
	  numremoved++;
	}
	else
	{ ptravail2 = (lcrzo_listelem *)ptravail2->pprecedent;
        }	
      }
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_duplicate_all(lcrzo_list *pliste,
				    int (*pfonccmpelem)(const void *peleminf,
							const void *pelemsup,
							const void *pinfos),
				    const void *pinfos)
{ lcrzo_er(lcrzo_list_remove_duplicate_range(pliste, 1, -1,
					     pfonccmpelem, pinfos));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_criteria_range(lcrzo_list *pliste,
				     lcrzo_int32 posdebut,
				     lcrzo_int32 posfin,
				     int (*pfonccritere)(const void *pelem,
							 const void *pinfos),
				     const void *pinfos)
{ lcrzo_int32 infdistance, supdistance, i;
  int infdirection, supdirection;
  lcrzo_listelem *ptravail;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  if (pfonccritere==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  if ( pliste->elemcount==0 && posdebut==+1 && posfin==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrangefast(pliste->elemcount, posdebut, posfin,
					  NULL, NULL, NULL, NULL,
					  &infdirection, &infdistance,
					  &supdirection, &supdistance));

  /*we go to the first element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, infdirection, infdistance, 0,
					 &ptravail));

  /*remove the elements in the range*/
  if (supdirection==LCRZO_PRIV_INCREASE)
  { ptravail = (lcrzo_listelem *)ptravail->pprecedent;
    for ( i=0 ; i<=supdistance; i++ )
    { if ( (*pfonccritere)(ptravail->pnext->pelem, pinfos) )
      { lcrzo_er(lcrzo_priv_list_remove_listelem(pliste, ptravail->pnext));
      }
      else
      { ptravail = (lcrzo_listelem *)ptravail->pnext;
      }
    }
  }
  else
  { ptravail = (lcrzo_listelem *)ptravail->pnext;
    for ( i=0 ; i<=supdistance; i++ )
    { if ( (*pfonccritere)(ptravail->pprecedent->pelem, pinfos) )
      { lcrzo_er(lcrzo_priv_list_remove_listelem(pliste,
						 ptravail->pprecedent));
      }
      else
      { ptravail = (lcrzo_listelem *)ptravail->pprecedent;
      }
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_remove_criteria_all(lcrzo_list *pliste,
				   int (*pfonccritere)(const void *pelem,
						       const void *pinfos),
				   const void *pinfos)
{ lcrzo_er(lcrzo_list_remove_criteria_range(pliste, 1, -1,
					    pfonccritere, pinfos));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_replace_pos(lcrzo_list *pliste,
			   lcrzo_int32 position,
			   const void *pelem)
{ lcrzo_int32 distance;
  int sens;
  lcrzo_listelem *ptravail;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  if (pelem==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, position, 0,
				     NULL, NULL, &sens, &distance));

  /*we go to the element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, sens, distance, 0, 
					 &ptravail));

  /*replace*/
  if ( pliste->pfunc_erase != NULL ) /*dispose d'une procedure*/
    (*pliste->pfunc_erase)(ptravail->pelem);
  memcpy(ptravail->pelem, pelem, pliste->sizeofelems);

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_replace_first(lcrzo_list *pliste, const void *pelem)
{ return(lcrzo_list_replace_pos(pliste, +1, pelem));
}

/*---------------------------------------------------------------*/
int lcrzo_list_replace_last(lcrzo_list *pliste, const void *pelem)
{ return(lcrzo_list_replace_pos(pliste, -1, pelem));
}

/*---------------------------------------------------------------*/
int lcrzo_list_switch(lcrzo_list *pliste,
		      lcrzo_int32 position1, 
		      lcrzo_int32 position2)
{ lcrzo_int32 distance1, distance2, pospositif1, pospositif2;
  int sens1, sens2;
  lcrzo_listelem *pprec1, *ptrav1, *psuiv1;
  lcrzo_listelem *pprec2, *ptrav2, *psuiv2;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, position1, 0,
				     NULL, &pospositif1, &sens1, &distance1));
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, position2, 0,
				     NULL, &pospositif2, &sens2, &distance2));

  /*pas besoin de permutter*/
  if (pospositif1==pospositif2) return(LCRZO_ERR_OK);

  /*we go to the elements*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, sens1, distance1, 0, 
					 &ptrav1));
  lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, sens2, distance2, 0, 
					 &ptrav2));

  /*on pointe aussi sur leurs precedents et suivants*/
  pprec1=(lcrzo_listelem *)ptrav1->pprecedent;
  psuiv1=(lcrzo_listelem *)ptrav1->pnext;
  pprec2=(lcrzo_listelem *)ptrav2->pprecedent;
  psuiv2=(lcrzo_listelem *)ptrav2->pnext;

  /*maintenant, on permutte*/
  if (pospositif1+1==pospositif2)
  { ptrav1->pprecedent=(lcrzo_listelem *)ptrav2;
    ptrav1->pnext=(lcrzo_listelem *)psuiv2;
    ptrav2->pprecedent=(lcrzo_listelem *)pprec1;
    ptrav2->pnext=(lcrzo_listelem *)ptrav1;
    pprec1->pnext=(lcrzo_listelem *)ptrav2;
    psuiv2->pprecedent=(lcrzo_listelem *)ptrav1;
  }
  else if (pospositif2+1==pospositif1)
  { ptrav2->pprecedent=(lcrzo_listelem *)ptrav1;
    ptrav2->pnext=(lcrzo_listelem *)psuiv1;
    ptrav1->pprecedent=(lcrzo_listelem *)pprec2;
    ptrav1->pnext=(lcrzo_listelem *)ptrav2;
    pprec2->pnext=(lcrzo_listelem *)ptrav1;
    psuiv1->pprecedent=(lcrzo_listelem *)ptrav2;
  }
  else /* cas standard ou les deux elems ne se suivent pas */
  { ptrav1->pprecedent=(lcrzo_listelem *)pprec2;
    ptrav1->pnext=(lcrzo_listelem *)psuiv2;
    ptrav2->pprecedent=(lcrzo_listelem *)pprec1;
    ptrav2->pnext=(lcrzo_listelem *)psuiv1;
    pprec1->pnext=(lcrzo_listelem *)ptrav2;
    pprec2->pnext=(lcrzo_listelem *)ptrav1;
    psuiv1->pprecedent=(lcrzo_listelem *)ptrav2;
    psuiv2->pprecedent=(lcrzo_listelem *)ptrav1;
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_invert_range(lcrzo_list *pliste,
			    lcrzo_int32 posdebut,
			    lcrzo_int32 posfin)
{ lcrzo_int32 pospositif1, pospositif2;
  lcrzo_int32 nbapermutter, i;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  if ( pliste->elemcount==0 && posdebut==+1 && posfin==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, posdebut, 0,
				     NULL, &pospositif1, NULL, NULL));
  lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, posfin, 0,
				     NULL, &pospositif2, NULL, NULL));

  if ( pospositif2>pospositif1 )
  { nbapermutter=pospositif2-pospositif1+1;
    /*calcul du nombre de permuttations a faire*/
    if (nbapermutter%2) /*impair*/
      nbapermutter=(nbapermutter-1)/2;
    else
      nbapermutter=(nbapermutter)/2;
    /*permuttations*/
    for ( i=0 ; i<nbapermutter ; i++)
    { lcrzo_er(lcrzo_list_switch(pliste, pospositif1+i, pospositif2-i));
    }
  }
  else
  { nbapermutter=pospositif1-pospositif2+1;
    /*calcul du nombre de permuttations a faire*/
    if (nbapermutter%2) /*impair*/
      nbapermutter=(nbapermutter-1)/2;
    else
      nbapermutter=(nbapermutter)/2;
    /*permuttations*/
    for ( i=0 ; i<nbapermutter ; i++)
    { lcrzo_er(lcrzo_list_switch(pliste, pospositif2+i, pospositif1-i));
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_invert_all(lcrzo_list *pliste)
{ lcrzo_er(lcrzo_list_invert_range(pliste, 1, -1));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_value_pos(lcrzo_list liste, lcrzo_int32 position,
			 void *pelem)
{ int sens;
  lcrzo_listelem *ptravail;
  lcrzo_int32 distance;

  /*verification des parametres et affectation des positions*/
  lcrzo_er(lcrzo_priv_manageelempos(liste.elemcount, position, 0,
				     NULL, NULL, &sens, &distance));

  lcrzo_er(lcrzo_priv_list_goto_listelem(&liste, sens, distance, 0, 
					 &ptravail));

  if (pelem!=NULL) memcpy(pelem, ptravail->pelem, liste.sizeofelems);
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_value_first(lcrzo_list liste, void *pelem)
{ return(lcrzo_list_value_pos(liste, +1, pelem));
}

/*---------------------------------------------------------------*/
int lcrzo_list_value_last(lcrzo_list liste, void *pelem)
{ return(lcrzo_list_value_pos(liste, -1, pelem));
}

/*---------------------------------------------------------------*/
int lcrzo_list_count(lcrzo_list liste, lcrzo_int32 *pnbelems)
{ if (pnbelems!=NULL) *pnbelems=liste.elemcount;
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_sort_range(lcrzo_list *pliste,
			  lcrzo_int32 startpos,
			  lcrzo_int32 endpos,
			  int (*pfonccmpelem)(const void *peleminf,
					      const void *pelemsup,
					      const void *pinfos),
			  const void *pinfos)
{ lcrzo_int32 startpo, endpo, i, j, idistance, nummini;
  int infsupdirection, idirection;
  lcrzo_listelem *ptrav1, *ptrav2, *pelemmini;

  /*parameters verification*/
  if (pliste==NULL) return(LCRZO_ERR_PANULLPTR);
  if (pfonccmpelem==NULL) return(LCRZO_ERR_PANULLPTR);
  /*verification des parametres et affectation des positions*/
  if ( pliste->elemcount==0 && startpos==+1 && endpos==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrange(pliste->elemcount, startpos, endpos,
				      NULL, &startpo, NULL, &endpo,
				      NULL, NULL, &infsupdirection, NULL));

  /*sort the elements in the range*/
  if (infsupdirection==LCRZO_PRIV_INCREASE)
  { for ( i=startpo ; i<=endpo ; i++ )
    { /*on va jusqu'a l'element i*/
      lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, i, 0, 
					 NULL, NULL, &idirection, &idistance));
      lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, idirection, idistance,
					     0, &ptrav1));
      /*now we try to find the mini*/
      nummini=i;
      pelemmini=ptrav1->pelem;
      ptrav2=(lcrzo_listelem *)ptrav1->pnext;
      for ( j=i+1 ; j<=endpo ; j++ )
      { /*printf("i=%ld, j=%ld %d\n", i, j, *(int*)(ptrav2->pelem));*/
        if ((*pfonccmpelem)(ptrav2->pelem, pelemmini, pinfos)==-1)
        { nummini=j;
          pelemmini=ptrav2->pelem;
        }
        ptrav2=(lcrzo_listelem *)ptrav2->pnext;
      }
      /*on permutte*/
      lcrzo_er(lcrzo_list_switch(pliste, i, nummini));
    }
  }
  else
  { for ( i=startpo ; i>=endpo ; i-- )
    { /*on va jusqu'a l'element i*/
      lcrzo_er(lcrzo_priv_manageelempos(pliste->elemcount, i, 0, 
					 NULL, NULL, &idirection, &idistance));
      lcrzo_er(lcrzo_priv_list_goto_listelem(pliste, idirection, idistance,
					     0, &ptrav1));
      /*now we try to find the mini*/
      nummini=i;
      pelemmini=ptrav1->pelem;
      ptrav2=(lcrzo_listelem *)ptrav1->pprecedent;
      for ( j=i-1 ; j>=endpo ; j-- )
      { /*printf("i=%ld, j=%ld %d\n", i, j, *(int*)(ptrav2->pelem));*/
        if ((*pfonccmpelem)(ptrav2->pelem, pelemmini, pinfos)==-1)
        { nummini=j;
          pelemmini=ptrav2->pelem;
        }
        ptrav2=(lcrzo_listelem *)ptrav2->pprecedent;
      }
      /*on permutte*/
      lcrzo_er(lcrzo_list_switch(pliste, i, nummini));
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_sort_all(lcrzo_list *pliste,
			int (*pfonccmpelem)(const void *peleminf,
					    const void *pelemsup,
					    const void *pinfos),
			const void *pinfos)
{ lcrzo_er(lcrzo_list_sort_range(pliste, 1, -1, pfonccmpelem, pinfos));
  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_search_range(lcrzo_list liste,
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos,
			    int (*pfoncrecherche)(const void *pelem,
						  const void *pinfos),
			    const void *pinfos,
			    lcrzo_int32 *pposelem,
			    void *pelem)
{ lcrzo_int32 infdistance, supdistance, i, infpospos;
  int infdirection, supdirection;
  lcrzo_listelem *ptravail;

  /*verification des parametres et affectation des positions*/
  if (pfoncrecherche==NULL) return(LCRZO_ERR_PANULLPTR);
  if ( liste.elemcount==0 && startpos==+1 && endpos==-1 )
    return(LCRZO_ERR_OKSEARCHNOTFOUND);
  lcrzo_er(lcrzo_priv_manageelemrange(liste.elemcount, startpos, endpos,
				      NULL, &infpospos, NULL, NULL,
				      &infdirection, &infdistance,
				      &supdirection, &supdistance));

  /*we go to the first element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(&liste, infdirection, infdistance, 0,
					 &ptravail));

  /*compare the elements in the range*/
  if (supdirection==LCRZO_PRIV_INCREASE)
  { for ( i=0 ; i<=supdistance; i++ )
    { if ( (*pfoncrecherche)(ptravail->pelem, pinfos) )
      { if (pposelem!=NULL) *pposelem=i+infpospos;
        if (pelem!=NULL) memcpy(pelem, ptravail->pelem, liste.sizeofelems);
        return(LCRZO_ERR_OK);
      }
      ptravail = (lcrzo_listelem *)ptravail->pnext;
    }
  }
  else
  { for ( i=0 ; i<=supdistance; i++ )
    { if ( (*pfoncrecherche)(ptravail->pelem, pinfos) )
      { if (pposelem!=NULL) *pposelem=infpospos-i;
        if (pelem!=NULL) memcpy(pelem, ptravail->pelem, liste.sizeofelems);
        return(LCRZO_ERR_OK);
      }
      ptravail = (lcrzo_listelem *)ptravail->pprecedent;
    }
  }

  return(LCRZO_ERR_OKSEARCHNOTFOUND);
}

/*---------------------------------------------------------------*/
int lcrzo_list_search_all(lcrzo_list liste,
			  int (*pfoncrecherche)(const void *pelem,
						const void *pinfos),
			  const void *pinfos,
			  lcrzo_int32 *pposelem,
			  void *pelem)
{ lcrzo_er(lcrzo_list_search_range(liste, 1, -1, pfoncrecherche, pinfos,
				   pposelem, pelem));
  return(LCRZO_ERR_OK);
}


/*---------------------------------------------------------------*/
int lcrzo_list_loop_range(lcrzo_list liste, 
			  lcrzo_int32 startpos,
			  lcrzo_int32 endpos,
			  int (*pfonc)(const void *pelem,
				       const void *pinfos),
			  const void *pinfos)
{ lcrzo_int32 infdistance, supdistance, i;
  int infdirection, supdirection;
  lcrzo_listelem *ptravail;

  /*verification des parametres et affectation des positions*/
  if (pfonc==NULL) return(LCRZO_ERR_PANULLPTR);
  if ( liste.elemcount==0 && startpos==+1 && endpos==-1 )
    return(LCRZO_ERR_OK);
  lcrzo_er(lcrzo_priv_manageelemrange(liste.elemcount, startpos, endpos,
				      NULL, NULL, NULL, NULL,
				      &infdirection, &infdistance,
				      &supdirection, &supdistance));

  /*we go to the first element*/
  lcrzo_er(lcrzo_priv_list_goto_listelem(&liste, infdirection, infdistance, 0,
					 &ptravail));

  /*compare the elements in the range*/
  if (supdirection==LCRZO_PRIV_INCREASE)
  { for ( i=0 ; i<=supdistance; i++ )
    { lcrzo_er(((*pfonc)(ptravail->pelem, pinfos)));
      ptravail = (lcrzo_listelem *)ptravail->pnext;
    }
  }
  else
  { for ( i=0 ; i<=supdistance; i++ )
    { lcrzo_er(((*pfonc)(ptravail->pelem, pinfos)));
      ptravail = (lcrzo_listelem *)ptravail->pprecedent;
    }
  }

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_list_loop_all(lcrzo_list liste,
			int (*pfonc)(const void *pelem,
				      const void *pinfos),
			const void *pinfos)
{ return(lcrzo_list_loop_range(liste, 1, -1, pfonc, pinfos));
}


/*---------------------------------------------------------------*/
/*---------------------------------------------------------------*/
void lcrzo_priv_listelem_data_free(void *pelem);
void lcrzo_priv_listelem_data_free(void *pelem)
{ lcrzo_listelem_data *plep;
  plep=(lcrzo_listelem_data *)pelem;
  lcrzo_data_free(plep->pdata);
}

/*---------------------------------------------------------------*/
int lcrzo_listelem_data_init(lcrzo_list *plist)
{ return(lcrzo_list_init(plist, sizeof(lcrzo_listelem_data),
			 &lcrzo_priv_listelem_data_free));
}

/*---------------------------------------------------------------*/
int lcrzo_listelem_data_add_pos(lcrzo_list *plist,
                                lcrzo_int32 pos,
                                lcrzo_uint32 datatype,
                                lcrzo_constdata data,
                                lcrzo_int32 datasize)
{ lcrzo_listelem_data lep;

  /*parameters verification*/
  if (plist==NULL) return(LCRZO_ERR_PANULLPTR);
  if (data==NULL) return(LCRZO_ERR_PANULLPTR);

  /*create the data*/
  lep.datatype=datatype;
  lep.datasize=datasize;
  lcrzo_er(lcrzo_data_alloc(datasize, &(lep.pdata)));
  memcpy(lep.pdata, data, datasize);

  /*add the data*/
  lcrzo_er(lcrzo_list_add_pos(plist, pos, &lep));

  return(LCRZO_ERR_OK);
}

/*---------------------------------------------------------------*/
int lcrzo_listelem_data_value_pos(lcrzo_list list,
                                  lcrzo_int32 pos,
				  lcrzo_uint32 *pdatatype,
                                  lcrzo_int32 datamaxsize,
                                  lcrzo_data data,
                                  lcrzo_int32 *pdatasize)
{ lcrzo_listelem_data lep;

  /*get the value*/
  lcrzo_er(lcrzo_list_value_pos(list, pos, &lep));

  if (pdatatype!=NULL) *pdatatype=lep.datatype;
  return(lcrzo_data_init_data(lep.pdata, lep.datasize,
			      datamaxsize, data, pdatasize));
}

/*---------------------------------------------------------------*/
int lcrzo_listelem_data_valuem_pos(lcrzo_list list,
                                   lcrzo_int32 pos,
                                   lcrzo_uint32 *pdatatype,
                                   lcrzo_data *pdata,
                                   lcrzo_int32 *pdatasize)
{ lcrzo_listelem_data lep;

  /*get the value*/
  lcrzo_er(lcrzo_list_value_pos(list, pos, &lep));

  if (pdatatype!=NULL) *pdatatype=lep.datatype;
  return(lcrzo_data_initm_data(lep.pdata, lep.datasize, pdata, pdatasize));
}

