/*us    Laurent Constantin's network library (lcrzo)
                 lcrzo_list module

  Functions herein allow to work on generic lists.
*/
/*fr    Bibliotheque reseau de Laurent Constantin (lcrzo)
                 Module lcrzo_list

  Les fonctions presentes dans ce module permettent de traiter des
listes generiques.
*/

/*---------------------------------------------------------------*/
/*us the definition of a lcrzo_list has to be public
   (because the compiler needs to know its size), 
   but you should not use its fields directly. */
/*fr la definition d'un lcrzo_list doit etre publique
   (car le compilateur doit connaitre sa taille), mais vous
   ne devriez pas utiliser directement ses champs. */
typedef struct lcrzo_listelem lcrzo_listelem;
struct lcrzo_listelem {
  lcrzo_listelem *pnext;
  lcrzo_listelem *pprecedent;
  void *pelem;
};
typedef struct {
  lcrzo_listelem *pnext;
  lcrzo_listelem *pprecedent;
  lcrzo_uint32 sizeofelems;
  lcrzo_int32 elemcount;
  void (*pfunc_erase)(void *pelem);
} lcrzo_list;

/*---------------------------------------------------------------*/
/*US************************************************************
 * Note about lcrzo_list :                                     *
 * A lcrzo_list is a double direction list. For example, a     *
 * containing 3 elements looks like this :                     *
 *                                                             *
 * .________________________________________________________.  *
 * | ,----------------------------------------------------. |  *
 * | |  _________   ___ +1 ___   ___ +2 ___   ___ +3 ___  | |  * 
 * | `->| lcrzo |-->|lcrzo_li|-->|lcrzo_li|-->|lcrzo_li|--' |  *
 * |____| _list |<--| stelem |<--| stelem |<--| stelem |<---'  *
 *      `-------'   `-- -3 --'   `-- -2 --'   `-- -1 --'       *
 *                       |            |            |           *
 *                       |            |            |           *
 *                       V            V            V           *
 *                     elem1        elem2        elem3         *
 *                                                             *
 * Elements can be accessed using positive position or         *
 * negative position. For example, the element +1 is the same  *
 * than the element -3.                                        *
 *                                                             *
 * Elements which are put in the list (elem1, elem2, etc.) are *
 * COPIES of user's values. So, if the user want to stock :    *
 *  - integers, a copy on sizeof(int) bytes is put in the list *
 *  - structures containing static objects, a copy on          *
 *    sizeof(the_struct) bytes is put in the list              *
 *  - structures containing dynamic objects (pointers), a copy *
 *    of sizeof(the_struct) bytes is put in the list, BUT the  *
 *    pointed items are not put because they are out of the    *
 *    structure                                                *
 * So, to be able to remove object of the above third          *
 * category, user has to give an erase function.               *
 *                                                             *
 *      Element to stock            Need an erase function ?   *
 *	      int                             no               *
 *	      char                            no               *
 *    struct {int a, char b}                  no               *
 *    int* (integer poitner)           yes (need 1 free)       *
 *    struct {int *a, int *b}          yes (need 2 free)       *
 *            etc.                                             *
 *                                                             *
 * This technology is easy to use, except for the third        *  
 * category.                                                   *
 ***************************************************************/
/*FR************************************************************
 * Note sur les lcrzo_list :                                   *
 * Une liste de type lcrzo_list est une liste a double sens de * 
 * parcours. Par exemple, une liste de 3 elements a la         *
 * structure suivante :                                        *
 *                                                             *
 * .________________________________________________________.  *
 * | ,----------------------------------------------------. |  *
 * | |  _________   ___ +1 ___   ___ +2 ___   ___ +3 ___  | |  * 
 * | `->| lcrzo |-->|lcrzo_li|-->|lcrzo_li|-->|lcrzo_li|--' |  *
 * |____| _list |<--| stelem |<--| stelem |<--| stelem |<---'  *
 *      `-------'   `-- -3 --'   `-- -2 --'   `-- -1 --'       *
 *                       |            |            |           *
 *                       |            |            |           *
 *                       V            V            V           *
 *                     elem1        elem2        elem3         *
 *                                                             *
 * Les elements peuvent etre accedes en utilisant un indice    *
 * positif ou un indice negatif. Par exemple, l'objet +1 est   *
 * le meme que l'objet -3.                                     *
 *                                                             *
 * Les elements mis dans la liste (elem1, elem2, etc.) sont    * 
 * des COPIES des elements passes par l'utilisateur. Cela      *
 * signifie que si l'utilisateur veut stocker :                *
 * - des entiers, une copie sur sizeof(int) octets est mise en *
 *   liste                                                     *
 * - des structures contenant des objets statiques, une copie  *
 *   sur sizeof(la_struct) octets est mise en liste            *
 * - des structures contenant des objets dynamiques (des       *
 *   pointeurs), une copie sur sizeof(la_struct) octets est    *
 *   mise en liste ; MAIS les objets pointes ne sont pas       *
 *   copies car ils sont hors de la structure.                 *
 * Afin de pouvoir effacer la troisieme categorie ci-dessus,   *
 * l'utilisateur est donc oblige de fournir une fonction       *
 * d'effacement.                                               *
 *                                                             *
 *     Element a stocker            Necessite une fonction     *
 *                                       d'effacement ?        *
 *	      int                             non              *
 *	      char                            non              *
 *    struct {int a, char b}                  non              *
 *   int* (pointeur sur un entier)    oui (besoin d'1 free)    *
 *    struct {int *a, int *b}         oui (besoin de 2 free)   *
 *            etc.                                             *
 *                                                             *
 * Cette technologie est simple d'utilisation, sauf dans le    *
 * troisieme cas ci-dessus qui necessite une fonction          *
 * d'effacement.                                               *
 ***************************************************************/

/*US*******************************************
 * Note about positions :                     *
 * Warning : 0 is an invalid value            *
 * Here are possible values for a position :  *
 *  +1 : first item                           *
 *  +2 : second item                          *
 *  ...                                       *
 * Moreover, we can count from the end :      *
 *  -1 : last item                            *
 *  -2 : last but one item                    *
 *  ...                                       *
 **********************************************/
/*FR*******************************************
 * Note sur les positions :                   *
 * Attention : 0 n'est pas une valeur valide  *
 * Voici des valeurs de position possibles :  *
 *  +1 : premier element                      *
 *  +2 : deuxieme element                     *
 *  ...                                       *
 * De plus, on peut partir de la fin :        *
 *  -1 : dernier element                      *
 *  -2 : avant dernier element                *
 *  ...                                       *
 **********************************************/

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_init
   Description :
     Initialize a lcrzo_list.
   Input parameter(s) :
     sizeofelems : size of elements which are to be put in the list
     *pfunc_erase : function needed to erase dynamic elements
                    This can be NULL
   Input/output parameter(s) :
   Output parameter(s) :
     *plist : lcrzo_list initialized
   Normal return values :
     LCRZO_ERR_OK : ok
   Examples :
    - list containing integers :
        lcrzo_list_init(&list, sizeof(int), NULL);
    - list containing structures :
        struct st{ int a; char b };
        lcrzo_list_init(&list, sizeof(struct st), NULL);
    - list containing structures with external pointers :
        struct st{ int a; char b; int *p};
        void erase(void *pelem) {free(pelem->p);}
        lcrzo_list_init(&list, sizeof(struct st), &erase);
*/
/*fr Nom : lcrzo_list_init
   Description :
     Initialise un lcrzo_list.
   Parametre(s) d'entree :
     sizeofelems : taille des elements qui seront mis dans la liste
     *pfunc_erase : fonction d'effacement des elements dynamiques
                    Peut valoir NULL
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *plist : lcrzo_list initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
   Exemples :
    - liste contenant des entiers :
        lcrzo_list_init(&liste, sizeof(int), NULL);
    - liste contenant des structures :
        struct st{ int a; char b };
        lcrzo_list_init(&liste, sizeof(struct st), NULL);
    - liste contenant des structures avec des pointeurs externes :
        struct st{ int a; char b; int *p};
        void efface(void *pelem) {free(pelem->p);}
        lcrzo_list_init(&liste, sizeof(struct st), &efface);
*/
int lcrzo_list_init(lcrzo_list *plist,
		    lcrzo_uint32 sizeofelems,
		    void (*pfunc_erase)(void *pelem));

/*us Name : lcrzo_list_close
   Description :
     Destroy a lcrzo_list.
   Input parameter(s) :
   Input/output parameter(s) :
     *plist : lcrzo_list to destroy
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_close
   Description :
     Detruit un lcrzo_list.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list a detruire
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_close(lcrzo_list *plist);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_add_pos
   Description :
     Add an element to the lcrzo_list.
   Input parameter(s) :
     position : position where the element will be added
     *pelem : element to add
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_add_pos
   Description :
     Ajoute un element a un lcrzo_list.
   Parametre(s) d'entree :
     position : position ou l'element sera ajoute
     *pelem : element a ajouter
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_add_pos(lcrzo_list *plist,
		       lcrzo_int32 position,
		       const void *pelem);

/*us Name : lcrzo_list_add_first
   Description :
     Add an element to the first position of a lcrzo_list.
   Input parameter(s) :
     *pelem : element to add
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_add_first
   Description :
     Ajoute un element a la premiere place d'un lcrzo_list.
   Parametre(s) d'entree :
     *pelem : element a ajouter
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_add_first(lcrzo_list *plist,
			 const void *pelem);

/*us Name : lcrzo_list_add_last
   Description :
     Add an element to the last position of a lcrzo_list.
   Input parameter(s) :
     *pelem : element to add
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_add_last
   Description :
     Ajoute un element a la derniere place d'un lcrzo_list.
   Parametre(s) d'entree :
     *pelem : element a ajouter
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_add_last(lcrzo_list *plist,
			const void *pelem);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_remove_pos
   Description :
     Remove an element from the lcrzo_list.
   Input parameter(s) :
     position : position of the removed element
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_pos
   Description :
     Enleve un element d'un lcrzo_list.
   Parametre(s) d'entree :
     position : position de l'element a enlever
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_pos(lcrzo_list *plist,
			  lcrzo_int32 position);

/*us Name : lcrzo_list_remove_first
   Description :
     Remove the first element from the lcrzo_list.
   Input parameter(s) :
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_first
   Description :
     Enleve le premier element d'un lcrzo_list.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_first(lcrzo_list *plist);

/*us Name : lcrzo_list_remove_last
   Description :
     Remove the last element from the lcrzo_list.
   Input parameter(s) :
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_last
   Description :
     Enleve le dernier element d'un lcrzo_list.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_last(lcrzo_list *plist);

/*us Name : lcrzo_list_remove_range
   Description :
     Remove a range of elements from the lcrzo_list.
   Input parameter(s) :
     startpos : position of the first element
     endpos : position of the last element
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_range
   Description :
     Enleve un intervalle d'elements d'un lcrzo_list.
   Parametre(s) d'entree :
     startpos : position du premier element
     endpos : position du dernier element
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_range(lcrzo_list *plist, 
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos);

/*us Name : lcrzo_list_remove_all
   Description :
     Remove all the elements from the lcrzo_list.
   Input parameter(s) :
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_all
   Description :
     Enleve tous les elements d'un lcrzo_list.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_all(lcrzo_list *plist);

/*us Name : lcrzo_list_remove_duplicate_range
   Description :
     Remove duplicate elements from a range of the lcrzo_list.
   Input parameter(s) :
     startpos : position of the first element
     endpos : position of the last element
     pfunc_compare : memory address of the function which will
                     be called for each element in the range 
                     [startpos->endpos]. This function compares
		     *peleminf and *pelemsup :
                      - if eleminf<elemsup, returns -1
                      - if eleminf>elemsup, returns +1
                      - if eleminf==elemsup, returns 0
		     For each call, the third parameter ('pinfos')
		     will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_compare.
	      This may be used to send information to *pfunc_compare.
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_duplicate_range
   Description :
     Enleve les doublons situes dans un intervalle d'un lcrzo_list.
   Parametre(s) d'entree :
     startpos : position du premier element
     endpos : position du dernier element
     pfunc_compare : adresse memoire de la fonction appelee pour chaque
                     element de l'intervalle [startpos->endpos]. Cette
		     fonction compare *peleminf et *pelemsup :
		      - si eleminf<elemsup alors retourne -1
		      - si eleminf>elemsup alors retourne +1
		      - si eleminf==elemsup alors retourne 0
		     Lors de chaque appel, le troisieme
		     parametre ('pinfos') est affecte avec la valeur 
		     de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_compare. Il peut 
              servir a envoyer des informations a pfunc_compare.
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_duplicate_range(lcrzo_list *plist,
				      lcrzo_int32 startpos,
				      lcrzo_int32 endpos,
				      int (*pfunc_compare)(const void*peleminf,
							   const void*pelemsup,
							   const void *pinfos),
				      const void *pinfos);

/*us Name : lcrzo_list_remove_duplicate_all
   Description :
     Remove duplicate elements of the lcrzo_list.
   Input parameter(s) :
     pfunc_compare : memory address of the function which will
                     be called for each element.
		     This function compares
		     *peleminf and *pelemsup :
                      - if eleminf<elemsup, returns -1
                      - if eleminf>elemsup, returns +1
                      - if eleminf==elemsup, returns 0
		     For each call, the third parameter ('pinfos')
		     will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_compare.
	      This may be used to send information to *pfunc_compare.
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_duplicate_all
   Description :
     Enleve les doublons d'un lcrzo_list.
   Parametre(s) d'entree :
     pfunc_compare : adresse memoire de la fonction appelee pour
                     chaque element. Cette
		     fonction compare *peleminf et *pelemsup :
		      - si eleminf<elemsup alors retourne -1
		      - si eleminf>elemsup alors retourne +1
		      - si eleminf==elemsup alors retourne 0
		     Lors de chaque appel, le troisieme
		     parametre ('pinfos') est affecte avec la valeur 
		     de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_compare. Il peut 
              servir a envoyer des informations a pfunc_compare.
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_duplicate_all(lcrzo_list *plist,
				    int (*pfunc_compare)(const void *peleminf,
							 const void *pelemsup,
							 const void *pinfos),
				    const void *pinfos);

/*us Name : lcrzo_list_remove_criteria_range
   Description :
     Remove elements matching a criteria, and in a range of the lcrzo_list.
   Input parameter(s) :
     startpos : position of the first element
     endpos : position of the last element
     pfunc_criteria : memory address of the function which will
                      be called for each element in the range 
                      [startpos->endpos]. This function checks
		      *pelem :
		       - if *pelem must be removed, returns 1
		       - else returns 0
  		      For each call, the second parameter ('pinfos')
		      will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_criteria.
	      This may be used to send information to *pfunc_criteria.
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_criteria_range
   Description :
     Enleve les elements verifiant un critere, et situes 
     dans un intervalle d'un lcrzo_list.
   Parametre(s) d'entree :
     startpos : position du premier element
     endpos : position du dernier element
     pfunc_criteria : adresse memoire de la fonction appelee
                      pour chaque element de l'intervalle
		      [startpos->endpos]. Cette fonction verifie
		      *pelem :
		      - si *pelem doit etre efface, retourne 1
		      - sinon retourne 0
		      Lors de chaque appel, le deuxieme
		      parametre ('pinfos') est affecte avec la valeur 
		      de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_criteria. Il peut 
              servir a envoyer des informations a pfunc_criteria.
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_criteria_range(lcrzo_list *plist,
				     lcrzo_int32 startpos,
				     lcrzo_int32 endpos,
				     int (*pfunc_criteria)(const void *pelem,
							   const void *pinfos),
				     const void *pinfos);

/*us Name : lcrzo_list_remove_criteria_all
   Description :
     Remove elements matching a criteria.
   Input parameter(s) :
     pfunc_criteria : memory address of the function which will
                      be called for each element. This function checks
		      *pelem :
		       - if *pelem must be removed, returns 1
		       - else returns 0
  		      For each call, the second parameter ('pinfos')
		      will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_criteria.
	      This may be used to send information to *pfunc_criteria.
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_remove_criteria_all
   Description :
     Enleve les elements verifiant un critere.
   Parametre(s) d'entree :
     pfunc_criteria : adresse memoire de la fonction appelee
                      pour chaque element. Cette fonction verifie
		      *pelem :
		      - si *pelem doit etre efface, retourne 1
		      - sinon retourne 0
		      Lors de chaque appel, le deuxieme
		      parametre ('pinfos') est affecte avec la valeur 
		      de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_criteria. Il peut 
              servir a envoyer des informations a pfunc_criteria.
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_remove_criteria_all(lcrzo_list *plist,
				   int (*pfunc_criteria)(const void *pelem,
							 const void *pinfos),
				   const void *pinfos);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_replace_pos
   Description :
     Replace an element in the lcrzo_list.
   Input parameter(s) :
     position : position of the replaced element
     *pelem : element which will be put in position
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_replace_pos
   Description :
     Remplace un element d'un lcrzo_list.
   Parametre(s) d'entree :
     position : position de l'element a remplacer
     *pelem : element a mettre a la position 'position'
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_replace_pos(lcrzo_list *plist,
			   lcrzo_int32 position,
			   const void *pelem);

/*us Name : lcrzo_list_replace_first
   Description :
     Replace the first element in the lcrzo_list.
   Input parameter(s) :
     *pelem : element which will be put in first place
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_replace_first
   Description :
     Remplace le premier element d'un lcrzo_list.
   Parametre(s) d'entree :
     *pelem : element a mettre en premier
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_replace_first(lcrzo_list *plist,
			     const void *pelem);

/*us Name : lcrzo_list_replace_last
   Description :
     Replace the last element in the lcrzo_list.
   Input parameter(s) :
     *pelem : element which will be put in first place
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_replace_last
   Description :
     Remplace le dernier element d'un lcrzo_list.
   Parametre(s) d'entree :
     *pelem : element a mettre en premier
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_replace_last(lcrzo_list *plist,
			    const void *pelem);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_replace_pos
   Description :
     Switch two elements in the lcrzo_list.
   Input parameter(s) :
     position1 : position of the first element
     position2 : position of the second element
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_replace_pos
   Description :
     Permutte deux elements d'un lcrzo_list.
   Parametre(s) d'entree :
     position1 : position du premier element
     position2 : position du deuxieme element
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_switch(lcrzo_list *plist,
		      lcrzo_int32 position1, 
		      lcrzo_int32 position2);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_invert_range
   Description :
     Invert the positions of elements in a given range
   Input parameter(s) :
     startpos : position of the first element
     endpos : position of the last element
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_invert_range
   Description :
     Inverse les elements situes dans un intervalle
   Parametre(s) d'entree :
     startpos : position du premier element
     endpos : position du dernier element
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_invert_range(lcrzo_list *plist,
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos);

/*us Name : lcrzo_list_invert_all
   Description :
     Invert the positions of elements
   Input parameter(s) :
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_invert_all
   Description :
     Inverse tous les elements
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_invert_all(lcrzo_list *plist);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_value_pos
   Description :
     Get the value of an element
   Input parameter(s) :
     list : lcrzo_list where to work
     position : position of the wanted element
   Input/output parameter(s) :
   Output parameter(s) :
     *pelem : receives a copy of the element
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_value_pos
   Description :
     Obtient la valeur d'un element
   Parametre(s) d'entree :
     list : lcrzo_list de travail
     position : position de l'element voulu
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pelem : recoit une copie de l'element
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_value_pos(lcrzo_list list,
			 lcrzo_int32 position,
			 void *pelem);

/*us Name : lcrzo_list_value_first
   Description :
     Get the value of the first element
   Input parameter(s) :
     list : lcrzo_list where to work
   Input/output parameter(s) :
   Output parameter(s) :
     *pelem : receives a copy of the element
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_value_first
   Description :
     Obtient la valeur du premier element
   Parametre(s) d'entree :
     list : lcrzo_list de travail
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pelem : recoit une copie de l'element
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_value_first(lcrzo_list list,
			   void *pelem);

/*us Name : lcrzo_list_value_last
   Description :
     Get the value of the last element
   Input parameter(s) :
     list : lcrzo_list where to work
   Input/output parameter(s) :
   Output parameter(s) :
     *pelem : receives a copy of the element
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_value_last
   Description :
     Obtient la valeur du dernier element
   Parametre(s) d'entree :
     list : lcrzo_list de travail
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pelem : recoit une copie de l'element
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_value_last(lcrzo_list list, 
			  void *pelem);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_count
   Description :
     Count the number of elements
   Input parameter(s) :
     list : lcrzo_list where to work
   Input/output parameter(s) :
   Output parameter(s) :
     *pnumberofelem : number of elements in list
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_count
   Description :
     Compte le nombre d'elements
   Parametre(s) d'entree :
     list : lcrzo_list de travail
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pnumberofelem : nombre d'elements dans list
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_count(lcrzo_list list,
		     lcrzo_int32 *pnumberofelem);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_sort_range
   Description :
     Sort elements from a range of the lcrzo_list.
   Input parameter(s) :
     startpos : position of the first element
     endpos : position of the last element
     pfunc_compare : memory address of the function which will
                     be called for each element in the range 
                     [startpos->endpos]. This function compares
                     *peleminf and *pelemsup :
                      - if eleminf<elemsup, returns -1
                      - if eleminf>elemsup, returns +1
                      - if eleminf==elemsup, returns 0
                     For each call, the third parameter ('pinfos')
                     will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_compare.
              This may be used to send information to *pfunc_compare.
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_sort_range
   Description :
     Trie les elements situes dans un intervalle d'un lcrzo_list.
   Parametre(s) d'entree :
     startpos : position du premier element
     endpos : position du dernier element
     pfunc_compare : adresse memoire de la fonction appelee pour chaque
                     element de l'intervalle [startpos->endpos]. Cette
                     fonction compare *peleminf et *pelemsup :
                      - si eleminf<elemsup alors retourne -1
                      - si eleminf>elemsup alors retourne +1
                      - si eleminf==elemsup alors retourne 0
                     Lors de chaque appel, le troisieme
                     parametre ('pinfos') est affecte avec la valeur 
                     de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_compare. Il peut 
              servir a envoyer des informations a pfunc_compare.
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_sort_range(lcrzo_list *plist,
			  lcrzo_int32 startpos,
			  lcrzo_int32 endpos,
			  int (*pfunc_compare)(const void *peleminf,
					       const void *pelemsup,
					       const void *pinfos),
			  const void *pinfos);

/*us Name : lcrzo_list_sort_all
   Description :
     Sort elements of the lcrzo_list.
   Input parameter(s) :
     pfunc_compare : memory address of the function which will
                     be called for each element. This function compares
                     *peleminf and *pelemsup :
                      - if eleminf<elemsup, returns -1
                      - if eleminf>elemsup, returns +1
                      - if eleminf==elemsup, returns 0
                     For each call, the third parameter ('pinfos')
                     will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_compare.
              This may be used to send information to *pfunc_compare.
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_sort_all
   Description :
     Trie les elements d'un lcrzo_list.
   Parametre(s) d'entree :
     pfunc_compare : adresse memoire de la fonction appelee pour chaque
                     element. Cette
                     fonction compare *peleminf et *pelemsup :
                      - si eleminf<elemsup alors retourne -1
                      - si eleminf>elemsup alors retourne +1
                      - si eleminf==elemsup alors retourne 0
                     Lors de chaque appel, le troisieme
                     parametre ('pinfos') est affecte avec la valeur 
                     de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_compare. Il peut 
              servir a envoyer des informations a pfunc_compare.
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_sort_all(lcrzo_list *plist,
			int (*pfunc_compare)(const void *peleminf,
					     const void *pelemsup,
					     const void *pinfos),
			const void *pinfos);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_search_range
   Description :
     Search elements matching a criteria, and in a range of the lcrzo_list.
   Input parameter(s) :
     list : lcrzo_list where to work
     startpos : position of the first element
     endpos : position of the last element
     pfunc_search : memory address of the function which will
                    be called for each element in the range 
                    [startpos->endpos]. This function checks
                    *pelem :
                     - if *pelem is like we want, returns 1
		     - else returns 0
		    For each call, the second parameter ('pinfos')
		    will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_search.
              This may be used to send information to *pfunc_search.
   Input/output parameter(s) :
   Output parameter(s) :
     *pposelem : position of the found element
     *pelem : copy of the found elem
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_search_range
   Description :
     Cherche les elements verifiant un critere, et situes 
     dans un intervalle d'un lcrzo_list.
   Parametre(s) d'entree :
     list : lcrzo_list de travail
     startpos : position du premier element
     endpos : position du dernier element
     pfunc_search : adresse memoire de la fonction appelee
                    pour chaque element de l'intervalle
                    [startpos->endpos]. Cette fonction verifie
		    *pelem :
                     - si *pelem correspond a ce que l'on
		       cherche, retourne 1
		     - sinon retourne 0
		    Lors de chaque appel, le deuxieme
		    parametre ('pinfos') est affecte avec la valeur 
		    de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_search. Il peut 
              servir a envoyer des informations a pfunc_search.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_search_range(lcrzo_list list,
			    lcrzo_int32 startpos,
			    lcrzo_int32 endpos,
			    int (*pfunc_search)(const void *pelem,
						const void *pinfos),
			    const void *pinfos,
			    lcrzo_int32 *pposelem,
			    void *pelem);

/*us Name : lcrzo_list_search_all
   Description :
     Search elements matching a criteria.
   Input parameter(s) :
     list : lcrzo_list where to work
     pfunc_search : memory address of the function which will
                    be called for each element. This function checks
                    *pelem :
                     - if *pelem is like we want, returns 1
		     - else returns 0
		    For each call, the second parameter ('pinfos')
		    will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc_search.
              This may be used to send information to *pfunc_search.
   Input/output parameter(s) :
   Output parameter(s) :
     *pposelem : position of the found element
     *pelem : copy of the found elem
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_search_all
   Description :
     Cherche les elements verifiant un critere.
   Parametre(s) d'entree :
     list : lcrzo_list de travail
     pfunc_search : adresse memoire de la fonction appelee
                    pour chaque element. Cette fonction verifie
		    *pelem :
                     - si *pelem correspond a ce que l'on
		       cherche, retourne 1
		     - sinon retourne 0
		    Lors de chaque appel, le deuxieme
		    parametre ('pinfos') est affecte avec la valeur 
		    de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc_search. Il peut 
              servir a envoyer des informations a pfunc_search.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_search_all(lcrzo_list list,
			  int (*pfunc_search)(const void *pelem,
						const void *pinfos),
			  const void *pinfos,
			  lcrzo_int32 *pposelem,
			  void *pelem);

/*---------------------------------------------------------------*/
/*us Name : lcrzo_list_loop_range
   Description :
     Call a function for each element in a
     given position range.
   Input parameter(s) :
     startpos : start position for the loop (cf. the above frame to
                know possible values)
     endpos : end position for the loop (cf. the above frame to
              know possible values)
     pfunc : memory address of the function which will be called
             for each element in the range [startpos->endpos]. For
             each call, the first parameter ('pelem') will be set
             with the element, and the second parameter ('pinfos')
             will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_loop_range
   Description :
     Appelle une fonction pour les elements
     situe dans un intervalle donne.
   Parametre(s) d'entree :
     startpos : position de debut de la boucle (cf. le cadre
                ci-dessus pour connaitre les valeurs valides)
     endpos : position de fin de la boucle (cf. le cadre
              ci-dessus pour connaitre les valeurs valides)
     pfunc : adresse memoire de la fonction appelee pour chaque
             element de l'intervalle [startpos->endpos]. Lors de
             chaque appel, le premier parametre ('pelem') est 
             affecte avec l'element courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_loop_range(lcrzo_list list, 
			  lcrzo_int32 startpos,
			  lcrzo_int32 endpos,
			  int (*pfunc)(const void *pelem,
				       const void *pinfos),
			  const void *pinfos);

/*us Name : lcrzo_list_loop_all
   Description :
     Call a function for each element.
   Input parameter(s) :
     pfunc : memory address of the function which will be called
             for each element. For
             each call, the first parameter ('pelem') will be set
             with the element, and the second parameter ('pinfos')
             will be set with the optional parameter below.
     pinfos : optional parameter (can be NULL) which will be
              used as the second parameter for *pfunc. This may be
              used to send information to *pfunc.
   Input/output parameter(s) :
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_loop_all
   Description :
     Appelle une fonction pour tous les elements.
   Parametre(s) d'entree :
     pfunc : adresse memoire de la fonction appelee pour chaque
             element. Lors de
             chaque appel, le premier parametre ('pelem') est 
             affecte avec l'element courant, et le deuxieme
             parametre ('pinfos') est affecte avec la valeur 
             de pinfos ci-dessous.
     pinfos : parametre optionnel (peut mettre NULL) qui sera
              passe en deuxieme parametre de pfunc. Il peut 
              servir a envoyer des informations a pfunc.
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_list_loop_all(lcrzo_list list,
			int (*pfunc)(const void *pelem,
				     const void *pinfos),
			const void *pinfos);


/*---------------------------------------------------------------*/
/*US************************************************************
 * Note about lcrzo_listelem_data_xxx :                        *
 * The folowing four functions allow to put variable sized     *
 * data in a lcrzo_list. They supersedes :                     *
 *   lcrzo_list_init                                           *
 *   lcrzo_list_add_pos                                        *
 *   lcrzo_list_value_pos                                      *
 ***************************************************************/
/*FR************************************************************
 * Note sur lcrzo_listelem_data_xxx :                          *
 * Les 4 fonctions suivantes permettent de stocker des donnees *
 * dans un lcrzo_list. Elles sont a utiliser en remplacement   *
 * de :                                                        *
 *   lcrzo_list_init                                           *
 *   lcrzo_list_add_pos                                        *
 *   lcrzo_list_value_pos                                      *
 ***************************************************************/

/*---------------------------------------------------------------*/
/*us this dynamic structure will be stored in the list */
/*fr cette structure dynamique est celle stockee dans la liste */
typedef struct {
  lcrzo_uint32 datatype;
  lcrzo_uint32 datasize;
  lcrzo_data   pdata; /*us pointer to data *//*fr pointeur sur les donnees */
} lcrzo_listelem_data;

/*---------------------------------------------------------------*/
/*us Name : lcrzo_listelem_data_init
   Description :
     Initialize a lcrzo_list.
   Input parameter(s) :
   Input/output parameter(s) :
   Output parameter(s) :
     *plist : lcrzo_list initialized
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_listelem_data_init
   Description :
     Initialise un lcrzo_list.
   Parametre(s) d'entree :
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *plist : lcrzo_list initialise
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_listelem_data_init(lcrzo_list *plist);

/*us Name : lcrzo_listelem_data_add_pos
   Description :
     Add an element to the lcrzo_list.
   Input parameter(s) :
     pos : position where the element will be added
     datatype : type of data (allows you to put several data
                types in the same list). You can use the value
		you want
     data : data to put in the list
     datasize : size of data
   Input/output parameter(s) :
     *plist : lcrzo_list where to work
   Output parameter(s) :
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_listelem_data_add_pos
   Description :
     Ajoute un element a un lcrzo_list.
   Parametre(s) d'entree :
     pos : position ou l'element sera ajoute
     datatype : type de data (permet de mettre plusieurs types
                de donnees dans la meme liste). Vous pouvez 
		utiliser la valeur de votre choix.
     data : donnees a mettre dans la liste
     datasize : taille de data
   Parametre(s) d'entree-sortie :
     *plist : lcrzo_list de travail
   Parametre(s) de sortie :
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_listelem_data_add_pos(lcrzo_list *plist,
				lcrzo_int32 pos,
				lcrzo_uint32 datatype,
				lcrzo_constdata data,
				lcrzo_int32 datasize);

/*us Name : lcrzo_list_value_pos
   Description :
     Get the value of an element
   Input parameter(s) :
     list : lcrzo_list where to work
     pos : position of the wanted element
     datamaxsize : max size which can be stored in data
   Input/output parameter(s) :
   Output parameter(s) :
     *pdatatype : type of this data
     data : output array.
            This array will be set with the element
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : datamaxsize reached before
                                 being able to store all the needed
                                 data in data
*/
/*fr Nom : lcrzo_list_value_pos
   Description :
     Obtient la valeur d'un element
   Parametre(s) d'entree :
     list : lcrzo_list de travail
     pos : position de l'element voulu
     dataoutmaxsize : taille maximale qui peut etre stockee
                      dans data
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdatatype : type des donnees a cette position
     data : tableau de sortie.
            Ce tableau recevra une copie de l'element
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
     LCRZO_ERR_OKDATATRUNCATED : datamaxsize a ete atteint 
                                 avant d'avoir pu stocker les
                                 donnees dans data
*/
int lcrzo_listelem_data_value_pos(lcrzo_list list,
				  lcrzo_int32 pos,
				  lcrzo_uint32 *pdatatype,
				  lcrzo_int32 datamaxsize,
				  lcrzo_data data,
				  lcrzo_int32 *pdatasize);

/*us Name : lcrzo_list_valuem_pos
   Description :
     Get the value of an element
   Input parameter(s) :
     list : lcrzo_list where to work
     pos : position of the wanted element
   Input/output parameter(s) :
   Output parameter(s) :
     *pdatatype : type of this data
     *pdata : pointer which will be malloced (so, the
              memory will have to be freed by the
              user with 'free(*pdata)').
	      The allocated memory will be set with the element
     *pdatasize : reached size of data
   Normal return values :
     LCRZO_ERR_OK : ok
*/
/*fr Nom : lcrzo_list_valuem_pos
   Description :
     Obtient la valeur d'un element
   Parametre(s) d'entree :
     list : lcrzo_list de travail
     pos : position de l'element voulu
   Parametre(s) d'entree-sortie :
   Parametre(s) de sortie :
     *pdatatype : type des donnees a cette position
     *pdata : pointeur qui sera alloue (la memoire
              devra etre liberee par l'utilisateur 
	      avec 'free(*pdata)').
	      Cette memoire allouee recevra une copie de l'element
     *pdatasize : taille atteinte par data
   Valeurs de retour normales :
     LCRZO_ERR_OK : ok
*/
int lcrzo_listelem_data_valuem_pos(lcrzo_list list,
				   lcrzo_int32 pos,
				   lcrzo_uint32 *pdatatype,
				   lcrzo_data *pdata,
				   lcrzo_int32 *pdatasize);

